# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from JSONApiPlugin import OpenStackApiModels as Models
import UwsgiRequestContext
import Tac
import Tracing
import json

from UrlMap import urlHandler

traceHandle = Tracing.Handle( 'OpenStackUwsgiServer' )
log = traceHandle.trace0
warn = traceHandle.trace1
info = traceHandle.trace2
trace = traceHandle.trace3
debug = traceHandle.trace4

SwitchInterface = Tac.Type( 'VirtualNetwork::Client::SwitchInterface' )

class BaseHandler:
   @staticmethod
   def _getRegion( mounts, region ):
      r = mounts[ 'openStackConfig' ].region.get( region )
      if r is None:
         raise UwsgiRequestContext.HttpNotFound( 'Region %s not found' % region )
      return r

   @staticmethod
   def _checkSyncLock( rc, mounts, region ):
      if not region:
         info( "No region specified" )
         return

      rConfig = BaseHandler._getRegion( mounts, region )
      rConfigSync = rConfig.syncStatus
      rStatus = mounts[ 'openStackStatus' ].regionStatus.get( region )
      if not rStatus:
         # Region has not been created
         info( "No region status for %s" % region )
         raise UwsgiRequestContext.HttpNotFound( 'Region %s not found' % region )

      if rConfig.requestId == "":
         # No sync in progress
         info( "No sync in progress" )
         return

      # If the sync ID is set, this neutron instance is performing a sync.
      syncId = rc.request_.get( 'HTTP_X_SYNC_ID' )
      syncStatus = Tac.Type( 'OpenStack::SyncStatus' )
      rStatusSync = rStatus.syncStatus
      if syncId:
         # If there's no sync in progress, the sync timed out and the neutron
         # controller is not yet aware
         if rStatusSync != syncStatus.syncInProgress:
            raise UwsgiRequestContext.HttpBadRequest( 'Sync not in progress' )
         # If the sync ID matches the config's request ID it has the sync lock
         if syncId == rConfig.requestId:
            info( "Sync lock held by %s, continuing sync" % syncId )
            rConfig.syncHeartbeat = Tac.now()
            return

      if rConfigSync == syncStatus.syncInProgress:
         if rStatusSync != syncStatus.syncTimedout: # pylint: disable=no-else-raise
            raise UwsgiRequestContext.HttpBadRequest(
                  'Region %s is syncing' % region )
         else:
            info( "Last sync timed out. Resetting status." )
            rConfig.syncStatus = syncStatus.unknown

   @staticmethod
   def _getQueryRegions( rc ):
      return rc.getUrlQuery().get( 'regionName' )

   @staticmethod
   def _getQueryTenants( rc ):
      return rc.getUrlQuery().get( 'tenantId' )

   @staticmethod
   def _getQueryNetworks( rc ):
      return rc.getUrlQuery().get( 'networkId' )

   @staticmethod
   def _getQueryTenant( rc, region ):
      tenants = BaseHandler._getQueryTenants( rc )
      if tenants is None:
         return None
      if len( tenants ) > 1:
         raise UwsgiRequestContext.HttpBadRequest( "Only one tenantId can be"
                                                  "specified" )

      try:
         tenant = region.tenant[ tenants[ 0 ] ]
      except KeyError:
         # pylint: disable-next=raise-missing-from
         raise UwsgiRequestContext.HttpNotFound( 'Tenant %s not found' %
                                                tenants[ 0 ] )

      return tenant

   @staticmethod
   def _getVlanPool( mounts, region ):
      vlanPool = []
      if not region:
         return vlanPool
      rStatus = mounts[ 'openStackStatus' ].regionStatus.get( region )
      cliConfig = mounts[ 'openStackAgentConfig' ]
      if not rStatus:
         return vlanPool
      driverType = Tac.Type( "OpenStack::TypeDriver" )
      if cliConfig.vlanTypeDriver == driverType.aristaVlan:
         vlanPool = rStatus.vlanPool
      return vlanPool

class Agent( BaseHandler ):
   regex = '/openstack/api/agent[/]'

   @staticmethod
   @urlHandler( ( 'GET', ), regex )
   def handleGet( rc, mounts ):
      # Log last UUID check for  each neutron worker
      osConfig = mounts[ 'openStackConfig' ]
      osConfig.clientHeartbeat[ rc.getRequesterIp() ] = Tac.now()

      agentStatus = mounts[ 'openStackAgentStatus' ]
      model = Models.AgentModel()
      model.fromSysdb( agentStatus )
      # TODO - this should be handled in from tac type, but unfortunately
      # we don't have a way to access an entity that is way outside this TAC
      # model just now.
      model.isLeader = \
            mounts[ 'clusterStatus' ].status[ 'default' ].isStandaloneOrLeader
      return model

class Region( BaseHandler ):
   regex = '/openstack/api/region[/{region}][/]'

   @staticmethod
   @urlHandler( ( 'GET', ), regex )
   def handleGet( rc, mounts, region=None ):
      regions = mounts[ 'openStackConfig' ].region
      regs = [ region ] if region else list( regions )

      ret = []
      for name in regs:
         model = Models.RegionModel()
         rConfig = regions.get( name )
         rStatus = mounts[ 'openStackStatus' ].regionStatus.get( name )
         if rConfig:
            model.fromSysdb( rConfig, rStatus )
            ret.append( model )
         else:
            warn( "Region:handleGet region %s not found" % name )
      return ret

   @staticmethod
   @urlHandler( ( 'POST', ), regex )
   def handlePost( rc, mounts, region=None ):
      BaseHandler._checkSyncLock( rc, mounts, region )
      data = rc.getRequestContent()
      data = json.loads( data )
      regions = mounts[ 'openStackConfig' ].region
      log( data )
      for item in data:
         model = Models.RegionModel()
         model.populateModelFromJson( item )
         key = model.name
         region = regions.get( key )
         if not region:
            region = regions.newMember( key )
         model.toSysdb( region )
      return {}

   @staticmethod
   @urlHandler( ( 'PUT', 'PATCH' ), regex )
   def handlePut( rc, mounts, region=None ):
      BaseHandler._checkSyncLock( rc, mounts, region )
      data = rc.getRequestContent()
      data = json.loads( data )
      regions = mounts[ 'openStackConfig' ].region
      if region:
         r = regions.get( region )
         if not r:
            raise UwsgiRequestContext.HttpNotFound( 'Region not found' )

      for item in data:
         model = Models.RegionModel()
         model.populateModelFromJson( item )
         key = model.name
         BaseHandler._checkSyncLock( rc, mounts, key )
         region = regions.get( key, None )
         if region:
            model.toSysdb( region )
         else:
            warn( "Region:handlePut() region %s not found" % key )
      return {}

   @staticmethod
   @urlHandler( ( 'DELETE', ), regex )
   def handleDelete( rc, mounts, region=None ):
      regions = mounts[ 'openStackConfig' ].region
      if region:
         r = regions.get( region )
         if not r:
            warn( 'Region %s not found' % region )
         else:
            del regions[ region ]
      else:
         data = rc.getRequestContent()
         data = json.loads( data )
         for item in data:
            r = item.get( 'name' )
            if regions.get( r ):
               del regions[ r ]
            else:
               warn( "Region:handleDelete() region %s not found" % r )
      return {}

class Sync( BaseHandler ):
   regex = '/openstack/api/region/{region}/sync[/]'

   @staticmethod
   @urlHandler( ( 'GET', ), regex )
   def handleGet( rc, mounts, region ):
      r = BaseHandler._getRegion( mounts, region )
      model = Models.SyncModel()
      model.fromSysdb( r )
      return model

   @staticmethod
   @urlHandler( ( 'POST', ), regex )
   def handlePost( rc, mounts, region ):
      # Map the requestor IP to this region
      osConfig = mounts[ 'openStackConfig' ]
      osConfig.clientRegion[ rc.getRequesterIp() ] = region

      BaseHandler._checkSyncLock( rc, mounts, region )
      r = BaseHandler._getRegion( mounts, region )
      data = rc.getRequestContent()
      data = json.loads( data )
      model = Models.SyncModel()
      model.populateModelFromJson( data )
      model.toSysdb( r )
      return {}

   @staticmethod
   @urlHandler( ( 'DELETE', ), regex )
   def handleDelete( rc, mounts, region ):
      BaseHandler._checkSyncLock( rc, mounts, region )
      r = BaseHandler._getRegion( mounts, region )
      r.requester = ""
      r.requestId = ""
      syncStatus = Tac.Type( 'OpenStack::SyncStatus' )
      r.syncStatus = syncStatus.syncComplete
      return {}

class ServiceEndPoint( BaseHandler ):
   regex = '/openstack/api/region/{region}/service-end-point[/]'

   @staticmethod
   @urlHandler( ( 'GET', ), regex )
   def handleGet( rc, mounts, region ):
      r = BaseHandler._getRegion( mounts, region )
      eps = list( r.serviceEndPoint.values() )

      ret = []
      for ep in eps:
         model = Models.ServiceEndPointModel()
         model.fromSysdb( ep )
         ret.append( model )

      return ret

   @staticmethod
   @urlHandler( ( 'POST', ), regex )
   def handlePost( rc, mounts, region ):
      BaseHandler._checkSyncLock( rc, mounts, region )
      r = BaseHandler._getRegion( mounts, region )
      data = rc.getRequestContent()
      data = json.loads( data )
      for item in data:
         # BUG232325: This bug is opened to address the issue that the
         # service endpoint object should not be deleted when handling a POST
         # request. The Arista ML2 driver should handle configuring a service
         # endpoint properly and not relying on deleting existing endpoint object
         # in this handler.
         name = item.get( 'name' )
         if r.serviceEndPoint.get( name ):
            del r.serviceEndPoint[ name ]
         model = Models.ServiceEndPointModel()
         model.populateModelFromJson( item )
         model.toSysdb( r )

      return {}

   @staticmethod
   @urlHandler( ( 'DELETE', ), regex )
   def handleDelete( rc, mounts, region ):
      BaseHandler._checkSyncLock( rc, mounts, region )
      r = BaseHandler._getRegion( mounts, region )
      data = rc.getRequestContent()
      data = json.loads( data )
      for item in data:
         name = item.get( 'name' )
         if r.serviceEndPoint.get( name ):
            del r.serviceEndPoint[ name ]
         else:
            warn( "ServiceEndPoint:handleDelete endpoint %s not found" % name )

      return {}

class Tenant( BaseHandler ):
   regex = '/openstack/api/region/{region}/tenant[/]'

   @staticmethod
   @urlHandler( ( 'GET', ), regex )
   def handleGet( rc, mounts, region ):
      r = BaseHandler._getRegion( mounts, region )
      tenants = BaseHandler._getQueryTenants( rc )
      if not tenants:
         tenants = list( r.tenant )

      ret = []
      for t in tenants:
         ten = r.tenant.get( t, None )
         if ten:
            model = Models.TenantModel()
            model.fromSysdb( ten )
            ret.append( model )
         else:
            warn( "Tenant:handleGet() tenant %s not found" % t )
      return ret

   @staticmethod
   @urlHandler( ( 'POST', ), regex )
   def handlePost( rc, mounts, region ):
      BaseHandler._checkSyncLock( rc, mounts, region )
      r = BaseHandler._getRegion( mounts, region )
      data = rc.getRequestContent()
      data = json.loads( data )
      for item in data:
         model = Models.TenantModel()
         model.populateModelFromJson( item )
         key = model.tenantId
         tenant = r.tenant.get( key )
         if not tenant:
            tenant = r.newTenant( key )
         model.toSysdb( tenant )
      return {}

   @staticmethod
   @urlHandler( ( 'PUT', 'PATCH' ), regex )
   def handlePut( rc, mounts, region ):
      BaseHandler._checkSyncLock( rc, mounts, region )
      r = BaseHandler._getRegion( mounts, region )
      data = rc.getRequestContent()
      data = json.loads( data )

      for item in data:
         model = Models.TenantModel()
         model.populateModelFromJson( item )
         key = model.tenantId
         t = r.tenant.get( key, None )
         if t:
            model.toSysdb( t )
         else:
            warn( "Tenant:handlePut() tenant %s not found" % key )
      return {}

   @staticmethod
   @urlHandler( ( 'DELETE', ), regex )
   def handleDelete( rc, mounts, region ):
      BaseHandler._checkSyncLock( rc, mounts, region )
      r = BaseHandler._getRegion( mounts, region )
      data = rc.getRequestContent()
      data = json.loads( data )
      for item in data:
         t = item.get( 'id' )
         if r.tenant.get( t ):
            del r.tenant[ t ]
         else:
            warn( "Tenant:handlePut() tenant %s not found" % t )
      return {}

class Network( BaseHandler ):
   regex = '/openstack/api/region/{region}/network[/]'

   @staticmethod
   @urlHandler( ( 'GET', ), regex )
   def handleGet( rc, mounts, region ):
      r = BaseHandler._getRegion( mounts, region )
      r = mounts[ 'openStackConfig' ].region.get( region )
      if r is None:
         raise UwsgiRequestContext.HttpNotFound( 'Region %s not found' % region )

      tenants = BaseHandler._getQueryTenants( rc )
      ret = []
      for network in r.network.values():
         if ( tenants and network.tenant is not None and
              network.tenant.id not in tenants ):
            continue
         model = Models.NetworkModel()
         model.fromSysdb( network )
         ret.append( model )

      return ret

   @staticmethod
   @urlHandler( ( 'POST', ), regex )
   def handlePost( rc, mounts, region ):
      BaseHandler._checkSyncLock( rc, mounts, region )
      r = BaseHandler._getRegion( mounts, region )
      data = rc.getRequestContent()
      data = json.loads( data )
      invalidTenants = set()
      for item in data:
         model = Models.NetworkModel()
         model.populateModelFromJson( item )
         netId = model.networkId
         tenant = model.tenantId
         t = r.tenant.get( tenant )
         if t is None:
            warn( "Network:handlePost() tenant %s not found" % tenant )
            invalidTenants.add( tenant )
            continue
         network = r.network.get( netId )
         if not network:
            network = r.newNetwork( netId, t )
         model.toSysdb( network )
      if invalidTenants:
         invalidTenantMessage = ( "Unknown tenant id%s %s" %
                                  ( 's' if len( invalidTenants ) > 1 else '',
                                    ", ".join( str( t ) for t in invalidTenants ) ) )
         raise UwsgiRequestContext.HttpBadRequest( invalidTenantMessage )
      return {}

   @staticmethod
   @urlHandler( ( 'PUT', 'PATCH' ), regex )
   def handlePut( rc, mounts, region ):
      BaseHandler._checkSyncLock( rc, mounts, region )
      r = BaseHandler._getRegion( mounts, region )
      data = rc.getRequestContent()
      data = json.loads( data )

      for item in data:
         model = Models.NetworkModel()
         model.populateModelFromJson( item )
         key = model.networkId
         tenant = model.tenantId
         t = r.tenant.get( tenant )
         if t is None:
            warn( "Network:handlePut() tenant '%s' not found" % tenant )
            continue
         n = r.network.get( key )
         if n:
            model.toSysdb( n )
         else:
            warn( "Network:handlePut() network '%s' not found" % key )

      return {}

   @staticmethod
   @urlHandler( ( 'DELETE', ), regex )
   def handleDelete( rc, mounts, region ):
      BaseHandler._checkSyncLock( rc, mounts, region )
      r = BaseHandler._getRegion( mounts, region )
      data = rc.getRequestContent()
      data = json.loads( data )
      for item in data:
         net = item.get( 'id' )
         if r.network.get( net ):
            del r.network[ net ]
         else:
            warn( "Network:handleDelete network '{}' not found in {}".format( net,
                                                                        r.name ) )

      return {}

class Instance( BaseHandler ):

   @staticmethod
   def _getQueryVms( rc ):
      return rc.getUrlQuery().get( 'vmId' )

   @staticmethod
   def _getQueryInstances( rc ):
      return rc.getUrlQuery().get( 'instanceId' )

   @staticmethod
   def _getQueryType( rc ):
      return rc.getUrlQuery().get( 'type' )[ 0 ]

   @staticmethod
   def _getVmList( region, tenants, vList=None ):
      if vList:
         vms = []
         for vm in vList:
            try:
               vms.append( region.vmInstance[ vm ].id )
            except KeyError:
               warn( "Vm:_getVmList vm %s not found" % vm )
         return vms

      if tenants:
         vms = []
         for vm in region.vmInstance.values():
            if vm.tenant and vm.tenant.id in tenants:
               vms.append( vm.id )
         return vms

      return list( region.vmInstance )

   @staticmethod
   def _getDhcpList( region, tenants, vList=None ):
      if vList:
         dhcps = []
         for d in vList:
            try:
               dhcps.append( region.dhcpInstance[ d ].id )
            except KeyError:
               warn( "Dhcp:_getDhcpList vm %s not found" % d )
         return dhcps

      if tenants:
         dhcps = []
         for d in region.dhcpInstance.values():
            if d.tenant and d.tenant.id in tenants:
               dhcps.append( d.id )
         return dhcps

      return list( region.dhcpInstance )

   @staticmethod
   def _getBaremetalList( region, tenants, bmList=None ):
      if bmList:
         bms = []
         for bm in bmList:
            try:
               bms.append( region.baremetalInstance[ bm ].id )
            except KeyError:
               warn( "Baremetal:_getBaremetalList baremetal %s not found" % bm )
         return bms

      if tenants:
         bms = []
         for bm in region.baremetalInstance.values():
            if bm.tenant and bm.tenant.id in tenants:
               bms.append( bm.id )
         return bms

      return list( region.baremetalInstance )

   @staticmethod
   def _getRouterList( region, tenants, rList=None ):
      if rList:
         routers = []
         for router in rList:
            try:
               routers.append( region.routerInstance[ router ].id )
            except KeyError:
               warn( "Router:_getRouterList router %s not found" % router )
         return routers

      if tenants:
         routers = []
         for router in region.routerInstance.values():
            if router.tenant and router.tenant.id in tenants:
               routers.append( router.id )
         return routers

      return list( region.routerInstance )

   @staticmethod
   # pylint: disable-next=inconsistent-return-statements
   def _getInstanceList( region, tenants, iType, instanceList=None ):
      if iType == 'vm':
         return Instance._getVmList( region, tenants, instanceList )
      elif iType == 'baremetal':
         return Instance._getBaremetalList( region, tenants, instanceList )
      elif iType == 'dhcp':
         return Instance._getDhcpList( region, tenants, instanceList )
      elif iType == 'router':
         return Instance._getRouterList( region, tenants, instanceList )

   @staticmethod
   def _getModelType( iType ): # pylint: disable=inconsistent-return-statements
      if iType == 'vm':
         return Models.VmModel
      elif iType == 'baremetal':
         return Models.BaremetalModel
      elif iType == 'dhcp':
         return Models.DhcpModel
      elif iType == 'router':
         return Models.RouterModel

   @staticmethod
   # pylint: disable-next=inconsistent-return-statements
   def _getInstanceCollection( region, iType ):
      if iType == 'vm':
         return region.vmInstance
      elif iType == 'baremetal':
         return region.baremetalInstance
      elif iType == 'dhcp':
         return region.dhcpInstance
      elif iType == 'router':
         return region.routerInstance

   @staticmethod
   # pylint: disable-next=inconsistent-return-statements
   def _createInstance( region, instanceId, iType ):
      if iType == 'vm':
         return region.newVmInstance( instanceId )
      elif iType == 'baremetal':
         return region.newBaremetalInstance( instanceId )
      elif iType == 'dhcp':
         return region.newDhcpInstance( instanceId )
      elif iType == 'router':
         return region.newRouterInstance( instanceId )

   @staticmethod
   def doHandleGet( rc, mounts, region, iType ):
      r = BaseHandler._getRegion( mounts, region )
      tenants = BaseHandler._getQueryTenants( rc )
      instances = Instance._getInstanceList( r, tenants, iType,
                                             Instance._getQueryInstances( rc ) )
      ModelType = Instance._getModelType( iType )
      instanceCollection = Instance._getInstanceCollection( r, iType )
      ret = []
      for iId in instances:
         model = ModelType()
         if iId in instanceCollection:
            model.fromSysdb( instanceCollection[ iId ] )
            ret.append( model )
         else:
            warn( "Instance:handleGet instance %s not found" % iId )

      return ret

   @staticmethod
   def doHandleInstance( rc, mounts, region, iType, exists=False ):
      BaseHandler._checkSyncLock( rc, mounts, region )
      r = BaseHandler._getRegion( mounts, region )
      tenant = BaseHandler._getQueryTenant( rc, r )
      legacyRequest = False
      if tenant:
         legacyRequest = True
      ModelType = Instance._getModelType( iType )
      instanceCollection = Instance._getInstanceCollection( r, iType )
      data = rc.getRequestContent()
      data = json.loads( data )
      invalidTenants = set()
      for item in data:
         model = ModelType()
         model.populateModelFromJson( item )
         instanceId = model.instanceId
         if not legacyRequest:
            tenantId = model.tenantId
            try:
               tenant = r.tenant[ tenantId ]
            except KeyError:
               invalidTenants.add( tenantId )
               continue
         if exists:
            instance = instanceCollection.get( instanceId, None )
         else:
            instance = Instance._createInstance( r, instanceId, iType )
         if instance:
            instance.tenant = tenant
            model.toSysdb( instance )
      if invalidTenants:
         invalidTenantMessage = ( "Unknown tenant id%s %s" %
                                  ( 's' if len( invalidTenants ) > 1 else '',
                                    ", ".join( invalidTenants ) ) )
         raise UwsgiRequestContext.HttpBadRequest( invalidTenantMessage )
      return {}

   @staticmethod
   def doHandlePost( rc, mounts, region, iType ):
      Instance.doHandleInstance( rc, mounts, region, iType, exists=False )

   @staticmethod
   def doHandlePut( rc, mounts, region, iType ):
      Instance.doHandleInstance( rc, mounts, region, iType, exists=True )

   @staticmethod
   def doHandleDelete( rc, mounts, region, iType ):
      BaseHandler._checkSyncLock( rc, mounts, region )
      r = BaseHandler._getRegion( mounts, region )
      instanceCollection = Instance._getInstanceCollection( r, iType )
      data = rc.getRequestContent()
      data = json.loads( data )
      invalidRequests = []
      for item in data:
         try:
            instId = item[ 'id' ]
            if instId in instanceCollection:
               del instanceCollection[ instId ]
            else:
               warn( "Instance:handleDelete instance %s not found" % (
                      item.get( 'id' ) ) )
         except KeyError:
            invalidRequests.append( item )

      if invalidRequests:
         raise UwsgiRequestContext.HttpBadRequest( 'Invalid DELETE request %s' % (
                             ', '.join( str( req ) for req in invalidRequests ) ) )
      return {}

class Vm( Instance ):
   regex = '/openstack/api/region/{region}/vm[/]'

   @staticmethod
   @urlHandler( ( 'GET', ), regex )
   def handleGet( rc, mounts, region ):
      return Instance.doHandleGet( rc, mounts, region, 'vm' )

   @staticmethod
   @urlHandler( ( 'POST', ), regex )
   def handlePost( rc, mounts, region ):
      return Instance.doHandlePost( rc, mounts, region, 'vm' )

   @staticmethod
   @urlHandler( ( 'PUT', 'PATCH' ), regex )
   def handlePut( rc, mounts, region ):
      return Instance.doHandlePut( rc, mounts, region, 'vm' )

   @staticmethod
   @urlHandler( ( 'DELETE', ), regex )
   def handleDelete( rc, mounts, region ):
      return Instance.doHandleDelete( rc, mounts, region, 'vm' )

class Baremetal( Instance ):
   regex = '/openstack/api/region/{region}/baremetal[/]'

   @staticmethod
   @urlHandler( ( 'GET', ), regex )
   def handleGet( rc, mounts, region ):
      return Instance.doHandleGet( rc, mounts, region, 'baremetal' )

   @staticmethod
   @urlHandler( ( 'POST', ), regex )
   def handlePost( rc, mounts, region ):
      return Instance.doHandlePost( rc, mounts, region, 'baremetal' )

   @staticmethod
   @urlHandler( ( 'PUT', 'PATCH' ), regex )
   def handlePut( rc, mounts, region ):
      return Instance.doHandlePut( rc, mounts, region, 'baremetal' )

   @staticmethod
   @urlHandler( ( 'DELETE', ), regex )
   def handleDelete( rc, mounts, region ):
      return Instance.doHandleDelete( rc, mounts, region, 'baremetal' )

class Dhcp( Instance ):
   regex = '/openstack/api/region/{region}/dhcp[/]'

   @staticmethod
   @urlHandler( ( 'GET', ), regex )
   def handleGet( rc, mounts, region ):
      return Instance.doHandleGet( rc, mounts, region, 'dhcp' )

   @staticmethod
   @urlHandler( ( 'POST', ), regex )
   def handlePost( rc, mounts, region ):
      return Instance.doHandlePost( rc, mounts, region, 'dhcp' )

   @staticmethod
   @urlHandler( ( 'PUT', 'PATCH' ), regex )
   def handlePut( rc, mounts, region ):
      return Instance.doHandlePut( rc, mounts, region, 'dhcp' )

   @staticmethod
   @urlHandler( ( 'DELETE', ), regex )
   def handleDelete( rc, mounts, region ):
      return Instance.doHandleDelete( rc, mounts, region, 'dhcp' )

class Router( Instance ):
   regex = '/openstack/api/region/{region}/router[/]'

   @staticmethod
   @urlHandler( ( 'GET', ), regex )
   def handleGet( rc, mounts, region ):
      return Instance.doHandleGet( rc, mounts, region, 'router' )

   @staticmethod
   @urlHandler( ( 'POST', ), regex )
   def handlePost( rc, mounts, region ):
      return Instance.doHandlePost( rc, mounts, region, 'router' )

   @staticmethod
   @urlHandler( ( 'PUT', 'PATCH' ), regex )
   def handlePut( rc, mounts, region ):
      return Instance.doHandlePut( rc, mounts, region, 'router' )

   @staticmethod
   @urlHandler( ( 'DELETE', ), regex )
   def handleDelete( rc, mounts, region ):
      return Instance.doHandleDelete( rc, mounts, region, 'router' )

class Port( BaseHandler ):
   # TODO - This model needs a lot of changes. It's not great to pass in the tenant
   # instead of the actual port instance in the toSysdb() function.
   # We also don't support moving ports, currently the PUT/PATCH methods only into
   # account updating the port name. We are also modifying the entity directly
   # which breaks our model. Most of this will go away once we flatten the TAC
   # model but in reality we want to move towards a system where an API Model does
   # not have to map to an exact entity.
   regex = '/openstack/api/region/{region}/port[/]'

   @staticmethod
   def _getQueryIds( rc ):
      return rc.getUrlQuery().get( 'id' )

   @staticmethod
   def _getQueryType( rc ):
      return rc.getUrlQuery().get( 'type' )

   @staticmethod
   @urlHandler( ( 'GET', ), regex )
   def handleGet( rc, mounts, region ):
      r = BaseHandler._getRegion( mounts, region )
      eType = Port._getQueryType( rc )

      # TODO remove these constraints once we flatten the TAC model
      if ( eType is None ) or ( len( eType ) > 1 ):
         raise UwsgiRequestContext.HttpBadRequest(
                                           "A single type must be specified" )

      eType = eType[ 0 ]
      ret = []
      entities = []
      ids = Port._getQueryIds( rc )
      if eType == 'vm':
         if not ids:
            ids = list( r.vmInstance )
         for vmId in ids:
            vm = r.vmInstance.get( vmId )
            if vm:
               entities.extend( vm.port.values() )
      elif eType == 'baremetal':
         if not ids:
            ids = list( r.baremetalInstance )
         for bmId in ids:
            bm = r.baremetalInstance.get( bmId )
            if bm:
               entities.extend( bm.port.values() )
      elif eType == 'router':
         if not ids:
            ids = list( r.routerInstance )
         for routerId in ids:
            rtr = r.routerInstance.get( routerId )
            if rtr:
               entities.extend( rtr.port.values() )
      elif eType == 'dhcp':
         if not ids:
            ids = list( r.dhcpInstance )
         for dhcpId in ids:
            dhcp = r.dhcpInstance.get( dhcpId )
            if dhcp:
               entities.extend( dhcp.port.values() )
      else:
         raise UwsgiRequestContext.HttpBadRequest(
                                          "Unknown type %s specified" % eType )

      for e in entities:
         model = Models.PortModel()
         model.fromSysdb( e, eType )
         ret.append( model )

      return ret

   @staticmethod
   @urlHandler( ( 'POST', ), regex )
   def handlePost( rc, mounts, region ):
      BaseHandler._checkSyncLock( rc, mounts, region )
      r = BaseHandler._getRegion( mounts, region )
      data = rc.getRequestContent()
      data = json.loads( data )
      invalidPorts = set()
      for item in data:
         model = Models.PortModel()
         model.populateModelFromJson( item )
         pId = model.portId
         iType = model.instanceType
         iId = model.instanceId
         if iType == 'vm':
            instance = r.vmInstance.get( iId )
         elif iType == 'baremetal':
            instance = r.baremetalInstance.get( iId )
         elif iType == 'router':
            instance = r.routerInstance.get( iId )
         elif iType == 'dhcp':
            instance = r.dhcpInstance.get( iId )
         else:
            warn( "Port:handlePost() unknown instance type %s" % iType )
            invalidPorts.add( pId )
            continue
         if instance is None:
            warn( f"Port:handlePost() instance {iType}:{iId} not found" )
            invalidPorts.add( pId )
            continue
         networkId = model.networkId
         network = r.network.get( networkId )
         if network is None:
            warn( "Port:handlePost() network %s not found" % networkId )
            invalidPorts.add( pId )
            continue
         model.toSysdb( r, pId, instance, network )
      if invalidPorts:
         invalidPortMessage = ( "No network or instance for port id%s %s" %
                               ( 's' if len( invalidPorts ) > 1 else '',
                                 ", ".join( str( p ) for p in invalidPorts ) ) )
         raise UwsgiRequestContext.HttpBadRequest( invalidPortMessage )
      return {}

   @staticmethod
   @urlHandler( ( 'PUT', 'PATCH' ), regex )
   def handlePut( rc, mounts, region ):
      BaseHandler._checkSyncLock( rc, mounts, region )
      r = BaseHandler._getRegion( mounts, region )
      data = rc.getRequestContent()
      data = json.loads( data )
      for item in data:
         model = Models.PortModel()
         model.populateModelFromJson( item )
         iType = item.get( 'instanceType' )
         iId = item.get( 'instanceId' )
         pId = item.get( 'id' )
         instance = None
         if iType and iId:
            if iType == 'vm':
               instance = r.vmInstance.get( iId )
            elif iType == 'baremetal':
               instance = r.baremetalInstance.get( iId )
            elif iType == 'router':
               instance = r.routerInstance.get( iId )
            elif iType == 'dhcp':
               instance = r.dhcpInstance.get( iId )
            else:
               warn( "Port:handlePut() unknown instance type %s for port %s" %
                     ( iType, pId ) )
               continue
         p = r.port.get( pId )
         if p is None:
            warn( "Port:handlePut() unkown port %s" % pId )
            continue
         pName = model.name
         if pName:
            p.portName = pName
         if instance:
            p.instance = instance

      return {}

   @staticmethod
   @urlHandler( ( 'DELETE', ), regex )
   def handleDelete( rc, mounts, region ):
      BaseHandler._checkSyncLock( rc, mounts, region )
      r = BaseHandler._getRegion( mounts, region )
      data = rc.getRequestContent()
      data = json.loads( data )
      for item in data:
         pId = item.get( 'id' )
         iType = item.get( 'instanceType' )
         iId = item.get( 'instanceId' )
         if iId is None:
            warn( "Port:handleDelete instanceId is None for port %s" % pId )
            continue
         if iType == 'vm':
            instance = r.vmInstance.get( iId )
         elif iType == 'baremetal':
            instance = r.baremetalInstance.get( iId )
         elif iType == 'router':
            instance = r.routerInstance.get( iId )
         elif iType == 'dhcp':
            instance = r.dhcpInstance.get( iId )
         else:
            warn( "Port:handleDelete unknown instance type %s for port %s" %
                  ( iType, pId ) )
            continue
         p = instance and instance.port.get( pId )
         if p is None:
            warn( "Port:handleDelete() unknown port %s" % pId )
            continue
         del instance.port[ pId ]
         del r.port[ pId ]
      return {}

class Segment( BaseHandler ):
   regex = '/openstack/api/region/{region}/segment[/]'

   @staticmethod
   @urlHandler( ( 'GET', ), regex )
   def handleGet( rc, mounts, region ):
      r = BaseHandler._getRegion( mounts, region )

      ret = []
      for segment in r.segment.values():
         model = Models.SegmentModel()
         model.fromSysdb( r, segment )
         ret.append( model )
      return ret

   @staticmethod
   @urlHandler( ( 'POST', ), regex )
   def handlePost( rc, mounts, region ):
      ''' Create a segment. '''
      BaseHandler._checkSyncLock( rc, mounts, region )
      r = BaseHandler._getRegion( mounts, region )
      vlanPool = BaseHandler._getVlanPool( mounts, region )
      invalidVlans = set()

      data = rc.getRequestContent()
      data = json.loads( data )
      invalidNetworks = set()
      for item in data:
         model = Models.SegmentModel()
         model.populateModelFromJson( item )
         networkId = model.networkId
         network = r.network.get( networkId )
         if not network:
            invalidNetworks.add( networkId )
            continue
         invalidVlan = model.toSysdb( r, network, vlanPool )
         if invalidVlan:
            invalidVlans.add( invalidVlan )
      if invalidVlans or invalidNetworks:
         invalidSegmentMessage = ( "VLAN segmentation id%s %s %s not available" %
                                  ( 's' if len( invalidVlans ) > 1 else '',
                                  ", ".join( str( vlan ) for vlan in invalidVlans ),
                                  "are" if len( invalidVlans ) > 1 else "is" )
                                  if invalidVlans else "" )
         invalidSegmentMessage += ( "{}Network {} not found".format(
            '. ' if invalidVlans else "", ', '.join( invalidNetworks ) )
             if invalidNetworks else "" )
         raise UwsgiRequestContext.HttpBadRequest( invalidSegmentMessage )
      return {}

   @staticmethod
   @urlHandler( ( 'DELETE', ), regex )
   def handleDelete( rc, mounts, region ):
      BaseHandler._checkSyncLock( rc, mounts, region )
      r = BaseHandler._getRegion( mounts, region )

      data = rc.getRequestContent()
      data = json.loads( data )
      for item in data:
         segmentId = item.get( 'id' )
         if segmentId not in r.segment:
            warn( "Segment:handleDelete segment %s not found" % segmentId )
            continue

         segment = r.segment[ segmentId ]
         network = r.network.get( segment.networkId )
         if network:
            if segmentId in network.staticSegment:
               del network.staticSegment[ segmentId ]
            elif segmentId in network.dynamicSegment:
               del network.dynamicSegment[ segmentId ]
         del r.segment[ segmentId ]

class Binding( BaseHandler ):
   legacyRegex = '/openstack/api/region/{region}/port/{port}/binding[/]'
   regex = '/openstack/api/region/{region}/portbinding[/]'

   @staticmethod
   def validatePort( r, portId ):
      p = r.port.get( portId )
      if not p:
         warn( "Binding:validatePort port %s not found" % portId )
         return False
      return True

   @staticmethod
   def validatePortBinding( binding, portId ):
      if not binding:
         warn( "Binding:validatePortBinding port %s not found" % portId )
         return False
      return True

   @staticmethod
   def validateSegments( region, data ):
      invalidSegs = set()
      if data.get( 'hostBinding' ):
         bModel = Models.PortToHostBindingModel()
         for hostBinding in data.get( 'hostBinding' ):
            bModel.populateModelFromJson( hostBinding )
            for segment in bModel.segments:
               if segment.segmentId not in region.segment:
                  warn( "Binding:segment %s not found" % segment.segmentId )
                  invalidSegs.add( segment.segmentId )
      elif data.get( 'switchBinding' ):
         bModel = Models.PortToSwitchInterfaceBindingModel()
         for switchBinding in data.get( 'switchBinding' ):
            bModel.populateModelFromJson( switchBinding )
            for segment in bModel.segments:
               segmentId = segment.segmentId
               if segment.segmentId not in region.segment:
                  warn( "Binding:segment %s not found" % segmentId )
                  invalidSegs.add( segmentId )

      return invalidSegs

   @staticmethod
   @urlHandler( ( 'GET', ), legacyRegex )
   @urlHandler( ( 'GET', ), regex )
   def handleGet( rc, mounts, region, port=None ):
      r = BaseHandler._getRegion( mounts, region )
      ret = []
      if port and port in r.portBinding:
         model = Models.PortBindingModel()
         model.fromSysdb( r, r.portBinding[ port ] )
         ret.append( model )
      elif port is None:
         for p in r.portBinding:
            model = Models.PortBindingModel()
            model.fromSysdb( r, r.portBinding[ p ] )
            ret.append( model )
      return ret

   @staticmethod
   @urlHandler( ( 'POST', ), legacyRegex )
   @urlHandler( ( 'POST', ), regex )
   def handlePost( rc, mounts, region, port=None ):
      ''' Create a port binding for the port. '''
      BaseHandler._checkSyncLock( rc, mounts, region )
      r = BaseHandler._getRegion( mounts, region )
      if port:
         portId = port
         if not Binding.validatePort( r, portId ):
            raise UwsgiRequestContext.HttpNotFound( 'Port %s not found' %
                                                    ( portId ) )

      data = rc.getRequestContent()
      data = json.loads( data )
      invalidPorts = set()
      invalidSegments = set()
      for item in data:
         model = Models.PortBindingModel()
         model.populateModelFromJson( item )
         if not port:
            portId = model.portId
            if not Binding.validatePort( r, portId ):
               invalidPorts.add( portId )
               continue
         invalidSegments.update( Binding.validateSegments( r, item ) )
         model.toSysdb( r, portId )

      if invalidPorts or invalidSegments:
         invalidPortMessage = ( "Unknown port id%s %s" %
                                 ( 's' if len( invalidPorts ) > 1 else '',
                                   ", ".join( str( p ) for p in invalidPorts ) ) if
                                  invalidPorts else "" )
         invalidPortMessage += ( "%sUnknown segment%s %s" %
                                   ( '. ' if invalidPorts else '',
                                     's' if len( invalidSegments ) > 1 else '',
                                     ", ".join( invalidSegments ) )
                                     if invalidSegments else "" )
         raise UwsgiRequestContext.HttpBadRequest( invalidPortMessage )

      return {}

   @staticmethod
   @urlHandler( ( 'DELETE', ), legacyRegex )
   @urlHandler( ( 'DELETE', ), regex )
   def handleDelete( rc, mounts, region, port=None ):
      BaseHandler._checkSyncLock( rc, mounts, region )
      r = BaseHandler._getRegion( mounts, region )
      data = rc.getRequestContent()
      data = json.loads( data )
      if port:
         portId = port
         binding = r.portBinding.get( portId )
         if not Binding.validatePortBinding( binding, portId ):
            return {}

         if not data:
            del r.portBinding[ portId ]
            return {}

      for item in data:
         if not port:
            model = Models.PortBindingModel()
            model.populateModelFromJson( item )
            portId = model.portId
            binding = r.portBinding.get( portId )
            if not Binding.validatePortBinding( binding, portId ):
               continue
         hostBindings = item.get( 'hostBinding', [] )
         for hostBinding in hostBindings:
            host = hostBinding[ 'host' ]
            if host in binding.portToHostBinding:
               del binding.portToHostBinding[ host ]
            else:
               warn( "Binding:handleDelete host %s not found" % host )

         switchBindings = item.get( 'switchBinding', [] )
         for switchBinding in switchBindings:
            switch = switchBinding[ 'switch' ]
            interface = switchBinding[ 'interface' ]
            switchInterface = SwitchInterface( switch, interface )
            if switchInterface in binding.portToSwitchInterfaceBinding:
               del binding.portToSwitchInterfaceBinding[ switchInterface ]
            else:
               warn( "Binding:handleDelete switch interface %s not found" %
                      switchInterface.name() )

         if( len( r.portBinding[ portId ].portToHostBinding ) == 0 and
             len( r.portBinding[ portId ].portToSwitchInterfaceBinding ) == 0 ):
            del r.portBinding[ portId ]

      return {}

def Plugin( ctx ):
   cdbMounts = {}
   sysdbMounts = {}

   cdbMounts[ 'openStackConfig' ] = (
      'openstack/openstackConfig',
      'OpenStack::OpenStackConfig',
      'w' )
   cdbMounts[ 'openStackStatus' ] = (
      'openstack/openstackStatus',
      'OpenStack::OpenStackStatus',
      'r' )

   sysdbMounts[ 'openStackAgentStatus' ] = 'mgmt/openstack/status'
   sysdbMounts[ 'clusterStatus' ] = 'controller/cluster/statusDir'

   return sysdbMounts, cdbMounts
