# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from functools import total_ordering
import os
import re

# Kernel Features
ROUTE_CACHE      =  1 # BUG103898
TAP_QDISK        =  2 # BUG102872
VLAN_1AD         =  3 # BUG101982
PERNS_IFINDEX    =  4 # BUG161232,BUG173228
IPV4_ADDR_LFT    =  5 # BUG145426
GRE_XNETNS       =  6 # BUG177285,BUG191296,BUG192293
CORE_NTFILE      =  7
TAP_SKB_ARRAY    =  8 # BUG255918
SET_IGMP_MAX_MEM =  9 # BUG235752, BUG284337
IPV6_ADDR_NOTIFY = 10 # BUG340648
IPV6_ENH_DAD     = 11 # BUG340651
I40E_DIS_SRC_PR  = 12 # BUG276522
VALID_CFI_BIT    = 13 # BUG553051
PIPE_EPOLL_FIX   = 14 # BUG553050, BUG617468
LKDTM_EXHAUST_STACK = 15 # BUG608441
LKDTM_SCSI_QUEUE_RQ = 16 # BUG609175
OVL_XINO_OPT = 17 # BUG652321, BUG640275

def _removedInVersion( featureVersion ):
   def _hasFeature( testVersion ):
      return testVersion < KernelVersion( featureVersion )

   return _hasFeature

def _addedInVersion( featureVersion ):
   def _hasFeature( testVersion ):
      return testVersion >= KernelVersion( featureVersion )

   return _hasFeature

_features_ref = {
   ROUTE_CACHE: _removedInVersion( '3.6' ), # 89aef8921bfb
   TAP_QDISK: _removedInVersion( '3.8' ), # 5d097109257c
   VLAN_1AD: _addedInVersion( '3.14' ), # a0cdfcf3936
   PERNS_IFINDEX: _addedInVersion( '3.7' ), # aa79e66eee5d
   IPV4_ADDR_LFT: _addedInVersion( '3.9' ), # 5c766d642bcaf
   GRE_XNETNS: _addedInVersion( '3.16' ), # b57708add314
   CORE_NTFILE: _addedInVersion( '3.7' ), # 2aa362c49c31
   TAP_SKB_ARRAY: _addedInVersion( '4.8' ), # 1576d9860599
   SET_IGMP_MAX_MEM: _addedInVersion( '4.6' ), # 815c52700746
   OVL_XINO_OPT: _addedInVersion( '4.17' ), # 795939a93e60
   IPV6_ADDR_NOTIFY: _addedInVersion( '4.18' ), # a2d481b326c9
   IPV6_ENH_DAD: _addedInVersion( '4.10' ), # adc176c54722
   I40E_DIS_SRC_PR: _addedInVersion( '4.15' ), # 64615b5418ba
   VALID_CFI_BIT: _addedInVersion( '5.0' ), # 1e33f015994f
   LKDTM_EXHAUST_STACK: _addedInVersion( '4.19' ), # 24cccab42c41a (5.2 backport)
   PIPE_EPOLL_FIX: _addedInVersion( '5.10' ), # 3b2018f9c9c0 in 5.10.62
   LKDTM_SCSI_QUEUE_RQ: _addedInVersion( '5.10' ) # d1f278da6b11 (5.15 backport)
}

@total_ordering
class KernelVersion:
   # The main version string contains digits and dots and ends with a
   # digit. Everything after that is localversion, which can be any
   # non-whitespace.
   versionRe = re.compile( r'([.\d]*\d)(\S*)' )

   @classmethod
   def fromEdut( cls, edut ):
      versionString, = edut.bashSuCmdIs( "uname -r" )
      return cls( versionString )

   def __init__( self, versionString ):
      match = self.versionRe.match( versionString )
      if not match:
         # pylint: disable-next=consider-using-f-string
         raise ValueError( "Invalid version string: %s" % versionString )

      version, self.localversion = match.groups()
      self.version = tuple( map( int, version.split( '.' ) ) )
      assert len( self.version ) > 1

   def __eq__( self, other ):
      return self.version == other.version

   def __lt__( self, other ):
      return self.version < other.version

   __hash__ = None

   def __str__( self ):
      versionStr = '.'.join( map( str, self.version ) )
      return versionStr + self.localversion

   def supports( self, feature ):
      return _features_ref[ feature ]( self )

current_version = KernelVersion( os.uname()[ 2 ] )

def supports( f ):
   return current_version.supports( f )
