# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''This module defines the card profiles builtin loader.

The loader is responsible for loading and parsing a group of card profiles
from their builtin YAML file definitions into a provided L1Profile:CardProfileDir.

Malformed card slot profile definitions as well as those already existing in the
destination L1Profile::CardProfileDir will be skipped.

Utilizes the common YAML loader infrastructure defiend in ArPyUtils.
'''

from L1ProfileBuiltinsLibrary.Card import (
   ParserV1,
)

import Tracing
from TypeFuture import TacLazyType
from YamlLoaderLibrary.Exceptions import ParsingError, SkipDocument
from YamlLoaderLibrary.Loader import LoaderBase
from YamlLoaderLibrary.Parser import ParserRegistry


CardProfileDescriptor = TacLazyType( 'L1Profile::CardProfileDescriptor' )
CardProfileSource = TacLazyType(
   'L1Profile::CardProfileSource::CardProfileSource' )

class CardProfileLoader( LoaderBase ):
   TraceHandle = Tracing.Handle( 'L1ProfileBuiltinsLibrary.Card.Loader' )

   def __init__( self, cardProfileDir, intfSlotProfileDir, skuBaseName=None ):
      '''The Loader implementation for the card profiles. Takes in both the dir to
      popuate with the loaded profiles along with the valid interface slot profiles
      that we can use.

      Note:
         The interface slot profiles must be loaded prior to loading the card
         profiles for the loader to be able to know which interface slot profiles
         referenced in the card profiles are actually valid.

      Args:
         cardProfileDir( Tac.Dir ): The TAC directory to store card profiles, we will
                                    write all results into this dir.
         intfSlotProfileDir( Tac.Dir ): The TAC directory already loaded with intf
                                        slot profiles. This is needed for us to
                                        ensure the builtin card profiles we try to
                                        load reference existing intf slot profiles.
         skuBaseName ( string ): The basename of the SKU. When this field is empty,
                                 we assume we are running on a modular system.
      '''
      self.cardProfileDir = cardProfileDir
      self.intfSlotProfileDir = intfSlotProfileDir
      self.skuBaseName = skuBaseName

      # Add the parsers that we support to a registry to pass to the parent
      parserRegistry = ParserRegistry()
      for parser in ( ParserV1.ParserV1, ):
         parserRegistry.register( parser() )
      super().__init__( parserRegistry )

   def loadDocument( self, parsedDocument ):
      '''Converts the parsed document describing a card profile into a model stored
      in the provided cardProfileDir.

      Raises:
         ParsingError if the card profile already exists or invalid interface slot
         profiles are referenced.
         SkipDocument if the document doesn't apply to the current skuBaseName.
      '''

      if parsedDocument.descriptor in self.cardProfileDir.cardProfile:
         # TODO BUG698933: It would be nice to check if there is a difference
         #                 between the two definitions. In such cases maybe
         #                 emitting an error trace is better.
         raise ParsingError( f'Profile {parsedDocument.name} already exists' )

      if self.skuBaseName and self.skuBaseName not in parsedDocument.applicability:
         raise SkipDocument( 'Profile not applicable to current SKU\n'
                             f'\tSKU: {self.skuBaseName}\n\tApplicability: '
                             f'{set( parsedDocument.applicability )}' )

      profileIntfSlotNames = { profile.name for profile in
                               parsedDocument.intfSlotProfile.values() }
      presentIntfSlotNames = set( desc.name for desc in
                                  self.intfSlotProfileDir.intfSlotProfile )
      invalidIntfSlotNames = profileIntfSlotNames - presentIntfSlotNames
      if invalidIntfSlotNames:
         raise ParsingError( 'Invalid interface slot profiles are referenced: '
                             f'{invalidIntfSlotNames}' )

      cardProfile = self.cardProfileDir.newCardProfile(
         CardProfileDescriptor( CardProfileSource.builtin,
                                parsedDocument.descriptor.name ) )
      cardProfile.copyFrom( parsedDocument )
