# Copyright (c) 2022, Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''This module define the interface slot profiles builtin loader.

The loader is responsible for loading and parsing a group of interface slot profiles
from their builtin YAML file definitions ( typically stored in
/usr/share/L1Profiles/intfSlot/builtin ) into a provided
L1Profile::InterfaceSlotProfileDir

Malformed interface slot profile definitions as well as those already existing in the
destination L1Profile::InterfaceSlotProfileDir will be skipped.

Utilizes the common YAML loader infrastructure defiend in ArPyUtils.
'''

from L1ProfileBuiltinsLibrary.IntfSlot import (
   ParserV1,
)

import Tracing
from TypeFuture import TacLazyType
from YamlLoaderLibrary.Exceptions import ParsingError
from YamlLoaderLibrary.Loader import LoaderBase
from YamlLoaderLibrary.Parser import ParserRegistry

InterfaceSlotProfileDescriptor = TacLazyType(
   'L1Profile::InterfaceSlotProfileDescriptor' )

class IntfSlotProfileLoader( LoaderBase ):
   TraceHandle = Tracing.Handle( 'L1ProfileBuiltinsLibrary.IntfSlot.Loader' )

   def __init__( self, intfSlotProfileDir ):
      '''The Loader implementation for the interface slot profiles. Takes in the dir
      to popuate with the loaded profiles.

      Args:
         intfSlotProfileDir ( InterfaceSlotProfileDir ): The destination entity which
                                                         will be loaded with the
                                                         parsed L1 interface slot
                                                         profile definitions.
      '''
      self.intfSlotProfileDir = intfSlotProfileDir

      # Add the parsers that we support to a registry to pass to the parent
      parserRegistry = ParserRegistry()
      for parser in ( ParserV1.ParserV1, ):
         parserRegistry.register( parser() )
      super().__init__( parserRegistry )

   def loadDocument( self, parsedDocument ):
      '''Converts the parsed document describing an interface slot profile into a
      model stored in the provided intfSlotProfileDir.

      Raises:
         ParsingError if the card profile already exists or invalid interface slot
         profiles are referenced.
         SkipDocument if the document doesn't apply to the current skuBaseName.
      '''

      if parsedDocument.descriptor in self.intfSlotProfileDir.intfSlotProfile:
         # TODO BUG698933: It would be nice to check if there is a difference
         #                 between the two definitions. In such cases maybe
         #                 emitting an error trace is better.
         raise ParsingError( f'Profile {parsedDocument.name} already exists' )

      intfSlotProfile = self.intfSlotProfileDir.newIntfSlotProfile(
         InterfaceSlotProfileDescriptor( self.intfSlotProfileDir.source(),
                                         parsedDocument.descriptor.name ) )
      intfSlotProfile.copyFrom( parsedDocument )
