# Copyright (c) 2022, Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import json
import jsonschema

import EthIntfLib
from TypeFuture import TacLazyType
from YamlLoaderLibrary.Exceptions import ParsingError
from YamlLoaderLibrary.Parser import ParserBase, getValidationError
from YamlLoaderLibrary.Versions import Version

INTF_SLOT_PROFILE_NAME_REGEX = r'^[A-Za-z0-9/_-]+$'

EnforcedCapabilities = TacLazyType( 'Hardware::Capabilities::EnforcedCapabilities' )
EnforcedCapabilitiesAction = TacLazyType(
   'Hardware::Capabilities::EnforcedCapabilitiesAction' )
InterfaceLaneSpecification = TacLazyType( 'L1Profile::InterfaceLaneSpecification' )
InterfaceSlotProfile = TacLazyType( 'L1Profile::InterfaceSlotProfile' )
InterfaceSlotProfileDescriptor = TacLazyType(
   'L1Profile::InterfaceSlotProfileDescriptor' )
InterfaceSlotProfileSource = TacLazyType(
   'L1Profile::InterfaceSlotProfileSource::InterfaceSlotProfileSource' )

class ParserV1( ParserBase ):
   version = Version( 1, 3 )

   schema = {
      '$schema': 'https://json-schema.org/draft/2020-12/schema',
      '$id': 'intfSlotProfileBuiltinV1',
      'title': 'Interface slot profile YAML definition V1',
      'description': ( 'The root object containing the definition of a '
                       'single interface slot profile' ),
      'patternProperties': {
         INTF_SLOT_PROFILE_NAME_REGEX: {
            'type': 'object',
            'properties': {
               'description': {
                  'description': 'The description of the interface slot profile',
                  'type': 'string',
                  'minLength': 1,
               },
               'interface-lanes': {
                  'description': ( 'A set of mappings from interface lane number to '
                                   'the specification for the desired interface to '
                                   'be generated.' ),
                  'type': 'object',
                  'patternProperties': {
                     '^[1-9][0-9]*$': {
                        'description': ( 'An interface lane specification which '
                                         'describes the name of the generated '
                                         'interface as well as its default '
                                         'configurations' ),
                        'type': 'object',
                        'properties': {
                           'root': {
                              'description': ( 'The host lane on the interface slot '
                                               'which the generated interface will '
                                               'point to.' ),
                              'type': 'integer',
                              'minimum': 1,
                              'maximum': 8,
                           },
                           'default': {
                              'description': ( 'Specifies the default behaviour for'
                                               'the generated interface' ),
                              'type': 'object',
                              'properties': {
                                 'speed': {
                                    'description': ( 'The default link mode ( '
                                                     'speed, lanes, duplex ) for '
                                                     'the generated interface.' ),
                                    'type': 'string',
                                    'pattern': r'(?i)^\d+[mg](-\d+)?'
                                               r'(\/(half|full|[hf]))?$',
                                 },
                                 'width': {
                                    'description': ( 'The number of lanes allocated '
                                                     'by default for the generated '
                                                     'interface.' ),
                                    'type': 'integer',
                                    'minimum': 1,
                                    'maximum': 8,
                                 },
                              },
                              'additionalProperties': False,
                           },
                           'capabilities': {
                              'description': ( 'Influence the genereted interface\'s'
                                               'capabilities' ),
                              'type': 'object',
                              'oneOf': [
                                 {
                                    "required": [ 'remove' ]
                                 },
                                 {
                                    "required": [ 'only' ]
                                 },
                              ],
                              'properties': {
                                 'remove': {
                                    'description': ( 'Link modes to remove from the'
                                                     'generated interface\'s '
                                                     'capabilties set' ),
                                    'type': 'array',
                                    'items': {
                                       'type': 'string',
                                       'pattern': r'(?i)^\d+[mg](-\d+)?'
                                                  r'(\/(half|full|[hf]))?$',
                                    },
                                    'minItems': 1,
                                 },
                                 'only': {
                                    'description': ( 'Link modes to override to the'
                                                     'generated interface\'s '
                                                     'capabilties set' ),
                                    'type': 'array',
                                    'items': {
                                       'type': 'string',
                                       'pattern': r'(?i)^\d+[mg](-\d+)?'
                                                  r'(\/(half|full|[hf]))?$',
                                    },
                                    'minItems': 1,
                                 },
                              },
                              'additionalProperties': False,
                           },
                        },
                        'additionalProperties': False,
                     },
                  },
                  'additionalProperties': False,
                  'minProperties': 1
               },
               'additionalProperties': False,
            },
            'additionalProperties': False,
            'required': [ 'description' ],
         },
      },
      'additionalProperties': False,
      'maxProperties': 1,
   }

   def validateDocument( self, documentSource, document ):
      try:
         # This is where the limitations of using a JSON schema with a YAML input
         # become painfully clear. YAML is sane and allows keys to be integers, JSON
         # is not and does not. The solution: dump the YAML dict to string and reload
         # it using the JSON infra which will automatically resolve these type
         # issues...
         jsonschema.validate( json.loads( json.dumps( document ) ), self.schema )
      except jsonschema.exceptions.ValidationError as error:
         raise getValidationError( f'{error}', document ) from error

   @staticmethod
   def parseDefaults( intfLaneSpec, desiredIntfLaneSpec ):
      '''Parse interface defaults from a YAML's document's inerface lane spec and
      populate the profile's interface lane spec.
      '''
      intfLaneSpecDefaults = desiredIntfLaneSpec.get( 'default' )
      if not intfLaneSpecDefaults:
         return

      if desiredDefaultSpeed := intfLaneSpecDefaults.get( 'speed' ):
         # Unfortunately, the conversion from "mode string" to EthLinkMode is not
         # safe and our only utility likes asserting.
         #
         # Any failure here is in all likelihood the result of inputs from the
         # profile definition.
         #
         # TODO BUG698943: Add a "safe" / "noexcept" version of the mode string
         #                 parsing infra.
         try:
            intfLaneSpec.defaultLinkMode = EthIntfLib.modeStrToEthIntfMode(
               '', desiredDefaultSpeed ).forcedLinkMode
         except Exception as e: # pylint: disable=broad-except
            raise ParsingError( 'Could not translate mode string '
                                f'{desiredDefaultSpeed} to EthLinkMode:\n'
                                f'{str(e)}',
                                intflane=intfLaneSpec.intfLane ) from e

      if desiredDefaultWidth := intfLaneSpecDefaults.get( 'width' ):
         for laneId in range( intfLaneSpec.root,
                              intfLaneSpec.root + desiredDefaultWidth ):
            intfLaneSpec.defaultLanes.add( laneId )

   @staticmethod
   def parseCapabilities( intfLaneSpec, desiredIntfLaneSpec ):
      '''Parse interface capabilities from a YAML's document's inerface lane spec and
      populate the profile's interface lane spec.
      '''
      intfLaneSpecCapabilities = desiredIntfLaneSpec.get( 'capabilities' )
      if not intfLaneSpecCapabilities:
         return

      removedCapabilities = intfLaneSpecCapabilities.get( 'remove' )
      if removedCapabilities:
         enforcedCapabilities = EnforcedCapabilities()
         enforcedCapabilities.action = EnforcedCapabilitiesAction.remove

         for removedCapString in removedCapabilities:
            try:
               mode = EthIntfLib.modeStrToEthIntfMode( '', removedCapString )
            except Exception as e: # pylint: disable=broad-except
               raise ParsingError( 'Could not translate mode string '
                                   f'{removedCapString} to EthLinkMode:\n'
                                   f'{str(e)}',
                                   intflane=intfLaneSpec.intfLane ) from e
            if mode in enforcedCapabilities.capabilities:
               raise ParsingError( 'Detected duplicate removed mode capability '
                                   f'{mode} on interface lane spec '
                                   f'{intfLaneSpec.intfLane}' )
            enforcedCapabilities.capabilities.add( mode )

         intfLaneSpec.enforcedCapabilities = enforcedCapabilities

      overridenCapabilities = intfLaneSpecCapabilities.get( 'only' )
      if overridenCapabilities:
         enforcedCapabilities = EnforcedCapabilities()
         enforcedCapabilities.action = EnforcedCapabilitiesAction.override

         for overridenCapString in overridenCapabilities:
            try:
               mode = EthIntfLib.modeStrToEthIntfMode( '', overridenCapString )
            except Exception as e: # pylint: disable=broad-except
               raise ParsingError( 'Could not translate mode string '
                                    f'{overridenCapString} to EthLinkMode:\n'
                                    f'{str(e)}',
                                    intflane=intfLaneSpec.intfLane ) from e
            if mode in enforcedCapabilities.capabilities:
               raise ParsingError( 'Detected duplicate overriden mode capability '
                                    f'{mode} on interface lane spec '
                                    f'{intfLaneSpec.intfLane}' )
            enforcedCapabilities.capabilities.add( mode )

         intfLaneSpec.enforcedCapabilities = enforcedCapabilities

   def parseDocument( self, document ):
      profileName, profileDefinition = next( iter( document.items() ) )

      # It is not the job of the parser to determine the source of the interface slot
      # profile.
      intfSlotProfile = InterfaceSlotProfile(
         InterfaceSlotProfileDescriptor( InterfaceSlotProfileSource.unknown,
                                         profileName ) )
      intfSlotProfile.description = profileDefinition[ 'description' ]

      intfLanesDefinition = profileDefinition.get( 'interface-lanes', {} )
      for desiredIntfLane, desiredIntfLaneSpec in intfLanesDefinition.items():
         # Needed since we can't validate that keys are integers using JSON schema,
         # only that they are string representations of integers.
         desiredIntfLane = int( desiredIntfLane )
         intfLaneSpec = intfSlotProfile.newIntfLaneSpec( desiredIntfLane )

         # Roots are 0 indexed in EOS. However, we expose them as 1 indexed numbers
         # in the YAML files to make them easier for humans to reason about.
         intfLaneSpec.root = desiredIntfLaneSpec.get( 'root', desiredIntfLane ) - 1

         self.parseDefaults( intfLaneSpec, desiredIntfLaneSpec )
         self.parseCapabilities( intfLaneSpec, desiredIntfLaneSpec )

      return intfSlotProfile
