# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from abc import ABCMeta, abstractmethod

from ArPyUtils.Types import ArException
import Tac

ModuleName = Tac.Type( 'Hardware::L1Topology::ModuleName::ModuleName' )

class ModuleBase( metaclass=ABCMeta ):
   '''
   A base class for all modules. New modules should subclass this class and
   implement the abstract methods and attributes below.
   '''

   @abstractmethod
   def buildTopology( self, module, env ):
      '''
      Function used to actually build the topology up. It is expected that you use
      FdlLib directives to build the topology up on the passed in fru

      Parameters
      -------
      module: The FruBase for the module to populate inventory models on.
      env: A dict of extra environment variables for the module to use.
      '''
      raise NotImplementedError

registeredModules = {}

def registerModule( moduleName ):
   '''
   A utility function used to register modules so that the L1 Topology Module Loader
   infrastructure can be made aware of them.

   Expected to be used as a decorator such as::

   @registerModule( ModuleName.name )
   class NewModule( ModuleBase ):
      ...

   Args
   ----
      moduleName:  The module name to register the decorated module class with.

   Returns
   -------
      A decorator function to actually register the class.

   Raises
   ------
      ArException if the moduleName is not in the Hardware::L1Topology::ModuleName
      enum, or if another module class is already registered with that name.
      The produced decorator will also raise an ArExpection the module class to
      register does not subclass ModuleBase.
   '''
   validNames = ModuleName.attributes
   if moduleName not in validNames:
      raise ArException( 'Must register modules with names from '
                         'Hardware::L1Topology::ModuleName',
                         invalidModuleName=moduleName,
                         validNames=validNames )

   def registerFunc( moduleCls ):
      if moduleName in registeredModules:
         raise ArException( 'Attempted to register multiple modules '
                            'with the same name',
                            moduleName=moduleName,
                            existingModule=registeredModules[ moduleName ],
                            offendingModule=moduleCls )
      if not issubclass( moduleCls, ModuleBase ):
         raise ArException( 'Attempted to register a module that does '
                            'not inherit from ModuleBase.',
                            offendingModule=moduleCls )

      registeredModules[ moduleName ] = moduleCls
      return moduleCls
   return registerFunc
