# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# TODO: This is a pretty nasty copy of the L1Topology FdlLib functions needed to
#       resolve some dependancy issues with the L1TopologyModule rpms.
#       See BUG831403; This should get removed with some other way to make the APIs
#       common between the two.

# pylint: disable=consider-using-f-string

from collections import namedtuple

import Tac
from TypeFuture import TacLazyType

PhyScope = TacLazyType( 'PhyEee::PhyScope' )

UNKNOWN_TRACE_ID = Tac.Value( 'Hardware::L1Topology::TraceId' ).unknownTraceId
POLARITY_DEFAULT = Tac.Type( "Inventory::Polarity" ).polarityDefault
POLARITY_SWAP = Tac.Type( "Inventory::Polarity" ).polaritySwap
SCOPE_SYS = "system"
SCOPE_LINE = "line"
XCVR_SLOT = "XcvrSlot"
XCVR_PAD = "XcvrPad"
XCVR_TYPES = ( XCVR_SLOT, XCVR_PAD )

TraceDesc = namedtuple(
   "TraceDesc",
   [ "systemComponentId", "systemSerdesId", "lineComponentId", "lineSerdesId",
     "traceLoss", "connectionType", "polaritySwap", "traceId" ] )

StaticSerdesLogicalLaneDesc = namedtuple(
   "StaticSerdesLogicalLaneDesc",
   [ "componentId", "serdesId", "serdesLogicalLane", "connectionType" ] )

StaticSerdesPolarityDesc = namedtuple(
   "StaticSerdesPolarityDesc",
   [ "componentId", "serdesId", "polaritySwap", "connectionType" ] )

CoreDesc = namedtuple(
   "CoreDesc",
   [ "coreId", "mode" ] )

class L1Component:
   """
   A descriptor for a single type of component. Describes all
   components of that type and role within the sku.

   Parameters
   ----------
   name : str
      Name of the component. This must match the name attribute of the component
      class in PhyEee/HwL1TopologyComponentLib/

   numComponents : int
      The number of components of this type. This is only used to ensure that we
      reference the correct number of components in the various csv inputs.

   mode : str, optional
      Name of the hardware mode the component is in. This generally describes the
      set of serdes in use by the component along with the role it is in.
      Must match an option in PhyEee/HwL1TopologyComponentLib/

   codeModes : L1CoreModes, optional
      The modes to initalized each of the cores in. Ignored unless the component's
      mode is "Core" and the component supports per-core modes. These per-core modes
      behave the same as the component mode described above just scoped to each core.

   subdomain: str, optional
      Name of the subdomain the component is in. This is used to tag entryPoints
      with subDomain. This should be set to a valid subdomain name, only if this
      component is the primary component in the subdomain else leave this as None.
      Supported Subdomains:
         BfnPhy: Tofino asic
         SandPhy: Jericho2 and later sand asics
         Crosspoint: Crosspoint chip
         AppFpga{chipId}: Application Fpga.
                          FruPlugin replaces {chipId} with FpgaId.
   """

   lineTraces = []
   """
   list( L1TraceSet )
      List of L1 Trace Set objects connecting to the line side
      of the component
   """

   systemTraces = []
   """
   list( L1TraceSet )
      List of L1 Trace Set objects connecting to the system side
      of the component
   """

   def __init__( self, name, numComponents, mode=None,
                 coreModes=None, subdomain=None ):
      self.name = name.replace( "-", "_" )
      self.numComponents = numComponents
      self.mode = mode
      self.coreModes = coreModes or L1CoreModes( [] )
      self.subdomain = subdomain or ""
      assert name
      assert numComponents

   def __str__( self ):
      return "<L1Component %s>" % self.name

class L1CoreModes:
   """
   A descriptor for the tuning modes that a component should be initialized in.

   Parameters
   ----------
   data: list( tuple )
      List of cores on a component and what modes to initialize them in. The tuple
      has the same data and format as the csv data and the CoreDescs parameters.
      It has the following attributes::
         coreId,int
         mode,str
   """

   def __init__( self, data ):
      # core descriptors based on csv data dump
      self.descs = [ CoreDesc( *row ) for row in data ]

   def __str__( self ):
      return "<L1CoreModes>"

class StaticSerdesLogicalLaneSet:
   """
   A descriptor of a set of SerDes and their corresponding static logical lane
   mappings.

   Parameters
   ----------

   data: list( tuple )
      The list of SerDes on the component which have static logical lanes
      associated with them. The tuple has the data and format as the CSV data and
      thus is represented by the named tuple StaticSerdesLogicalLaneDesc which has
      the following attributes:
         ComponentId,int
         SerdesId,int
         serdesLogicalLane,int
         connectionType,str

      See StaticSerdesLogicalLaneDesc or AID9682 for more detail on each of the
      attributes.
   """
   def __init__( self, data ):
      self.laneDescs = [ StaticSerdesLogicalLaneDesc( *row ) for row in data ]

   def __str__( self ):
      return "<StaticSerdesLogicalLaneSet>"

class StaticSerdesPolaritySet:
   """
   A descriptor of a set of SerDes and their corresponding static polarity map.

   Parameters
   ----------

   data: list( tuple )
      The list of SerDes on the component which have a static polarity map
      associated with them. The tuple has the data and format as the CSV data and
      thus is represented by the named tuple StaticSerdesPolarityDesc which has
      the following attributes:
         ComponentId,int
         SerdesId,int
         polaritySwap,bool
         connectionType,str

      See StaticSerdesPolarityDesc or AID9682 for more detail on each of the
      attributes.
   """
   def __init__( self, data ):
      self.laneDescs = [ StaticSerdesPolarityDesc( *row ) for row in data ]

   def __str__( self ):
      return "<StaticSerdesPolaritySet>"

class L1TraceSet:
   """
   A descriptor of a set of traces. Each unique pair of chip types that
   are connected by traces will need a trace set to describe those traces.

   Parameters
   ----------
   system : L1Component
      Id of the system side component

   line : L1Component
      Id of the system side trace

   data: list( tuple )
      List of traces between the system and line components. The tuple has
      the data and format as the csv data and the TraceDescs parameters.
      It has the following attributes::

         systemComponentId,int
         systemSerdesId,int
         lineComponentId,int
         lineSerdesId,int
         traceLoss,float
         connectionType,str
         polaritySwap,bool

      See TraceDesc or AID6523 for more detail on each of the attributes.
   """
   def __init__( self, system, line, data ):
      # the component on the system side of the trace
      self.systemComponent = system
      # the component on the line side of the trace
      self.lineComponent = line
      # trace descriptors based on csv data dump
      self.descs = [ TraceDesc( *row, getNextTraceId() ) for row in data ]

   def __str__( self ):
      return "<L1TraceSet from {} (system) to {} (line)>".format(
                   self.systemComponent.name, self.lineComponent.name )

_traceId = 0

def resetTraceId():
   global _traceId
   _traceId = 0

def getNextTraceId():
   global _traceId
   _traceId += 1
   return _traceId

# Make sure every system trace set has been processed for a component
# We know a trace set has been processed if its traceIds are in the traceDir
def _checkTraceSets( topologyDir, component ):
   traceDir = topologyDir.newEntity( 'Inventory::L1TraceDir', 'L1Traces' )
   wasProcessed = True
   for traceSet in component.systemTraces:
      assert traceSet, "%s has no system-side traces" % str( traceSet )
      wasProcessed &= all( traceDesc.traceId in traceDir.l1Trace
                           for traceDesc in traceSet.descs )
   return wasProcessed

def _createTraces( topologyDir, traceSet ):
   traceDir = topologyDir.newEntity( 'Inventory::L1TraceDir', 'L1Traces' )
   for trace in traceSet.descs:
      polarity = POLARITY_SWAP if trace.polaritySwap else POLARITY_DEFAULT
      traceDir.newL1Trace( trace.traceId, polarity, trace.traceLoss )

def _findStaticSerdesDir( topologyDir, phyComponent ):
   staticMapDir = topologyDir.get( 'StaticSerdesMap' )
   if not staticMapDir:
      return None
   dirName = phyComponent.name
   if phyComponent.mode:
      dirName += "-%s" % phyComponent.mode
   dirName += "-static"
   return staticMapDir.get( dirName )

def _createStaticSerdesDir( topologyDir, phyComponent ):
   # This asssumes no two component sets share the same name and mode
   # We may need to revisit this directory naming in the future
   # TODO BUG664547: assert this is the case
   staticMapDir = topologyDir.mkdir( 'StaticSerdesMap' )
   dirName = phyComponent.name
   if phyComponent.mode:
      dirName += "-%s" % phyComponent.mode
   dirName += "-static"
   assert dirName not in staticMapDir
   staticSerdesDir = staticMapDir.newEntity(
                          'Inventory::L1TopologyStaticSerdesComponentDir',
                          dirName )
   staticSerdesDir.chipType = phyComponent.name
   staticSerdesDir.mode = phyComponent.mode or ''
   for coreId, mode in phyComponent.coreModes.descs:
      staticSerdesDir.coreMode[ coreId ] = mode or ''
   return staticSerdesDir

def _findComponentDir( topologyDir, phyComponent ):
   dirName = phyComponent.name
   if phyComponent.mode:
      dirName += "-%s" % phyComponent.mode

   return topologyDir.get( dirName )

def _createComponentDir( topologyDir, phyComponent ):
   # This asssumes no two component sets share the same name and mode
   # We may need to revisit this directory naming in the future
   # TODO BUG664547: assert this is the case
   dirName = phyComponent.name
   if phyComponent.mode:
      dirName += "-%s" % phyComponent.mode
   l1ComponentDir = _findComponentDir( topologyDir, phyComponent )
   if not l1ComponentDir:
      l1ComponentDir = topologyDir.newEntity( 'Inventory::L1ComponentTopoDir',
                                              dirName )
      l1ComponentDir.chipType = phyComponent.name
      l1ComponentDir.mode = phyComponent.mode or ''
      l1ComponentDir.subdomain = phyComponent.subdomain
      for coreId, mode in phyComponent.coreModes.descs:
         l1ComponentDir.coreMode[ coreId ] = mode or ''
   else:
      toAssert = True
      toAssert &= l1ComponentDir.chipType == phyComponent.name
      toAssert &= l1ComponentDir.mode in ( phyComponent.mode, '' )
      toAssert &= l1ComponentDir.subdomain == phyComponent.subdomain
      for coreId, mode in phyComponent.coreModes.descs:
         toAssert &= ( l1ComponentDir.coreMode[ coreId ] == mode ) or \
               ( l1ComponentDir.coreMode[ coreId ] == '' )
      assert toAssert, \
         f'Could not find L1 Topology component dir for {phyComponent}'
   return l1ComponentDir

def _createPhy( topologyDir, phyComponent ):
   l1ComponentDir = _createComponentDir( topologyDir, phyComponent )

   # The trace has a boolean tx attribute that says whether the system side of the
   # trace is tx or rx. The line side of the trace must be determined according to
   # the system side.

   # iterate through all the line traces and add them to the associate chip
   # The line side of a component is the system side of the traces.
   # We can populate the tx attribute's value directly.
   if phyComponent.lineTraces:
      for traceSet in phyComponent.lineTraces:
         for desc in traceSet.descs:
            l1ComponentDir.newL1ComponentTopo( desc.systemComponentId )
            chip = l1ComponentDir.l1ComponentTopo[ desc.systemComponentId ]
            if desc.connectionType == "rx":
               chip.lineRxSerdesId[ desc.systemSerdesId ] = desc.traceId
            elif desc.connectionType == "tx":
               chip.lineTxSerdesId[ desc.systemSerdesId ] = desc.traceId
            else:
               assert False, "Invalid Connection Type: %s" % desc.connectionType

   # The line side of the trace must be determined according to the tx attribute
   # describing the system side of the trace.
   # Most links are mapped rx <-> tx and tx <-> rx. Thus the system side of each
   # component ( which connects to the line side of the associated trace ) should be
   # the reverse of the tx attribute for that trace.
   # Traces to the XcvrSlots are mapped through as rx <-> rx and tx <-> tx.
   # Therefore, the system side of XcvrSlot component ( connected to the line side of
   # the associated trace ) should use the tx attribute for that trace

   # iterate through all the system traces and add them to the associate chip
   if phyComponent.systemTraces:
      for traceSet in phyComponent.systemTraces:
         for desc in traceSet.descs:
            l1ComponentDir.newL1ComponentTopo( desc.lineComponentId )
            chip = l1ComponentDir.l1ComponentTopo[ desc.lineComponentId ]
            if phyComponent.name in XCVR_TYPES:
               if desc.connectionType == "rx":
                  chip.systemRxSerdesId[ desc.lineSerdesId ] = desc.traceId
               elif desc.connectionType == "tx":
                  chip.systemTxSerdesId[ desc.lineSerdesId ] = desc.traceId
               else:
                  assert False, "Invalid Connection Type: %s" % desc.connectionType
            else:
               if desc.connectionType == "rx":
                  chip.systemTxSerdesId[ desc.lineSerdesId ] = desc.traceId
               elif desc.connectionType == "tx":
                  chip.systemRxSerdesId[ desc.lineSerdesId ] = desc.traceId
               else:
                  assert False, "Invalid Connection Type: %s" % desc.connectionType

   # The number of unique component ids must equal to the number of components
   assert len( l1ComponentDir.l1ComponentTopo ) == phyComponent.numComponents

# Start from the asic in the l1 topology graph.
# When we process a component, we will create all the traces on the line side of the
# chip. For each trace set, we check the next downstream component on the line side.
# We will process the downstream component immediately if all of the that component's
# system side traces are populated. Finally, at the very end we handle the population
# of the inventory model for the current component.

# This guarantees that we visit each component only once. Since we visit each
# component only once and populate line side serdes for that component,  we will
# visit each trace set only once as well. Since everything is downstream from the
# asic, we are also guaranteed to reach each component with a breadth first search
# downstream from the asic.
def _handleComponent( topologyDir, l1Component ):
   for traceSet in l1Component.lineTraces:
      _createTraces( topologyDir, traceSet )

      downstreamComponent = traceSet.lineComponent
      # process a component if all system side traces to the component
      # are populated. This guarantees it will not be processed twice
      if _checkTraceSets( topologyDir, downstreamComponent ):
         _handleComponent( topologyDir, downstreamComponent )

   # since handle component is only called on a component once all its
   # system traces are populated, and we populate all line trace above
   # by the time we are here all traces are populated for this component
   _createPhy( topologyDir, l1Component )

def createL1Topology( componentOrDomain, asicTopology ):
   """This method creates the L1Topology inventory. It traverses a trace graph
   of L1Components and L1TraceSets in order and creates chips and traces for
   each piece.

   Lets take the following basic topology with an asic, xcvr and phy where some
   ports go through the phy and some do not::

      +--------+     +-------+     +--------+
      |        |     |       |     |        |
      |        +-----+  Phy  +-----+        |
      |        |     |       |     |        |
      |  Asic  |     +-------+     |  Xcvr  |
      |        |                   |        |
      |        +-------------------+        |
      |        |                   |        |
      +--------+                   +--------+

   The Fdl will need to create a graph of L1Components and L1TraceSets to represent
   this front panel topology along the following lines::

      +-------------+   +---------+   +-------------+   +---------+   +-------------+
      |             |   |         |   |             |   |         |   |             |
      |             +---+ L1Trace +---+ L1Component +---+ L1Trace +---+             |
      |             |   |   Set   |   |    Phy      |   |   Set   |   |             |
      |             |   |         |   |             |   |         |   |             |
      | L1Component |   +---------+   +-------------+   +---------+   | L1Component |
      |    Asic     |                                                 |    Xcvr     |
      |             |                 +---------+                     |             |
      |             |                 |         |                     |             |
      |             +-----------------+ L1Trace +---------------------+             |
      |             |                 |   Set   |                     |             |
      |             |                 |         |                     |             |
      +-------------+                 +---------+                     +-------------+

   This topology is used by FdlLib in order to create the inventory models.

   Typical usage for the above example::

      j2Component = FdlLibL1.L1Component( "Jericho2", 1 )
      xcvrComponent = FdlLibL1.L1Component( "XcvrSlot", 36 )
      babbageComponent = FdlLibL1.L1Component( "Babbage", 8, "Gearbox" )

      j2ToXcvr = FdlLibL1.L1TraceSet( j2Component, xcvrComponent,
                                      <@j2ToXcvrTraceInfo@> )
      j2ToBabbage = FdlLibL1.L1TraceSet( j2Component, babbageComponent,
                                         <@j2ToBabbageTraceInfo@> )
      babbageToXcvr = FdlLibL1.L1TraceSet( babbageComponent, xcvrComponent,
                                           <@babbageToXcvrTraceInfo@> )

      j2Component.lineTraces = [ j2ToXcvr, j2ToBabbage ]
      xcvrComponent.systemTraces = [ j2ToXcvr, babbageToXcvr ]
      babbageComponent.systemTraces = [ j2ToBabbage ]
      babbageComponent.lineTraces = [ babbageToXcvr ]
      FdlLibL1.createL1Topology( system.component, jericho2Component )

   Parameters
   ----------
   componentOrDomain : Tac::Dir
      The containing directory. May be Inventory::MainPowerDomain (modular)
      or Tac::Dir (fixed sys)

   asicTopology : L1Component
      The L1Component objct for the component in the front panel topology farthest
      to the system side. This will be used as a starting point for the traversal,
      which will then go toward the line side as far as possible. Typically this
      component will be the asic. The API assumes all components are connected
      ( directly or through other components ) to the line side of this component.
   """
   resetTraceId()
   topologyDir = componentOrDomain.mkdir( 'L1Topology' )
   topologyDir.newEntity( 'Inventory::L1Topology::ModuleDir', 'ModuleDir' )
   _handleComponent( topologyDir, asicTopology )

def addStaticSerdesLaneMap( componentOrDomain, phyComponent, staticLaneSet, scope ):
   """This method creates the L1Topology static serdes map inventory if it doesn't
   already exist, and populates the static serdes lane map.
   Most products will not require this; static mappings are intended to be used
   when we want to override the dynamic mapping algorithms (e.g. when they might
   incorrectly resolve the mappings, or we do not want them to change in the
   future). Consult the appropriate PHY team if unsure about a specific product
   requiring static mappings.

   Parameters
   ----------
   componentOrDomain : Tac::Dir
      The containing directory. May be Inventory::MainPowerDomain (modular)
      or Tac::Dir (fixed sys)

   component : L1Component
      The L1Component object for the component that has static lane maps.

   staticLaneSet : StaticSerdesLogicalLaneSet
      The set of static lane mappings to apply for the component.

   scope : str
      The scope on the component the static lane mappings are for.
      Should either be SCOPE_LINE or SCOPE_SYS.
   """
   assert staticLaneSet, "Can't add lane maps if none are given..."
   assert phyComponent.name not in XCVR_TYPES, \
          "Lanemapping not support by XcvrSlots"

   topologyDir = componentOrDomain.mkdir( 'L1Topology' )
   staticComponentMapDir = _findStaticSerdesDir( topologyDir, phyComponent )
   if not staticComponentMapDir:
      staticComponentMapDir = _createStaticSerdesDir( topologyDir, phyComponent )
   if scope == SCOPE_SYS:
      phyScope = PhyScope.phyScopeSystem
   elif scope == SCOPE_LINE:
      phyScope = PhyScope.phyScopeLine
   else:
      assert False, "Invalid scope: %s" % scope

   for desc in staticLaneSet.laneDescs:
      component = \
         staticComponentMapDir.newComponentStaticSerdesMaps( desc.componentId )
      componentLaneMap = component.newStaticSerdesLaneMap( phyScope )
      if desc.connectionType == "rx":
         componentLaneMap.rxLogicalLane[ desc.serdesId ] = desc.serdesLogicalLane
      elif desc.connectionType == "tx":
         componentLaneMap.txLogicalLane[ desc.serdesId ] = desc.serdesLogicalLane
      else:
         assert False, "Invalid Connection Type: %s" % desc.connectionType

def addStaticSerdesPolarityMap( componentOrDomain, phyComponent,
                                staticPolaritySet, scope ):
   """This method creates the L1Topology static serdes map inventory if it doesn't
   already exist, and populates the static serdes polarity map.
   Most products will not require this; static mappings are intended to be used
   when we want to override the dynamic mapping algorithms (e.g. when they might
   incorrectly resolve the mappings, or we do not want them to change in the
   future). Consult the appropriate PHY team if unsure about a specific product
   requiring static mappings.

   Parameters
   ----------
   componentOrDomain : Tac::Dir
      The containing directory. May be Inventory::MainPowerDomain (modular)
      or Tac::Dir (fixed sys)

   component : L1Component
      The L1Component object for the component that has static polarity maps.

   staticPolaritySet : StaticSerdesPolaritySet
      The set of static polarity mappings to apply for the component.

   scope : str
      The scope on the component the static polarity mappings are for.
      Should either be phyScopeLine or phyScopeSystem.
   """
   assert staticPolaritySet, "Can't add polarity maps if none are given..."
   assert phyComponent.name not in XCVR_TYPES, \
          "Polarity mapping not support by XcvrSlots"

   topologyDir = componentOrDomain.mkdir( 'L1Topology' )
   staticComponentMapDir = _findStaticSerdesDir( topologyDir, phyComponent )
   if not staticComponentMapDir:
      staticComponentMapDir = _createStaticSerdesDir( topologyDir, phyComponent )
   if scope == SCOPE_SYS:
      phyScope = PhyScope.phyScopeSystem
   elif scope == SCOPE_LINE:
      phyScope = PhyScope.phyScopeLine
   else:
      assert False, "Invalid scope: %s" % scope

   for desc in staticPolaritySet.laneDescs:
      component = \
         staticComponentMapDir.newComponentStaticSerdesMaps( desc.componentId )
      componentPolarityMap = component.newStaticSerdesPolarityMap( phyScope )
      polarity = POLARITY_SWAP if desc.polaritySwap else POLARITY_DEFAULT
      if desc.connectionType == "rx":
         componentPolarityMap.rxPolarity[ desc.serdesId ] = polarity
      elif desc.connectionType == "tx":
         componentPolarityMap.txPolarity[ desc.serdesId ] = polarity
      else:
         assert False, "Invalid Connection Type: %s" % desc.connectionType

