# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from L1Topology.ModuleLib.Base import registerModule, ModuleBase, ModuleName
import L1Topology.ModuleLib.FdlLibClone as L1Topology

@registerModule( ModuleName.qsa )
class QsaModule( ModuleBase ):
   '''
   The definition for the simple QSA case. Really only has a single trace.
   '''

   def buildTopology( self, module, env ):
      padComponent = L1Topology.L1Component( "XcvrPad", 1 )
      cageComponent = L1Topology.L1Component( "XcvrSlot", 1 )

      padToCage = L1Topology.L1TraceSet( padComponent, cageComponent, [
         [ 0, 0, 1, 0, 1.0, 'rx', False ],
         [ 0, 0, 1, 0, 1.0, 'tx', False ],
      ] )

      padComponent.lineTraces = [ padToCage ]
      cageComponent.systemTraces = [ padToCage ]

      L1Topology.createL1Topology( module.component, padComponent )

@registerModule( ModuleName.dumbarton )
class DumbartonModule( ModuleBase ):
   '''
   The definition for the dumbarton OQA module. Swaps the polarity on its traces.
   '''

   def buildTopology( self, module, env ):
      padComponent = L1Topology.L1Component( "XcvrPad", 1 )
      cageComponent = L1Topology.L1Component( "XcvrSlot", 1 )

      padToCage = L1Topology.L1TraceSet( padComponent, cageComponent, [
         [ 0, 0, 1, 0, 1.0, 'rx', True ],
         [ 0, 0, 1, 0, 1.0, 'tx', True ],
         [ 0, 1, 1, 1, 1.0, 'rx', True ],
         [ 0, 1, 1, 1, 1.0, 'tx', True ],
         [ 0, 2, 1, 2, 1.0, 'rx', True ],
         [ 0, 2, 1, 2, 1.0, 'tx', True ],
         [ 0, 3, 1, 3, 1.0, 'rx', True ],
         [ 0, 3, 1, 3, 1.0, 'tx', True ],
      ] )

      padComponent.lineTraces = [ padToCage ]
      cageComponent.systemTraces = [ padToCage ]

      L1Topology.createL1Topology( module.component, padComponent )

@registerModule( ModuleName.blanco )
class BlancoModule( ModuleBase ):
   '''
   The definition for the blanco module.

   This module is pretty unique in a number of ways. It loops the signals from one
   input back to the n + 4th lane so the signals loop back into the box. However,
   unlike a loopback tester, it actually loops the tx pins back to the tx pins and
   vice versa.

   Then, on systems that support this special module we have a matching set of traces
   that link together two xcvr ports so the module can be used to convert 2x QSFP56s
   into a sinlge QSFP-DD.

   For good measure the connections also have polarity swaps!
   '''

   def buildTopology( self, module, env ):
      # TODO: See BUG514518; We can't use the same component even though its a
      #       self-to-self connection since we get stuck in infinite recursion
      egressComponent = L1Topology.L1Component( "XcvrPad", 1 )
      ingressComponent = L1Topology.L1Component( "XcvrPad", 1 )

      # TODO: See BUG601960; we don't currently know the loss that blanco adds
      #       internal to the module itself, so just use 1 for now.
      padToPad = L1Topology.L1TraceSet( egressComponent, ingressComponent, [
         [ 0, 0, 0, 4, 1.0, 'rx', True ],
         [ 0, 0, 0, 4, 1.0, 'tx', True ],
         [ 0, 1, 0, 5, 1.0, 'rx', True ],
         [ 0, 1, 0, 5, 1.0, 'tx', True ],
         [ 0, 2, 0, 6, 1.0, 'rx', True ],
         [ 0, 2, 0, 6, 1.0, 'tx', True ],
         [ 0, 3, 0, 7, 1.0, 'rx', True ],
         [ 0, 3, 0, 7, 1.0, 'tx', True ],
      ] )

      egressComponent.lineTraces = [ padToPad ]
      ingressComponent.systemTraces = [ padToPad ]

      L1Topology.createL1Topology( module.component, egressComponent )

@registerModule( ModuleName.mraT )
class MraTModule( ModuleBase ):
   '''
   The definition for an MRA BASE-T module.

   MRA stands for Multi-Rate Adapter. MRA BASE-T xcvr rate adapts and converts to
   BASE-T. This xcvr can take an SFP port and turn it into an RJ45 port using the
   phy within the transceiver to perform rate adaptation. It's a single lane phy that
   supports 10G on the system side, and 100M, 1G, 2.5G, 5G and 10G on the line side.
   We only advertise 100M, 1G and 10G based on customer request.
   '''
   def buildTopology( self, module, env ):
      padComponent = L1Topology.L1Component( "XcvrPad", 1 )
      phyComponent = L1Topology.L1Component( "MraTPhy", 1 )
      cageComponent = L1Topology.L1Component( "XcvrSlot", 1 )

      padToPhy = L1Topology.L1TraceSet( padComponent, phyComponent, [
         [ 0, 0, 1, 0, 1.0, 'rx', False ],
         [ 0, 0, 1, 0, 1.0, 'tx', False ],
      ] )

      phyToCage = L1Topology.L1TraceSet( phyComponent, cageComponent, [
         [ 1, 0, 2, 0, 1.0, 'rx', False ],
         [ 1, 0, 2, 0, 1.0, 'tx', False ],
      ] )

      padComponent.lineTraces = [ padToPhy ]
      phyComponent.systemTraces = [ padToPhy ]

      phyComponent.lineTraces = [ phyToCage ]
      cageComponent.systemTraces = [ phyToCage ]

      L1Topology.createL1Topology( module.component, padComponent )

@registerModule( ModuleName.lpo )
class LpoModule( ModuleBase ):
   '''
   The definition for the LPO module. It swaps polarity.

   See BUG924511, we need moduleProperties to be something customizable by any module
   long term.

   Tracelist supports the only LPO Module we support today.
   '''

   def buildTopology( self, module, env ):
      padComponent = L1Topology.L1Component( "XcvrPad", 1 )
      cageComponent = L1Topology.L1Component( "XcvrSlot", 1 )
      padToCage = L1Topology.L1TraceSet( padComponent, cageComponent, [
         [ 0, 0, 1, 0, 1.0, 'rx', False ],
         [ 0, 0, 1, 0, 1.0, 'tx', True ],
         [ 0, 1, 1, 1, 1.0, 'rx', False ],
         [ 0, 1, 1, 1, 1.0, 'tx', True ],
         [ 0, 2, 1, 2, 1.0, 'rx', False ],
         [ 0, 2, 1, 2, 1.0, 'tx', True ],
         [ 0, 3, 1, 3, 1.0, 'rx', False ],
         [ 0, 3, 1, 3, 1.0, 'tx', True ],
         [ 0, 4, 1, 4, 1.0, 'rx', False ],
         [ 0, 4, 1, 4, 1.0, 'tx', True ],
         [ 0, 5, 1, 5, 1.0, 'rx', False ],
         [ 0, 5, 1, 5, 1.0, 'tx', True ],
         [ 0, 6, 1, 6, 1.0, 'rx', False ],
         [ 0, 6, 1, 6, 1.0, 'tx', True ],
         [ 0, 7, 1, 7, 1.0, 'rx', False ],
         [ 0, 7, 1, 7, 1.0, 'tx', True ],
      ] )

      padComponent.lineTraces = [ padToCage ]
      cageComponent.systemTraces = [ padToCage ]

      L1Topology.createL1Topology( module.component, padComponent )
