# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing

t0 = Tracing.trace0

# Maximum level possible today is 3 i.e.,
# LoadBalance ( level 1 ) --> Label ( level 2 ) --> Destination ( level 3 )
# If we hit a level above this, then that's a new requirement from
# clients. Keeping the max allowed level to 5 to allow for future
# growth of destination chains.
MAX_LEVEL = 5

def traverseAndValidateMultiTable( nextObj, destTable, labelTable,
                                   lbTable, level, action=None,
                                   expectedDestTypes=None ):
   """Recursively traverse L2Rib output tables by following
   nextObj. Optionally perform 'action' at each level by passing
   nextObj, nextObjEntry and level to 'action' before return at each
   level. A valid host destination must terminate for a nextObj whose
   tableType is tableTypeDest. NOTE: This method accesses different
   smash tables. There can be race condition where the nextObj is
   valid but the entry to which it is pointing to in the table no
   longer exists. For this reason, the nextObjEntry is passed in to
   the action method.
   """
   if nextObj.tableType == 'tableTypeInvalid' or level > MAX_LEVEL:
      # This is unexpected but can happen due to context-switching and
      # smash tables getting updated without CLI being aware of it.
      t0( "Unexpected tableType", nextObj.tableType, " or level", level )
      return False
   if nextObj.tableType == 'tableTypeDest':
      # tableTypeDest is the terminating nextObj for our recursion.
      destEntry = destTable.dest.get( nextObj.objId )
      if not destEntry:
         return False
      if action:
         action( nextObj, destEntry, level )
      # If the caller is looking for specific dest types, validate the dest type,
      # otherwise return true if any dest was found
      if expectedDestTypes:
         return destEntry.destType in expectedDestTypes
      else:
         return True
   # Non-terminating table types like label, load-balance will recurse
   # down the tree by incrementing level on each step till they hit a
   # terminating destination.
   if nextObj.tableType == 'tableTypeLabel':
      labelEntry = labelTable.label.get( nextObj.objId )
      if not labelEntry:
         return False
      if action:
         action( nextObj, labelEntry, level )
      return traverseAndValidateMultiTable( labelEntry.next, destTable, labelTable,
                                            lbTable, level+1, action,
                                            expectedDestTypes=expectedDestTypes )
   if nextObj.tableType == 'tableTypeLoadBalance':
      lbEntry = lbTable.lb.get( nextObj.objId )
      if not lbEntry:
         return False
      if isinstance( lbEntry, Tac.Type( "L2Rib::LoadBalanceAndSource" ) ):
         lbEntry = lbEntry.lb
      if action:
         action( nextObj, lbEntry, level )
      for lbNext in lbEntry.next.values():
         if not traverseAndValidateMultiTable( lbNext, destTable, labelTable,
                                               lbTable, level + 1, action,
                                               expectedDestTypes=expectedDestTypes ):
            return False
      return True
   return False
