#!/usr/bin/env python3
# Copyright (c) 2008, 2009 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=superfluous-parens

import Tac
from LacpConstants import * # pylint: disable=wildcard-import
from TableOutput import TableFormatter, Format
from functools import reduce # pylint: disable=redefined-builtin

SubIntfIdType = Tac.Type( "Arnet::SubIntfId" )

def computeLacpState( state ):
   s = ""
   def stateFlag( flag ):
      return( state & flag )
   s += stateFlag( StateActivity ) and "A" or "P"
   s += stateFlag( StateTimeout ) and "S" or "L"
   s += stateFlag( StateAggregation ) and "G" or "I"
   s += stateFlag( StateSynchronization ) and "s+" or "s-"
   s += stateFlag( StateCollecting ) and "C" or ""
   s += stateFlag( StateDistributing ) and "D" or ""
   s += stateFlag( StateDefaulted ) and "d" or ""
   s += stateFlag( StateExpired ) and "X" or ""
   return( s )

def dot43sysid( priority, macAddr ):
   addrStr = macAddr.replace(":","-")
   return( "%04X,%s"%( priority, addrStr )) # pylint: disable=consider-using-f-string

def lagPortId( lpid ):
   # pylint: disable-next=redefined-builtin
   str = "("+dot43sysid( lpid.sysPriority, lpid.sysId ) + ","
   # pylint: disable-next=consider-using-f-string
   str += "%04X,%04X,%04X)"%(lpid.key,lpid.portPriority,lpid.portId)
   return( str )

def lagPortIdLessThan( l1, l2 ):
   return (( l1.sysPriority < l2.sysPriority ) or
           (( l1.sysPriority == l2.sysPriority ) and
            (( l1.sysId < l2.sysId ) or
             (( l1.sysId == l2.sysId ) and
              # Can stop at key.  No need to compare ports, they
              # are always 0 for aggregates.
              ( l1.key < l2.key)))))

def dot43lagId( lagId ):
   if lagPortIdLessThan( lagId.remoteId, lagId.localId ):
      l1 = lagId.remoteId
      l2 = lagId.localId
   else:
      l1 = lagId.localId
      l2 = lagId.remoteId
   return( "["+lagPortId( l1 )+"," + lagPortId( l2 ) + "]" )

def isManagement( intfName ):
   return intfName.startswith( "Management" )

def isInternal( intfName ):
   return intfName.startswith( "Internal" )

def intfSupportsLag( intfName ):
   return ( Tac.Value( 'Arnet::EthIntfId' ).isEthIntfId( intfName ) and
            not SubIntfIdType.isSubIntfId( intfName ) )

# pylint: disable-next=redefined-builtin,inconsistent-return-statements
def printList( list, title, lineLen, returnValue = False ):
   titleLen = len( title )
   tableLen = lineLen - ( titleLen + 1 )
   # More than one line needed
   if ( reduce( lambda tot, item: tot+len( item) +1, list, 0 ) > tableLen ):
      listLengths = [ len(x) for x in sorted( list, key=len, reverse=True ) ]
      width = 0
      columns = 0
      for length in listLengths:
         if width + length + 1 > tableLen - 1:
            break
         width += length + 1
         columns += 1
      t = TableFormatter( tableWidth = lineLen )
      f = Format( justify = 'left' )
      f.padLimitIs( True )
      f.noPadLeftIs( True )
      t.formatColumns( *[f]*(columns+1) )
      # pylint: disable-next=consider-using-enumerate
      for cellCount in range( len( list ) ):
         if cellCount == 0:
            t.startRow( rowFormat=f )
            t.newCell( title )
         elif cellCount % columns == 0:
            t.startRow( rowFormat=f )
            t.newCell( titleLen * ' ' )
         t.newCell( list[ cellCount ] )
      nameList = t.output( ).rstrip( )
   else:
      nameList = title
      for name in list:
         nameList += "%s " % name # pylint: disable=consider-using-f-string
   if returnValue:
      return nameList
   else:
      print( nameList )

