# Copyright (c) 2006, 2007 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""Utility functions for userspace access to legacy device resources.

This module provides utility functions for direct read and write access to system
memory and I/O ports.
"""

import Pci

def _getIoResource( readOnly=False ):
   return Pci.PseudoMmapResource( "/dev/port", readOnly=readOnly )

def _getMemResource( readOnly=False ):
   # /dev/mem needs to use the address value passed in directly when calling seek()
   # on the file - we need ignoreLength to avoid Python trying to reduce it to a
   # valid index range. This fixes the issues experienced when trying to address the
   # upper 2GiB of address space on 32-bit EOS.
   # Note: you must pass check=False to the functions called on this resource.
   return Pci.PseudoMmapResource( "/dev/mem", readOnly=readOnly, ignoreLength=True )

def ioRead8( addr ):
   return _getIoResource( readOnly=True ).read8( addr )

def ioRead32( addr ):
   return _getIoResource( readOnly=True ).read32( addr )

def ioWrite8( addr, value ):
   _getIoResource().write8( addr, value )

def ioWrite32( addr, value ):
   _getIoResource().write32( addr, value )

def memRead8( addr ):
   return _getMemResource( readOnly=True ).read8( addr, check=False )

def memRead32( addr ):
   return _getMemResource( readOnly=True ).read32( addr, check=False )

def memWrite8( addr, value ):
   _getMemResource().write8( addr, value, check=False )

def memWrite32( addr, value ):
   _getMemResource().write32( addr, value, check=False )
