#!/usr/bin/env python3
# Copyright (c) 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import ExtensionMgrLib, PyClient, Tac, Tracing
import optparse, os, re, sys # pylint: disable=deprecated-module

t0 = Tracing.trace0

usage = """
LoadExtensionStatus -f <extension-status-file> -d <extensions-dir> [options]

Populates Sysdb with extension status, including information about all
extension files in the extensions directory and the information in the
extension status file, which is expected to be output from the LoadExtensions
program.
"""

def parseLoadedExtensionFile( lines, extensionConfig, extensionStatus, err ):
   # This code is purposely tolerant of badly formatted data.  Lines that are
   # not properly formatted are ignored.  Each line is expected to contain 9
   # tab-separated columns: name, format, presence, presenceDetail, status,
   # statusDetail, signedStr, installed rpms and mountpoints. 
   # The last two columns are two space-separated
   # lists of rpm names that were installed by this extension
   # and mountpoints mounted by this extension. 
   blankRe = re.compile( "^\\s*$" )
   lineNum = 0
   index = 0

   deps = set() 

   for line in lines:
      lineNum += 1
      if line.startswith( '#' ):
         continue
      if blankRe.match( line ) is not None:
         continue
      m = line.split( '\t' )
      if m is None or len( m ) < 10:
         # pylint: disable-next=consider-using-f-string
         msg = "Parse error on line %d: %s\n" % ( lineNum, line )
         err.write( msg )
         continue
      [ _name, boot, pType, presence, presenceDetail, status,
        statusDetail, signedStr, rpm, mountpoint ] = m
      t0( "Parsed status:", _name, ",", boot, ",", pType, ",",
          presence, ",", presenceDetail, ",", status, ",", statusDetail, ",",
          signedStr, ",", rpm, ",", mountpoint )
      info = ExtensionMgrLib.latestExtensionForName( _name, extensionStatus )
      if not info:
         # pylint: disable-next=consider-using-f-string
         err.write( "No Sysdb state found for extension %s\n" % _name )
         key = Tac.Value( "Extension::InfoKey", _name, 'formatUnknown', 1 )
         info = extensionStatus.info.newMember( key )
         info.presence = 'absent'
         info.presenceDetail = presenceDetail
         info.status = 'notInstalled'
         info.statusDetail = statusDetail
         info.boot = boot
         continue
      # I think it's better to leave presence/presenceDetail alone since the
      # code that populated Sysdb's notion of this state ran later than the
      # code that wrote the status file.
      # info.presence = presence
      # info.presenceDetail = presenceDetail
      info.status = status
      info.statusDetail = statusDetail
      info.boot = boot
      # On bootup, there is no need to restart any agent, so clear those install
      # "side-effects".
      info.agentsToRestart.clear()
      if status in ( 'installed', 'forceInstalled' ):
         item = extensionConfig.installation.newMember( _name, pType, index )
         item.force = bool( status == 'forceInstalled' )
         if signedStr == 'notSigned':
            item.signatureIgnored = True
         index += 1
         ExtensionMgrLib.updateStatus( extensionStatus, info.primaryPkg,
                                       install=True )
      rpms = rpm.split( " " )
      for r in rpms:
         deps.add( r )
         rpmPtr = info.package.get( r )
         if rpmPtr is None:
            # pylint: disable-next=consider-using-f-string
            err.write( "Error on line %d: no rpm %s in extension" % ( lineNum, r ) )
         else:
            info.installedPackage.addMember( rpmPtr )
      
      mountpoints = mountpoint.split()
      for idx, m in enumerate( mountpoints ):
         info.mountpoints[ idx ] = m
         
   for info in extensionStatus.info.values():
      if info.filename in deps:
         if info.status == 'notInstalled':
            info.status = 'installed'
         rpmPtr = info.package.get( info.filename )
         info.installedPackage.addMember( rpmPtr )

class LoadExtensionStatusError( Exception ):
   pass

def loadExtensionStatus( path, extensionConfig, extensionStatus, err ):
   try:
      f = open( path ) # pylint: disable=consider-using-with
      lines = f.readlines()
      parseLoadedExtensionFile( lines, extensionConfig, extensionStatus, err )
   except Exception as e:
      t0( "Failed to load extension status file", path, ":", e )
      raise LoadExtensionStatusError( str( e ) ) # pylint: disable=raise-missing-from

def main( argv, stderr=sys.stderr, stdout=sys.stdout ):
   parser = optparse.OptionParser( usage=usage )
   parser.add_option( "-d", "--dir", action="store",
      help="the extensions directory" )
   parser.add_option( "-f", "--config", action="store",
      help="the config file" )   
   parser.add_option( "-s", "--sysname", action="store", default="ar",
      help="system name (default: %default)" )
   options, args = parser.parse_args( args=argv )
   if len( args ) != 0:
      parser.error( "Unrecognized options: " + " ".join( args ) )
   if not options.config:
      parser.error( "Must specify -f <config file>" )
   if not options.dir:
      parser.error( "Must specify -d <extensions directory>" )

   pc = PyClient.PyClient( options.sysname, "Sysdb" )
   root = pc.agentRoot()
   status = root.entity[ ExtensionMgrLib.statusPath() ]
   config = root.entity[ ExtensionMgrLib.configPath() ]

   if status.initializationStatus != 'notInitialized':
      msg = "Error: extension status has already been initialized."
      raise LoadExtensionStatusError( msg )

   def updateStatus( state, details='' ):
      status.initializationStatus = state 
      status.initializationStatusDetail = details

   updateStatus( 'inProgress' )

   try:
      ExtensionMgrLib.readExtensionsDir( options.dir, status,
         continueOnError=True )
   except Exception as e:
      # pylint: disable-next=consider-using-f-string
      msg = "Error reading extensions directory: %s" % e
      updateStatus( 'failed', msg )
      raise LoadExtensionStatusError( msg ) # pylint: disable=raise-missing-from
   
   try:
      if os.path.exists( options.config ):
         loadExtensionStatus( options.config, config, status, stderr )
   except Exception as e:
      msg = f"Error reading extensions status file {options.config}: {e}"
      updateStatus( 'failed', msg )
      raise LoadExtensionStatusError( msg ) # pylint: disable=raise-missing-from

   updateStatus( 'completed' )

def name():
   return 'LoadExtensionStatus'
