# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import collections
from ctypes import cdll, c_char_p
import linecache
import os
import stat
import sys
import weakref

import BasicCliModes
import BasicCliSession
import CliAaa
import CliCommon
import CliDynamicSymbol
import CliGlobal
import CliPatchTac
import CliPatchSubprocess
import CliSession
from ConfigAgentPluginContext import ConfigAgentPluginContext as Ctx
import Plugins
import Tac
import TacSigint
import Tracing
import Url
import CliExtensions
import SessionUrlUtil as SUU
import WaitForWarmup
import GitLib

th = Tracing.defaultTraceHandle()
t0 = th.trace0
t1 = th.trace1
t2 = th.trace2
t3 = th.trace3
# RunActivities thread tracing:
t4 = th.trace4  # minimal
t5 = th.trace5  # verbose

# Cli plugins loaded in this process:
__CliPluginsLoaded__ = False

ALIAS_TYPE = Tac.Type( 'Cli::Config::Alias' )

libc = cdll.LoadLibrary( "libc.so.6" )
c_getenv = libc.getenv
c_getenv.argtypes = ( c_char_p, )
c_getenv.restype = c_char_p

pluginsLoadedHook = CliExtensions.CliHook()
newCliSessionHook = CliExtensions.CliHook()

class ConfigFile:
   def __init__( self, f, trackLineNo=True ):
      # file_ should always be a generator like object. This is because
      # multiline input will read from the ConfigFile and it should continue
      # reading from where it was previously reading from instead of the beginning
      self.file_ = f if isinstance( f, collections.abc.Iterator ) else iter( f )
      if trackLineNo:
         self.lineNo = 0
      else:
         self.lineNo = None
      self.currentLine = None

   def __iter__( self ):
      for line in self.file_:
         if self.lineNo is not None:
            self.lineNo += 1
            self.currentLine = line
         yield line

   def close( self ):
      # Some tests pass a list of config cmds and not a true file-like object
      if hasattr( self.file_, 'close' ):
         self.file_.close()

class AaaCliConfigReactor( Tac.Notifiee ):
   notifierTypeName = "Cli::AaaCliConfig"

   def __init__( self, notifier ):
      Tac.Notifiee.__init__( self, notifier )
      # BUG1869: since Tac.Notifiee.__init__ doesn't call handlers when
      # in immediate-notification mode, I have to do it.
      self.handleAaaProvider()

   @Tac.handler( 'aaaProvider' )
   def handleAaaProvider( self ):
      # pylint: disable-msg=W0212
      CliAaa._selectProvider( self.notifier_.aaaProvider )

class ConfigLastChangedReactor( Tac.Notifiee ):
   notifierTypeName = "Tac::ConfigCounter"

   def __init__( self, notifier, configHistory, redundancyStatus ):
      Tac.Notifiee.__init__( self, notifier )
      self.configHistory = configHistory
      self.redundancyStatus = redundancyStatus
      # since no config can change if we restart, we don't need to worry about
      # syncing input/output

   @Tac.handler( 'counter' )
   def handleCounter( self ):
      # When under SSO redundancy, ConfigAgent running on the standby supervisor will
      # get incoming ConfigCounter updates from standby Sysdb when the config changes
      # on active Sysdb. Make sure that we don't try to update the timestamp on
      # standby, otherwise we hit a write to read-only exception. If we're using RPR
      # redundancy, it's fine to update the timestamp
      if ( self.redundancyStatus.mode == 'active' or
           self.redundancyStatus.protocol != 'sso' ):
         self.configHistory.runningConfigLastChanged = Tac.now()

class MainCli:
   def __init__( self, entityManager,
                 plugins=None, noPlugins=None, standalone=False,
                 runSessionManager=False,
                 allowActiveMounts=True, redundancyStatus=None, agent=None,
                 callback=None ):
      t1( "Cli agent initialized" )
      CliDynamicSymbol.setEntityManager( entityManager )
      self.pluginsComplete_ = False
      CliPatchTac.init()
      CliPatchSubprocess.init()

      self.configAgentPlugins_ = None
      self.configAgentPluginCtx_ = None
      # In cohab testing, we don't have redundancyStatus from ConfigAgent,
      # so just use the one from the entity manager. This field is used by
      # ConfigAgentPlugins.
      self.redundancyStatus_ = ( redundancyStatus or
                                 entityManager.redundancyStatus() )
      # weakref to Agent object, if set
      self.agent_ = agent

      # things that are mounted
      self.cliConfig_ = None
      self.cliConfigHistory_ = None
      self.aaaCliConfigReactor_ = None
      self.configLastChangedReactor_ = None
      # end things that are mounted

      runSessionManager = runSessionManager or standalone
      # Allow writes - even though writes are allowed initially,
      # some tests start Cli with different sysname multiple times.
      # So we make sure we keep writes enabled during initialization.
      CliGlobal.enableWrite( True )

      def _sessionMounted():
         willRunSessionMgr = runSessionManager or standalone
         self.loadPlugins( entityManager,
                           plugins, noPlugins,
                           allowActiveMounts,
                           callback=callback,
                           redundancyStatus=redundancyStatus,
                           runSessionManager=willRunSessionMgr )

      # These are required for loading CliPlugin (ConfigMounts)
      CliSession.doMounts( entityManager, bootstrap=False,
                           block=False, callback=_sessionMounted )

   def redundancyStatus( self ):
      return self.redundancyStatus_

   def loadPlugins( self, entityManager, plugins=None, noPlugins=False,
                    allowActiveMounts=True, callback=None,
                    runSessionManager=False, redundancyStatus=None ):
      if not entityManager:
         return

      t0( "loading plugins" )

      loadingPlugins = False

      # delay mountGroup for plugins until after CliSession does its own
      # mounts
      mountGroup = entityManager.mountGroup()
      self.cliConfig_ = mountGroup.mount( "cli/config", "Cli::Config", "wi" )
      self.cliConfigHistory_ = mountGroup.mount( "cli/configHistory",
                                                 "Cli::ConfigHistory", "w" )
      aaaCliConfig = mountGroup.mount( "cli/input/aaa", "Cli::AaaCliConfig", "r" )

      self.loadConfigAgentPlugins( entityManager, mountGroup )
      self.loadAgentPlugins( entityManager )
      # We want to skip loading plugins more than once in our breadth tests
      if not ( __CliPluginsLoaded__ and
               __CliPluginsLoaded__() == entityManager ):
         if __CliPluginsLoaded__:
            t0( "Reloading CliPlugins for", entityManager )
         t3( "Loading cli plugins" )
         if not noPlugins:
            loadingPlugins = True
            Plugins.loadPlugins( "CliPlugin", context=entityManager,
                                 plugins=plugins )
            CliDynamicSymbol.maybeLoadAllDynamicCliPlugins() # in debug cases
            Plugins.loadPlugins( "CliShellPlugin" )
            # CliPlugins register show tech commands.
            # Under the covers this calls extract_stack, which populates.
            # linecache as a side-effect.
            # Drop the cache, we don't really need it after the stack extraction.
            linecache.clearcache()
         Url.setEntityManager( entityManager )

      def finish():
         global __CliPluginsLoaded__
         if loadingPlugins:
            __CliPluginsLoaded__ = weakref.ref( entityManager )
            t3( "cli plugins loaded" )

         self.aaaCliConfigReactor_ = AaaCliConfigReactor( aaaCliConfig )
         self.configLastChangedReactor_ = ConfigLastChangedReactor(
            Tac.singleton( 'Tac::ConfigCounter' ),
            self.cliConfigHistory_,
            self.redundancyStatus_ )

         def saveSessionMountsCompleted():

            def cleanConfigCompleted():
               # start all ConfigAgentPlugin statemachines when CLI has started
               self.configAgentPluginCtx_.startAllStateMachines()

               t3( 'clean-config created' )

               CliGlobal.enableWrite( False )

               t3( 'saveSessionMountsCompleted - enter' )
               self.pluginsComplete_ = True
               t3( 'plugins marked as completed' )
               GitLib.maybeCreateGitRepo( entityManager.sysname() )

               if callback:
                  callback( self )

            if runSessionManager:
               t3( 'starting CliSession Cohab Agent' )
               CliSession.startCohabitingAgent( entityManager,
                                                redundancyStatus=redundancyStatus,
                                                block=False,
                                                callback=cleanConfigCompleted )
            else:
               cleanConfigCompleted()

         pluginsLoadedHook.notifyExtensions( entityManager )
         SUU.doMountsForSaveSession( entityManager, CliSession.sessionStatus,
                                     saveSessionMountsCompleted )

      mountGroup.close( finish )

   def pluginsComplete( self ):
      t3( 'pluginsComplete - status:', self.pluginsComplete_ )
      return self.pluginsComplete_

   def activityLockMonitorEnabled( self ):
      return self.cliConfig_.activityLockSyslogMsg

   def loadConfigAgentPlugins( self, entityManager, mountGroup ):
      t3( "loading ConfigAgentPlguins" )
      # Get weak reference to self to avoid circular referencing
      self.configAgentPluginCtx_ = Ctx( entityManager.sysname(),
                                        entityManager, mountGroup,
                                        weakref.proxy( self ) )
      self.configAgentPlugins_ = Plugins.loadPlugins(
         'ConfigAgentPlugin',
         context=self.configAgentPluginCtx_ )

   def loadAgentPlugins( self, entityManager ):
      t3( "loading AgentPlugins" )
      # load the AgentGroupPlugins so that we know which all
      # agent plugins need to be loaded.
      pluginGroups = []

      Plugins.loadPlugins( "AgentGroupPlugin", context=pluginGroups )

      if not pluginGroups:
         # return if no pluginGroups need to be loaded
         return

      # There will not be need to instantiate agentPluginContext
      # when Cli derives from Agent as these are member variables
      # of CAgent class
      agentPluginContext = Tac.newInstance( "Agent::Plugin::Context" )
      agentPluginLoader = Tac.newInstance( "Agent::Plugin::Loader" )

      localDir = "localAgentPlugin"
      entityManager.root().parent.newEntity( 'Tac::Dir', localDir )
      entityManager.localRootPathIs( "/" + entityManager.sysname() +
             "/" + localDir + "/" )
      agentPluginContext.agentName = "ConfigAgent"
      agentPluginContext.entityManager = entityManager.cEntityManager()
      agentPluginContext.createAllLocalEntities = True

      # load plugins
      for group in pluginGroups:
         agentPluginLoader.loadPluginGroup( 'AgentPlugin', group,
               agentPluginContext, "", "" )

   def loadDynamicAliases( self ):
      """ Takes all of the executable scripts in session.cliConfig.commandsPath
      and turns them into aliases of the form 'alias foo bash /path/to/foo'. These
      aliases are not persistent. """

      # Clear existing dynamic aliases.
      self.cliConfig_.dynamicAlias.clear()

      # Check to see if path exists and is valid in our file system.
      path = self.cliConfig_.commandsPath
      try:
         if os.path.isdir( path ):
            scripts = os.listdir( path )

            # Turn each script into an alias.
            for script in scripts:
               # Do not add 'alias' as a dynamic alias.
               if script == "alias":
                  continue

               fullPath = os.path.join( path, script )

               # Only add scripts that are executable, so that we avoid
               # unnecesary clutter of the alias list.
               if ( os.path.exists( fullPath )
                    and stat.S_IXUSR & os.stat( fullPath )[ stat.ST_MODE ] ):
                  cmd = 'bash ' + fullPath
                  dynamicAlias = ALIAS_TYPE( script )
                  # Cannot use 0 as index because of a Tacc bug.
                  dynamicAlias.originalCmd[ 1 ] = cmd
                  self.cliConfig_.dynamicAlias[ script ] = dynamicAlias
      except ( AttributeError, TypeError, OSError ):
         import traceback # pylint: disable=import-outside-toplevel
         exc = traceback.format_exc()
         print( exc )

configUpgradeCallbacks = CliExtensions.CliHook()

def registerConfigUpgradeCallback( callback ):
   configUpgradeCallbacks.addExtension( callback )

def loadConfigFromFile( f, entityManager,
                        initialModeClass=BasicCliModes.GlobalConfigMode,
                        privLevel=CliCommon.DEFAULT_PRIV_LVL,
                        disableAaa=False,
                        disableGuards=False,
                        skipConfigCheck=False,
                        isEapiClient=False,
                        startupConfig=False,
                        secureMonitor=False,
                        aaaUser=None,
                        autoComplete=True,
                        abortOnError=False,
                        echo='' ):
   """Loads a sequence of CLI commands from a URL, ignoring any errors that
   occur in individual lines of the file."""
   BasicCliSession.drainReadlineWorkQueue()

   # if echo, we don't print line number.
   configFile = ConfigFile( f, not bool( echo ) )

   # Note that if this call is made from the context of a current CLI session,
   # we do not re-use the current session but create a new one.  This isolates
   # the current session from changes to the current mode caused by the commands
   # in the config file.
   session = BasicCliSession.Session( initialModeClass,
                                      entityManager,
                                      privLevel=privLevel,
                                      configFile=configFile,
                                      disableAaa=disableAaa,
                                      disableAutoMore=True,
                                      disableGuards=disableGuards,
                                      startupConfig=startupConfig,
                                      secureMonitor=secureMonitor,
                                      skipConfigCheck=skipConfigCheck,
                                      autoComplete=autoComplete,
                                      standalone=entityManager.isLocalEm(),
                                      isEapiClient=isEapiClient,
                                      aaaUser=aaaUser )

   newCliSessionHook.notifyExtensions( session )

   # Since the top-level mode for the new session may be GlobalConfigMode, not
   # UnprivMode, the "end" or "exit" commands, if they were to appear in the
   # config file, could cause us to "fall off" the root of the mode tree.  To
   # prevent this, we set the top-level mode's parent to point to itself, so
   # that the "end" or "exit" commands leave the session in the top-level mode.
   # Note that the session's current mode might not be the top-level mode in
   # some cases, so we must find the top mode.
   topMode = session.mode
   while topMode.parent_ is not None:
      topMode = topMode.parent_
   topMode.parent_ = topMode

   errors = 0
   # cli cmd 'trace <agent> setting <facility/level>' writes to C's environ.
   # Python's environ started with a copy of it, but has a life of its own later.
   c_trace = c_getenv( b"TRACE" )
   # this is a hack to minimize tracing overhead when it's not enabled
   savedTraceFunc = Tracing._traceLineSuppressed # pylint: disable-msg=W0212
   if not c_trace:
      Tracing._traceLineSuppressed = lambda x, y: True # pylint: disable-msg=W0212
   try: # pylint: disable-msg=too-many-nested-blocks
      firstLine = True
      for line in configFile:
         if firstLine:
            firstLine = False
            if line.startswith( '#!' ):
               continue
         if echo == 'E':
            # pylint: disable-next=consider-using-f-string
            print( '[%s]' % Tac.now(), session.mode.prompt + line.strip( "\n" ) )
            sys.stdout.flush()
         elif echo == 'e':
            print( session.mode.prompt + line.strip( "\n" ) )
            sys.stdout.flush()
         try:
            session.runCmd( line, expandAliases=True )
            if session.errors_:
               errors += 1
               if abortOnError:
                  raise CliCommon.LoadConfigError()
            TacSigint.check()
         except CliCommon.LoadConfigError:
            raise
         except: # pylint: disable-msg=W0702
            errors += 1
            session.handleCliException( sys.exc_info(), line,
                                        lineNo=configFile.lineNo,
                                        handleKeyboardInterrupt=False )
            if abortOnError:
               # pylint: disable-next=raise-missing-from
               raise CliCommon.LoadConfigError()

      if session.mode != topMode:
         # For self-constructed config files, people may have forgotten
         # the final 'end' command. This may cause group-change modes
         # to not actually commit the changes. We detect it and automatically
         # add the 'end' command.
         try:
            while session.mode != topMode:
               session.gotoParentMode()
         except: # pylint: disable-msg=W0702
            errors += 1
            session.handleCliException( sys.exc_info(), 'end' )

      if startupConfig:
         configUpgradeCallbacks.notifyExtensions( session )

      if not entityManager.isLocalEm():
         # We need to make sure that we don't exit until our attrlog has flushed to
         # Sysdb. This ensures that LoadConfig does not set /ar/Sysdb/Sysdb/status.
         # startupConfigStatus=completed too early (when CliShell exits).
         WaitForWarmup.wait( entityManager, [ 'Sysdb' ], sleep=True )

   except OSError as e:
      errors += 1
      if hasattr( f, "name" ):
         session.addError( f"Error loading file {f.name} ({e.strerror})" )
      else:
         # pylint: disable-next=consider-using-f-string
         session.addError( "Error (%s)" % e.strerror )
      # We can land here because a cli command handler did not catch an IOError (say
      # because of EPERM). Since 'config replace' will not see the returned error
      # count, make sure we at least raise an exception.
      if abortOnError:
         raise CliCommon.LoadConfigError( 'Failed to load config file' )
   finally:
      Tracing._traceLineSuppressed = savedTraceFunc # pylint: disable-msg=W0212

      # GlobalConfigMode.onExit would lead to generation of config-man traps
      # if session.configHistory is set.
      session.configHistory = None
      session.mode.onExit()

      # Explicitly close to speed up freeing resources instead of waiting for GC
      session.close()

   return errors

def loadConfig( f,
                session,
                initialModeClass=BasicCliModes.GlobalConfigMode,
                privLevel=None,
                disableAaa=None,
                disableGuards=None,
                skipConfigCheck=None,
                isEapiClient=None,
                startupConfig=None,
                secureMonitor=None,
                aaaUser=None,
                autoComplete=None,
                abortOnError=None,
                echo='' ):
   # Similar to loadConfigFromFile but default to the current session flags.
   # This is easier to use.
   if privLevel is None:
      privLevel = session.privLevel_
   if disableAaa is None:
      disableAaa = session.disableAaa_
   if disableGuards is None:
      disableGuards = session.disableGuards_
   if isEapiClient is None:
      isEapiClient = session.isEapiClient()
   if skipConfigCheck is None:
      skipConfigCheck = session.skipConfigCheck()
   if startupConfig is None:
      startupConfig = session.startupConfig()
   if secureMonitor is None:
      secureMonitor = session.secureMonitor()
   if aaaUser is None:
      aaaUser = session.aaaUser()
   if autoComplete is None:
      autoComplete = session.autoComplete_
   return loadConfigFromFile( f, session.entityManager,
                              initialModeClass=initialModeClass,
                              privLevel=privLevel,
                              disableAaa=disableAaa,
                              disableGuards=disableGuards,
                              skipConfigCheck=skipConfigCheck,
                              isEapiClient=isEapiClient,
                              startupConfig=startupConfig,
                              secureMonitor=secureMonitor,
                              aaaUser=aaaUser,
                              autoComplete=autoComplete,
                              abortOnError=abortOnError,
                              echo=echo )
