# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
from ApiBaseModels import BaseModel, Str, Bool, Int

class ApiSettingsModel( BaseModel ):
   apiVersion = Str( inputOk=False, description='Current version of MCS API' )
   apiTitle = Str( inputOk=False, description='Title of MCS API' )
   apiDescription = Str( inputOk=False, description='Description of MCS API' )

   # pylint: disable-msg=arguments-differ
   def fromSysdb( self, apiSettings ): # pylint: disable=arguments-renamed
      self.apiVersion = apiSettings.apiVersion
      self.apiTitle = apiSettings.apiTitle

class RPModel ( BaseModel ):
   deviceId = Str( apiName='chassis-id', description="Device MACId" )
   intfName = Str( apiName='interface-name', description="Interface Name" )
   rp = Str( apiName='reservation-percent', description="Reservation Percentage" )

   # pylint: disable-msg=arguments-differ
   def toSysdb( self, apiConfig ): # pylint: disable=arguments-renamed
      rpKey = Tac.Value( "Mcs::DeviceAndIntfId", self.deviceId, self.intfName )
      apiConfig.newReservationPercentage( rpKey )
      apiConfig.reservationPercentage[ rpKey ].percentValue = (
        round( float( self.rp ), 2 )
      )

class OuiModel ( BaseModel ):
   oui = Int( description="OUI Id" )
   vendorName = Str( description="Vendor Name" )

   # pylint: disable-msg=arguments-differ
   def toSysdb( self, apiConfig ): # pylint: disable=arguments-renamed
      oui = apiConfig.vendor.newMember( Tac.Value( "Mcs::Oui",
                                        self.oui ) )
      oui.vendorName = self.vendorName


class AgentStatusModel( BaseModel ):
   enabled = Bool( inputOk=False, description='Status of MCS Agent' )

   # pylint: disable-msg=arguments-differ

   def fromSysdb( self, status ): # pylint: disable=arguments-renamed
      self.enabled = status.enabled

class ApiStatusModel( BaseModel ):
   # Represents Same API state as Agent Status for now.
   # Can be expanded to represent holistic API status including agent status, cvx
   # client status, etc
   enabled = Bool( inputOk=False, description='Status of MCS API' )
   syncStatus = Str( description='Status sync' )

   # pylint: disable-msg=arguments-differ

   def fromSysdb( self, status ): # pylint: disable=arguments-renamed
      self.enabled = status.enabled
      self.syncStatus = status.syncInProgress

