# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import BothTrace
import Tac
import re
from McsHttpErrorCodes import errorCodes
import Toggles.McsToggleLib
from TypeFuture import TacLazyType

log = BothTrace.tracef0
warn = BothTrace.tracef1
error = BothTrace.tracef2
info = BothTrace.tracef3
debug = BothTrace.tracef4

ROUTE_OWNER_MCS = 'mcs'

VENDOR_NAME = 'Arista Networks'
MCS_BOUNDARY_INTF = 'McsBoundaryIntf'

FS = Tac.Type( 'Mcs::FlowProgramStatus' )

def errorTemplate( ec ):
   return {}

EthSpeed = Tac.Type( 'Interface::EthSpeed' )
EthSpeedApi = TacLazyType( "Interface::EthSpeedApi" )
DevAndIntfType = Tac.Type( "Mcs::DeviceAndIntfId" )
ChStateType = Tac.Type( 'Mcs::ApiConfig::ConsistentHashState' )

def get_bandwidth_num( bandwidthStr ):
   """ Convert bandwidth string to number
   """
   # returns bandwidth in kbits/second
   if hasattr( EthSpeed, bandwidthStr ):
      return EthSpeedApi.speedKbps( bandwidthStr )
   return 0

def isMcastIpAddr( addr ):
   regMulGrp = r'^((22[4-9]|23\d)){1}(\.(25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)){3}$'
   return re.match( regMulGrp, addr ) is not None

def validateSourceIP( addr ):
   '''
   Validate of Source IP is within acceptible Unicast source IP ranges
   we are also excluding 0.0.0.0 as valid unicast IP
   '''
   if not addr or not isinstance( addr, str ):
      ec = '127'
      error( errorCodes[ ec ] )
      return False, ec

   regx = r'^(([0-9]|[1-9][0-9]|1[0-9][0-9]|2[0-4][0-9]|25[0-5])\.){3}'\
           '([0-9]|[1-9][0-9]|1[0-9][0-9]|2[0-4][0-9]|25[0-5])$'
   if re.match( regx, addr ) is not None:
      if not isMcastIpAddr( addr ) and addr != '255.255.255.255'\
         and addr != '0.0.0.0':
         return True, ''
      ec = '125'
      error( errorCodes[ ec ].replace( '<address>', addr ) )
      return False, ec
   ec = '127'
   error( errorCodes[ ec ] )
   return False, ec

def validateMcastGroup( addr ):
   '''
      Validate if Multicast destIp mentioned is within acceptible
      multicast range 224.0.0.0-239.255.255.255
   '''
   if not addr or not isinstance( addr, str ):
      ec = '129'
      error( errorCodes[ ec ] )
      return False, ec
   if isMcastIpAddr( addr ):
      return True, ''
   ec = '126'
   error( errorCodes[ ec ].replace( '<address>', addr ) )
   return False, ec

def findMissingKeys( expected_keys, flow_keys ):
   missingKeys = expected_keys - set( flow_keys )
   if missingKeys:
      ec = '131'
      error( errorCodes[ ec ].replace( '<key>', str( missingKeys ) ) )
      return ec, missingKeys
   return '', missingKeys

def validateBandwidthType( bwType ):
   return bwType in [ 'k', 'm', 'g' ]


def validateClearRoutes( data ):
   allClear = False
   routesToClear = []
   isValid = True
   errorCode = ''
   if data.get( 'ipRoutes' ):
      if data[ 'ipRoutes' ].get( 'all' ):
         allClear = True
         routesToClear = []
      else:
         allClear = False
         routes = data[ 'ipRoutes' ].get( 'selected' )
         if routes:
            for r in routes:
               if not validateSourceIP( r[ 'sourceIP' ] )[ 0 ]:
                  errorCode = validateSourceIP( r[ 'sourceIP' ] )[ 1 ]
                  isValid = False
               elif not validateMcastGroup( r[ 'destinationIP' ] )[ 0 ]:
                  isValid = False
                  errorCode = validateMcastGroup( r[ 'destinationIP' ] )[ 1 ]
               else:
                  routesToClear.append( ( r[ 'sourceIP' ], r[ 'destinationIP' ] ) )
         else:
            isValid = False
            errorCode = "131"
   else:
      isValid = False
      errorCode = "131"
   return isValid, allClear, routesToClear, errorCode

def validateOui( data ):
   oui = data.get( 'oui' )
   vendor = data.get( 'vendorName' )
   if not oui or not vendor:
      return False, ''

   if not isinstance( vendor, str ) or not isinstance( oui, str ):
      return False, ''

   dlim = ''
   if '.' in oui:
      dlim = '.'

   if '-' in oui:
      dlim = '-'

   if ':' in oui:
      dlim = ':'

   if dlim:
      oui = oui.replace( dlim, '' )

   if len( oui ) > 6:
      return False, ''
   elif not oui.isalnum():
      return False, ''
   else:
      return True, oui

def get_dotted_mac( mac ):
   '''
   Parse X.X.X, X-X-X-X or X:X:X:X mac addresses to X.X.X
   '''
   if not mac:
      return ''
   regx = r'^([0-9A-Fa-f]{2}[:-]){5}([0-9A-Fa-f]{2})|([0-9a-fA-F]{4}\.'\
          r'[0-9a-fA-F]{4}\.[0-9a-fA-F]{4})$'
   if re.match( regx, mac ):
      if ':' in mac:
         rgtmp = mac.split( ':' )
         return rgtmp[ 0 ] + rgtmp[ 1 ] + '.' + rgtmp[ 2 ] + rgtmp[ 3 ] + '.' +\
               rgtmp[ 4 ] + rgtmp[ 5 ]
      elif '-' in mac:
         rgtmp = mac.split( '-' )
         return rgtmp[ 0 ] + rgtmp[ 1 ] + '.' + rgtmp[ 2 ] + rgtmp[ 3 ] + '.' +\
               rgtmp[ 4 ] + rgtmp[ 5 ]
      else:
         return mac

   return ''



def get_colon_mac( mac ):
   '''
   Parse X.X.X, X-X-X-X or X:X:X:X mac addresses to X:X:X:X
   '''
   if not mac:
      return ''
   regx = r'^([0-9A-Fa-f]{2}[:-]){5}([0-9A-Fa-f]{2})|([0-9a-fA-F]{4}\.'\
           r'[0-9a-fA-F]{4}\.[0-9a-fA-F]{4})$'
   if re.match( regx, mac ):
      if '-' in mac:
         return mac.replace( '-', ':' )
      elif '.' in mac:
         tmp = mac.replace( '.', '' )
         tmp = ':'.join( tmp[ i : i + 2 ] for i in range( 0, 12, 2 ) )
         return tmp
      else:
         return mac

   return ''

def isSameMac( mac1, mac2 ):
   return get_dotted_mac( mac1 ) == get_dotted_mac( mac2 )

def validateBandwidth( bw ):
   return isinstance( bw, int ) and bw >= 0 and bw.bit_length() <= 32

def replaceAll( text, wordMap ):
   for key, value in wordMap.items():
      text = text.replace( key, str( value ) )
   return text

def validatePort( port ):
   port = port.lower().capitalize()
   regx = r'Ethernet(\d+|\d+\/\d+|\d+\/\d+\/\d+)$|Vlan\d+$'
   if not re.match( regx, port ):
      return False
   return True

def checkMissingKeyAndValidateRpValue( data ):
   expected_keys = [ 'chassis-id', 'interface-name', 'reservation-percent' ]
   missingKeys = set( expected_keys ) - set( data )
   if missingKeys:
      ec = '131'
      error( errorCodes[ ec ].replace( '<key>', str( missingKeys ) ) )
      return False, ec
   try:
      rp = float( data[ 'reservation-percent' ] )
   except ValueError:
      ec = '139'
      error( errorCodes[ ec ] )
      return False, ec
   if not( rp >= 0.0 and rp <= 1.0 ): # pylint: disable=chained-comparison
      ec = '138'
      error( errorCodes[ ec ] )
      return False, ec
   return True, ''

def rangeToCanonicalIntfs( intfName ):
   regex1 = r'Ethernet?(?P<start>\d+)-(?P<end>\d+)$'
   regex2 = r'Ethernet?(?P<module>\d+)\/(?P<start>\d+)-(?P<end>\d+)$'
   regex3 =  \
         r'Ethernet?(?P<module>\d+)\/(?P<start>\d+)-(?P<end>\d+)\/(?P<lane>\d+)$'
   intfList = []
   startName = 'Ethernet'
   if re.match( regex1, intfName ):
      m = re.match( regex1, intfName )
      rangeStart = m.group( 'start' )
      rangeEnd = m.group( 'end' )
      for x in range( int( rangeStart ), int( rangeEnd ) + 1 ):
         intfList.append( startName + str( x ) )
   elif re.match( regex2, intfName ):
      m = re.match( regex2, intfName )
      moduleNo = m.group( 'module' )
      rangeStart = m.group( 'start' )
      rangeEnd = m.group( 'end' )
      for x in range( int( rangeStart ), int( rangeEnd ) + 1 ):
         intfList.append( startName + moduleNo + '/' + str( x ) )
   elif re.match( regex3, intfName ):
      m = re.match( regex3, intfName )
      moduleNo = m.group( 'module' )
      laneNo = m.group( 'lane' )
      rangeStart = m.group( 'start' )
      rangeEnd = m.group( 'end' )
      for x in range( int( rangeStart ), int( rangeEnd ) + 1 ):
         intfNameTmp = startName + moduleNo + '/' + str( x )
         intfNameTmp += '/' + laneNo
         intfList.append( intfNameTmp )
   return intfList

def validateRpChassisId( data, mcsMounts ):
   devices = []
   if data[ 'chassis-id' ] == 'all':
      for deviceId in mcsMounts.switchStatusDir:
         devices.append( deviceId )
      return [], devices
   # pylint: disable-next=unnecessary-comprehension
   devices.extend( [ dev for dev in data[ 'chassis-id' ].split( ',' ) ] )
   validDevices = devices[ : ]
   ecList = set()
   for deviceId in devices:
      tDeviceId = get_colon_mac( deviceId ).replace( ':', '-' )
      if not tDeviceId or tDeviceId not in mcsMounts.switchStatusDir:
         ec = '137'
         validDevices.remove( deviceId )
         ecList.add( ec )
         error( errorCodes[ ec ].replace( '<deviceId>', deviceId ) )
   return list( ecList ), validDevices

def validateRpInterfaceId( data, mcsMounts, deviceId ):
   interfacesList = []
   cdbDev = get_colon_mac( deviceId ).replace( ':', '-' )
   mcsClientStatus = mcsMounts.switchStatusDir.get( cdbDev )
   if data[ 'interface-name' ] == 'all':
      if mcsClientStatus:
         for intf in mcsClientStatus.intfSpeed:
            if intf.startswith( 'Ethernet' ):
               interfacesList.append( intf )
      return [], interfacesList
   if not data[ 'interface-name' ]:
      ec = '156'
      error( errorCodes[ ec ] )
      return [ ec ], []
   # pylint: disable-next=unnecessary-comprehension
   interfacesList.extend( [ intf for intf in data[ 'interface-name' ].
         split( ',' ) ] )
   validIntfList = []
   ecList = set()
   for intf in interfacesList:
      if not intf.startswith( 'Ethernet' ):
         ec = '156'
         ecList.add( ec )
         error( errorCodes[ ec ] )
         continue
      tList = []
      if '-' in intf:
         tList = rangeToCanonicalIntfs( intf )
      else:
         tList.append( intf )
      if not tList:
         ec = '233'
         ecList.add( ec )
         error( errorCodes[ ec ].replace( '<deviceId>', deviceId ) )
         continue
      for intfId in tList:
         if intfId not in mcsClientStatus.intfSpeed:
            ec = '231'
            error( errorCodes[ ec ].replace( '<interface-name>', intfId ).replace(
               '<deviceId>', deviceId ) )
            ecList.add( ec )
            continue
         validIntfList.append( intfId )
   return list( ecList ), validIntfList

def validateIntfNotInBoundaryIntf( apiConfig, devIntfId, name='' ):
   dev, intf = devIntfId.split( '-' )
   for key, value in apiConfig.boundaryInterface.items():
      if DevAndIntfType( dev, intf ) in value.interface:
         if name and name == key:
            continue
         ec = '257'
         msg = errorCodes[ ec ].replace( '<intfId>', devIntfId )
         error( msg.replace( '<bIntf>', key ) )
         return False, ec
   return True, ''

def validateDeviceID( apiConfig, interfaceID, name='' ):
   """
   Validates if we received a valid ingress interface ID.
   Validity determined on expected format <deviceId>-<port>, precense of macID and
   valid macID for device, precense of port and valid port(EthernetXX/vlanXX)
   Also validate if given interfaceID is already part of certain boundary interface
   """
   if interfaceID and isinstance( interfaceID, str ):
      device, sep, intfId = interfaceID.partition( '-' )
      if sep:
         if not device:
            ec = '132'
            error( errorCodes[ ec ] )
            return False, ec, '', ''
         if not intfId:
            ec = '133'
            error( errorCodes[ ec ] )
            return False, ec, '', ''
         device = get_colon_mac( device )
         if not device:
            ec = '137'
            error( errorCodes[ ec ].replace( '<deviceId>', interfaceID.split( '-'
               )[ 0 ] ) )
            return False, ec, '', ''
         if not validatePort( intfId ):
            ec = '184'
            error( errorCodes[ ec ] )
            return False, ec, '', ''
      else:
         if ( Toggles.McsToggleLib.toggleMcsBoundaryIntfEnabled() and
               interfaceID.isalnum() ):
            # This will be executed only when interfaceID is boundary interface
            # and is only called to validate boundary interface name from POST
            # API of add/modify/del sender/receiver
            valid, ec = validateBoundaryIntfName( interfaceID )
            if valid:
               if interfaceID not in apiConfig.boundaryInterface:
                  ec = '252'
                  valid = False
                  error( errorCodes[ ec ].replace( '<bIntf>', interfaceID ) )
            return valid, ec, '', ''
         ec = '183'
         error( errorCodes[ ec ] )
         return False, ec, '', ''
   else:
      ec = '182'
      error( errorCodes[ ec ] )
      return False, ec, '', ''

   # Validating deviceID, if it part of boundary interface
   valid, ec = validateIntfNotInBoundaryIntf( apiConfig, interfaceID, name )
   if not valid:
      return False, ec, '', ''
   return True, "", device, intfId

def validateLabel( label ):
   if len( label ) > 253:
      return False, '232'
   else:
      return True, ''

def validateTc( tc ):
   if int( tc ) < 0 or int( tc ) > 7:
      return False, '237'
   else:
      return True, ''

def validateDscp( dscp ):
   if int( dscp ) < 0 or int( dscp ) > 63:
      return False, '236'
   else:
      return True, ''

def validateApplyPolicy( applyPolicy ):
   if not isinstance( applyPolicy, bool ):
      return False, '239'
   else:
      return True, ''

def validateBoundaryIntfName( name ):
   if ( not isinstance( name, str ) or not name.startswith( 'McsBoundaryIntf' ) or
         not name.isalnum() or len( name ) > 64 ):
      ec = '249'
      error( errorCodes[ ec ].replace( '<name>', str( name ) ) )
      return False, ec
   return True, ''

def get_flow_programmed_template( bw, destinationIP, deviceId, iif, oifs, sourceIP ):
   return {
            "activityState": False,
            "activityTime": "",
            "deviceActive": True,
            "bandwidth": bw,
            "destinationIP": destinationIP,
            "deviceId": deviceId,
            "iif": iif,
            "oifs": oifs,
            "sourceIP": sourceIP
        }

def getFlowBwProgram( agentStatus ):
   flows = []
   bwProgrammed = {}
   for mcastKey in agentStatus.flowProgrammed:
      bwValue = agentStatus.flowProgrammed[ mcastKey ].bw.value
      devices = agentStatus.flowProgrammed[ mcastKey ].device

      for deviceValue in devices.values():
         deviceId = get_dotted_mac( deviceValue.ethAddr )
         if deviceId not in bwProgrammed:
            bwProgrammed[ deviceId ] = {}

         iif = deviceValue.iif
         if iif not in bwProgrammed[ deviceId ]:
            bwProgrammed[ deviceId ][ iif ] = { 'rxTotal': 0, 'txTotal': 0 }
         bwProgrammed[ deviceId ][ iif ][ 'rxTotal' ] += bwValue

         oifs = []
         for oif in deviceValue.oif:
            if oif not in bwProgrammed[ deviceId ]:
               bwProgrammed[ deviceId ][ oif ] = { 'rxTotal': 0, 'txTotal': 0 }
            bwProgrammed[ deviceId ][ oif ][ 'txTotal' ] += bwValue
            oifs.append( oif )
         flows.append( get_flow_programmed_template( bwValue, mcastKey.group,
                                             deviceId, iif, oifs, mcastKey.source ) )

   return flows, bwProgrammed

def getImpactedReasons( agentStatus, mcastKey, device, intf ):
   reasons = []
   impRcv = agentStatus.impactedRcv.get( mcastKey )
   if impRcv:
      # Collect impacted receivers by device
      for impRcvDev in impRcv.rcvsByDev.values():
         if ( impRcvDev.receivers.get( device ) and
              impRcvDev.receivers[ device ].recvIntfs.get( intf ) ):
            reasons.append( 'device down ' + get_dotted_mac( impRcvDev.device ) )

      # Collect impacted receivers by link
      for impRcvLink in impRcv.rcvsByLink.values():
         if ( impRcvLink.receivers.get( device ) and
              impRcvLink.receivers[ device ].recvIntfs.get( intf ) ):
            sDev = get_dotted_mac( impRcvLink.link.self )
            sIntf = impRcvLink.link.selfIntf
            nDev = get_dotted_mac( impRcvLink.link.neighbor )
            nIntf = impRcvLink.link.neighborIntf
            if not sIntf:
               reasons.append( f'interface down {nDev}-{nIntf}' )
            elif not nIntf:
               reasons.append( f'interface down {sDev}-{sIntf}' )
            else:
               reasons.append( f'link down {sDev}-{sIntf} {nDev}-{nIntf}' )
   if Toggles.McsToggleLib.toggleDeviceMaintenanceEnabled():
      devSg = Tac.Value( "Mcs::DeviceSg", device, mcastKey )
      for maintDev, mds in agentStatus.maintenanceDeviceStatus.items():
         if devSg in mds.maintImpRcv:
            maintDev = get_dotted_mac( maintDev )
            reasons.append( f'maintenance {maintDev}' )
   return reasons

def verifyKeysInFilters( filterData ):
   ec = ''
   validKeys = [ 'sourceIP', 'destinationIP', 'deviceId', 'impacted', 'maintenance' ]
   if diffInKey := set( filterData ).difference( validKeys ):
      error( f"Invalid key {diffInKey} in requests" )
      ec = '187'
   return ec

def getStoredReceivers( apiConfig, agentStatus, filters ):
   storedReceivers = {}
   messages = []

   if not isinstance( filters, list ):
      error( f"Invalid request with {filters}" )
      messages.append( '187' )
      return storedReceivers, messages

   def getReceiverBoundaryIntf( key, devIntf ):
      if ( Toggles.McsToggleLib.toggleMcsBoundaryIntfEnabled() and
            mcastKey in apiConfig.boundaryReceiver ):
         for bIntfName in apiConfig.boundaryReceiver[ key ].receiver:
            if devIntf in apiConfig.boundaryInterface[ bIntfName ].interface:
               return bIntfName
      return ""

   #pylint: disable-msg=too-many-nested-blocks
   for data in filters:
      if ec := verifyKeysInFilters( data ):
         messages.append( ec )
         continue

      srcFilter = data.get( 'sourceIP' )
      if srcFilter:
         _, ec = validateSourceIP( srcFilter )
         if ec:
            if ec not in messages:
               messages.append( ec )
            return storedReceivers, messages

      grpFilter = data.get( 'destinationIP' )
      if grpFilter:
         _, ec = validateMcastGroup( grpFilter )
         if ec:
            if ec not in messages:
               messages.append( ec )
            return storedReceivers, messages

      deviceFilter = None
      deviceId = data.get( 'deviceId' )
      if deviceId:
         deviceFilter = get_colon_mac( deviceId )
         if not deviceFilter:
            ec = '137'
            if ec not in messages:
               messages.append( ec )
            return storedReceivers, messages

      if Toggles.McsToggleLib.toggleDeviceMaintenanceEnabled():
         mds = None
         maintenanceFilter = data.get( 'maintenance' )
         if maintenanceFilter:
            maintenanceFilter = get_colon_mac( maintenanceFilter )
            if not maintenanceFilter:
               ec = '137'
               if ec not in messages:
                  messages.append( ec )
               return storedReceivers, messages
            elif maintenanceFilter not in agentStatus.maintenanceDeviceStatus:
               return storedReceivers, messages
            mds = agentStatus.maintenanceDeviceStatus[ maintenanceFilter ]

      impactedFilter = data.get( 'impacted' )
      if Toggles.McsToggleLib.toggleDeviceMaintenanceEnabled():
         log( f"{srcFilter=}, {grpFilter=}, {deviceFilter=}, {impactedFilter=}, "
              f"{maintenanceFilter=}" )
      else:
         log( f"{srcFilter=}, {grpFilter=}, {deviceFilter=}, {impactedFilter=}" )

      for mcastKey in apiConfig.mcastReceiver:
         receivers = apiConfig.mcastReceiver[ mcastKey ].receivers
         source = mcastKey.source
         group = mcastKey.group

         if srcFilter and source != srcFilter:
            continue
         if grpFilter and group != grpFilter:
            continue

         for device in receivers:
            if deviceFilter and deviceFilter != device:
               continue
            if Toggles.McsToggleLib.toggleDeviceMaintenanceEnabled():
               if maintenanceFilter and Tac.Value( "Mcs::DeviceSg",
                     device, mcastKey ) not in mds.maintImpRcv:
                  continue
            for recvIntf in receivers[ device ].recvIntfs:
               trackingId = receivers[ device ].recvIntfs[ recvIntf ].trackingId
               if mcastKey not in agentStatus.flowProgrammed:
                  continue
               fp = agentStatus.flowProgrammed[ mcastKey ]
               if device not in fp.device:
                  continue
               if recvIntf not in fp.device[ device ].oif:
                  continue

               impacted = 'ok'
               reasons = getImpactedReasons( agentStatus, mcastKey, device,
                                             recvIntf )
               if reasons:
                  impacted = 'impacted'

               # Filter only impacted receiver, if it is set
               if impactedFilter and impacted == 'ok':
                  continue

               if impacted not in storedReceivers:
                  storedReceivers[ impacted ] = {}
               if source not in storedReceivers[ impacted ]:
                  storedReceivers[ impacted ][ source ] = {}
               if group not in storedReceivers[ impacted ][ source ]:
                  storedReceivers[ impacted ][ source ][ group ] = {}
               if trackingId not in storedReceivers[ impacted ][ source ][ group ]:
                  storedReceivers[ impacted ][ source ][ group ][ trackingId ] = {}
               tIdObj = storedReceivers[ impacted ][ source ][ group ][ trackingId ]
               devIntf = DevAndIntfType( device, recvIntf )
               # If the receiver belongs to a boundary interface, add it in a
               # separate object
               msg = []
               success = True
               if ( fs := fp.status.get( devIntf ) ) and fs != FS.noError:
                  msg = [ f'{Tac.enumValue( FS, fs )}' ]
                  success = False
               devObj = {
                  "destinationIP": group,
                  "deviceId": device,
                  "messages": msg,
                  "receiverIntfIds": [ recvIntf ],
                  "sourceIP": source,
                  "success": success,
                  "trackingID": trackingId,
                  "action": impacted
               }
               dev = device
               if bIntf := getReceiverBoundaryIntf( mcastKey, devIntf ):
                  dev = f"{device}-{bIntf}"
                  devObj.update( { 'boundaryInterface': bIntf } )
               if not tIdObj.get( dev ):
                  tIdObj[ dev ] = devObj
               else:
                  if recvIntf not in tIdObj[ dev ][ 'receiverIntfIds' ]:
                     tIdObj[ dev ][ 'receiverIntfIds' ].append( recvIntf )

               if reasons:
                  tIdObj[ dev ][ 'impactedReasons' ] = list(
                     set( tIdObj[ dev ].get( 'impactedReasons', [] ) + reasons ) )

   #pylint: enable-msg=too-many-nested-blocks
   return storedReceivers, messages

def get_flows_active_template( destination, devId, iif, oifs, srcIp, vrfName, bw ):
   return {
            'bandwidth': bw,
            'bwType': 'k',
            'destination': destination,
            'deviceId': devId,
            'iif': iif,
            'oifs': oifs,
            'sourceIP': srcIp,
            'vrf': vrfName
   }

def getActiveFlows( mcsControllerMounts, filters ):
   activeFlows = []
   messages = []
   # validate filters is a list type
   if not isinstance( filters, list ):
      error( f"Invalid request with {filters}" )
      messages.append( '187' )
      return activeFlows, messages

   # validate data in filters
   # pylint: disable=too-many-nested-blocks
   for data in filters:
      if ec := verifyKeysInFilters( data ):
         messages.append( ec )
         return activeFlows, messages

      srcFilter = data.get( 'sourceIP' )
      if srcFilter:
         _, ec = validateSourceIP( srcFilter )
         if ec:
            if ec not in messages:
               messages.append( ec )
            return activeFlows, messages

      grpFilter = data.get( 'destinationIP' )
      if grpFilter:
         _, ec = validateMcastGroup( grpFilter )
         if ec:
            if ec not in messages:
               messages.append( ec )
            return activeFlows, messages

      deviceFilter = None
      deviceId = data.get( 'deviceId' )
      if deviceId:
         deviceFilter = get_colon_mac( deviceId )
         if not deviceFilter:
            ec = '137'
            if ec not in messages:
               messages.append( ec )
            return activeFlows, messages

      # pylint: disable-msg=R1702
      topoStatus = mcsControllerMounts.cdbTopology
      activeFlowsStatus = mcsControllerMounts.cdbActiveFlows
      for deviceId in activeFlowsStatus:
         if deviceFilter and not isSameMac( deviceId, deviceFilter ):
            continue
         for mcastKey in activeFlowsStatus[ deviceId ].route:
            if ( ROUTE_OWNER_MCS not in
               activeFlowsStatus[ deviceId ].route[ mcastKey ].referrer ):
               continue
            s = mcastKey.source
            g = mcastKey.group

            if srcFilter and s != srcFilter:
               continue
            if grpFilter and g != grpFilter:
               continue

            igmpGroup = Tac.newInstance( "Arnet::IpGenAddr", g )
            igmpSource = Tac.newInstance( "Arnet::IpGenAddr", s )

            inIntf = [ k for k, v in
                  activeFlowsStatus[ deviceId ].route[ mcastKey ].iif.items()
                     if ROUTE_OWNER_MCS in v.referrer ]
            oifs = [ k for k, v in
                  activeFlowsStatus[ deviceId ].route[ mcastKey ].oif.items()
                  if ROUTE_OWNER_MCS in v.referrer ]
            if not inIntf:
               continue

            iif = inIntf[ 0 ]
            destination = {
                           'isIgmpEnabled': False,
                           'destinationIP': g,
                           'igmpSnooping': []
                        }
            for vlanId in activeFlowsStatus[ deviceId ].igmp:
               afIgmp = activeFlowsStatus[ deviceId ].igmp[ vlanId ].group
               if igmpGroup in afIgmp:
                  if igmpSource in afIgmp[ igmpGroup ].source:
                     destination[ 'isIgmpEnabled' ] = True
                     afIgmpSrc = afIgmp[ igmpGroup ].source[ igmpSource ]
                     igmpOifs = list( afIgmpSrc.includeIntf )
                     destination[ 'igmpSnooping' ].append( {
                                 'vlanId': vlanId,
                                 'interfaces': igmpOifs
                                 } )
                     # Checking whether iif and oifs are in same vlan
                     if ( iif.startswith( 'Vlan' ) and
                         int( iif.split( 'Vlan' )[ 1 ] ) == vlanId and
                         iif not in oifs ):
                        # The oif is in vlan
                        oifs.append( iif )
            flowBw = 0
            if activeFlowsStatus[ deviceId ].route[ mcastKey ].bw:
               flowBw = activeFlowsStatus[ deviceId ].route[ mcastKey ].bw.value
            flow = get_flows_active_template( destination,
                                             get_dotted_mac( deviceId ), iif,
                                             oifs, s, 'default', flowBw )
            devId = get_colon_mac( deviceId )
            host = topoStatus.host.get( devId )
            hostName = host.hostname if host else ''
            flow[ 'network-device-name' ] = hostName

            activeFlows.append( flow )
   return activeFlows, messages

# Generates a simple error message given the FailProgramStatus string
def getFpsMsg( fps ):
   errToDesc = {
      'noBandwidth': 'Not enough bandwidth available',
      'noLink': 'Receiver is not a direct connection or '
                'cannot be reached via 1 hop spine',
      'noReceiver': 'Sender does not have any connections to receiver',
      'noPathDiversity': 'No path diversity',
      'noInterface': 'Interface does not exist on the device',
      'maintenanceDevice': 'The only connections include maintenance device',
      'noMaintReroute': 'Failure to reroute maintenance-impacted receiver',
   }
   return errToDesc.get( fps, '' )

def validateConsistentHashingPost( data ):
   allKeys = [ 'mode', 'weight' ]
   missingKeys = set( allKeys ) - set( data )
   msgs = []
   if len( missingKeys ) == len( allKeys ):
      msgs.append( '131' )
      error( errorCodes[ '131' ].replace(
         '<key>', f'{allKeys[ 0 ]} or {allKeys[ 1 ]}' ) )
      return not msgs, msgs
   if 'mode' in data:
      mode = data[ 'mode' ]
      validModes = [ ChStateType.disabled, ChStateType.consistent ]
      if mode not in validModes:
         msgs.append( '241' )
         error( errorCodes[ '241' ].replace( '<mode>', mode ) )
   if 'weight' in data:
      weight = data[ 'weight' ]
      if not isinstance( weight, int ) or weight < 1 or weight > 2048:
         msgs.append( '240' )
         error( errorCodes[ '240' ].replace( '<weight>', str( weight ) ) )
   return not msgs, msgs

def getDevName( devId, topoStatus ):
   name = ''
   devId = get_colon_mac( devId )
   if topoStatus.host.get( devId ):
      name = topoStatus.host.get( devId ).hostname
   return name

def validateDeviceMaintenancePost( data ):
   keys = [ 'deviceId', 'maintenance' ]
   missingKeys = set( keys ) - set( data )
   if missingKeys:
      ec = '131'
      error( errorCodes[ ec ].replace( '<key>', str( missingKeys ) ) )
      return False, [ ec ]
   dev = get_colon_mac( data[ 'deviceId' ] ).replace( ':', '-' )
   if not dev:
      ec = '137'
      error( errorCodes[ ec ].replace( '<deviceId>', data[ 'deviceId' ] ) )
      return False, [ ec ]
   return True, []

def validateTrackingId( data ):
   if isinstance( data, int ) and data >= 0 and data.bit_length() <= 32:
      return True, ''
   else:
      ec = '254'
      error( errorCodes[ ec ] )
      return False, ec

def validateTransactionId( data ):
   if isinstance( data, str ):
      return True, ''
   else:
      ec = '255'
      error( errorCodes[ ec ] )
      return False, ec
