# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import random

import BothTrace
import Tac
import Tracing

info = BothTrace.tracef0
warn = BothTrace.tracef1
error = BothTrace.tracef2
log = BothTrace.tracef3
bv = BothTrace.Var
__defaultTraceHandle__ = Tracing.Handle( 'McsStateHAReplicate' )

apiSyncStatus = Tac.Type( "Mcs::ApiSyncStatus" )

class UpdatingApiConfig:
   def __init__( self, status, sync=False ):
      self.status = status
      self._sync = sync

   def __enter__( self ):
      if self._sync:
         info( "Setting syncInProgress to", bv( apiSyncStatus.started ) )
         self.status.syncInProgress = apiSyncStatus.started

   def __exit__( self, exc, excVal, tb ):
      if self._sync:
         info( "Setting syncInProgress to", bv( apiSyncStatus.finished ) )
         self.status.syncInProgress = apiSyncStatus.finished
      if tb:
         raise excVal.with_traceback( tb )

   def __call__( self, func ):
      def wrapper( *args, **kwargs ):
         with self:
            return func( *args, **kwargs )
      return wrapper

class McsStateReplicate:
   def __init__( self, apiConfig, clusterStatus, apiStatus, agentStatus,
                 mcsStatusHA, em ):
      self.apiConfig = apiConfig
      self.clusterStatus = clusterStatus
      self.apiStatus = apiStatus
      self.agentStatus = agentStatus
      self.mcsStatusHA = mcsStatusHA
      self.apiStatus.syncInProgress = apiSyncStatus.notStarted
      self.replicationSm = None
      self.em = em

   def replicate( self ):
      return UpdatingApiConfig( self.apiStatus )

   def sync( self ):
      return UpdatingApiConfig( self.apiStatus, sync=True )

   def syncApiConfig( self, sync=True ):
      log( "mountDone:", bv( self.agentStatus.mountDone ),
           ", Mcs agent enabled:", bv( self.agentStatus.enabled ) )
      if not self.agentStatus.mountDone:
         info( "Mount is not ready. Not starting API Config sync process" )
         return
      if self.apiStatus.syncInProgress == apiSyncStatus.finished:
         info( "API Config sync already completed" )
         return
      info( "Start API Config sync" )

      # Run the code if the cluster is leader.
      if self.clusterStatus.status[ 'default' ].isStandalone:
         info( 'CVX is not leader. Do not update API config' )
         self.apiStatus.syncInProgress = apiSyncStatus.finished
         return

      info( 'Updating ApiConfig after CVX leadership change' )
      # Sort clients based on the update count
      updateCount = sorted( self.mcsStatusHA.items(),
                            key=lambda x: x[ 1 ].updateCount )
      # Find clients that are in sync
      clientsInSync = [ client for client, entity in self.mcsStatusHA.items()
                        if entity.clientInSync ]
      info( "clientsInSync:", bv( clientsInSync ),
            "updateCount:", bv( updateCount ) )

      # Now select client from clientsInSync or updateCount
      client = random.choice( clientsInSync ) if clientsInSync else (
         updateCount[ -1 ][ 0 ] if updateCount else [] )
      info( "Selecting client", bv( client ), " as a source to sync API config" )
      if not ( client and self.mcsStatusHA[ client ].updateCount ):
         warn( "There is no data available in the client for sync" )
         self.apiStatus.syncInProgress = apiSyncStatus.finished
         return

      newApiConfig = self.mcsStatusHA[ client ]
      self.replicationSm = Tac.newInstance( "Mcs::McsApiReplicationSm",
                                           self.apiStatus,
                                           self.apiConfig,
                                           newApiConfig,
                                           self.em.cEntityManager() )
