#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import os
from TypeFuture import TacLazyType

mgmtSecurityConfigType = "Mgmt::Security::Config"
SslConstant = TacLazyType( "Mgmt::Security::Ssl::Constants" )
CertLocation = TacLazyType( "Mgmt::Security::Ssl::CertLocation" )
KeyLocation = TacLazyType( "Mgmt::Security::Ssl::KeyLocation" )

# Method to get the auto cert profile status corresponding to an ssl profile.
def getAutoCertProfileStatusFromSslProfile( sslProfile, allAutoCertProfileStatus ):
   # check if ssl profile is using an auto cert
   if ( not sslProfile or not sslProfile.certKeyPair or
        not sslProfile.certKeyPair.certFile or
        sslProfile.certKeyPair.certLocation != CertLocation.autoCerts ):
      return None

   # Get the auto cert profile name from the auto cert file name
   autoCertProfileStatusName = getAutoCertProfileNameFromAutoCert(
       sslProfile.certKeyPair.certFile )

   # Return the auto cert profile status
   return allAutoCertProfileStatus.profileStatus.get( autoCertProfileStatusName )

# Method to get the keyFile and keyLocation from an auto cert profile status.
# If keyLocation in autoCertProfileStatus is 'auto' the key is in autokeys dir,
# with filename similar to autoCertProfile's name else key is in keys dir, with
# with filename as value of keyLocation.
def getKeyInfoFromAutoCertProfileStatus( autoCertProfileStatus ):
   if autoCertProfileStatus and autoCertProfileStatus.keyLocation:
      if autoCertProfileStatus.keyLocation == SslConstant.autoCertKeyMgmt:
         keyFile = f"{autoCertProfileStatus.name}.key"
         return ( keyFile, KeyLocation.autoKeys )
      else:
         keyFile = autoCertProfileStatus.keyLocation
         return ( keyFile, KeyLocation.keys )

   return None

# Get the auto cert profile name from a corresponding auto cert file,
# autoCertProfile name is same as auto cert file name without any extension.
def getAutoCertProfileNameFromAutoCert( certFile ):
   return os.path.splitext( certFile )[ 0 ]

# Give an ssl profile configured with an auto cert profile, return the key
# information.
def getKeyInfoFromAutoCertConfiguredSslProfile( profileConfig,
                                                allAutoCertProfileStatus ):
   # Get the corresponding auto cert profile
   autoCertProfileStatus = getAutoCertProfileStatusFromSslProfile(
      profileConfig,
      allAutoCertProfileStatus
   )
   # return the key information
   return getKeyInfoFromAutoCertProfileStatus( autoCertProfileStatus )
