#!/usr/bin/env python3
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import array
import Cell
import FpgaUtil     # pylint: disable-msg=F0401
import os
import Pci
import PicassoInit  # pylint: disable-msg=F0401
import re
import sys
import time
import Tac
import Tracing
from functools import cache

class ExtraTracing:
   def __init__( self, s ):
      self.s = s
      self.bak = None

   def __enter__( self ):
      self.bak = os.environ.get( 'TRACE' )
      os.environ[ 'TRACE' ] = (
         self.s if self.bak is None
         else f"{self.bak},{self.s}" )
      Tracing.traceFacilityManager.doReadEnvironment()

   def __exit__( self, exc_type, exc_val, exc_tb ):
      if self.bak is None:
         os.environ.pop( 'TRACE' )
      else:
         os.environ[ 'TRACE' ] = self.bak
      Tracing.traceFacilityManager.doReadEnvironment()

def formatVersion( rawValue ):
   byte0 = rawValue >> 24
   byte1 = ( rawValue >> 16 ) & 0xff
   bytes23 = rawValue & 0xffff
   return f"{byte0:x}.{byte1:x}.{bytes23:x}"

_enableOutputToConsole = False
def printMsg( *args, toConsole=False ):
   print( time.asctime(), *args, file=sys.stderr )
   if _enableOutputToConsole and toConsole:
      FpgaUtil.printToConsole( ' '.join( args ) + '\n' )

def sid():
   try:
      with open( "/proc/cmdline" ) as f:
         return re.search( r'sid=(\w+) ', f.read() ).group( 1 )
   # pylint: disable-msg=bare-except
   except:
      return ""

firmwareBasePath = '/usr/share/Microsemi/'
firmwareMap = { 'OtterLake':     'firmware-1.B.8C.pmc',
                'OtterLakeS':    'firmware-1.B.8C.pmc',
                'Redwood':       'firmware-1.B.8C.pmc',
                'HorseshoeLake': 'firmware-1.B.8C.pmc',
                'HorseshoeLakeS':'firmware-1.B.8C.pmc',
                'Narwhal':       'firmware-1.B.8C.pmc',
                'NarwhalClock':  'firmware-1.B.8C.pmc',
                'Penguin':       'firmware-1.B.8C.pmc',
                'PenguinClock':  'firmware-1.B.8C.pmc',
}

firmwareConfigMap = {
   'OtterLake': 'OtterLake.pmc',
   'OtterLakeS': 'OtterLake.pmc',
   'Redwood': 'Redwood.pmc',
   'HorseshoeLake': 'HorseshoeLake.pmc',
   'HorseshoeLakeS': 'HorseshoeLake.pmc',
   'Narwhal': 'Tundra.pmc',
   'NarwhalClock': 'Tundra.pmc',
   'Penguin': 'Tundra.pmc',
   'PenguinClock': 'Tundra.pmc',
}

supSwitchMepMap = {
   'OtterLake': '0000:05:00.1',
   'OtterLakeS': '0000:05:00.1',
   'Redwood': '0000:05:00.1',
   'HorseshoeLake': '0000:05:00.1',
   'HorseshoeLakeS': '0000:05:00.1',
   'Narwhal': '0000:05:00.1',
   'NarwhalClock': '0000:05:00.1',
   'Penguin': '0000:05:00.1',
   'PenguinClock': '0000:05:00.1',
}

def firmwareFile():
   if sid() in firmwareMap:
      return firmwareMap[ sid() ]
   else:
      return ""

class MicrosemiConsts:
   GAS_DONE = 2
   GAS_INPROGRESS = 1
   MRPC_BG_STAT_IDLE = 0
   MRPC_BG_STAT_DONE = 2
   SWITCHTEC_MAX_PORTS = 48
   firmwareImage = firmwareBasePath + firmwareFile()
   microsemiVendorID = 0x11f8
   microsemiDeviceIDs = [ 0x8533, 0x8534, 0x8532 ]
   microsemiBinaryHeaderSize = 630
   downloadHeaderSize = 0x10
   def configImage( self ):
      fileName = firmwareConfigMap.get( sid() )
      if fileName:
         fileName = firmwareBasePath + fileName
      return fileName

class MicrosemiGAS:
   GAS_INPUT_DATA = 0
   GAS_OUTPUT_DATA = 0x400
   GAS_COMMAND = 0x800
   GAS_STATUS = 0x804
   GAS_RETURNVALUE = 0x808
   GAS_BLOCKSIZE = 0x400
   GAS_VendorTableRevision = 0x2010
   GAS_ActiveFirmwareAddress = 0x2204
   GAS_ActiveFirmwareVersion = 0x2208
   GAS_ActiveConfigAddress = 0x2210
   GAS_ActiveConfigVersion = 0x2214
   GAS_ActiveConfigBuild = 0x2218
   GAS_InactiveFirmwareAddress = 0x221C
   GAS_InactiveFirmwareVersion = 0x2220
   GAS_InactiveConfigAddress = 0x2228
   GAS_InactiveConfigVersion = 0x222c
   GAS_InactiveConfigBuild = 0x2230

class MicrosemiDownloadSubcommand:
   MRPC_DNLD_Toggle = 2
   MRPC_DNLD_ToggleFirmware = 1 << 8
   MRPC_DNLD_ToggleConfig = 1 << 16

class MCRPC_P2PSubcommand:
   MRPC_P2P_BIND = 0
   MRPC_P2P_UNBIND = 1
   MRPC_P2P_INFO = 3

class MRPC_LTSSMSubcommand:
   MRPC_LTSSM_LogDump = 7

class MicrosemiMRPC( MicrosemiDownloadSubcommand,
                     MCRPC_P2PSubcommand,
                     MRPC_LTSSMSubcommand ):
   MRPC_DIAG_PMC_START = 0
   MRPC_TWI = 1
   MRPC_VGPIO = 2
   MRPC_PWM = 3
   MRPC_DIETEMP = 4
   MRPC_FWDNLD = 5
   MRPC_FWLOGRD = 6
   MRPC_PMON = 7
   MRPC_PORTLN = 8
   MRPC_PORTARB = 9
   MRPC_MCOVRLY = 10
   MRPC_STACKBIF = 11
   MRPC_PORTPARTP2P = 12
   MRPC_DIAG_TLP_INJECT = 13
   MRPC_DIAG_TLP_GEN = 14
   MRPC_DIAG_PORT_EYE = 15
   MRPC_DIAG_POT_VHIST = 16
   MRPC_DIAG_PORT_LTSSM_LOG = 17
   MRPC_DIAG_PORT_TLP_ANL = 18
   MRPC_DIAG_PORT_LN_ADPT = 19
   MRPC_NT_MCG_CAP = 23
   MRPC_TACHO = 24
   MRPC_SMBUS = 26
   MRPC_RESET = 27
   MRPC_LNKSTAT = 28
   MRPC_SES = 30
   MRPC_ECHO = 65

class MicrosemiGasOps:
   # Region 6 at 16k (0x10000)
   # NTB Memoory Mapped Registers, page 174 in Microsemi Device Specifications
   # See 17.8, page 480 for format of the region
   MRPC_NTB_BASE = 0x10000
   MRPC_NTB_CONTROL_OFFSET = 0x4000
   MRPC_NTB_CONTROL_BLOCK_SIZE = 0x2000
   MRPC_NTB_REQUESTER_TABLE_OFFSET = 0x400

   def NTxBASE( self, x ):
      # Address of NTx Control Registers area
      # see 17.8 in Microsemi Device Specifications
      return self.MRPC_NTB_BASE + \
         self.MRPC_NTB_CONTROL_OFFSET + x * self.MRPC_NTB_CONTROL_BLOCK_SIZE

   def requesterTable( self, x ):
      # See 17.8.2 in Microsemi Device Specifications
      return self.NTxBASE( x ) + self.MRPC_NTB_REQUESTER_TABLE_OFFSET

   def requesterMask( self, origDevice, origFunction, destDevice, destFunction ):
      valid = 1
      # See page 489, Table 296, in Microsemi Device Specifications
      return ( ( origDevice << 3 | origFunction ) << 16 |
               ( destDevice << 3 | destFunction ) << 1 |
               valid )

   MRPC_NTOP_LOCK = 1
   MRPC_NTOP_CONF = 2
   MRPC_NTOP_RESET = 3

   def ntOp( self, x ):
      # See 17.8.2 in Microsemi Device Specifications
      return self.NTxBASE( x ) + 4

   MRPC_PARTITIONS = [ 0, 1 ]
   MRPC_NTBAR = 4             # Physical BAR4 used for NTB
   MRPC_NTBARSETUPSIZE = 0x10 # Page 482 of DevSpec
   MRPC_NTBARSETUP_VALID = 1
   MRPC_NT_TYPE_64BIT = 0b100
   MRPC_NT_LUT_WIN_ENABLE = 0b10000

   # The following two values mean that
   # window size is 2GB and least
   # 31 bits are offset within a window
   # See table 293 in device spec, offsets 0x4-0x7
   MRPC_NT_WIN_DIR_SIZE = 0x80000000
   MRPC_NT_DIR_XLATE_POS = 31
   MRPC_NT_DIR_XLATE_BASE_ADDR_HIGH = 64 // 4

class MicrosemiBase( MicrosemiMRPC, MicrosemiGAS, MicrosemiGasOps, MicrosemiConsts ):
   microsemiBar = int( 0 )
   debug = False
   dataEndpoint_ = None
   microsemiFunctionBar = None
   microsemiFunctionBar_ = None

   def __init__( self, mepAddr, verbose=True, mmio=False  ):
      self.mepAddr = Pci.Address( mepAddr )
      self.verbose = verbose
      self.mmio = mmio
      self.size = 0
      self.bdf = 0
      self.prot = 0
      self.device = None
      self.dmamem = None

   @classmethod
   def fromSid( cls, sidl, **kwargs ):
      mepAddr = supSwitchMepMap.get( sidl )
      if mepAddr:
         return cls( mepAddr, **kwargs )
      return None

   @classmethod
   def forSupervisor( cls, **kwargs ):
      sidl = sid()
      if sidl:
         return cls.fromSid( sidl, **kwargs )
      return None

   def managementEndpointBar( self ):
      if self.microsemiFunctionBar is None:
         self.microsemiFunctionBar = Pci.Device (
               self.mepAddr ).resource( self.microsemiBar )
      return self.microsemiFunctionBar

   def read8( self, offset ):
      return self.managementEndpointBar().read8( offset )

   def read32( self, offset ):
      return self.managementEndpointBar().read32( offset )

   def write8( self, offset, value ):
      return self.managementEndpointBar().write8( offset, value )

   def write32( self, offset, value ):
      return self.managementEndpointBar().write32( offset, value )

   def hardwareIsPresent( self ):
      return os.path.exists( "/sys/bus/pci/devices/" + str( self.mepAddr ) )

   def showRegs( self ):
      return ( f"INPUT={self.read32( self.GAS_INPUT_DATA ):x}, "
               f"OUTPUT={self.read32( self.GAS_OUTPUT_DATA ):x}, "
               f"COMMAND={self.read32( self.GAS_COMMAND ):x}, "
               f"STATUS={self.read32( self.GAS_STATUS ):x}, "
               f"RET_VALUE={self.read32( self.GAS_RETURNVALUE ):x}" )

   def gasWait( self, timeout=600 ):
      def status():
         status = self.read32( self.GAS_STATUS )
         assert status == self.GAS_DONE or status == self.GAS_INPROGRESS, \
            f"MRPC FAULT: {self.showRegs()}"
         return status
      Tac.waitFor( lambda: status() == self.GAS_DONE,
                   timeout=timeout )
      return self.read32( self.GAS_RETURNVALUE )


   # PFX Device Specification, chapter 7.7.2:
   # The process for using the MRPC is as follows:
   # 1. Write the input data parameters as defined in the command data structure
   #    into the Input Data region.
   # 2. Write the command values into command registers. This will begin the command
   #    execution by the CPU. MRPC status will be set to 0x1 ( MRPC_STAT_INPROGRESS )
   #    and the firmware calls the related command handler to process the command.
   #    If an unsupported command is sent, MRPC status will be set to
   #    0xFF( MRPC_STAT_ERROR ).
   # 3. After the command handler done, firmware will set the MRPC status to 0x2
   # ( MRPC_STAT_DONE ) and send MRPC Command Complete event to the host. The host
   # should read the MRPC status value after receiving the MRPC Command Complete
   # event, until it reads 0x2 ( MRPC_STAT_DONE )

   def doGas( self,
              argument, cmd ):
      if self.debug:
         printMsg( f"cmd = {cmd:x}, arg = {argument:x}" )
      self.write32( self.GAS_INPUT_DATA, argument )
      self.write32( self.GAS_COMMAND, cmd )
      return self.gasWait()

   def getImageInfo( self, fileName ):
      with open( fileName, 'rb' ) as f:
         magic = f.read( 3 )
         assert magic == b"PMC", f"Wrong image given - {fileName}"
         f.read( 1 )
         values = array.array( 'i' )
         values.fromfile( f, self.microsemiBinaryHeaderSize )
         length = values[ 0 ]
         imgType = values[ 1 ]
         loadAddr = values[ 2 ]
         version = values[ 3 ]
         vendor = values[ 16 ] >> 24
         revision = self.swap16(
            values[ 17 ] >> 16 ) << 16 | self.swap16( values[ 18 ] )
         return length, imgType, loadAddr, version, vendor, revision


   def rescan( self ):
      # MEP at <bus>:00.1
      mepAddr = self.mepAddr
      # Upstream port at <bus>:00.0
      uspAddr = Pci.Address( mepAddr.domain,
                             mepAddr.bus, mepAddr.slot, 0 )

      def write( path, v ):
         try:
            open( path, "w" ).write( v )
         except OSError as e:
            printMsg( f"Error writing {path}: {e.strerror}" )

      # Re-enable the MEP (it might have become disabled)
      # This is a workaround for us seeing the MEP disabled after reset on Tundra
      path = f"/sys/bus/pci/devices/{mepAddr}/enable"
      if os.path.exists( path ):
         write( path, "1" )

      # Find the parent bridge linking to the USP.
      parent = \
         os.path.basename(
            os.path.dirname(
               os.readlink(
                  f"/sys/bus/pci/devices/{uspAddr}" ) ) )

      # Remove USP + MEP
      for addr in [ uspAddr, mepAddr ]:
         path = f"/sys/bus/pci/devices/{addr}/remove"
         if os.path.exists( path ):
            write( path, "1" )

      # Finally, rescan parent
      write( f"/sys/bus/pci/devices/{parent}/rescan", "1" )

   @cache
   def mep( self ):
      MEP = Tac.Type( "Microsemi::MEP" )
      return MEP( self.mepAddr.value() )

   @cache
   def mrpcDriver( self ):
      driver = self.mep().mrpcDriver( "Mmio" if self.mmio else "Client" )

      assert driver is not None, \
         f"Failed to instantiate mrpcDriver for MEP {self.mepAddr!r}"

      return driver

   def mrpcEnq( self, req ):
      self.mrpcDriver().rq.enq( req )

   class MRPCFailure( Exception ):

      def __init__( self, req ):
         super().__init__( "MRPC Failure", req )

      @property
      def req( self ):
         return self.args[ 1 ]

      def __str__( self ):
         return "command={!r}, subCmd={!r}, status={!r}, result={!r}".format(
            self.req.command,
            getattr( self.req, 'subCmd', None ),
            self.req.completion.status,
            getattr( self.req, 'result', self.req.completion.result ) )

   @classmethod
   def mrpcSync( cls, req, expectedResult=0 ):
      Tac.waitFor( lambda: req.completion.isFinished )
      if req.completion.isFailure or (
            # Most, if not all, MRPCs are expected to finish with
            # result=0. Checked by default.
            expectedResult is not None and
            req.completion.result != expectedResult ):
         raise cls.MRPCFailure( req )

   def mrpcExec( self, req, **args ):
      self.mrpcEnq( req )
      self.mrpcSync( req, **args )

   def firmwareDownload( self, fileName, BDF=None ):
      GetStatusRequest = \
         Tac.Type( "Microsemi::MRPC::FwDnld::GetStatusRequest" )
      DnldStatus = \
         Tac.Type( "Microsemi::MRPC::FwDnld::DnldStatus" )
      BgProcStatus = \
         Tac.Type( "Microsemi::MRPC::FwDnld::BgProcStatus" )
      DownloadRequest = \
         Tac.Type( "Microsemi::MRPC::FwDnld::DownloadRequest" )

      imageLength = os.path.getsize( fileName )

      def requestStatus():
         req = GetStatusRequest()
         self.mrpcExec( req )
         return req

      def printStatus( status ):
         msg = ( f"Download status = '{status.dnldStatus}', "
                 f"background status = '{status.bgProcStatus}'" )
         printMsg( msg, toConsole=True )

      # Firmware 1.9.63, used in manufacturing, is slow to move from
      # initial state (e.g. ReadyForDownload) to DownloadInProgress,
      # which should happen after the first block. It will, but takes
      # 4-5 blocks to get there.
      #
      # We require a transition to DownloadInProgress at any, but
      # /some/ point during the bulk transfer (dnldReadyToResume), and
      # then expect to stay there until after the last block. Final
      # checks in dnldComplete will enforce this.
      prevDnldStatus = None

      # Before first block
      def dnldReadyToBegin( dnldStatus, bgProcStatus ):
         nonlocal prevDnldStatus
         prevDnldStatus = dnldStatus
         return bgProcStatus in [ BgProcStatus.BgProcIdle,
                                  BgProcStatus.BgProcFinished,
                                  BgProcStatus.BgProcWriteError ]

      # After any but the last block
      def dnldReadyToResume( dnldStatus, bgProcStatus ):
         nonlocal prevDnldStatus
         assert dnldStatus in { prevDnldStatus,
                                DnldStatus.DownloadInProgress }
         prevDnldStatus = dnldStatus
         assert bgProcStatus in [ BgProcStatus.BgProcInProgress,
                                  BgProcStatus.BgProcFinished ]
         return bgProcStatus == BgProcStatus.BgProcFinished

      # After last block
      def dnldComplete( dnldStatus, bgProcStatus ):
         assert prevDnldStatus == DnldStatus.DownloadInProgress
         if dnldStatus == DnldStatus.DownloadInProgress:
            assert bgProcStatus == BgProcStatus.BgProcInProgress
            return False
         else:
            assert dnldStatus in [
               DnldStatus.FirmwareCompleteBeforeToggle,
               DnldStatus.ConfigCompleteBeforeToggle ]
            assert bgProcStatus == BgProcStatus.BgProcFinished
            return True

      def waitCompletion( testFn ):
         prev = None
         def ready():
            nonlocal prev
            status = requestStatus()
            if ( prev is None or
                 status.dnldStatus != prev.dnldStatus or
                 status.bgProcStatus != prev.bgProcStatus ):
               printStatus( status )
               prev = status
            return testFn( status.dnldStatus, status.bgProcStatus )
         Tac.waitFor( ready )

      with open( fileName, 'rb' ) as f, ExtraTracing( "Microsemi::MRPC/0" ):
         waitCompletion( dnldReadyToBegin )
         offset = None
         req = DownloadRequest.first( f.fileno(), "doNotToggle" )
         while req:
            assert req.imgLength == imageLength
            offset = req.imgOffset
            self.mrpcExec( req )
            waitCompletion( dnldReadyToResume if not req.isLast
                            else dnldComplete )
            printMsg( f"Wrote chunk at {req.imgOffset} length {req.dataLength} "
                      f"of total {req.imgLength}" )

            if req.isLast:
               break # skip else

            req = req.next( f.fileno() )
         else:
            assert False, \
               f"req=None at offset {offset}/{imageLength}"

      printMsg( f"File {fileName} downloaded.", toConsole=True )

   def firmwareToggle( self, toggleFirmware=False, toggleConfig=False ):
      ToggleRequest = \
         Tac.Type( "Microsemi::MRPC::FwDnld::ToggleRequest" )
      req = ToggleRequest( int( toggleFirmware ), int( toggleConfig ) )
      self.mrpcExec( req )

   def updateConfig( self, fileName=None, force=False ):
      if fileName is None:
         fileName = self.configImage()
      if self.hardwareIsPresent() and os.path.exists( fileName ):
         ( _, _, _, version, _, revision ) = self.getImageInfo( fileName )
         hwRevision = self.read32( self.GAS_VendorTableRevision )
         hwVersion = self.read32( self.GAS_ActiveConfigVersion )
         versionMsg = ( f"Configuration version in file is {version:X}.{revision:X} "
                        f"and in hardware is {hwVersion:X}.{hwRevision:X}" )
         printMsg( versionMsg )
         if revision != hwRevision or version != hwVersion or force:
            printMsg( f"{versionMsg}, updating...", toConsole=True )
            printMsg( f"Updating config from {fileName}..." )
            self.firmwareDownload( fileName )
            printMsg( "Config toggle and reboot required" )
            return True # ask NorCal to toggle config and restart supervisor
         else:
            printMsg( "Microsemi config is up to date" )
      return False # dont ask NorCal to toggle config and restart supervisor

   def updateFirmware( self, force=False ):
      if self.hardwareIsPresent() and os.path.exists( self.firmwareImage ) :
         ( _, _, _,
           version, _, _ ) = self.getImageInfo( self.firmwareImage )
         hwVersion = self.read32( self.GAS_ActiveFirmwareVersion )
         if version != hwVersion or force:
            printMsg( f"Firmware version in file is {version:X} and in hardware "
                      f"is {hwVersion:X}, updating...", toConsole=True )
            printMsg( f"Updating firmware from {self.firmwareImage}..." )
            self.firmwareDownload( self.firmwareImage )
            printMsg( "Firmware toggle and reboot required" )
            return True # ask NorCal to toggle firmware and restart supervisor
         else:
            printMsg( "Microsemi firmware is up to date" )
      return False # dont ask NorCal to toggle firmware and restart supervisor

   def readGroup4( self, offset ):
      value = self.read32( offset )
      return ( value & 0xff,
               ( value >> 4 ) & 0xf,
               ( value >> 8 ) & 0xf,
               ( value >> 12 ) & 0xf,
               ( value >> 16 ) & 0xf,
               ( value >> 10 ) & 0xf,
               ( value >> 24 ) & 0xf,
               ( value >> 28 ) & 0xf )

   def readGroup8( self, offset ):
      value = self.read32( offset )
      return ( value & 0xff,
               ( value >> 8 ) & 0xff,
               ( value >> 16 ) & 0xff,
               ( value >> 24 ) & 0xff )

   def readGroup16( self, offset ):
      value = self.read32( offset )
      return ( value & 0xffff,
               ( value >> 16 ) & 0xffff )

   def cleanBufs( self ):
      for offset in range( 0, self.GAS_OUTPUT_DATA - 4, 4 ):
         self.write32( offset, 0 )

   def getBits( self, value, fro, to ):
      return ( value >> to ) & ( ( 1<< ( fro - to + 1 ) ) - 1 )

   def allPorts( self,
                 partition=0 ):
      ports = dict()
      assert self.doGas( partition << 8 | self.MRPC_P2P_INFO,
                         self.MRPC_PORTPARTP2P ) == 0
      for offset in range( self.GAS_OUTPUT_DATA, self.GAS_OUTPUT_DATA + 0x10, 4 ):
         ( p0, s0, p1, s1 ) = self.readGroup8( offset )
         if ( p0 != 0 and p0 != 255 and s0 != 0 ):
            ports[ p0 ] = 1
         if ( p1 != 0 and p1 != 255 and s1 != 0 ):
            ports[ p1 ] = 1
      return list( ports )

   def swap16( self, x ):
      return ( ( x >> 8 ) & 0xff ) | ( ( x << 8 ) & 0xff00 )

   def getVersion( self, offset ):
      rawValue = self.read32( offset )
      return formatVersion( rawValue )

   def linksStates( self ):
      LinkStatRequest = Tac.Type( "Microsemi::MRPC::LinkStatRequest" )
      LinkStatPortMap = Tac.Type( "Microsemi::MRPC::LinkStatPortMap" )

      req = LinkStatRequest( LinkStatPortMap.any )
      self.mrpcExec( req )
      return req.output.linkStat

   def NTPartitionGetInfo( self, nt ):
      part = dict()

      ntBase = self.NTxBASE( nt )
      part[ 'ntBase' ] = ntBase

      ( part[ 'ntStatL' ], part[ 'ntStatH' ],
        part[ 'lockedId' ], part[ 'partID' ] ) = self.readGroup8( ntBase )
      ( part[ 'ntOpc' ], _ ) = self.readGroup16( ntBase + 4 )
      part[ 'ntStat' ] = part[ 'ntStatH' ] << 8 | part[ 'ntStatL' ]
      part[ 'ntControl' ] = self.read32( ntBase + 8 )
      ( part[ 'barNum' ], part[ 'barOffset' ] ) = self.readGroup16(
         ntBase + 0xc )
      ( part[ 'ntErrorIndex' ], part[ 'ntError' ] ) = self.readGroup16(
         ntBase + 0x10 )
      part[ 'ntTableError' ] = self.read32( ntBase + 0x18 )
      part[ 'ntRequesterError' ] = self.read32( ntBase + 0x20 )
      for n in self.MRPC_PARTITIONS:
         value = self.read32( self.requesterTable( nt ) + 4 * n )
         part[ f'Requester{n}' ] = {
            'value': value,
            'enabled': value & 1 == 1,
            'NTProxy': value >> 1 & 0xf,
            'RequesterID': value >> 16
         }
      return part

   def NTBBarGetInfo( self, nt, ntBarNo ):
      bar = dict()
      dw = list( range( 4 ) )
      for idx in range( 0, 4 ):
         ( barNum, barOffset ) = self.readGroup16( self.NTxBASE( nt ) + 0xc )
         dw[ idx ] = self.read32( self.NTxBASE( nt ) + barOffset +
                                  ntBarNo * 0x10 + idx * 4 )

         if dw[ 0 ] & 1:
            bar[ 'valid' ] = True

            if dw[ 0 ] & 0b110 == 0:
               bar[ 'mode' ] = "32-bit "
            if dw[ 0 ] & 0b110 == 0b100:
               bar[ 'mode' ] = "64-bit "

            bar[ 'prefetch' ] = bool( dw[ 0 ] & 0b1000 )

            if dw[ 0 ] & 0b10000:
               bar[ 'mappingType' ] = "NT-Direct "
               bar[ 'NtTranslationPosition' ] = 1 << ( dw[ 1 ] & 0x3F )
               bar[ 'NtTranslationSize' ] = ( dw[ 1 ] >> 12 ) * 4 * 1024
               bar[ 'NtDestinationPartition' ] = dw[ 2 ] & 0x3f
               bar[ 'BaseAddr' ] = dw[ 3 ] << 32 | ( ( dw[ 2 ] >> 12 ) << 12 )
            else:
               if dw[ 0 ] & 0b100000:
                  bar[ 'mappingType' ] = "NT-LUT "
               else:
                  del bar[ 'valid' ]

         # This is for debugging
         bar[ 'dw' ] = dw
         bar[ 'barNum' ] = barNum
         bar[ 'barOffset' ] = barOffset
      return bar

   def bind( self,
             port,
             DSP=1, # DSPs 1,2,3 exists on eval board
             partition = 0 ):
      status = self.doGas( self.MRPC_P2P_BIND | int( partition ) << 8 |
                           int( DSP ) << 16 | int( port ) << 24,
                           self.MRPC_PORTPARTP2P )
      return status

   def ntReset( self ):
      for nt in self.MRPC_PARTITIONS:
         def inReset( x ):
            self.write32( self.ntOp( x ), self.MRPC_NTOP_RESET )
            return self.NTPartitionGetInfo( x )[ 'ntStat' ] == 5
         Tac.waitFor( lambda x=nt: inReset( x ), timeout=60 )

   def ntEnable( self ):
      def setNTBar( nt ):
         dw = [ self.MRPC_NT_LUT_WIN_ENABLE | self.MRPC_NT_TYPE_64BIT |
                self.MRPC_NTBARSETUP_VALID,
                self.MRPC_NT_WIN_DIR_SIZE | self.MRPC_NT_DIR_XLATE_POS,
                None,
                self.MRPC_NT_DIR_XLATE_BASE_ADDR_HIGH ]
         # Set target partition
         dw[ 2 ] = 1 - nt
         barOffset = self.MRPC_NTBAR * self.MRPC_NTBARSETUPSIZE
         for idx in range( 0, 4 ):
            self.write32( self.NTxBASE( nt ) + 0xc, barOffset )
            self.write32( self.NTxBASE( nt ) + barOffset +
                          self.MRPC_NTBAR * self.MRPC_NTBARSETUPSIZE + idx * 4,
                          dw[ idx ] )

      def ntIsSet( nt ):
         NTBInfo = self.NTBBarGetInfo( nt, self.MRPC_NTBAR )
         partInfo = self.NTPartitionGetInfo( nt )
         return ( 'valid' in NTBInfo and
                  partInfo[ 'Requester0' ][ 'enabled' ] and
                  partInfo[ 'Requester1' ][ 'enabled' ] )

      def ntSetup():
         for nt in self.MRPC_PARTITIONS:
            if not ntIsSet( nt ):
               self.write32( self.ntOp( nt ), self.MRPC_NTOP_LOCK )
               setNTBar( nt )
               # Translate BDF 0:0.0 -> x:0.1 (Root Complex)
               self.write32( self.requesterTable( nt ) + 0,
                             self.requesterMask( 0, 0, 1, 0 ) )
               # Translate BDF 0:3.0 -> x:1.1 (Host Bridge)
               self.write32( self.requesterTable( nt ) + 4,
                             self.requesterMask( 3, 0, 1, 1 ) )
               self.write32( self.ntOp( nt ), self.MRPC_NTOP_CONF )
         return ntIsSet( 0 ) and ntIsSet( 1 )

      # Setup NTBs to access between sup1 and sup2
      try:
         Tac.waitFor( ntSetup, timeout=5 * 60 )
         assert ntIsSet( 0 ) and ntIsSet( 1 )
      # pylint: disable-msg=bare-except
      except:
         for nt in self.MRPC_PARTITIONS:
            if not ntIsSet( nt ):
               printMsg( f"Microsemi: NTB{nt} is not set, peer is missing" )

   def initPicasso( self ):
      if Cell.cellId() == 1:
         busNo = 0x5
      else:
         busNo = 0x43
      PicassoInit.picassoSetNtBus( busNo, ntFunctionNo=1 )

   def doStartup( self, hitlessBoot ):
      global _enableOutputToConsole
      _enableOutputToConsole = True
      if self.hardwareIsPresent():
         self.mmio = True
         if not os.path.exists( "/mnt/flash/skipEepromUpgrade" ) and not hitlessBoot:
            printMsg( "Updating microsemi eeprom if necessary" )
            # BUG603403: Use only one single firmwareToggle command, after updating
            # both config and firmware, to avoid toggling config when firmware update
            # fails.
            configIsUpdated = self.updateConfig()
            firmwareIsUpdated = self.updateFirmware()
            if ( configIsUpdated or firmwareIsUpdated ):
               self.firmwareToggle( toggleFirmware=firmwareIsUpdated,
                                    toggleConfig=configIsUpdated )
               printMsg( "Microsemi EEPROM Upgraded", toConsole=True )
               Tac.run( [ 'reboot' ] )
               # We don't want NorCalInit to continue execution before the switch
               # is rebooted. As such, we do an infinite loop here to prevent the
               # script from going forward.
               while True:
                  time.sleep( 1 )

         self.initPicasso()

         Tac.run( [ "/sbin/modprobe", "microsemi", "microsemi_debug=0" ] )

   def startup( self, *args, **kwargs ):
      with ExtraTracing( f"{Tracing.defaultTraceHandle().name}/0" ):
         self.doStartup( *args, **kwargs )
