# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
#
# Common tools for reading and parsing data about eMMC flash devices.

from __future__ import absolute_import, division, print_function
import binascii
import os
import six
import Tac

CPUBOARDS_WITH_EMMC = [
   'BelvedereCpu',
   'Crow',
   'Grackle',
   'IslandCpu',
   'Magpie',
   'Mendocino',
   'NewportCpu',
   'NewportPrimeCpu',
   'PrairieIslandCpu',
   'RaspberryIslandCpu',
   'Woodpecker',
]

CROW_PRODUCTS_WITHOUT_EMMC = (
   r'Caspar|'
   r'Cazadero|'
   r'Clearlake$|'
   r'Doran|'
   r'Forestville|'
   r'Guerneville|'
   r'Larkfield|'
   r'Yreka'
)

# Lookup table for flash vendor ID to vendor name
MANUFACTURER_ID_TABLE = {
   '0x03' : 'Toshiba',
   '0x11' : 'Toshiba',
   '0x13' : 'Micron',
   '0x15' : 'Samsung',
   '0x45' : 'Western Digital',
   '0x70' : 'Kingston',
   '0x89' : 'Tricor',
   '0xf6' : 'Smart Modular',
   '0xfe' : 'Micron'
}

# Start/end bit positions of each field within the eMMC flash device CID
# register. Indexed starting at LSB. Offsets for each field in CID register are
# from: http://rere.qmqm.pl/~mirq/JESD84-A44.pdf, page 132
MANUFACTURER_ID_START = 128
MANUFACTURER_ID_END   = 120

PRODUCT_NAME_START    = 104
PRODUCT_NAME_END      = 56

FWREV_UPPER_START     = 56
FWREV_UPPER_END       = 52

FWREV_LOWER_START     = 52
FWREV_LOWER_END       = 48

SERIAL_NUM_START      = 48
SERIAL_NUM_END        = 16

# Read CID register for flash device. Contains information about
# manufacturer, serial number, etc. encoded in a 32-character hex string,
# indexed starting at LSB.
def readCidRegister( device ):
   try:
      with open( os.path.join( "/sys/block", device, "device/cid" ) ) as f:
         return f.read()
   except IOError:
      return ''

# Returns the size of a storage device rounded up to the nearest GB.
def readDeviceSize( device ):
   BYTES_PER_GB = 1000 ** 3

   try:
      fdisk_output = Tac.run( [ "/sbin/fdisk", "-l", "/dev/" + device ], asRoot=True,
                              stdout=Tac.CAPTURE )
   except Tac.SystemCommandError:
      return None

   if "No such file" in fdisk_output:
      return None

   sizeBytes = fdisk_output.split('\n')[0].split(' ')[4]

   # pylint: disable=round-builtin
   return int( round( float( sizeBytes ) / BYTES_PER_GB ) )

# Converts a bit position in a CID register to a character index within the CID
# string returned from the sysfs eMMC driver that contains that bit. The CID
# register is indexed starting at the LSB, but the hex string that represents
# the CID contents is indexed from the MSB.
def __cidBitPosToCharPos( bitPos ):
   CID_NUM_BITS = 128

   if bitPos < 0 or bitPos > CID_NUM_BITS:
      return -1

   return ( CID_NUM_BITS - bitPos ) // 4

def manufacturerId( cid ):
   return '0x' + cid[ __cidBitPosToCharPos( MANUFACTURER_ID_START ) :
                      __cidBitPosToCharPos( MANUFACTURER_ID_END ) ].lower()

def manufacturerName( cid ):
   manfId = manufacturerId( cid ).lower()
   if manfId in MANUFACTURER_ID_TABLE:
      return MANUFACTURER_ID_TABLE[ manfId ]

   return 'Unknown'

def productName( cid ):
   name = cid[ __cidBitPosToCharPos( PRODUCT_NAME_START ) :
               __cidBitPosToCharPos( PRODUCT_NAME_END ) ]
   return six.ensure_str( binascii.unhexlify( name ) )

def firmwareRevision( cid ):
   firmwareRevisionUpper = str( int( \
         cid[ __cidBitPosToCharPos( FWREV_UPPER_START ) :
              __cidBitPosToCharPos( FWREV_UPPER_END ) ], 16 ) )
   firmwareRevisionLower = str( int( \
         cid[ __cidBitPosToCharPos( FWREV_LOWER_START ) :
              __cidBitPosToCharPos( FWREV_LOWER_END ) ], 16 ) )

   return firmwareRevisionUpper + '.' + firmwareRevisionLower

def serialNumber( cid ):
   return cid[ __cidBitPosToCharPos( SERIAL_NUM_START ) :
               __cidBitPosToCharPos( SERIAL_NUM_END ) ]
