# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import os
import re
import glob
import sysconfig
import Tracing
from PassiveMountNamingLib import mountProfileParseAgentName
from PluginTree import PluginTree
t1 = Tracing.trace1  #verbose

class MountProfileParserBase:
   def __init__( self, entMgr, addDefaults, fileName, root ):
      self.fileName = fileName
      self.root = root

   TEMPLATE_RE = re.compile( r"\{.*?\}" )
   AGENTNAME = "agentname"
   MOUNTGROUP = "mountgroup"
   CONDITION = "condition"
   BLOCKER = "blocker"
   PRIORITY = "priority"
   DEFAULT_MOUNTGROUP = "default"
   ALL_MOUNTGROUP = "all"
   INCLUDE = "include"
   PRIO_LOW = 99
   PRIO_HIGH = 1
   COMMENT = "#"
   SUBAGENT = "subagent"
   ALIAS = "alias"
   ROOTFLAGS = "rootflags"
   NODEFAULTMOUNTGROUP = "nodefaultmountgroup"
   MOUNTFILTER = "mountfilter"

   def parseAgentName( self, procTitleLine ):
      return mountProfileParseAgentName( self.AGENTNAME, procTitleLine )

   def run( self, ):
      with open( self.fileName ) as f:
         line = f.readline()
         while line.startswith( self.COMMENT ) or not line.strip():
            line = f.readline()

         procTitleLine = line 
         return self.parseAgentName( procTitleLine )

class MountProfileLoaderBase:
   def __init__( self, parserClass=MountProfileParserBase,
                 pluginType="SysdbMountProfiles" ):
      self.parserClass = parserClass
      self.defaultPath = os.path.join( sysconfig.get_config_var( 'LIBDIR' ),
                                       pluginType + "/" )
      self.agentMountProfileDir = None
      self.passiveProfDir = None
      self.passiveStatusDir = None
      self.pluginTree = PluginTree( "/bld/deps", pluginType )

   def loadProfiles( self, path=None, addDefaults=True, entMgr=None ):
      # If the path is a file, load the file.  If it is a directory, load files there
      searchPath = path or self.defaultPath
      if os.path.isdir( searchPath ):
         fpList = glob.glob( os.path.join( searchPath, "*" ) )
      else:
         fpList = glob.glob( searchPath )

      mpL = []
      inWs = 'A4_CHROOT' in os.environ
      for filePath in fpList:
         if inWs and ';' in filePath:
            # Note from BUG154909:
            # When an rpm is installed, it creates temporary files of the format 
            # '[file];XXXXXX', which is being incorrectly pulled in by the glob.
            # The file got renamed to the proper file later.  This is not a problem 
            # on a switch, as all rpms are installed during boot-up.  Here we skip
            # loading it since this profile won't be needed by current alive agents.
            t1( "skipping temporary mount profile ", filePath )
            continue

         if os.path.isdir( filePath ):
            t1( "skipping directory ", filePath )
            continue

         fileName = os.path.basename( filePath )
         fileDir = os.path.dirname( os.path.abspath( filePath ) )
         if self.pluginTree.canLoad( fileDir, fileName ):
            mp = self.parserClass( entMgr, addDefaults, filePath,
                                   self.passiveProfDir ).run()

            t1( "FileName: " + filePath + " , Agent name: " + str( mp ) )
            if mp is not None:
               mpL.append( mp )
         else:
            # pylint: disable-next=consider-using-f-string
            t1( "Denied loading %s for package %s" % ( filePath,
                                                       self.pluginTree.package ) )
      return mpL
