#!/usr/bin/env python3
# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import glob
import os
import sys
import sysconfig

def defaultPreinitDir():
   """Returns the platform-specific directory where preinits are stored. This is
   `/usr/lib/preinit` for 32-bit platforms and `/usr/lib64/preinit` for 64-bit
   platforms."""
   return os.path.join( sysconfig.get_config_var( 'LIBDIR' ), 'preinit' )

def defaultMountProfileDir():
   """Returns the platform-specific directory where mount profiles are stored. This
   is `/usr/lib/SysdbMountProfiles` for 32-bit platforms and
   `/usr/lib64/SysdbMountProfiles` for 64-bit platforms."""
   return os.path.join( sysconfig.get_config_var( 'LIBDIR' ), 'SysdbMountProfiles' )

def handleExtraPreinits( preinitPaths ):
   """Given a list of paths to preinit files or directories of preinits, load these
   paths into the appropriate environment variables to override standard preinit
   loading. If preinitPaths is None, then forcibly unset the relevant environment
   variables.

   Note that this function may ONLY be called inside EntityManager code. If you are
   looking for similar functionality in a test, look at the `preInitProfiles`
   argument for EntityManager."""
   if preinitPaths:
      defaultDir = defaultPreinitDir()
      fullProfiles = set()
      for path in preinitPaths:
         if os.path.isabs( path ):
            expanded = glob.glob( path )
            # pylint: disable-next=consider-using-f-string
            assert expanded, ( "Unable to find preinit profile matching path '%s'"
            % path )
            fullProfiles.update( expanded )
         else:
            # Make sure that all relative paths can be resolved relative to the
            # default preinit directory
            defaultPath = os.path.join( defaultDir, path )
            expanded = glob.glob( defaultPath )
            # pylint: disable-next=consider-using-f-string
            assert expanded, ( "Given relative path to preinit profile '%s'. All "
            "profiles outside of '%s' must be provided with an absolute path"
            % ( path, defaultDir ) )
            fullProfiles.update( expanded )
      os.environ[ 'SYSDB_EXTRA_PREINIT_PATH' ] = ':'.join( fullProfiles )
      os.environ[ 'MFI_NO_DEFAULT_PREINIT' ] = '1'
   else:
      # Ensure that test env vars are unset if they aren't going to be used
      for var in ( 'SYSDB_EXTRA_PREINIT_PATH', 'MFI_NO_DEFAULT_PREINIT' ):
         oldVal = os.environ.pop( var, None )
         if oldVal is not None:
            # pylint: disable-next=consider-using-f-string
            print( 'WARNING: Removing variable %s from environment. This is '
               'expected if the current test instantiates several EntityManagers '
               'in the same process (e.g. a new EM for each test case). '
               'Otherwise, this is likely affecting your test. Previous value: %s'
               % ( var, oldVal ), file=sys.stderr )
