# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Cell
from EbraTestBridgePlugin.FwdingEtba import (
   forwardingBridgeInit,
   mountNexthopResolver,
)
from IpLibConsts import DEFAULT_VRF
import FibUtils
import os
import QuickTrace
import Tac
import Tracing
import SharedMem
import Smash
from TypeFuture import TacLazyType

from Agent import Agent

# This file is going to be deleted soon, so just ignore these errors for now
# pylint: disable-msg=no-member
# pylint: disable-msg=attribute-defined-outside-init

th = Tracing.Handle( "MplsEtbaAgent" )
t0 = th.trace0

clientMountName = "mpls-etba"
TunnelProgrammingStatus = TacLazyType( "Tunnel::Hardware::TunnelProgrammingStatus" )
FecMode = Tac.Type( "Smash::Fib::FecMode" )
HardwareStatusInstance = TacLazyType( "MplsEtba::HardwareStatusInstance" )

class MplsEtbaContext:
   def __init__( self, entityManager, bridgeStatus ):
      self._entityManager = entityManager
      self._smashEntityManager = SharedMem.entityManager( entityManager.sysname(),
                                                          entityManager.isLocalEm() )
      self.brStatus_ = bridgeStatus

   def em( self ):
      return self._entityManager

   def sEm( self ):
      return self._smashEntityManager

def startHwStatusPopulator(
      hwStatusRoot, scheduler, lfibStatus, arpSmash, bridgingConfig, smashBrStatus,
      ethIntfStatusDir, hwStatus, lfibHwStatus, nhgHwVrfStatus,
      proactiveArpNexthopConfig, *,
      instance=HardwareStatusInstance.unknownInstance ):
   """Start SMs for the new hardware resolver (aid/12121)"""

   hwStatusRoot.viaKeyViaSetKeySm = (
      instance,
      lfibStatus,
      hwStatusRoot.viaKeyViaSetKeyColl,
      hwStatusRoot.backupViaKeyViaSetKeyColl,
   )
   hwStatusRoot.viaSetKeyRouteKeySm = (
      instance,
      lfibStatus,
      hwStatusRoot.viaSetKeyRouteKeyColl,
   )
   hwStatusRoot.arpKeyMonitorSm = (
      instance,
      lfibStatus,
      hwStatusRoot.arpKeyViaKeySetMap
   )
   hwStatusRoot.l3IntfMonitorSm = (
      instance,
      lfibStatus,
      hwStatusRoot.l3IntfViaKeySetMap
   )
   hwStatusRoot.hostKeyMonitorSm = (
      instance,
      lfibStatus,
      arpSmash,
      bridgingConfig,
      hwStatusRoot.l3IntfViaKeySetMap,
      hwStatusRoot.arpKeyViaKeySetMap,
      hwStatusRoot.hostKeyViaKeySetMap
   )
   hwStatusRoot.hardwareViaCollSm = (
      instance,
      lfibStatus,
      bridgingConfig,
      smashBrStatus,
      arpSmash,
      hwStatusRoot.l3IntfViaKeySetMap,
      hwStatusRoot.arpKeyViaKeySetMap,
      hwStatusRoot.hostKeyViaKeySetMap,
      hwStatusRoot.hwViaColl
   )
   hwStatusRoot.l2IntfMonitorSm = (
      instance,
      hwStatusRoot.hwViaColl,
      hwStatusRoot.l2IntfViaKeySetMap,
   )
   hwStatusRoot.hardwareViaUsableSm = (
      instance,
      hwStatusRoot.hwViaColl,
      ethIntfStatusDir,
      hwStatusRoot.l2IntfViaKeySetMap,
      hwStatusRoot.hwViaStatusMap,
   )
   hwStatusRoot.hardwareAdjacencyCollSm = (
      instance,
      lfibStatus,
      hwStatusRoot.hwViaColl,
      hwStatusRoot.viaSetKeyRouteKeyColl,
      hwStatusRoot.hwViaStatusMap,
      hwStatusRoot.viaKeyViaSetKeyColl,
      hwStatusRoot.backupViaKeyViaSetKeyColl,
      hwStatusRoot.hwAdjColl,
      hwStatusRoot.viaSetKeyAdjIndexMap,
   )

   hwStatusRoot.hardwareRouteSm = (
      instance,
      lfibStatus,
      hwStatusRoot.hwAdjColl,
      hwStatusRoot.viaSetKeyRouteKeyColl,
      nhgHwVrfStatus,
      scheduler,
      hwStatus,
      lfibHwStatus,
   )
   hwStatusRoot.hardwareAdjacencyWriterSm = (
      instance,
      hwStatusRoot.viaSetKeyAdjIndexMap,
      hwStatusRoot.hwAdjColl,
      scheduler,
      hwStatus,
   )
   hwStatusRoot.viaSetStatusSm = (
      instance,
      hwStatusRoot.hwAdjColl,
      scheduler,
      lfibHwStatus,
   )
   hwStatusRoot.mcastViaStatusSm = (
      instance,
      lfibStatus,
      hwStatusRoot.viaKeyViaSetKeyColl,
      hwStatusRoot.backupViaKeyViaSetKeyColl,
      hwStatusRoot.hwViaStatusMap,
      scheduler,
      lfibHwStatus,
   )
   if instance != HardwareStatusInstance.egressInstance:
      # Skip ProactiveArpSm for decap LFIB, as egress vias are not relevant there
      hwStatusRoot.proactiveArpSm = (
         hwStatus,
         proactiveArpNexthopConfig,
         Tac.newInstance( 'Arx::SmControl' )
      )

def startMplsEtbaAgent( context, scheduler ):
   em = context.em()
   sEm = context.sEm()
   lfibStatus = sEm.getTacEntity( 'mpls/transitLfib' )
   decapLfibStatus = sEm.getTacEntity( 'mpls/decapLfib' )
   arpSmash = sEm.getTacEntity( 'arp/status' )
   smashBrStatus = sEm.getTacEntity( 'bridging/status' )
   bridgingConfig = em.entity( 'bridging/config' )
   ethIntfStatusDir = em.entity( 'interface/status/eth/intf' )

   hwStatus = em.entity( 'routing/hardware/mpls/status' )
   lfibHwStatus = sEm.getTacEntity( 'hardware/mpls/lfib' )
   # Sysdb HW status is not exported for decapLfib, but SMs use it internally
   decapHwStatus = context.decapHwStatus = Tac.newInstance(
      'Mpls::Hardware::Status', 'decapHwStatus' )
   decapLfibHwStatus = sEm.getTacEntity( 'hardware/mpls/decapLfib' )
   nhgHwStatus = em.entity( 'routing/hardware/nexthopgroup/status' )
   nhgHwVrfStatus = nhgHwStatus.vrfStatus[ DEFAULT_VRF ]
   proactiveArpNexthopConfig = sEm.getTacEntity( 'arp/proactive/config/mpls' )

   routingHardwareStatus = em.entity( 'routing/hardware/status' )
   routingHardwareStatus.mplsPushSupported = True

   # Transit LFIB
   startHwStatusPopulator(
      context.agentRoot.transitHwStatusRoot, scheduler, lfibStatus, arpSmash,
      bridgingConfig, smashBrStatus, ethIntfStatusDir, hwStatus,
      lfibHwStatus, nhgHwVrfStatus, proactiveArpNexthopConfig,
      instance=HardwareStatusInstance.transitInstance )
   # Decap LFIB (Egress)
   startHwStatusPopulator(
      context.agentRoot.decapHwStatusRoot, scheduler, decapLfibStatus,
      arpSmash, bridgingConfig, smashBrStatus, ethIntfStatusDir,
      decapHwStatus, decapLfibHwStatus, nhgHwVrfStatus, None,
      instance=HardwareStatusInstance.egressInstance )

class MplsEtba( Agent ):
   def mountsComplete( self ):
      t0( "call forwardingBridgeInit" )
      forwardingBridgeInit( self.context, clientName=clientMountName )

      startMplsEtbaAgent( self.context, self.agentScheduler() )

      self.setupArfaTypes()
      t0( "init complete" )

   def setupArfaTypes( self ):
      Tac.Type( "Arfa::ArfaTracing" ).init( "MplsEtba" )

   def doInit( self, entityManager ):
      qtFileDefault = "MplsEtba%s.qt" % \
         ( '' if 'QUICKTRACEDIR' in os.environ else '-%d' )
      qtFile = os.environ.get( 'QTFILE', qtFileDefault )
      if qtFile:
         qtSizeSpec = (
            128, # BTRACE_CRIT
            128, # BTRACE_ERROR
            128, # BTRACE_WARN
            32, # BTRACE_INIT
            1024, # BTRACE_INFO
            512, # BTRACE_DEBUG
            2, # BTRACE_DEBUG6
            2, # BTRACE_DEBUG7
            2048, # BTRACE_DEBUG8 (FTRACE)
            2, # BTRACE_VERBOSE
         )
         QuickTrace.initialize( qtFile, sizes=','.join( map( str, qtSizeSpec ) ) )

      root = entityManager.root()
      parent = root.parent
      mplsEtbaDir = parent[ 'MplsEtba' ]
      self.createLocalEntity( "EthIntfStatusDir",
                              "Interface::EthIntfStatusDir",
                              "interface/status/eth/intf" )
      mg = entityManager.mountGroup()
      mg.mount( "arp/monitorednexthop/input/status/mplsEtba",
                "Arp::MonitoredNexthop::Input::Status", "wc" )
      mg.mount( "bridging/config", "Bridging::Config", "r" )
      mg.mount( "l3/config", "L3::Config", "r" )
      mg.mount( "l3/intf/config", "L3::Intf::ConfigDir", "r" )
      mg.mount( "routing/hardware/mpls/backup-status",
                "Mpls::Hardware::Status", "wS" )
      mg.mount( "routing/hardware/mpls/backup-tunnel-status",
                "Mpls::Hardware::Status", "wS" )
      mg.mount( "routing/hardware/mpls/status", "Mpls::Hardware::Status", "wS" )
      mg.mount( "routing/hardware/mpls/tunnel-status",
                "Mpls::Hardware::Status", "wS" )
      mg.mount( "routing/hardware/nexthopgroup/status",
                "Routing::Hardware::NexthopGroupStatus", "r" )
      mg.mount( "routing/hardware/status", "Routing::Hardware::Status", "w" )
      mg.mount( "routing/mpls/status", "Mpls::Status", "rS" )
      mg.mount( "routing/vrf/routingInfo/status", "Tac::Dir", "r" )
      mg.mount( "routing6/vrf/routingInfo/status", "Tac::Dir", "r" )
      mg.mount( "bfd/status/peer", "Bfd::StatusPeer", "r" )
      mg.mount( Cell.path( 'vrf/vrfNameStatus' ),
                           'Vrf::VrfIdMap::NameToIdMapWrapper', 'r' )
      mg.mount( "bridging/etba/status", "Bridging::Etba::Status", "r" )

      mountNexthopResolver( mg, clientName=clientMountName )
      mg.mount( Cell.path( 'ip/vrf/status/local' ), 'Ip::AllVrfStatusLocal', 'r' )
      mg.mount( "ip/vrf/status/global", "Ip::AllVrfStatusGlobal", "r" )

      arpShadow = Smash.mountInfo( "shadow" )
      fwdInfo = FibUtils.forwardingStatusInfo( "keyshadow" )
      fwdInfoReader = FibUtils.forwardingStatusInfo( "reader" )
      routeInfo = FibUtils.routeStatusInfo( "keyshadow" )
      shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
      shmemEm.doMount( 'tunnel/tunnelFib', 'Tunnel::TunnelFib::TunnelFib',
                                         Smash.mountInfo( 'keyshadow' ) )
      shmemEm.doMount(
         'tunnel/tunnelRibs/status/0', 'Tunnel::TunnelTable::TunnelRib',
         Smash.mountInfo( 'keyshadow' ) )

      shmemEm.doMount( 'mpls/transitLfib', 'Mpls::LfibStatus',
                       Smash.mountInfo( 'keyshadow' ) )
      shmemEm.doMount( 'mpls/decapLfib', 'Mpls::LfibStatus',
                       Smash.mountInfo( 'keyshadow' ) )
      shmemEm.doMount( "routing/status", "Smash::Fib::RouteStatus", routeInfo )
      shmemEm.doMount( "routing6/status", "Smash::Fib6::RouteStatus", routeInfo )
      shmemEm.doMount( "arp/status", "Arp::Table::Status", arpShadow )

      proactiveArpMountInfo = (
         Smash.mountInfo( 'writer',
                          collectionInfo=[ ( 'request', 1024 ) ] ) )
      shmemEm.doMount( 'arp/proactive/config/mpls',
                       'Arp::ProactiveArp::ClientConfig',
                       proactiveArpMountInfo )
      shmemEm.doMount( "forwarding/srte/status", "Smash::Fib::ForwardingStatus",
                   fwdInfoReader )
      shmemEm.doMount( "forwarding/status", "Smash::Fib::ForwardingStatus", fwdInfo )
      shmemEm.doMount( "forwarding/unifiedStatus", "Smash::Fib::ForwardingStatus",
                   fwdInfo )
      shmemEm.doMount( "forwarding6/status", "Smash::Fib6::ForwardingStatus",
                       fwdInfo )
      shmemEm.doMount( "forwarding6/unifiedStatus", "Smash::Fib6::ForwardingStatus",
                   fwdInfo )
      shmemEm.doMount( "forwardingGen/unifiedStatus",
                       "Smash::FibGen::ForwardingStatus", fwdInfo )
      shmemEm.doMount( "forwardingGen/status",
                       "Smash::FibGen::ForwardingStatus", fwdInfo )

      lfibHwStatusMountInfo = (
         Smash.mountInfo( 'writer',
            collectionInfo=[ ( 'unprogrammedRoute', 1024 ),
                             ( 'viaSet', 1024 ),
                             ( 'mcastVia', 1024 ) ] ) )
      shmemEm.doMount( "hardware/mpls/lfib", "Mpls::LfibHwStatus",
                       lfibHwStatusMountInfo )
      shmemEm.doMount( "hardware/mpls/decapLfib", "Mpls::LfibHwStatus",
                       lfibHwStatusMountInfo )

      # pylint: disable=attribute-defined-outside-init
      shmemEm.doMount( "bridging/status",
                       "Smash::Bridging::Status",
                       Smash.mountInfo( "shadow" ) )
      self.brStatus_ = Tac.newInstance( "Bridging::Status", "brStatus" )
      self.smashBrStatus_ = shmemEm.getEntity[ "bridging/status" ]
      # start a shim to populate Bridging::Status from changes in the Smash
      # table for the benefit of plugins that look at brStatus_ directly.  Once
      # these plugins are rewritten to use smash, we can eliminate this reactor
      # and the Bridging::Status altogether.
      self.shim_ = Tac.Type( "Bridging::ReaderFdbStatusSm" )( self.brStatus_,
                                                              self.smashBrStatus_ )
      self.context = MplsEtbaContext( entityManager, self.brStatus_ )
      self.context.agentRoot = mplsEtbaDir.createEntity(
         "MplsEtbaAgent::MplsEtbaAgentRoot", "agentRoot" )

      self.localEntitiesCreatedIs( True )
      t0( "close mg" )
      mg.close( self.mountsComplete )

      # Create a CPU Usage quicktracer
      self._cpuUsageCollectorSm = Tac.newInstance(
            "CpuUsage::CpuUsageCollectorSm", "MplsEtba" )
