# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ForwardingHelper import ForwardingHelper
from TypeFuture import TacLazyType
import Tac
import Tracing

th = Tracing.Handle( 'EbraTestBridgeMpls' )
t2 = th.trace2
t3 = th.trace3
t8 = th.trace8

# Tac Types
FecAdjType = TacLazyType( 'Smash::Fib::AdjType' )
FecId = TacLazyType( 'Smash::Fib::FecId' )
FecIdIntfId = TacLazyType( 'Arnet::FecIdIntfId' )
HwL2Via = TacLazyType( 'Mpls::Hardware::L2Via' )
HwVia = TacLazyType( 'Mpls::Hardware::Via' )
IntfId = TacLazyType( 'Arnet::IntfId' )
MplsLabelAction = TacLazyType( 'Arnet::MplsLabelAction' )

class MplsEtbaForwardingHelper( ForwardingHelper ):
   '''
   Helper object to assist in MPLS forwarding functions.  This is used to share
   certain specific forwarding-related code with other modules, and also used
   internally for MPLS routing.
   '''

   def getPopSwapResolvedNexthopInfo( self, dstIp, intf=None ):
      t8( 'getPopSwapResolvedNexthopInfo: Getting info for dstIp', dstIp )
      ret = self.getResolvedNexthopInfo( dstIp, intf )
      return ( ret.resolved, ret.dstMac, ret.nexthopIp, ret.intf, ret.route )

   def extFecViaList( self, extFecLfibVia ):
      '''
      Return a list containing a single Mpls::HwVia object that represents the
      provided Mpls::LfibExtFecVia, extFecLfibVia.

      The MplsHardwareStatusSm stores a mapping of label to active LfibViaSet
      for LFIB routes. For viaTypeExternalFec via sets, LfibViaSet has an has an
      LfibExtFecVia, but needs an Mpls::HwVia that can be recursively
      resolved / flattened by resolveHierarchical().
      '''
      func = 'extFecViaList'

      validActions = ( MplsLabelAction.pop, MplsLabelAction.forward )
      if extFecLfibVia.labelAction not in validActions:
         t8( func, 'mplsExternalFec adjacency', extFecLfibVia.key,
             'has invalid labelAction', extFecLfibVia.labelAction )
         self._counts[ 'extFecInvalidLabelAction' ].inc()
         return None

      via = HwVia()
      l2Via = HwL2Via()
      l2Via.labelAction = extFecLfibVia.labelAction
      l2Via.payloadType = extFecLfibVia.payloadType
      # l2Via.outLabel has default value
      # l2Via.macAddr has default value
      via.l2Via = l2Via
      fecIdIntfId = FecId.fecIdToHierarchicalIntfId( extFecLfibVia.fecId )
      via.l3Intf = fecIdIntfId
      return [ via ]

def mplsEtbaForwardingHelperFactory( bridge ):
   return MplsEtbaForwardingHelper(
         bridgingStatus=bridge.brStatus,
         vrfRoutingStatus=bridge.vrfRoutingStatus_,
         vrfRouting6Status=bridge.vrfRouting6Status_,
         forwardingStatus=bridge.forwardingStatus_,
         forwarding6Status=bridge.forwarding6Status_,
         forwardingGenStatus=bridge.forwardingGenStatus_,
         srTeForwardingStatus=bridge.srTeForwardingStatus_,
         srTeSegmentListTunnelTable=bridge.srTeSegmentListTunnelTable,
         nhgEntryStatus=bridge.nhgEntryStatus_,
         arpSmash=bridge.arpSmash_,
         tunnelFib=bridge.tunnelFib_,
         trie4=bridge.vrfTrie_,
         trie6=bridge.vrfTrie6_,
         vrfNameStatus=bridge.vrfNameStatus_,
         intfConfigDir=bridge.intfConfigDir_,
         ethIntfStatusDir=bridge.ethIntfStatusDir_
      )

def mplsEtbaForwardingHelperKwFactory( **kwargs ):
   '''
   This factory function should only be used for tests that are mocking
   or using only small pieces of the Helper.  Actual plugin should use the
   mplsEtbaForwardingHelperFactory( bridge ) factory function instead.
   '''
   args = [
      "bridgingStatus",
      "vrfRoutingStatus",
      "vrfRouting6Status",
      "forwardingStatus",
      "forwarding6Status",
      "forwardingGenStatus",
      "srTeForwardingStatus",
      "srTeSegmentListTunnelTable",
      "nhgEntryStatus",
      "arpSmash",
      "tunnelFib",
      "trie4",
      "trie6",
      "vrfNameStatus",
      "intfConfigDir",
      "ethIntfStatusDir"
   ]
   defaultKwArgs = { k: None for k in args }
   defaultKwArgs.update( kwargs )
   return MplsEtbaForwardingHelper( **defaultKwArgs )
