# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string
# pylint: disable=singleton-comparison

import Tac
import Tracing
import BothTrace
import collections
import re
import signal
import socket
import sys
from ClientCommonLib import (
   errorSubTlvLspPingRetCodeStr,
   errorTlvLspPingRetCodeStr,
   getThreadLocalData,
   lspPingRetCodeStr,
   LspPingReturnCode,
   LspPingTypeBgpLu,
   LspPingTypeGeneric,
   LspPingTypeLdp,
   LspPingTypeMldp,
   LspPingTypePwLdp,
   LspPingTypeRsvp,
   LspPingTypeSr,
   LspPingTypeVpn,
   receiveMessage,
   setThreadLocalData,
   sendViaSocket,
)

from PingModel_pb2 import ( # pylint: disable=no-name-in-module
   ErrInfo as Err,
   MplsPing,
)
from PseudowireLib import (
   pwPingCcMsg,
   pwPingCcValToEnum,
)
LspPingNhgModeAllEntries, LspPingNhgModeOneEntry = 0, 1
igpProtocolType = Tac.Type( 'LspPing::LspPingSrIgpProtocol' )
# --------------------------------
# BothTrace Short hand variables
# --------------------------------
__defaultTraceHandle__ = Tracing.Handle( "MplsPingClientLib" )
bv = BothTrace.Var
bt8 = BothTrace.trace8

# ---------------------------------------------------------
#           Ping render helpers
# ---------------------------------------------------------
def getProtocolStr( protocol ):
   protocolToStrDict = {
      LspPingTypeBgpLu : "BGP labeled unicast",
      LspPingTypeVpn : "VPN",
      LspPingTypeLdp : "LDP",
      LspPingTypeMldp : "MLDP",
      LspPingTypeSr : "Segment-Routing",
      LspPingTypeGeneric : "Generic",
      LspPingTypeRsvp : "RSVP",
      LspPingTypePwLdp : "LDP pseudowire",
   }
   if protocol in protocolToStrDict:
      return protocolToStrDict[ protocol ]
   return ""

def getIgpProtocolStr( igpProtocol ):
   igpProtocolToStrDict = {
      igpProtocolType.isis : "IS-IS",
      igpProtocolType.ospf : "OSPF"
   }
   if igpProtocol in igpProtocolToStrDict:
      return igpProtocolToStrDict[ igpProtocol ]
   return ""

def renderStatisticsHeader( pingModel, protocol ):
   if getThreadLocalData( 'printStats' ):
      return

   if protocol == LspPingTypeVpn:
      prefix = pingModel.prefix
      print( "\n--- VPN MPLS route " + prefix + ": lspping statistics ---" )
   elif protocol in [ LspPingTypeLdp, LspPingTypeMldp,
                      LspPingTypeGeneric, LspPingTypeBgpLu, LspPingTypeRsvp ]:
      prefix = pingModel.prefix
      protocolStr = getProtocolStr( protocol )
      print( "\n--- {} target fec {} : lspping statistics ---".format(
             protocolStr, prefix ) )
   elif protocol == LspPingTypeSr:
      prefix = pingModel.prefix
      algorithm = pingModel.algorithm
      protocolStr = getProtocolStr( protocol )
      algoStr = f", algorithm {algorithm}" if algorithm else ""
      print( "\n--- {} target fec {}{} : lspping statistics ---".format(
             protocolStr, prefix, algoStr ) )
   elif protocol == LspPingTypePwLdp:
      pwLdpName = pingModel.pwLdpName
      protocolStr = getProtocolStr( protocol )
      print( "\n--- {} {} : lspping statistics ---".format( protocolStr,
                                                            pwLdpName ) )
   setThreadLocalData( 'printStats', True )

def getPingProtocolModel( pingHdr ):
   # add for more protocols as capi is supported for them.
   if pingHdr.protocol == LspPingTypeVpn:
      return pingHdr.vpnModel
   elif pingHdr.protocol in [ LspPingTypeLdp, LspPingTypeMldp, LspPingTypeGeneric ]:
      return pingHdr.genericModel
   elif pingHdr.protocol == LspPingTypeSr:
      return pingHdr.srModel
   elif pingHdr.protocol == LspPingTypeBgpLu:
      return pingHdr.bgpLuModel
   elif pingHdr.protocol == LspPingTypeRsvp:
      return pingHdr.rsvpModel
   elif pingHdr.protocol == LspPingTypePwLdp:
      return pingHdr.pwLdpModel
   else:
      return None

def sendOrRenderPingErr( err, sock=None ):
   if sock:
      errInfo = Err( err=err )
      sendViaSocket( sock, MplsPing( errInfo=errInfo ) )
   else:
      print( err )

def renderRequest( pingHdr ):
   output = ""
   if pingHdr.protocol == LspPingTypeVpn:
      output += ( 'LSP ping to VPN Prefix {}, Route Distinguisher {}'.format(
                            pingHdr.vpnModel.prefix, pingHdr.vpnModel.rd ) )
   elif pingHdr.protocol in [ LspPingTypeLdp, LspPingTypeMldp, LspPingTypeGeneric ]:
      protocolStr = getProtocolStr( pingHdr.protocol )
      output += ( 'LSP ping to {} route {}'.format( protocolStr,
                                                   pingHdr.genericModel.prefix ) )
   elif pingHdr.protocol == LspPingTypeSr:
      algoName = pingHdr.srModel.algorithm
      protocolStr = getProtocolStr( pingHdr.protocol )
      algoStr = f", algorithm {algoName}" if algoName else ""
      if pingHdr.srModel.igpProtocolType:
         igpProtocol = pingHdr.srModel.igpProtocolType
         output += ( 'LSP ping to {} {} route {}{}'.format( getIgpProtocolStr(
                                                            igpProtocol ),
                                                            protocolStr,
                                                            pingHdr.srModel.prefix,
                                                            algoStr ) )
      else:
         output += ( 'LSP ping to {} route {}{}'.format( protocolStr,
                                                        pingHdr.srModel.prefix,
                                                        algoStr ) )
   elif pingHdr.protocol == LspPingTypeBgpLu:
      protocolStr = getProtocolStr( pingHdr.protocol )
      output += ( 'LSP ping to {} tunnel {}'.format( protocolStr,
                                                    pingHdr.bgpLuModel.prefix ) )
   elif pingHdr.protocol == LspPingTypeRsvp:
      lspId = pingHdr.rsvpModel.lspId
      session = pingHdr.rsvpModel.session
      tunnel = pingHdr.rsvpModel.tunnel
      subTunnelId = pingHdr.rsvpModel.subTunnelId
      if session:
         # session could be name or Id
         sessionStr = f"#{session}" if session.isdigit() else session
         lspStr = f" LSP #{lspId}" if lspId >= 0 else ""
         output += f"LSP ping to RSVP session {sessionStr}{lspStr}"
      else:
         output += f"LSP ping to RSVP tunnel {tunnel}"
         if subTunnelId:
            subTunnelStr = f" sub-tunnel #{subTunnelId}"
            output += subTunnelStr
   elif pingHdr.protocol == LspPingTypePwLdp:
      pwLdpName = pingHdr.pwLdpModel.pwLdpName
      pwNeighbor = pingHdr.pwLdpModel.pwNeighbor
      pwId = pingHdr.pwLdpModel.pwId
      pwLabel = pingHdr.pwLdpModel.pwLabel
      pwCCType = pingHdr.pwLdpModel.pwCCType
      output += f"LSP ping to LDP pseudowire {pwLdpName}"
      if pwNeighbor:
         output += f'\nNeighbor {pwNeighbor}'
      if pwId:
         # vplsAdPwGenId will have format of [VPLS ID, Router ID, PE Address]
         if re.search( r"\[.*\]", pwId ):
            output += f", pseudowire {pwId}"
         else:
            output += f", pseudowire id {pwId}"
      if pwLabel:
         output += f", pseudowire label {pwLabel}"
      if pwCCType:
         output += f"\n   CC type: {pwPingCcMsg[ pwCCType ]}"
   # Rendering timeout and interval for each capi-supported protocol.
   output += ( '\n   timeout is {}ms, interval is {}ms'.format(
                  pingHdr.timeout * 1000, pingHdr.interval * 1000 ) )
   print( output )

# ---------------------------------------------------------
#           Ping model generation helpers
# ---------------------------------------------------------

def getPingModel():
   if getThreadLocalData( 'cache' ) is not None:
      return
   else:
      try:
         s = getThreadLocalData( 's' )
         s.settimeout( 1 ) # set timeout to return
         clientsocket = getThreadLocalData( 'cs' )
         if not clientsocket:
            clientsocket, _ = s.accept()
            bt8( "Create client socket ", bv( clientsocket.getsockname() ) )
            setThreadLocalData( 'cs', clientsocket )
         clientsocket.settimeout( 3 ) # set timeout to return
         msg = receiveMessage( clientsocket )
         setThreadLocalData( 'cache', msg )
         if not msg:
            raise socket.timeout
         storeAndRenderModel( render=getThreadLocalData( 'render' ) )
      except socket.timeout:
         # either a process is spawned by the cli or a thread is
         # spawned by binary LspTraceroute.
         p = getThreadLocalData( 'p' )
         t = getThreadLocalData( 't' )
         if not p and not t:
            # either process or thread should be present
            bt8( "No process or thread present" )
            setThreadLocalData( 'cmdBreak', True )
         if p and p.poll() != None: # subprocess is not running
            bt8( "Subprocess was created and is not running anymore" )
            setThreadLocalData( 'cmdBreak', True )
         if t and not t.is_alive(): # thread is not running
            bt8( "Thread was created and is not running anymore" )
            setThreadLocalData( 'cmdBreak', True )

def getDataFromClientSocket():
   while True:
      if getThreadLocalData( 'cache' ):
         storeAndRenderModel( getThreadLocalData( 'render' ) )
      clientsocket = getThreadLocalData( 'cs' )
      if not clientsocket:
         break
      msg = receiveMessage( clientsocket )
      if not msg:
         break
      setThreadLocalData( 'cache', msg )

def renderAndCleanPingSocket():
   ''' Cleanup client and server socket and also render statistics 
       in case KbINT or 'q' was pressed and data remains in socket buffer'''
   sock = getThreadLocalData( 's' )
   # Before exiting, get all remaining data from client socket buffer
   # as we might have pressed kbInt ( Ctrl + C ) and there might be
   # remaining data in sock buffer
   # We can also reach here by pressing `q` which generates SIGPIPE in cli but
   # is not propagated to child process running LspPing/LspTraceroute binary.
   # NOTE: `q` becomes effective only when the wrapping/paging happens
   # ( i.e --More-- shows up ) at cli.
   # Pressing `q` before registers it but the cli only takes it into effect when
   # wrapping/paging happens i.e. --More-- apprears on CLI.
   # Hence, send the signal to child process to stop and generate stats if any.
   proc = getThreadLocalData( 'p' )
   if proc:
      proc.send_signal( signal.SIGINT )
   getDataFromClientSocket()
   if sock:
      bt8( "Clean up server socket ", bv( sock.getsockname() ) )
      sock.shutdown( socket.SHUT_RDWR )
      sock.close()
   setThreadLocalData( 'cs', None )
   setThreadLocalData( 's', None )

def storeAndRenderModel( render=False ):
   # pylint: disable=no-member
   # pylint: disable-next=import-outside-toplevel
   from CliDynamicSymbol import CliDynamicPlugin
   MplsUtilModel = CliDynamicPlugin( "MplsUtilModel" )
   BgpLuModel = MplsUtilModel.BgpLuModel
   ErrInfo = MplsUtilModel.ErrInfo
   GenericModel = MplsUtilModel.GenericModel
   MplsPingHdr = MplsUtilModel.MplsPingHdr
   MplsPingReply = MplsUtilModel.MplsPingReply
   MplsPingStatistics = MplsUtilModel.MplsPingStatistics
   MplsPingStatisticsSummary = MplsUtilModel.MplsPingStatisticsSummary
   MplsPingVia = MplsUtilModel.MplsPingVia
   PwLdpModel = MplsUtilModel.PwLdpModel
   RsvpModel = MplsUtilModel.RsvpModel
   SrModel = MplsUtilModel.SrModel
   VpnModel = MplsUtilModel.VpnModel

   ping = MplsPing()
   ping.ParseFromString( getThreadLocalData( 'cache' ) )
   if ping.HasField( 'mplsPingProtocol' ):
      mplsPingProtocol = ping.mplsPingProtocol
      pingHdr = MplsPingHdr( protocol=str( mplsPingProtocol.protocol ),
                             timeout=mplsPingProtocol.timeout,
                             interval=mplsPingProtocol.interval )
      setThreadLocalData( 'pingHdr', pingHdr )
      setThreadLocalData( 'printStats', False )
   elif ping.HasField( 'bgpLuModel' ):
      bgpLuModel = ping.bgpLuModel
      pingHdr = getThreadLocalData( 'pingHdr' )
      # Parent model instance for ping which contains all other submodel
      # instances. There can be only one pingModel instance per cli cmd output.
      pingModel = (
         BgpLuModel( prefix=str( bgpLuModel.prefix ) ) )
      pingHdr.bgpLuModel = pingModel
      if render:
         renderRequest( pingHdr )
   elif ping.HasField( 'vpnModel' ):
      vpnModel = ping.vpnModel
      pingHdr = getThreadLocalData( 'pingHdr' )
      # Parent model instance for ping which contains all other submodel
      # instances. There can be only one pingModel instance per cli cmd output.
      pingModel = (
         VpnModel( prefix=str( vpnModel.prefix ),
                   rd=str( vpnModel.rd ) ) )
      pingHdr.vpnModel = pingModel
      if render:
         renderRequest( pingHdr )
   elif ping.HasField( 'genericModel' ):
      genericModel = ping.genericModel
      pingHdr = getThreadLocalData( 'pingHdr' )
      pingModel = GenericModel( prefix=str( genericModel.prefix ) )
      pingHdr.genericModel = pingModel
      if render:
         renderRequest( pingHdr )
   elif ping.HasField( 'srModel' ):
      srModel = ping.srModel
      pingHdr = getThreadLocalData( 'pingHdr' )
      pingModel = SrModel( prefix=str( srModel.prefix ),
                           algorithm=str( srModel.algorithm ),
                           igpProtocolType=str( srModel.igpProtocolType ) )
      pingHdr.srModel = pingModel
      if render:
         renderRequest( pingHdr )
   elif ping.HasField( 'rsvpModel' ):
      rsvpModel = ping.rsvpModel
      pingHdr = getThreadLocalData( 'pingHdr' )
      pingModel = RsvpModel( prefix=rsvpModel.prefix,
                             lspId=rsvpModel.lspId,
                             session=rsvpModel.session,
                             tunnel=rsvpModel.tunnel,
                             subTunnelId=rsvpModel.subTunnelId )
      pingHdr.rsvpModel = pingModel
      if render:
         renderRequest( pingHdr )
   elif ping.HasField( 'pwLdpModel' ):
      pwLdpModel = ping.pwLdpModel
      pingHdr = getThreadLocalData( 'pingHdr' )
      pwCCType = pwPingCcValToEnum[ pwLdpModel.pwCCType ]
      pingModel = PwLdpModel( pwLdpName=pwLdpModel.pwLdpName,
                              pwNeighbor=pwLdpModel.pwNeighbor,
                              pwId=pwLdpModel.pwId,
                              pwLabel=pwLdpModel.pwLabel,
                              pwCCType=pwCCType )
      pingHdr.pwLdpModel = pingModel
      if render:
         renderRequest( pingHdr )
   elif ping.HasField( 'mplsPingVia' ):
      mplsPingVia = ping.mplsPingVia
      pingHdr = getThreadLocalData( 'pingHdr' )
      pingModel = getPingProtocolModel( pingHdr )
      labels = list( map( int, mplsPingVia.labelStack ) )
      viaModel = MplsPingVia( resolved=mplsPingVia.resolved,
                              nextHopIp=str( mplsPingVia.nextHopIp ),
                              interface=str( mplsPingVia.interface ),
                              labelStack=labels )
      # Rsvp specific fields. For other protocols these fields will be empty so we
      # don't affect existing test models
      if mplsPingVia.lspId >= 0:
         viaModel.lspId = mplsPingVia.lspId
      if mplsPingVia.subTunnelId:
         viaModel.subTunnelId = mplsPingVia.subTunnelId

      if mplsPingVia.statisticsVia:
         pingHdr.statsModel += ( [ viaModel ] )
      else:
         pingHdr.viaModel += ( [ viaModel ] )
      if render:
         # render the most recently added via from the list
         if mplsPingVia.statisticsVia:
            renderStatisticsHeader( pingModel, pingHdr.protocol )
            renderVia( pingHdr.statsModel[ -1 ] )
         else:
            renderVia( pingHdr.viaModel[ -1 ] )
   elif ping.HasField( 'mplsPingStatisticsSummary' ):
      mplsPingStatisticsSummary = ping.mplsPingStatisticsSummary
      pingHdr = getThreadLocalData( 'pingHdr' )
      pingHdr.statsModel[ -1 ].pingStats = (
         MplsPingStatisticsSummary(
            packetsTransmitted=mplsPingStatisticsSummary.packetsTransmitted,
            packetsReceived=mplsPingStatisticsSummary.packetsReceived,
            packetLoss=mplsPingStatisticsSummary.packetLoss,
            replyTime=mplsPingStatisticsSummary.replyTime ) )
      if render:
         renderPingStatsSummary( pingHdr.statsModel[ -1 ].pingStats )
   elif ping.HasField( 'mplsPingStatistics' ):
      mplsPingStatistics = ping.mplsPingStatistics
      pingHdr = getThreadLocalData( 'pingHdr' )
      pingHdr.statsModel[ -1 ].pingStats.mplsStatistics += (
         [ MplsPingStatistics(
                        destination=str( mplsPingStatistics.destination ),
                        roundTripTimeMin=mplsPingStatistics.roundTripTimeMin,
                        roundTripTimeMax=mplsPingStatistics.roundTripTimeMax,
                        roundTripTimeAvg=mplsPingStatistics.roundTripTimeAvg,
                        roundTripTimeMinUs=mplsPingStatistics.roundTripTimeMinUs,
                        roundTripTimeMaxUs=mplsPingStatistics.roundTripTimeMaxUs,
                        roundTripTimeAvgUs=mplsPingStatistics.roundTripTimeAvgUs,
                        oneWayDelayMin=mplsPingStatistics.oneWayDelayMin,
                        oneWayDelayMax=mplsPingStatistics.oneWayDelayMax,
                        oneWayDelayAvg=mplsPingStatistics.oneWayDelayAvg,
                        packetsReceived=mplsPingStatistics.packetsReceived )
          ] )
      if render:
         renderPingStats( pingHdr.statsModel[ -1 ].pingStats.mplsStatistics[ -1 ] )
   elif ping.HasField( 'mplsPingReply' ):
      mplsPingReply = ping.mplsPingReply
      pingHdr = getThreadLocalData( 'pingHdr' )
      pingHdr.viaModel[ -1 ].pingReply = (
         MplsPingReply( replyHost=str( mplsPingReply.replyHost ),
                        sequence=mplsPingReply.sequence,
                        roundTripTime=mplsPingReply.roundTripTime,
                        roundTripTimeUs=mplsPingReply.roundTripTimeUs,
                        oneWayDelay=mplsPingReply.oneWayDelay,
                        retCode=str( mplsPingReply.retCode ) ) )
      if render:
         renderPingReply( pingHdr.viaModel[ -1 ].pingReply )
   elif ping.HasField( 'errInfo' ):
      errInfo = ping.errInfo
      pingHdr = getThreadLocalData( 'pingHdr' )
      pingHdr.errInfo += [ ErrInfo( error=errInfo.err ) ]
      if render:
         print( errInfo.err )
   else:
      # should never come here as all models covered up.
      bt8( "Received unknown model", bv( ping.SerializeToString() ) )
   sys.stdout.flush()
   # pylint: enable=no-member

# ---------------------------------------------------------
#               general render helpers
# ---------------------------------------------------------

def renderPingReply( replyPktInfo,
                     expectedRetCode=LspPingReturnCode.repRouterEgress ):
   string = "Reply from " + str( replyPktInfo.replyHost )
   string += ": seq="
   string += str( replyPktInfo.sequence )
   string += ", rtt="
   string += str( replyPktInfo.roundTripTime ) + '.'
   string += str( replyPktInfo.roundTripTimeUs )
   string += "ms" # millisecond
   string += ", 1-way="
   string += str( replyPktInfo.oneWayDelay // 1000 ) + '.'
   string += str( replyPktInfo.oneWayDelay % 1000 )
   string += "ms" # millisecond
   if expectedRetCode:
      string += ","
      if replyPktInfo.retCode == expectedRetCode:
         string += " success: "
      else:
         string += " error: "
      if replyPktInfo.errorSubTlvMap:
         string += errorSubTlvLspPingRetCodeStr( replyPktInfo )
      elif replyPktInfo.errorTlvMap:
         string += errorTlvLspPingRetCodeStr( replyPktInfo )
      else:
         string += lspPingRetCodeStr( replyPktInfo.retCode )
   print( '   ' + string )

def lspPingReplyStr( replyPktInfo, expectedRetCode=None ):
   if replyPktInfo:
      string = "Reply from " + str( replyPktInfo.replyHost )
      string += ": seq="
      string += str( replyPktInfo.seqNum )
      string += ", rtt="
      string += str( replyPktInfo.roundTrip // 1000 ) + '.'
      string += str( replyPktInfo.roundTrip % 1000 )
      string += "ms" # millisecond
      string += ", 1-way="
      string += str( replyPktInfo.oneWayDelay // 1000 ) + '.'
      string += str( replyPktInfo.oneWayDelay % 1000 )
      string += "ms" # millisecond
      if expectedRetCode:
         string += ","
         if replyPktInfo.retCode == expectedRetCode:
            string += " success: "
         else:
            string += " error: "
         if replyPktInfo.errorSubTlvMap:
            string += errorSubTlvLspPingRetCodeStr( replyPktInfo )
         elif replyPktInfo.errorTlvMap:
            string += errorTlvLspPingRetCodeStr( replyPktInfo )
         else:
            string += lspPingRetCodeStr( replyPktInfo.retCode )
   else:
      string = "Request timeout"
   return string

# ---------------------------------------------------------
#           VPN ping render helpers
# ---------------------------------------------------------

def lspPingVpnReplyRender( clientId, replyPktInfo, renderArgs ):
   lspPingStaticReplyRender( clientId, replyPktInfo, renderArgs )

def lspPingVpnStatisticsRender( clientIds, time, txPkts, replyInfo,
                                delayInfo, renderArgs ):
   clientIdToVias, prefix, unresolvedVias = renderArgs
   if not txPkts:
      return

   print( "\n--- VPN MPLS route " + prefix + ": lspping statistics ---" )
   lspPingStatisticsRender( clientIds, time, txPkts, replyInfo, delayInfo,
                            clientIdToVias, unresolvedVias )

# ---------------------------------------------------------
#           static ping render helpers
# ---------------------------------------------------------

def renderVia( viaInfo ):
   # A valid viaInfo.lspId means we need to print via header
   # lspId will either be >=0 or None
   if isinstance( viaInfo.lspId, int ):
      print( f'LSP {viaInfo.lspId}' )
   if viaInfo.subTunnelId:
      print( f'SubTunnel {viaInfo.subTunnelId}' )
   labelStr = f'label stack: {viaInfo.labelStack}'
   output = 'Via {}, {}, {}'.format( viaInfo.nextHopIp,
                                     viaInfo.interface,
                                     labelStr )
   print( output )

def printVias( vias, resolved=True ):
   for via in vias:
      intf = '%s, ' % via[ 2 ] if resolved else ''
      if isinstance( via[ 1 ], int ):
         labels = [ via[ 1 ] ]
      else:
         labels = list( via[ 1 ] )
      labelStr = f'label stack: {labels}'
      print( f'Via {via[ 0 ]}, {intf}{labelStr}' )

def lspPingStaticReplyRender( clientId, replyPktInfo, renderArgs ):
   vias = renderArgs[ 0 ][ clientId ]
   printVias( vias )
   print( '   ' + lspPingReplyStr(
                     replyPktInfo,
                     expectedRetCode=LspPingReturnCode.repRouterEgress ) )

def lspPingStaticStatisticsRender( clientIds, time, txPkts, replyInfo,
                                   delayInfo, renderArgs ):
   clientIdToVias, prefix, unresolvedVias = renderArgs
   if not txPkts:
      return

   print( "\n--- static MPLS push-label route " + prefix +
          ": lspping statistics ---" )
   lspPingStatisticsRender( clientIds, time, txPkts,
                            replyInfo, delayInfo, clientIdToVias, unresolvedVias )

def renderPingStatsSummary( pingStats ):
   string = '   '
   string += str( pingStats.packetsTransmitted ) + " packets transmitted, "
   string += str( pingStats.packetsReceived ) + " received, "
   string += str( pingStats.packetLoss ) + "% packet loss, time "
   string += str( pingStats.replyTime ) + "ms"
   print( string )

def renderPingStats( pingStats ):
   string = '   '
   string += ( '{} received from {}, rtt min/max/avg '
               '{}.{}/{}.{}/{}.{} ms, 1-way min/max/avg '
               '{}/{}/{} ms'.format( pingStats.packetsReceived,
                                     pingStats.destination,
                                     pingStats.roundTripTimeMin,
                                     pingStats.roundTripTimeMinUs,
                                     pingStats.roundTripTimeMax,
                                     pingStats.roundTripTimeMaxUs,
                                     pingStats.roundTripTimeAvg,
                                     pingStats.roundTripTimeAvgUs,
                                     pingStats.oneWayDelayMin / 1000, # us to ms conv
                                     pingStats.oneWayDelayMax / 1000,
                                     pingStats.oneWayDelayAvg / 1000 ) )
   print( string + "\n" )

def lspPingStatisticsRender( clientIds, time, txPkts,
                             replyInfo, delayInfo, clientIdToVias, unresolvedVias ):
   for clientId in clientIds:
      # sum up all recorded RTTs to figure out the total # of echo replies
      if clientId in replyInfo:
         recvNum = sum( len( v ) for v in replyInfo[ clientId ].values() )
      else:
         recvNum = 0
      lossRate = 100 - recvNum * 100 // txPkts[ clientId ]
      string = ''
      vias = clientIdToVias.get( clientId )
      if vias is not None:
         printVias( vias )
         string += '   '
      string += str( txPkts[ clientId ] ) + " packets transmitted, "
      string += str( recvNum ) + " received, "
      string += str( lossRate ) + "% packet loss, time " + str( time ) + "ms"
      print( string )
      if clientId in replyInfo:
         ll = []
         for host, rtts in replyInfo[ clientId ].items():
            delays = delayInfo[ clientId ][ host ]
            if rtts:
               ll.append( '{} received from {}, rtt min/max/avg '
                          '{}/{}/{} ms, 1-way min/max/avg {}/{}/{} ms'
                          .format( len( rtts ), host,
                                   min( rtts ),
                                   max( rtts ),
                                   ( sum( rtts ) / len( rtts ) ),
                                   min( delays ),
                                   max( delays ),
                                   ( sum( delays ) / len( delays ) ) ) )
            print( '   ' + ', '.join( ll ) )
      print()

   if unresolvedVias:
      printVias( unresolvedVias, resolved=False )
      print( '   Not resolved' )
      print()

# ---------------------------------------------------------
#           nhg ping render helpers
# ---------------------------------------------------------

def printTunnels( tunnels, backup=False ):
   for tunnel in sorted( tunnels, key=lambda entry: ( entry.entryIndex,
                                                      entry.nexthop ) ):
      print( f'{"Backup entry" if backup else "Entry"} {tunnel.entryIndex}' )
      print( f'   Via {tunnel.nexthop}' )

def printUnresolvedTunnels( tunnels, backup=False ):
   string = 'Backup entry' if backup else 'Entry'
   maxLineLen = 66
   start = len( string )
   lineLen = start
   for tunnel in sorted( tunnels ):
      l = len( ' %d' % tunnel )
      if lineLen + l > maxLineLen:
         string += '\n' + ' ' * start
         lineLen = start    # reset lineLen
      string += ' %d' % tunnel
      lineLen += l
   print( string )

def lspPingNhgReplyRender( clientId, replyPktInfo, renderArgs ):
   tunnels = renderArgs[ 0 ][ clientId ]
   clientIdBaseToNhgName = renderArgs[ 4 ]
   prefix = renderArgs[ 5 ]
   nhgNameToNhgTunnelIdx = renderArgs[ 7 ]
   if prefix is not None and clientId in clientIdBaseToNhgName:
      nhgName = clientIdBaseToNhgName[ clientId ]
      if nhgNameToNhgTunnelIdx:
         print( f'\n{prefix}: nexthop-group tunnel index '
                f'{nhgNameToNhgTunnelIdx[ nhgName ]} (nexthop-group name: '
                f'{clientIdBaseToNhgName[ clientId ]})' )
      else:
         print( f'\n{prefix}: nexthop-group route (nexthop-group name: {nhgName})' )
   printTunnels( tunnels, backup=renderArgs[ 8 ] )
   print( '   ' + lspPingReplyStr( replyPktInfo,
                               expectedRetCode=LspPingReturnCode.repRouterEgress ) )

def lspPingNhgStatisticsRender( clientIds, time, txPkts, replyInfo,
                                delayInfo, renderArgs ):
   ( clientIdToTunnels, nhgNames, nhgNameToClientIds,
     nhgNameToUnresolvedTunnels, _, _, mode, nhgNameToNhgTunnelIdx,
     backup ) = renderArgs
   if not txPkts:
      return

   for nhgName in nhgNames:
      entryInfo = ''
      if mode == LspPingNhgModeOneEntry:
         entryTunnel = clientIdToTunnels[ nhgNameToClientIds[ nhgName ][ 0 ] ][ 0 ]
         entryInfo = ( f' {"backup entry " if backup else "entry "}'
                       f'{entryTunnel.entryIndex}' )
      if nhgNameToNhgTunnelIdx:
         print( "\n--- nexthop-group tunnel index %d, "
                "nexthop-group %s%s: lspping statistics ---" %
                ( nhgNameToNhgTunnelIdx[ nhgName ], nhgName, entryInfo ) )
      else:
         print( "\n--- nexthop-group %s%s: lspping statistics ---" %
                ( nhgName, entryInfo ) )
      for clientId in nhgNameToClientIds[ nhgName ]:
         # sum up all recorded RTTs to figure out the total # of echo replies
         if clientId in replyInfo:
            recvNum = sum( len( v ) for v in replyInfo[ clientId ].values() )
         else:
            recvNum = 0
         lossRate = 100 - recvNum * 100 // txPkts[ clientId ]
         string = ''
         tunnels = clientIdToTunnels[ clientId ]
         if tunnels is not None:
            printTunnels( tunnels, backup=backup )
            string += '   '
         string += str( txPkts[ clientId ] ) + " packets transmitted, "
         string += str( recvNum ) + " received, "
         string += str( lossRate ) + "% packet loss, time " + str( time ) + "ms"
         print( string )
         if clientId in replyInfo:
            ll = []
            for host, rtts in replyInfo[ clientId ].items():
               delays = delayInfo[ clientId ][ host ]
               ll.append( '{} received from {}, rtt min/max/avg '
                          '{}/{}/{} ms, 1-way min/max/avg {}/{}/{} ms'
                          .format( len( rtts ), host,
                                   min( rtts ),
                                   max( rtts ),
                                   ( sum( rtts ) / len( rtts ) ),
                                   min( delays ),
                                   max( delays ),
                                   ( sum( delays ) / len( delays ) ) ) )
               print( '   ' + ', '.join( ll ) ) # what about the line is too long
         print()

      # print unresolved tunnels if any
      if nhgNameToUnresolvedTunnels[ nhgName ]:
         printUnresolvedTunnels( nhgNameToUnresolvedTunnels[ nhgName ],
                                 backup=backup )
         print( '   Not resolved or configured\n' )

# ---------------------------------------------------------
#           pwLdp ping render helpers
# ---------------------------------------------------------

def lspPingPwLdpReplyRender( clientId, replyPktInfo, renderArgs ):
   vias = renderArgs[ 0 ][ clientId ]
   printVias( vias )
   print( '   ' +
         lspPingReplyStr( replyPktInfo,
                          expectedRetCode=LspPingReturnCode.repRouterEgress ) )

def lspPingPwLdpStatisticsClientRender( clientId, vias, time, numPktsSent,
                                            replyHostRtts, oneWayDelays ):
   recvNum = 0 if replyHostRtts is None or not replyHostRtts else \
       sum( len( rtts ) for rtts in replyHostRtts.values() )

   lossRate = 100 - recvNum * 100 // numPktsSent

   string = str( numPktsSent ) + " packets transmitted, "
   string += str( recvNum ) + " received, "
   string += str( lossRate ) + "% packet loss, time " + str( time ) + "ms"
   print( string )

   # Can we really have multiple hosts in response?
   if replyHostRtts is not None:
      ll = []
      for host, rtts in replyHostRtts.items():
         delays = oneWayDelays[ host ]
         ll.append( '{} received from {}, rtt min/max/avg '
                    '{}/{}/{} ms, 1-way min/max/avg {}/{}/{} ms'
                    .format( len( rtts ), host,
                             min( rtts ),
                             max( rtts ),
                             ( sum( rtts ) / len( rtts ) ),
                             min( delays ),
                             max( delays ),
                             ( sum( delays ) / len( delays ) ) ) )
         print( ', '.join( ll ) )

def lspPingPwLdpStatisticsRender( clientIds, time, txPkts, replyInfo,
                                  delayInfo, renderArgs ):
   clientIdToVias = renderArgs[ 0 ]
   pwName = renderArgs[ 1 ]
   if not txPkts:
      return

   print( "\n--- LDP pseudowire %s : lspping statistics ---" % ( pwName ) )

   for clientId in clientIds:
      packetsSent = txPkts.get( clientId )
      if packetsSent != None and packetsSent > 0:
         vias = clientIdToVias.get( clientId )
         lspPingPwLdpStatisticsClientRender( clientId, vias, time, packetsSent,
                                           replyInfo.get( clientId ),
                                           delayInfo.get( clientId ) )
         print()

# ---------------------------------------------------------
#           SrTe ping render helpers
# ---------------------------------------------------------

SrTePingRenderArgs = collections.namedtuple( 'SrTePingRenderArgs',
                                             'clientIdToTunnels endpoint color' )

def printSrTeTunnels( tunnels ):
   '''
   Prints the Tunnels of the form:
   Segment List Label Stack: [21, 22, 32]
       Via 1.1.1.2
   '''
   for tunnel in tunnels:
      via, segmentList = tunnel
      segmentListRepr = str( list( segmentList ) )
      print( f'Segment list label stack: {segmentListRepr}' )
      print( f'   Via: {via}' )

def lspPingSrTeReplyRender( clientId, replyPktInfo, renderArgs ):
   tunnels = renderArgs[ 0 ][ clientId ]
   printSrTeTunnels( tunnels )
   print( '   ' + lspPingReplyStr( replyPktInfo,
                               expectedRetCode=LspPingReturnCode.repRouterEgress ) )

def lspPingSrTeReplyRenderWithoutCode( clientId, replyPktInfo, renderArgs ):
   tunnels = renderArgs[ 0 ][ clientId ]
   printSrTeTunnels( tunnels )
   print( '   ' + lspPingReplyStr( replyPktInfo ) )

def lspPingSrTeStatisticsRender( clientIds, time, txPkts, replyInfo,
                                 delayInfo, renderArgs ):
   clientIdToTunnels, endpoint, color = renderArgs
   if not txPkts:
      return

   print( "\n--- SR-TE Policy endpoint: {} color: {} lspping statistics ---".format(
            str( endpoint ), color ) )
   for clientId in clientIds:
      if clientId in replyInfo:
         recvNum = sum( len( v ) for v in replyInfo[ clientId ].values() )
      else:
         recvNum = 0
      lossRate = 100 - recvNum * 100 // txPkts[ clientId ]
      tunnels = clientIdToTunnels[ clientId ]
      if tunnels is not None:
         printSrTeTunnels( tunnels )
      string = '   '
      string += str( txPkts[ clientId ] ) + " packets transmitted, "
      string += str( recvNum ) + " received, "
      string += str( lossRate ) + "% packet loss, time " + str( time ) + "ms"
      print( string )
      if clientId in replyInfo:
         ll = []
         for host, rtts in replyInfo[ clientId ].items():
            delays = delayInfo[ clientId ][ host ]
            ll.append( '{} received from {}, rtt min/max/avg '
                       '{}/{}/{} ms, 1-way min/max/avg {}/{}/{} ms'
                       .format( len( rtts ), host,
                                min( rtts ),
                                max( rtts ),
                                ( sum( rtts ) / len( rtts ) ),
                                min( delays ),
                                max( delays ),
                                ( sum( delays ) / len( delays ) ) ) )
            print( '   ' + ', '.join( ll ) )
      print()

# ---------------------------------------------------------
#           raw ping render helpers
# ---------------------------------------------------------

def lspPingRawReplyRender( clientId, replyPktInfo, fec ):
   print( lspPingReplyStr( replyPktInfo,
                           expectedRetCode=LspPingReturnCode.repRouterEgress ) )

def lspPingRawStatisticsClientRender( clientId, time, numPktsSent, replyHostRtt ):
   recvNum = 0 if replyHostRtt is None or not replyHostRtt else \
       sum( len( rtts ) for rtts in replyHostRtt.values() )
   lossRate = 100 - recvNum * 100 // numPktsSent
   string = str( numPktsSent ) + " packets transmitted, "
   string += str( recvNum ) + " received, "
   string += str( lossRate ) + "% packet loss, time " + str( time ) + "ms"
   print( string )
   print()

def lspPingRawStatisticsRender( clientIds, time, txPkts, replyInfo, delayInfo, fec ):
   if not txPkts:
      return

   print( "\n--- %s : lspping statistics ---" % fec )
   for clientId in clientIds:
      numPktsSent = txPkts.get( clientId )
      if numPktsSent > 0:
         lspPingRawStatisticsClientRender( clientId, time, numPktsSent,
                                           replyInfo.get( clientId ) )

# ---------------------------------------------------------
#           RSVP ping render helpers
# ---------------------------------------------------------

def lspPingRsvpReplyRender( clientId, replyPktInfo, renderArgs ):
   # renderArgs[ 3 ] is a dictionary which stores the lspIds associated with
   # its corresponding clientID. lspId refers to the spCliId which can be specified
   # in the RSVP ping command with the `lsp` argument.
   # If there is no entry in renderArgs[ 3 ], it means that the `lsp` argument in
   # the ping command was specified and there is no need to specify the LSP in the
   # output.
   clientIdToLspId = renderArgs[ 3 ]
   if clientIdToLspId:
      lspId = clientIdToLspId[ clientId ]
      print( f'LSP {lspId}' )
   clientIdToSubTunnelId = renderArgs[ 4 ]
   if clientIdToSubTunnelId and clientIdToSubTunnelId[ clientId ] != 0:
      print( f'SubTunnel {clientIdToSubTunnelId[ clientId ]}' )
   # Get the vias from clientId
   vias = renderArgs[ 0 ][ clientId ]
   printVias( vias )
   expectedRetCode = LspPingReturnCode.repRouterEgress
   print( '   ' + lspPingReplyStr( replyPktInfo, expectedRetCode ) )

def lspPingRsvpStatisticsClientRender( clientId, vias, time, numPktsSent,
                                       replyHostRtts, oneWayDelays, clientIdToLsp,
                                       clientIdToSubTunnel ):
   recvNum = 0 if replyHostRtts is None or not replyHostRtts else \
       sum( len( rtts ) for rtts in replyHostRtts.values() )

   lossRate = 100 - recvNum * 100 // numPktsSent

   if clientIdToLsp:
      print( f'LSP {clientIdToLsp[ clientId ]}' )
   if clientIdToSubTunnel and clientIdToSubTunnel[ clientId ] != 0:
      print( f'SubTunnel {clientIdToSubTunnel[ clientId ]}' )

   if vias is not None:
      printVias( vias )
   string = str( numPktsSent ) + " packets transmitted, "
   string += str( recvNum ) + " received, "
   string += str( lossRate ) + "% packet loss, time " + str( time ) + "ms"
   print( '   ' + string )

   if replyHostRtts is not None:
      ll = []
      for host, rtts in replyHostRtts.items():
         delays = oneWayDelays[ host ]
         ll.append( '{} received from {}, rtt min/max/avg '
                    '{}/{}/{} ms, 1-way min/max/avg {}/{}/{} ms'
                    .format( len( rtts ), host,
                             min( rtts ),
                             max( rtts ),
                             ( sum( rtts ) / len( rtts ) ),
                             min( delays ),
                             max( delays ),
                             ( sum( delays ) / len( delays ) ) ) )
         print( '   ' + ', '.join( ll ) )

def lspPingRsvpStatisticsRender( clientIds, time, txPkts, replyInfo, delayInfo,
                                 renderArgs ):
   clientIdToVias = renderArgs[ 0 ]
   prefix = renderArgs[ 1 ]
   protocol = renderArgs[ 2 ]
   clientIdToLsp = renderArgs[ 3 ]
   clientIdToSubTunnel = renderArgs[ 4 ]
   if not txPkts:
      return

   print( f"\n--- {protocol} target fec {prefix} : lspping statistics ---" )

   for clientId in clientIds:
      packetsSent = txPkts.get( clientId )
      if packetsSent != None and packetsSent > 0:
         vias = clientIdToVias.get( clientId )
         lspPingRsvpStatisticsClientRender( clientId, vias, time, packetsSent,
                                            replyInfo.get( clientId ),
                                            delayInfo.get( clientId ),
                                            clientIdToLsp, clientIdToSubTunnel )
         print()

# -----------------------------------------------------------------
#        LDP, MLDP and Segment Routing Ping Render helpers
# -----------------------------------------------------------------

def lspPingLabelDistReplyRender( clientId, replyPktInfo, renderArgs ):
   vias = renderArgs[ 0 ][ clientId ]
   printVias( vias )
   print( '   ' +
          lspPingReplyStr( replyPktInfo,
                           expectedRetCode=LspPingReturnCode.repRouterEgress ) )

def lspPingLabelDistStatisticsClientRender( vias, time, numPktsSent,
                                            protocol, replyHostRtts, oneWayDelays ):
   recvNum = 0 if replyHostRtts is None or not replyHostRtts else \
       sum( len( rtts ) for rtts in replyHostRtts.values() )

   lossRate = 100 - recvNum * 100 // numPktsSent

   if vias is not None:
      printVias( vias )
   string = str( numPktsSent ) + " packets transmitted, "
   string += str( recvNum ) + " received"
   if protocol != 'MLDP':
      string += ", " + str( lossRate ) + "% packet loss"
   string += ", time " + str( time ) + "ms"
   print( '   ' + string )

   if replyHostRtts is not None:
      ll = []
      for host, rtts in sorted( replyHostRtts.items() ):
         delays = oneWayDelays[ host ]
         ll.append( '{} received from {}, rtt min/max/avg '
                    '{}/{}/{} ms, 1-way min/max/avg {}/{}/{} ms'
                    .format( len( rtts ), host,
                             min( rtts ),
                             max( rtts ),
                             ( sum( rtts ) / len( rtts ) ),
                             min( delays ),
                             max( delays ),
                             ( sum( delays ) / len( delays ) ) ) )
         print( '   ' + ', '.join( ll ) )

def lspTracerouteLabelDistStatisticsClientRender( vias, time, numPktsSent,
                                                  protocol, replyHostRtts ):
   recvNum = ( 0 if replyHostRtts is None or not replyHostRtts
               else sum( len( rtts ) for rtts in replyHostRtts.values() ) )
   lossRate = 100 - recvNum * 100 // numPktsSent

   if vias:
      printVias( vias )
   lossStr = ''
   if protocol != 'mldp':
      lossStr = f", {lossRate}% packet loss"
   stats = ( '{sent} packets transmitted, {recv} received{loss}, '
             'time {time}ms' ).format( sent=numPktsSent,
                                       recv=recvNum,
                                       loss=lossStr,
                                       time=time )
   print( '   ' + stats )

   if replyHostRtts:
      for host, rtts in sorted( replyHostRtts.items() ):
         if rtts:
            print( '   {} received from {}, rtt min/max/avg '
                   '{}/{}/{} ms'
                   .format( len( rtts ), host,
                            min( rtts ),
                            max( rtts ),
                            ( sum( rtts ) / len( rtts ) ) ) )

def lspPingLabelDistStatisticsRender( clientIds, time, txPkts, replyInfo,
                                      delayInfo, renderArgs ):
   clientIdToVias = renderArgs[ 0 ]
   prefix = renderArgs[ 1 ]
   protocol = renderArgs[ 2 ]
   if not txPkts:
      return

   print( f"\n--- {protocol} target fec {prefix} : lspping statistics ---" )

   for clientId in clientIds:
      packetsSent = txPkts.get( clientId )
      if packetsSent != None and packetsSent > 0:
         vias = clientIdToVias.get( clientId )
         lspPingLabelDistStatisticsClientRender( vias, time,
                                                 packetsSent, protocol,
                                                 replyInfo.get( clientId ),
                                                 delayInfo.get( clientId ) )
         print()

def lspTracerouteLabelDistStatisticsRender( time, txPkts, replyHostRtts,
                                            protocol, prefix, vias ):
   if txPkts > 0:
      print( "\n--- {} target fec {} : lsptraceroute statistics ---".format(
             protocol.upper(), prefix ) )
      lspTracerouteLabelDistStatisticsClientRender( vias, time, txPkts, protocol,
                                                    replyHostRtts )
      print()

# -----------------------------------------------------------------
#        BGP LU Ping Render helpers
# -----------------------------------------------------------------

def lspPingBgpLuReplyRender( clientId, replyPktInfo, renderArgs ):
   vias = renderArgs[ 0 ][ clientId ]
   printVias( vias )
   print( '   ' +
          lspPingReplyStr( replyPktInfo,
                           expectedRetCode=LspPingReturnCode.repRouterEgress ) )

def lspPingBgpLuStatisticsClientRender( clientId, vias, time, numPktsSent,
                                        replyHostRtts, oneWayDelays ):
   recvNum = ( 0 if replyHostRtts is None or not replyHostRtts else
               sum( len( rtts ) for rtts in replyHostRtts.values() ) )

   lossRate = 100 - recvNum * 100 // numPktsSent

   if vias is not None:
      printVias( vias )
   string = ( '{} packets transmitted, {} received, '
              '{}% packet loss, time {}ms' ).format( numPktsSent, recvNum,
                                                     lossRate, time )
   print( '   ' + string )

   if replyHostRtts is not None:
      hostInfo = []
      for host, rtts in replyHostRtts.items():
         delays = oneWayDelays[ host ]
         hostInfo.append( '{} received from {}, rtt min/max/avg '
                          '{}/{}/{} ms, 1-way min/max/avg {}/{}/{} ms'
                          .format( len( rtts ), host,
                                   min( rtts ),
                                   max( rtts ),
                                   ( sum( rtts ) / len( rtts ) ),
                                   min( delays ),
                                   max( delays ),
                                   ( sum( delays ) / len( delays ) ) ) )
         print( '   ' + ', '.join( hostInfo ) )

def lspPingBgpLuStatisticsRender( clientIds, time, txPkts, replyInfo,
                                  delayInfo, renderArgs ):
   clientIdToVias, prefix, protocol, unresolvedVias = renderArgs
   if not txPkts:
      return

   print( f"\n--- {protocol} target fec {prefix} : lspping statistics ---" )

   for clientId in clientIds:
      packetsSent = txPkts.get( clientId )
      if packetsSent != None and packetsSent > 0:
         vias = clientIdToVias.get( clientId )
         lspPingBgpLuStatisticsClientRender( clientId, vias, time, packetsSent,
                                             replyInfo.get( clientId ),
                                             delayInfo.get( clientId ) )
         print()

   if unresolvedVias:
      printVias( unresolvedVias, resolved=False )
      print( '   Not resolved' )
      print()
