# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Tac
import errno
from IpLibConsts import DEFAULT_VRF
from ClientCore import ping
from ClientCommonLib import (
   resolveNexthop,
   resolveNexthopDefaultVrf,
   getNhgId, 
   labelStackToList,
   resolveHierarchical,
   isIpv6Addr, 
   getTunnelNhgName,
   resolveVpnTunnel,
   _resolveSrTePolicyTunnels,
   getStaticFec,
   getVpnFec,
   isNexthopGroupVia, 
   getNexthopGroupId,
   getNhgIdToName,
   isNexthopGroupTunnelVia,
   DynTunnelIntfId,
   resolveNhgTunnelFibEntry,
   getProtocolIpFec,
   getPwLdpFec,
   IPV4, IPV6, MldpInfo,
   getBgpLuTunnelFibEntry,
   getNhAndLabelFromTunnelFibVia,
   validateBgpLuResolvedPushVia,
   LspPingTypeSr,
   LspPingTypeOspfSr,
   LspPingTypeVpn,
   LspPingTypeBgpLu,
   sendViaSocket,
   generateMplsEntropyLabel,
   getIntfPrimaryIpAddr,
   fillEntropyLabelPlaceholders,
   getNhgTunnelLabelStack,
)

from MplsPingClientLib import (
   lspPingStaticReplyRender,
   lspPingStaticStatisticsRender,
   lspPingNhgReplyRender,
   lspPingNhgStatisticsRender,
   lspPingSrTeReplyRender,
   lspPingSrTeReplyRenderWithoutCode,
   lspPingSrTeStatisticsRender,
   lspPingRawReplyRender,
   lspPingRawStatisticsRender,
   LspPingNhgModeAllEntries, 
   LspPingNhgModeOneEntry,
   sendOrRenderPingErr,
   SrTePingRenderArgs,
)

from ClientState import getGlobalState
from ForwardingHelper import getNhgSize
from PingModel_pb2 import ( # pylint: disable=no-name-in-module
   GenericModel,
   BgpLuModel,
   MplsPing,
   MplsPingProtocol,
   PwLdpModel,
   SrModel,
   VpnModel,
)
from PseudowireLib import (
   PwPingCcType,
   PwPingCvType,
   pwPingCcEnumToVal,
   pwPingCvEnumToVal,
   pwPingCcMsg,
   pwPingCvMsg,
)

import collections
from TypeFuture import TacLazyType

IpAddr = TacLazyType( 'Arnet::IpAddr' )
IpGenAddr = TacLazyType( 'Arnet::IpGenAddr' )
IpGenPrefix = TacLazyType( 'Arnet::IpGenPrefix' )
MplsLabel = TacLazyType( 'Arnet::MplsLabel' )
MplsLabelStack = TacLazyType( 'Arnet::MplsLabelOperation' )
NexthopGroupType = TacLazyType( 'Routing::NexthopGroup::NexthopGroupType' )
TunnelId = TacLazyType( 'Tunnel::TunnelTable::TunnelId' )
VplsAdPwGenId = TacLazyType( 'Pseudowire::VplsAdPwGenId' )
igpProtocolType = TacLazyType( 'LspPing::LspPingSrIgpProtocol' )

# ---------------------------------------------------------
#                   LspPing VPN
# ---------------------------------------------------------

def handleLspPingVpn( prefix, mount, sock, src=None, dst=None, smac=None,
                      dmac=None, vrf=None, interval=0, count=None,
                      verbose=False, label=None, interface=None, entry=None,
                      tc=None, genOpqVal=None, sourceAddrOpqVal=None,
                      groupAddrOpqVal=None, jitter=None, responderAddr=None,
                      standard=None, size=None, padReply=False, tos=None,
                      rd=None, **kwargs ):
   protocol = kwargs[ 'type' ]
   mplsPingProtocol = MplsPingProtocol( protocol=protocol, timeout=5,
                                        interval=interval )
   sendViaSocket( sock, MplsPing( mplsPingProtocol=mplsPingProtocol ) )
   if not rd:
      err = 'Route Distinguisher not found for prefix'
      sendOrRenderPingErr( err, sock )
      return errno.EINVAL
   vpnModel = VpnModel( prefix=prefix, rd=rd )
   sendViaSocket( sock, MplsPing( vpnModel=vpnModel ) )
   # get rdNbo value from string
   routeDist = Tac.Value( 'Arnet::RouteDistinguisher' )
   routeDist.stringValue = rd
   rdNbo = routeDist.rdNbo

   fec, err, errVal = getVpnFec( mount, prefix )
   if fec is None:
      sendOrRenderPingErr( err, sock )
      return errVal

   if not fec.via:
      err = 'No adj found for prefix'
      sendOrRenderPingErr( err, sock )
      return errno.EINVAL

   state = getGlobalState()
   unresolvedPushVia = []

   # filter unresolved fec vias
   for idx in range( len( fec.via ) ): # pylint: disable=consider-using-enumerate
      via = fec.via[ idx ]
      if not MplsLabel( via.mplsLabel ).isValid():
         continue
      nexthopIntf, nexthopEthAddr = resolveNexthop( mount, state, via.hop )
      if not ( nexthopIntf and nexthopEthAddr ):
         # reset ipv4/6RoutingSim set in resolveNexthop if nexthop info is
         # not returned as for vpn tunnel route could be v4/v6 and
         # could be in default vrf as well
         state.ipv4RoutingSim = None
         state.ipv6RoutingSim = None
         nexthopIntf, nexthopEthAddr = resolveNexthopDefaultVrf( mount, state,
                                                                 via.hop )
         if not ( nexthopIntf and nexthopEthAddr ):
            unresolvedPushVia.append( ( via.hop, via.mplsLabel, nexthopIntf ) )
            continue

   viaInfo = []
   ipv = IPV6 if isIpv6Addr( prefix ) else IPV4

   tunnelToAdjacencies, err = resolveVpnTunnel( mount, fec.fecId )
   if tunnelToAdjacencies == errno.EINVAL:
      sendOrRenderPingErr( err, sock )
      return errno.EINVAL

   clientIdToVia = {}
   for idx, tunnel in enumerate( sorted( tunnelToAdjacencies.keys() ) ):
      # If MPLS EL push is enabled, we want to add the ELI and EL to the tunnel's
      # labelStack and save that to be used for rendering and for the ping request.
      labelStack, nhMac, l3Intf = tunnelToAdjacencies[ tunnel ]
      # Convert from a tuple to a list to operate on
      labelStack = list( labelStack )
      # Update the tunnel tuple  with the entropy label stack
      # Tunnel looks like tuple( tuple( nh, tuple( labelStack ) ) )
      tunnel = ( ( tunnel[ 0 ][ 0 ], tuple( labelStack ) ), )
      clientIdToVia[ idx ] = [ ( tunnel[ 0 ][ 0 ], labelStack, l3Intf ) ]
      viaInfo.append( ( l3Intf, labelStack, nhMac ) )

   return ping( mount, state, src, dst, smac, interval, count, viaInfo,
                None, None, # capi supported so no replyHandlers passed
                ( clientIdToVia, prefix, unresolvedPushVia ), ipv=ipv,
                tc=tc, standard=standard, size=size,
                protocol=LspPingTypeVpn, padReply=padReply, tos=tos, rd=rdNbo,
                sock=sock, json=kwargs.get( 'json' ) )

# ---------------------------------------------------------
#                   LspPing BGP LU
# ---------------------------------------------------------

def handleLspPingBgpLu( prefix, mount, sock, src=None, dst=None,
                        smac=None, dmac=None,
                        vrf=None, interface=None, interval=1, count=None,
                        label=None, verbose=False, entry=None, tc=None,
                        nexthop=None, standard=None, size=None, padReply=False,
                        egressValidateAddress=None, multiLabel=1, tos=None,
                        **kwargs ):
   protocol = kwargs[ 'type' ]
   mplsPingProtocol = MplsPingProtocol( protocol=protocol, timeout=5,
                                        interval=interval )
   sendViaSocket( sock, MplsPing( mplsPingProtocol=mplsPingProtocol ) )
   bgpLuModel = BgpLuModel( prefix=prefix )
   sendViaSocket( sock, MplsPing( bgpLuModel=bgpLuModel ) )
   state = getGlobalState()
   viaInfo, clientIdToVias = [], {}
   resolvedPushVia = collections.defaultdict( list )
   unresolvedPushVia = []
   ipv = IPV6 if isIpv6Addr( str( prefix ) ) else IPV4

   tunnelFibEntry, err = getBgpLuTunnelFibEntry( mount, prefix )
   if err:
      sendOrRenderPingErr( err, sock )
      return errno.EINVAL

   for tunnelVia in tunnelFibEntry.tunnelVia.values():
      nextHopAndLabel, err = getNhAndLabelFromTunnelFibVia( mount, tunnelVia )
      if err:
         sendOrRenderPingErr( err, sock )
         return errno.EINVAL
      nexthopIp = nextHopAndLabel.nextHopIp
      labels = nextHopAndLabel.label
      intfId = nextHopAndLabel.intfId

      # If nexthop and/or label stack are specified, use only the via that has
      # that nexthop/label stack.
      if nexthop and IpGenAddr( nexthop ) != nexthopIp:
         continue
      if label and label != labels:
         continue

      nexthopAddr = nexthopIp.v4Addr if ipv == IPV4 else nexthopIp.v6Addr
      # Map L3 nexthop to L2 nexthop
      nexthopIntf, nexthopEthAddr = resolveNexthop( mount, state, nexthopAddr,
                                                    intf=intfId )
      if not ( nexthopIntf and nexthopEthAddr ): # pylint: disable=no-else-continue
         unresolvedPushVia.append( ( nexthopIp, labels, nexthopIntf ) )
         continue
      else:
         key = ( nexthopIntf, tuple( labels ), nexthopEthAddr )
         resolvedPushVia[ key ].append( ( nexthopIp, labels, nexthopIntf ) )

   if nexthop:
      err = validateBgpLuResolvedPushVia( resolvedPushVia, nexthop, label )
      if err:
         sendOrRenderPingErr( err, sock )
         return errno.EINVAL

   for idx, ( key, val ) in enumerate( resolvedPushVia.items() ):
      clientIdToVias[ idx ] = val
      viaInfo.append( key )

   if not viaInfo:
      err = 'Failed to find a valid output interface'
      sendOrRenderPingErr( err, sock )
      return errno.EINVAL

   return ping( mount, state, src, dst, smac, interval, count, viaInfo,
                None, None, # capi supported so no replyHandlers passed
                renderArg=( clientIdToVias, prefix, unresolvedPushVia ),
                ipv=ipv, tc=tc, standard=standard, size=size,
                protocol=LspPingTypeBgpLu, padReply=padReply, tos=tos,
                sock=sock, json=kwargs.get( 'json' ) )

# ---------------------------------------------------------
#                   LspPing nexthop-group
# ---------------------------------------------------------

def handleLspPingNhg( nhgNames, mount, prefix=None, src=None, dst=None, 
                      smac=None, dmac=None, vrf=None, interface=None, interval=1, 
                      count=None, label=None, verbose=False, entry=None, tc=None,
                      nhgNameToNhgTunnelIdx=None, standard=None,
                      size=None, padReply=False, egressValidateAddress=None,
                      multiLabel=1, tos=None, backup=False, **kwargs ):
   state = getGlobalState()
   idxBase = 0
   viaInfo, clientIdToTunnels, clientIdBaseToNhgName = [], {}, {}
   nhgNameToClientIds, nhgNameToUnresolvedTunnels = {}, {}
   if not isinstance( nhgNames, list ):
      nhgNames = [ nhgNames ]

   #pylint: disable-msg=too-many-nested-blocks
   for nhgName in nhgNames:
      nhgId = getNhgId( nhgName, mount )
      if nhgId is None:
         print( 'Nexthop-group %s does not exist.' % nhgName )
         return errno.EINVAL
      # we are expecting a programmed MPLS nexthop-group
      nhgConfig = mount.routingNhgConfig.nexthopGroup[ nhgName ]
      if nhgConfig.type != NexthopGroupType.mpls:
         print( 'Nexthop-group %s is not in MPLS type.' % nhgName )
         return errno.EINVAL
      # mapping from adj (meaning labelStack + l2Adj) to tunnels
      adjToTunnels = {}
      unresolvedTunnels = []
      # XXX VRF
      hwVrfStatus = mount.routingHwNexthopGroupStatus.vrfStatus.get( DEFAULT_VRF )
      if not hwVrfStatus:
         print( 'Cannot find next-hop group status.' )
         return errno.EINVAL
      if nhgId not in hwVrfStatus.nexthopGroupAdjacency:
         print( 'Nexthop-group %s is not programmed.' % nhgName )
         return errno.EINVAL

      nhgAdj = hwVrfStatus.nexthopGroupAdjacency.get( nhgId )
      nhgAdjVia = nhgAdj.backupVia if backup else nhgAdj.via
      entrySize = getNhgSize( nhgConfig, isBackup=backup )

      if not nhgAdjVia:
         print( f'Via for Nexthop-group {nhgName} cannot be resolved' )
         return errno.EINVAL

      if entry is None or entry < 0:
         mode = LspPingNhgModeAllEntries
         entries = list( range( entrySize ) )
      else:
         mode = LspPingNhgModeOneEntry
         entries = [ entry ]

      TunnInfo = collections.namedtuple( 'tunnelInfo', 'entryIndex nexthop' )
      for tunEntry in entries:
         entryResolved = False
         via = nhgAdjVia.get( tunEntry )
         if via and via.l2Via.vlanId != 0:
            # resolved tunnel
            ipv = IPV6 if isIpv6Addr( via.hop.stringValue ) else IPV4
            key = ( getNhgTunnelLabelStack( nhgConfig, tunEntry, backup ),
                    via.l2Via.macAddr, via.l3Intf )
            tunnInfo = TunnInfo( tunEntry, str( via.hop ) )
            adjToTunnels.setdefault( key, [] ).append( tunnInfo )
            entryResolved = True
         elif via and via.nextLevelFecIndex:
            ipv = IPV6 if isIpv6Addr( via.route.stringValue ) else IPV4
            nextLevelFecId = via.nextLevelFecIndex
            nextHopAndLabels = resolveHierarchical( mount, fecId=nextLevelFecId )
            if nextHopAndLabels:
               for nhAndlabel in nextHopAndLabels:
                  nexthop = nhAndlabel.nextHopIp
                  nhIntf, nhMac = resolveNexthop( mount, state, nexthop )
                  if not nhIntf or not nhMac:
                     continue
                  key = ( getNhgTunnelLabelStack( nhgConfig, tunEntry, backup ),
                          nhMac, nhIntf )
                  tunnInfo = TunnInfo( tunEntry, str( nexthop ) )
                  adjToTunnels.setdefault( key, [] ).append( tunnInfo )
                  entryResolved = True
         if not entryResolved:
            unresolvedTunnels.append( tunEntry )

      nhgNameToUnresolvedTunnels[ nhgName ] = unresolvedTunnels

      if not adjToTunnels:
         if len( entries ) == entrySize:
            errMsg = 'no entry resolved'
         else:
            errMsg = 'entry %d not resolved' % entries[ 0 ]
         print( f'nexthop-group {nhgName}: {errMsg}' )
         return errno.EINVAL

      tunnelsToAdj = { tuple( v ) : k for k, v in adjToTunnels.items() }
      sortedKeys = sorted( tunnelsToAdj.keys(),
                           key=lambda entries: ( entries[ 0 ].entryIndex,
                                                 entries[ 0 ].nexthop ) )
      nhgNameToClientIds[ nhgName ] = []
      clientIdBaseToNhgName[ idxBase ] = nhgName
      for idx, entries in enumerate( sortedKeys ):
         clientIdToTunnels[ idxBase + idx ] = entries
         labelStack, nhMacAddr, l3Intf = tunnelsToAdj[ entries ]
         viaInfo.append( ( l3Intf, labelStackToList( labelStack ), nhMacAddr ) )
         nhgNameToClientIds[ nhgName ].append( idxBase + idx )
      idxBase += len( sortedKeys )

   #pylint: enable-msg=too-many-nested-blocks
   return ping( mount, state, src, dst, smac, interval, count, viaInfo,
                lspPingNhgReplyRender, lspPingNhgStatisticsRender,
                ( clientIdToTunnels, nhgNames, nhgNameToClientIds,
                  nhgNameToUnresolvedTunnels, clientIdBaseToNhgName,
                  prefix, mode, nhgNameToNhgTunnelIdx, backup ),
                ipv=ipv, tc=tc, standard=standard, size=size, padReply=padReply,
                egressValidateAddress=egressValidateAddress, tos=tos,
                lookupLabelCount=multiLabel )

# ---------------------------------------------------------
#                LspPing Nexthop Group Tunnel
# ---------------------------------------------------------

def handleLspPingNhgTunnel( endpoint, mount, src=None, dst=None, smac=None,
                            dmac=None, vrf=None, interface=None, interval=1,
                            count=None, label=None, verbose=False, entry=None,
                            tc=None, standard=None, size=None, padReply=False,
                            egressValidateAddress=None, multiLabel=1, tos=None,
                            backup=False, **kwargs ):
   ( nhgName, tunnelIndex, err ) = getTunnelNhgName( mount, endpoint )
   if err is not None:
      print( err )
      return errno.EINVAL

   if egressValidateAddress == 'default':
      egressValidateAddress = endpoint

   return handleLspPingNhg( [ nhgName ], mount, prefix=endpoint, src=src, 
                            dst=dst, smac=smac, dmac=dmac, vrf=vrf, entry=entry, 
                            tc=tc, interval=interval, count=count, verbose=verbose, 
                            nhgNameToNhgTunnelIdx={ nhgName : tunnelIndex },
                            standard=standard, size=size, padReply=padReply,
                            egressValidateAddress=egressValidateAddress,
                            multiLabel=multiLabel, backup=backup, tos=tos )

# ---------------------------------------------------------
#                   LspPing static
# ---------------------------------------------------------

def handleLspPingStatic( prefix, mount, src=None, dst=None, smac=None, 
                         dmac=None, vrf=None, interval=1, count=None, verbose=False,
                         label=None, interface=None, entry=None, tc=None,
                         standard=None, size=None, padReply=False,
                         egressValidateAddress=None, multiLabel=1, tos=None,
                         **kwargs ):
   fec, err = getStaticFec( mount, prefix )
   if fec is None:
      return err

   if not fec.via:
      print( 'No adj found for prefix' )
      return errno.EINVAL

   state = getGlobalState()
   resolvedPushVia = {}
   unresolvedPushVia = []
   nhgNames = []
   nhgNameToNhgTunnelIdx = {}

   for idx in range( len( fec.via ) ): # pylint: disable=consider-using-enumerate
      via = fec.via[ idx ]
      if isNexthopGroupVia( via ):
         nexthopGroupId = getNexthopGroupId( via )
         nhgName = getNhgIdToName( nexthopGroupId, mount )
         if not nhgName:
            print( 'No nexthop-group with id %d' % nexthopGroupId )
            return errno.EINVAL
         nhgNames.append( nhgName )
         continue
      if isNexthopGroupTunnelVia( via ):
         tunnelId = DynTunnelIntfId.tunnelId( via.intfId )
         ( nhgName, tunnelIndex, err ) = resolveNhgTunnelFibEntry( mount, tunnelId )
         if err is not None:
            print( err )
            return errno.EINVAL
         nhgNameToNhgTunnelIdx[ nhgName ] = tunnelIndex
         nhgNames.append( nhgName )
         continue
      if not MplsLabel( via.mplsLabel ).isValid():
         continue
      nexthopIntf, nexthopEthAddr = resolveNexthop( mount, state, via.hop )
      if not ( nexthopIntf and nexthopEthAddr ):
         unresolvedPushVia.append( ( via.hop, via.mplsLabel ) )
         continue
      key = ( nexthopIntf, via.mplsLabel, nexthopEthAddr )
      resolvedPushVia.setdefault( key, [] ).append( ( via.hop, via.mplsLabel,
                                                      nexthopIntf ) )

   if nhgNames:
      return handleLspPingNhg( nhgNames, mount, prefix=prefix, src=src, dst=dst, 
                               smac=smac, dmac=dmac, vrf=vrf, interval=interval, 
                               count=count, verbose=verbose, tc=tc,
                               nhgNameToNhgTunnelIdx=nhgNameToNhgTunnelIdx,
                               standard=standard, size=size, padReply=padReply,
                               egressValidateAddress=egressValidateAddress,
                               multiLabel=multiLabel, tos=tos )
   if not resolvedPushVia:
      print( 'No via resolved' )
      return errno.EINVAL

   viaInfo, clientIdToVia = [], {}
   for idx, ( key, val ) in enumerate( resolvedPushVia.items() ):
      clientIdToVia[ idx ] = val
      viaInfo.append( key )
   ipv = IPV6 if isIpv6Addr( prefix ) else IPV4

   return ping( mount, state, src, dst, smac, interval, count, viaInfo,
                lspPingStaticReplyRender, lspPingStaticStatisticsRender, 
                ( clientIdToVia, prefix, unresolvedPushVia ), ipv=ipv, tc=tc,
                standard=standard, size=size, padReply=padReply, tos=tos,
                egressValidateAddress=egressValidateAddress,
                lookupLabelCount=multiLabel )

# ---------------------------------------------------------
#                   LspPing Ldp Pseudowire
# ---------------------------------------------------------

def handleLspPingPwLdp( pwLdpName, mount, sock, prefix=None, src=None, dst=None,
                        smac=None, dmac=None, vrf=None, interface=None, interval=1, 
                        count=None, label=None, verbose=False, entry=None, tc=None,
                        standard=None, size=None, padReply=False, tos=None,
                        **kwargs ):
   protocol = kwargs[ 'type' ]
   json = kwargs.get( 'json' )
   # send Ping protocol before resolving pwLdp fec
   mplsPingProtocol = MplsPingProtocol( protocol=protocol, timeout=5,
                                        interval=interval )
   sendViaSocket( sock, MplsPing( mplsPingProtocol=mplsPingProtocol ) )

   pwLdpInfo, nextHopAndLabel, err = getPwLdpFec( mount, pwLdpName )
   pwLdpModel = PwLdpModel( pwLdpName=pwLdpName )
   if pwLdpInfo is not None:
      pwLdpModel.pwNeighbor = pwLdpInfo.remoteRouterId.stringValue
      if pwLdpInfo.pwId:
         pwLdpModel.pwId = str( pwLdpInfo.pwId )
      else:
         vplsAdPwGenId = VplsAdPwGenId( stringValue=pwLdpInfo.pwGenId )
         pwLdpModel.pwId = '[VPLS ID {}, Router ID {}, PE Address {}]'.format(
                              vplsAdPwGenId.vplsIdStrep(),
                              IpAddr( vplsAdPwGenId.localAddress ).stringValue,
                              IpAddr( vplsAdPwGenId.peAddress ).stringValue )
      if pwLdpInfo.vcLabel:
         pwLdpModel.pwLabel = pwLdpInfo.vcLabel
   if err is not None:
      sendOrRenderPingErr( err, sock )
      return errno.EINVAL
   nexthop = nextHopAndLabel.nextHopIp
   labels = nextHopAndLabel.label
   intfId = nextHopAndLabel.intfId

   # Currently, we only support LSP ping as CV type
   if not pwLdpInfo.cvTypes & pwPingCvEnumToVal[ PwPingCvType.lspPing ]:
      msg = ( 'Neighbor does not support using {} as a connectivity verification'
              ' mechanism.' ).format( pwPingCvMsg[ PwPingCvType.lspPing ] )
      sendOrRenderPingErr( msg, sock )
      return errno.EOPNOTSUPP

   state = getGlobalState()
   # Print the CC Type being used and update pwLdpInfo to identify active CC type.
   # Currently, we support Control Word or RA label as CC Types. For TTL Expiry,
   # our hardware does not support pipe mode at ingress. We need pipe mode at both
   # ingress and PHP node to correctly process the VC label.
   if pwLdpInfo.controlWord:
      if pwLdpInfo.ccTypes & pwPingCcEnumToVal[ PwPingCcType.cw ]:
         pwLdpModel.pwCCType = pwPingCcEnumToVal[ PwPingCcType.cw ]
         pwLdpInfo = pwLdpInfo._replace(
                        ccTypes=pwPingCcEnumToVal[ PwPingCcType.cw ] )
      elif pwLdpInfo.ccTypes & pwPingCcEnumToVal[ PwPingCcType.ra ]:
         # Add router alert label to the stack
         labels.append( MplsLabel.routerAlert )
         pwLdpModel.pwCCType = pwPingCcEnumToVal[ PwPingCcType.ra ]
         pwLdpInfo = pwLdpInfo._replace(
                        ccTypes=pwPingCcEnumToVal[ PwPingCcType.ra ] )
      else:
         msg = ( 'Neighbor does not support using either {} or {} as connectivity'
                 ' check mechanisms.' ).format( pwPingCcMsg[ PwPingCcType.cw ],
                                                pwPingCcMsg[ PwPingCcType.ra ] )
         sendOrRenderPingErr( msg, sock )
         return errno.EOPNOTSUPP
   else:
      if pwLdpInfo.ccTypes & pwPingCcEnumToVal[ PwPingCcType.ra ]:
         # Add router alert label to the stack
         labels.append( MplsLabel.routerAlert )
         pwLdpModel.pwCCType = pwPingCcEnumToVal[ PwPingCcType.ra ]
         pwLdpInfo = pwLdpInfo._replace(
                        ccTypes=pwPingCcEnumToVal[ PwPingCcType.ra ] )
      else:
         msg = ( 'Neighbor does not support using {} as a connectivity'
                 ' check mechanism.' ).format( pwPingCcMsg[ PwPingCcType.ra ] )
         sendOrRenderPingErr( msg, sock )
         return errno.EOPNOTSUPP

   # send pwLdp model after filled in all fields
   sendViaSocket( sock, MplsPing( pwLdpModel=pwLdpModel ) )

   # Add VC label to the stack and remove Implicit Null if it's there
   labels.append( pwLdpInfo.vcLabel )
   if labels[ 0 ] == MplsLabel.implicitNull:
      labels = labels[ 1 : ]

   # Map L3 nexthop to L2 nexthop
   resolvedPushVia = {}
   unresolvedPushVia = []
   nexthopIntf, nexthopEthAddr = resolveNexthop( mount, state, str( nexthop ),
                                                 intf=intfId )

   if not ( nexthopIntf and nexthopEthAddr ):
      unresolvedPushVia.append( ( nexthop, labels ) )
   else:
      key = ( nexthopIntf, tuple( labels ), nexthopEthAddr )
      resolvedPushVia.setdefault( key, [] ).append(
         ( nexthop, labels, nexthopIntf ) )

   viaInfo, clientIdToVias = [], {}
   for idx, ( key, val ) in enumerate( resolvedPushVia.items() ):
      clientIdToVias[ idx ] = val
      viaInfo.append( key )

   if not viaInfo:      # No resolved interface
      print( "Failed to find a valid output interface" )
      return errno.EINVAL

   ipv = IPV6 if isIpv6Addr( pwLdpInfo.remoteRouterId ) else IPV4

   return ping( mount, state, src, dst, smac, interval, count, viaInfo,
                None, None, # capi supported so no replyHandlers passed
                ( clientIdToVias, pwLdpName, pwLdpInfo ), protocol=protocol,
                ipv=ipv, pwLdpInfo=pwLdpInfo, tc=tc, standard=standard, size=size,
                padReply=padReply, tos=tos, json=json, sock=sock )

# ---------------------------------------------------------
#                   LspPing Sr-Te Tunnels
# ---------------------------------------------------------

def handleLspPingSrTe( endpoint, mount, color, trafficAf=None, src=None, 
                       dst=None, smac=None, dmac=None, vrf=None, interface=None,
                       interval=1, count=None, label=None, verbose=False, tc=None,
                       standard=None, size=None, padReply=False, tos=None,
                       egressValidateAddress=None, **kwargs ):
   clientIdToTunnels = []
   viaInfo = []
   ipv = IPV6 if isIpv6Addr( str( endpoint ) ) else IPV4
   if trafficAf:
      ipv = IPV6 if trafficAf == 'v6' else IPV4
   tunnelToAdjacencies = _resolveSrTePolicyTunnels( mount, endpoint, color,
                                                    trafficAf=trafficAf )
   if tunnelToAdjacencies == errno.EINVAL:
      return errno.EINVAL

   state = getGlobalState()

   for tunnel in sorted( tunnelToAdjacencies.keys() ):
      # If MPLS EL push is enabled, we want to add the ELI and EL to the tunnel's
      # labelStack and save that to be used for rendering and for the ping request.
      labelStack, nhMac, l3Intf = tunnelToAdjacencies[ tunnel ]
      # Convert from a tuple to a list to operate on
      labelStack = list( labelStack )
      # Update the tunnel tuple  with the entropy label stack
      # Tunnel looks like tuple( tuple( nh, tuple( labelStack ) ) )
      tunnel = ( ( tunnel[ 0 ][ 0 ], tuple( labelStack ) ), )
      clientIdToTunnels.append( tunnel )
      viaInfo.append( ( l3Intf, labelStack, nhMac ) )

   renderArgs = SrTePingRenderArgs( clientIdToTunnels=clientIdToTunnels,
                                    endpoint=endpoint, color=color )

   replyRenderMethod = ( lspPingSrTeReplyRender if egressValidateAddress else
                         lspPingSrTeReplyRenderWithoutCode )
   # If the user does not provide any egress address, we consider the endpoint
   # as the egressValidateAddress by default.
   if egressValidateAddress == 'default':
      egressValidateAddress = endpoint

   return ping( mount, state, src, dst, smac, interval, count, viaInfo,
                replyRenderMethod, lspPingSrTeStatisticsRender,
                renderArgs, ipv=ipv, tc=tc, standard=standard, size=size,
                padReply=padReply, egressValidateAddress=egressValidateAddress,
                tos=tos )

# ---------------------------------------------------------
#                   LspPing raw
# ---------------------------------------------------------

def handleLspPingRaw( prefix, mount, label, smac, dmac, interface, src, dst,
                      interval, count, verbose, nexthop, tc, standard=None,
                      size=None, padReply=False, tos=None, **kwargs ):
   state = getGlobalState()
   if dmac is None or interface is None:
      if nexthop:
         interface, dmac = resolveNexthop( mount, state, nexthop )
      if interface is None or dmac is None:
         print( "Failed to find a valid output interface" )
         return errno.EINVAL

   # Translation from default Ipv4 to default Ipv6 if not given
   ipv = IPV4
   if ( prefix and isIpv6Addr( prefix ) or
        nexthop and isIpv6Addr( nexthop ) or
        src and isIpv6Addr( src ) ):
      ipv = IPV6
   viaInfo = [ ( interface, label, dmac ) ]

   return ping( mount, state, src, dst, smac, interval, count, viaInfo,
                lspPingRawReplyRender, lspPingRawStatisticsRender, prefix, ipv=ipv,
                tc=tc, standard=standard, size=size, padReply=padReply, tos=tos )

# ---------------------------------------------------------
#                   LspPing Rsvp
# ---------------------------------------------------------

def handleLspPingRsvp( prefix, mount, sock, src=None, dst=None, smac=None,
                       dmac=None, vrf=None, interval=0, count=None, verbose=False,
                       label=None, interface=None, entry=None, tc=None, 
                       session=None, lsp=None, standard=None, size=None,
                       padReply=False, tos=None, **kwargs ):
   '''
      For RSVP nexthop info is calculated on each ping iteration.
      Thus passing viaInfo as None which will be calculated later in below
      ping function.
   '''
   protocol = kwargs[ 'type' ]
   json = kwargs.get( 'json' )
   tunnel = kwargs.get( 'tunnel' )
   subTunnelId = kwargs.get( 'sub_tunnel_id' )
   # render Ping header, timeout is the default value
   mplsPingProtocol = MplsPingProtocol( protocol=protocol, timeout=5,
                                        interval=interval )
   sendViaSocket( sock, MplsPing( mplsPingProtocol=mplsPingProtocol ) )
   # for RSVP protocol pingModel is sent inside ping process
   return ping( mount, None, src, dst, smac, interval, count, None,
                None, None, # with capi support no render functions needed
                None, rsvpTunnel=tunnel, rsvpSubTunnel=subTunnelId,
                session=session, lsp=lsp, protocol=protocol, tc=tc,
                standard=standard, size=size, padReply=padReply, tos=tos, json=json,
                sock=sock )

# --------------------------------------------------------------------
#                   LspPing LDP, MLDP and Segment Routing
# --------------------------------------------------------------------
def flexAlgoIdToName( mount, algoId ):
   if algoId:
      fad = mount.flexAlgoConfig.definition[ algoId ]
      if fad:
         return fad.name
   return None

def handleLspPingLdpMldpSr( prefix, mount, sock, src=None, dst=None, smac=None,
                            dmac=None, vrf=None, interval=0, count=None, 
                            verbose=False, label=None, interface=None, entry=None, 
                            tc=None, genOpqVal=None, sourceAddrOpqVal=None,
                            groupAddrOpqVal=None, jitter=None, responderAddr=None,
                            standard=None, size=None, padReply=False, tos=None,
                            **kwargs ):
   protocol = kwargs[ 'type' ]
   algorithm = kwargs.get( 'algorithm' )
   json = kwargs.get( 'json' )
   # srTunnel will be one of these values - 'isis', 'ospf' or None
   srTunnel = kwargs.get( 'srTunnel' )
   # The implementation of 'generic' protocol ping supports ISIS-SR, OSPF-SR
   # and LDP tunnel endpoints.
   nextHopsAndLabels, err, errVal, igpProtocol = getProtocolIpFec( mount, prefix,
                                                      protocol, genOpqVal,
                                                      sourceAddrOpqVal,
                                                      groupAddrOpqVal, algorithm,
                                                      allowEntropyLabel=True,
                                                      igpProtocol=srTunnel )

   mplsPingProtocol = MplsPingProtocol( protocol=protocol, timeout=5,
                                        interval=interval )
   sendViaSocket( sock, MplsPing( mplsPingProtocol=mplsPingProtocol ) )

   if protocol == LspPingTypeSr:
      if igpProtocol == igpProtocolType.ospf:
         protocol = LspPingTypeOspfSr
      algoName = flexAlgoIdToName( mount, algorithm )
      srModel = SrModel( prefix=prefix, algorithm=algoName,
                        igpProtocolType=igpProtocol )
      sendViaSocket( sock, MplsPing( srModel=srModel ) )
   else:
      genericModel = GenericModel( prefix=prefix )
      sendViaSocket( sock, MplsPing( genericModel=genericModel ) )
   state = getGlobalState()
   
   ipv = IPV6 if isIpv6Addr( prefix ) else IPV4
   if not nextHopsAndLabels:
      sendOrRenderPingErr( err, sock )
      return errVal

   # always use first via from given fecVia list
   nhlEntry = nextHopsAndLabels[ 0 ]
   nextHopIp = nhlEntry.nextHopIp
   intfId = nhlEntry.intfId
   labelStack = ( nhlEntry.label if isinstance( nhlEntry.label, list ) else
                  [ nhlEntry.label ] )

   # We don't need to keep the implicit null if there are other labels
   if any( label != 3 for label in labelStack ):
      labelStack = [ l for l in labelStack if l != 3 ]

   nextHopIntf, nextHopEthAddr = resolveNexthop( mount, state, nextHopIp,
                                                 intf=intfId )

   # Map L3 nexthop to L2 nexthop
   viaInfo, clientIdToVia = [], {}
   if ( nextHopIntf and nextHopEthAddr ):
      udpPam = state.lspPingClientRootUdpPam
      genPrefix = IpGenPrefix( str( prefix ) )
      el = generateMplsEntropyLabel(
            srcIp=( src or getIntfPrimaryIpAddr( mount, nextHopIntf, ipv ) ),
            dstIp=( genPrefix.v4Addr if ipv == IPV4 else genPrefix.v6Addr ),
            udpSrcPort=( 0 if not udpPam else udpPam.rxPort ) )
      fillEntropyLabelPlaceholders( labelStack, el )

      labelTup = tuple( labelStack )
      info = ( nextHopIntf, labelTup, nextHopEthAddr )
      viaInfo.append( info )
      clientIdToVia[ 0 ] = [ ( nextHopIp, labelTup, nextHopIntf ) ]
   else:
      errOutput = "Failed to find a valid output interface for next-hop {}".format(
                                                                          nextHopIp )
      sendOrRenderPingErr( errOutput, sock )
      return errno.EINVAL

   mldpInfo = MldpInfo( genOpqVal, sourceAddrOpqVal, groupAddrOpqVal,
                        jitter, responderAddr )

   return ping( mount, state, src, dst, smac, interval, count, viaInfo,
                None, None, # capi supported so no replyHandlers passed
                ( clientIdToVia, prefix, None ), protocol=protocol,
                ipv=ipv, tc=tc, mldpInfo=mldpInfo, standard=standard, size=size,
                padReply=padReply, tos=tos, sock=sock, algorithm=algorithm,
                json=json )
