# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliMatcher
import Tac

TEXT_MODE = 1
JSON_MODE = 2

# For policies configured via CLI, we add the following prefix to the policy name
# and used it as the key for servicePolicy collection. This will help us to identify
# the policy in the global policy space.
cliPrefix = 'CLI+'
separator = '+'

# L3 static policy config
MssL3PolicyAction = Tac.Type( 'MssL3::MssAction' )
MssL3V2PolicyAction = Tac.Type( 'MssL3V2::MssAction' )

MssL3PolicyModifier = Tac.Type( 'MssL3::MssPolicyModifier' )
mssL3ModifierVerbatim = MssL3PolicyModifier.verbatim
mssL3ModifierForwardOnly = MssL3PolicyModifier.forwardOnly
mssL3ModifierReverseOnly = MssL3PolicyModifier.reverseOnly

MssL3V2PolicyModifier = Tac.Type( 'MssL3V2::MssPolicyModifier' )
mssL3V2ModifierVerbatim = MssL3V2PolicyModifier.verbatim
mssL3V2ModifierForwardOnly = MssL3V2PolicyModifier.forwardOnly
mssL3V2ModifierReverseOnly = MssL3V2PolicyModifier.reverseOnly

PolicyState = Tac.Type( 'MssL3V2::MssPolicyState' )
LocalData = Tac.Type( 'MssL3V2::LocalData' )

# This is used to show user friendly name in the policy status output. Note that MSS
# doesn't support every protocol listed here.
ipProtocol = {
   1   : 'ICMP',     # Internet Control Message Protocol.
   2   : 'IGMP',     # Internet Group Management Protocol.
   4   : 'IPIP',     # IPIP tunnels (older KA9Q tunnels use 94).
   6   : 'TCP',      # Transmission Control Protocol.
   8   : 'EGP',      # Exterior Gateway Protocol.
   12  : 'PUP',      # PUP protocol.
   17  : 'UDP',      # User Datagram Protocol.
   22  : 'IDP',      # XNS IDP protocol.
   29  : 'TP',       # SO Transport Protocol Class 4.
   41  : 'IPV6',     # IPv6 header.
   43  : 'ROUTING',  # IPv6 routing header.
   44  : 'FRAGMENT', # IPv6 fragmentation header.
   46  : 'RSVP',     # Reservation Protocol.
   47  : 'GRE',      # General Routing Encapsulation.
   50  : 'ESP',      # Encapsulating security payload.
   51  : 'AH',       # Authentication header.
   58  : 'ICMPV6',   # ICMPv6.
   59  : 'NONE',     # IPv6 no next header.
   60  : 'DSTOPTS',  # IPv6 destination options.
   92  : 'MTP',      # Multicast Transport Protocol.
   98  : 'ENCAP',    # Encapsulation Header.
   103 : 'PIM',      # Protocol Independent Multicast.
   108 : 'COMP',     # Compression Header Protocol.
   132 : 'SCTP',     # Stream Control Transmission Protocol.
   137 : 'MPLS',     # MPLS in IP.
   255 : 'RAW'       # Raw IP packets.
}

l3Protocol = {
   1: 'ICMP'
}

l4Protocol = {
   6: 'TCP', 17: 'UDP'
}

# Maps TAC enum to user displayable enum name
stateMap = {
   'Internal Redirect': 'IR',
   'dryrunPending': 'allocationPending',
   'dryrunActive': 'allocationSuccess',
   'dryrunError': 'allocationError',
   'pending': 'pending',
   'active': 'active',
   'error': 'error'
}

# Maps user displayable enums to displayble strings
stateMapDisplayString = {
   'Internal Redirect': 'IR',
   'allocationPending': 'Allocation Pending',
   'allocationSuccess': 'Allocation Success',
   'allocationError': 'Allocation Error',
   'pending': 'Pending',
   'active': 'Active',
   'error': 'Error'
}

# Maps user displayable enums to abbreviations
abbreviatedStateMap = {
   'IR': 'IR',
   'allocationPending': 'AP',
   'allocationSuccess': 'AS',
   'allocationError': 'AE',
   'pending': 'P',
   'active': 'A',
   'error': 'E'
}

def displayStringForState( state, forCli=True ):
   strState = str( state )
   if strState.startswith( 'ip' ):
      strState = strState[ 2: ]
   elif strState.startswith( 'rule' ):
      strState = strState[ 4: ]

   if forCli: # Get value to display in CLI
      return stateMapDisplayString.get( strState, strState )
   else: # Get value to populate in JSON
      return stateMap.get( strState, strState )

def abbreviateState( state ):
   return abbreviatedStateMap.get( state, state )

# Common token definitions (used in Mss, MssClient CLI)
#------------------------------------------------------------------------------
# MssDeviceTokens
#------------------------------------------------------------------------------
matcherMssDevice = CliMatcher.KeywordMatcher( 'device',
      helpdesc='Set service device properties' )

#------------------------------------------------------------------------------
# Show policy keywords
#------------------------------------------------------------------------------
matcherMssStaticPolicySource = CliMatcher.KeywordMatcher( 'static',
      helpdesc='Policies configured via CLI' )

def getHostname( topologyStatus, chassisId ):
   def getName():
      for hostId, host in topologyStatus.host.items():
         if hostId.lower() == chassisId:
            return host.hostname
      return ''

   return '' if not topologyStatus else getName()

class PolicyFilter:
   def __init__( self, policy="", source="", device="", vsys="", vrf="",
                 switch="", ipv4=None, active=False, group=False ):
      self.policy = policy
      self.source = source
      self.device = device
      self.vsys = vsys
      self.vrf = vrf
      self.switch = switch
      self.ipv4 = ipv4
      self.active = active
      self.group = group

   def isAllowed( self, policy, source, device, vsys='', vrf='', ips=None,
                  switches=None, state=None ):
      allowed = (
         lambda actual, filterValue: actual == filterValue if filterValue else True )
      ipIncluded = bool( self.ipv4 & ips ) if self.ipv4 else True
      switchIncluded = self.switch in switches if self.switch else True
      activePolicy = state == PolicyState.active if self.active else True
      groupPolicy = policy.startswith(
         LocalData.internalRedirectPolicyName ) if self.group else True

      return allowed( policy, self.policy ) and allowed( source, self.source ) \
         and allowed( device, self.device ) and allowed( vsys, self.vsys ) \
         and allowed( vrf, self.vrf ) and switchIncluded and ipIncluded \
         and activePolicy and groupPolicy

   def areIpsNeeded( self ):
      return self.ipv4

   def areSwitchesNeeded( self ):
      return self.switch

   def isPolicyGroupAllowed( self, policies, source, device ):
      return any( self.isAllowed( i, source, device ) for i in policies )

#------------------------------------------------------------------------------------
# Command class for syntax of traffic inspection config. Used by both Mss and MssPM
#           [ no|default ] traffic inspection {local|local outbound}
#------------------------------------------------------------------------------------
class TrafficInspectionCmd( CliCommand.CliCommandClass ):
   syntax = 'traffic inspection TYPES'
   noOrDefaultSyntax = 'traffic inspection ...'
   data = {
      'traffic' : 'Traffic type for policy enforcement',
      'inspection': 'Traffic type for policy enforcement',
      'TYPES' : CliCommand.setCliExpression( {
         'local' : 'Consider East-West traffic',
         'outbound' : 'Consider Northbound traffic' } )
   }

   @staticmethod
   def handler( mode, args ):
      local = "local" in args
      outbound = "outbound" in args
      if outbound and not local:
         mode.addError( "Only Northbound traffic inspection is not supported" )
         return
      mode.trafficInspectionIs( local, outbound )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noTrafficInspection()

#------------------------------------------------------------------------------------
# Command class for syntax of
#           direction (forward | forward reverse)
# This used in multiple places of static policy command
#------------------------------------------------------------------------------------
class DirectionCmd( CliCommand.CliCommandClass ):
   syntax = 'direction TYPES'
   noOrDefaultSyntax = 'direction ...'
   data = {
      'direction': 'Choose the traffic direction for enforcement',
      'TYPES' : CliCommand.setCliExpression( {
         'forward': 'Enforcement of only forward direction traffic',
         'reverse': 'Enforcement of only reverse direction traffic' } )
   }

   @staticmethod
   def handler( mode, args ):
      forward = 'forward' in args
      reverse = 'reverse' in args
      if reverse and not forward:
         mode.addError( "Enforcement of only reverse direction is not supported. "
                        "Consider inverting the match criteria" )
         return
      mode.directionIs( forward, reverse )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noDirection()
