#!/usr/bin/env python3
# Copyright (c) 2015-2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
#
# pylint: disable=R1702

# pylint: disable=consider-using-f-string

''' MSS monitors service device policies using each vendors API and automates
    network logical topology changes to put the service device in the traffic
    path for hosts identified in the service device policies.
'''

import os
import time
import uuid
import traceback
import threading
import Agent
import Tac
from collections import defaultdict
from contextlib import contextmanager
from Arnet import IpGenAddrWithMask
from ControllerdbEntityManager import Controllerdb
from MssPolicyMonitor import PluginController
from MssPolicyMonitor import Lib, Logger
from MssPolicyMonitor.Lib import MssDeviceName
from MssPolicyMonitor.Lib import ACTIVE, SHUTDOWN, DRY_RUN, ZONE_A, ZONE_B
from MssPolicyMonitor.Lib import PORT_CHANNEL, MLAG
from MssPolicyMonitor.Lib import INTERCEPT_ZONE, NON_INTERCEPT_ZONE
from MssPolicyMonitor.Lib import VERBATIM, FORWARD_ONLY
from MssPolicyMonitor.Lib import UNCLASSIFIED_ZONE, b0, b1, b2, b3, b4, v
from MssPolicyMonitor.Lib import __defaultTraceHandle__  #pylint: disable-msg=W0611
from MssPolicyMonitor import Reactors
from MssPolicyMonitor.Reactors import ActivityLock, deviceConfigComplete
from MssPolicyMonitor.PluginLib import ServiceDeviceRoutingTables
from MssPolicyMonitor.Error import DeviceStateMutexError

Tac.activityManager.useEpoll = True
ServiceDeviceLinkType = Tac.Type('MssPolicyMonitor::ServiceDeviceLink')
InterceptSpecType = Tac.Type('MssPolicyMonitor::InterceptSpec')
ServiceInterfaceType = Tac.Type('Mss::ServiceInterface')
InterceptFlowType = Tac.Type('Mss::InterceptFlowInfo') 
SwitchIntfPairType = Tac.Type('Mss::SwitchIntfPair')
IpGenAddrType = Tac.Type('Arnet::IpGenAddr')
NullIpAddr = Tac.Value('Arnet::IpGenAddrWithMask')
MssGlobals = Tac.Type( 'Mss::MssGlobals' )
MssL3MssPolicyModifier = Tac.Type( 'MssL3::MssPolicyModifier' )
TaskSchedulerRoot = Tac.Type( 'Ark::TaskSchedulerRoot' )

deviceStateMutexes = {}  # key=serviceDeviceId, value=threading.RLock object
servicePolicyCache = {}
SOURCE_ZONE = 'srcZone'
DEST_ZONE = 'dstZone'

HAPairDeviceId = Tac.Type( 'MssPolicyMonitor::HAPairDeviceId' )

def populatePolicyL4Services( rawPolicy, devicePolicy ):
   devicePolicy.dstL4App.clear()
   for protocol, portList in rawPolicy.dstL4Services.items():
      l4app = devicePolicy.newDstL4App( protocol )
      for port in portList:
         l4app.port.add( str( port ) )


def assignZoneType( devPolicyZone, zoneType ):
   if zoneType == Lib.VIRTUAL_WIRE:
      devPolicyZone.zoneType = Lib.VIRTUAL_WIRE_ZONE
   elif zoneType == Lib.LAYER3:
      devPolicyZone.zoneType = Lib.LAYER3_ZONE
   else:
      devPolicyZone.zoneType = Lib.UNKNOWN_ZONE_TYPE

def delta( startTime ):
   return  'time=%.3fs' % ( time.time() - startTime )


def genServiceInterfaceUuid():
   return str( uuid.uuid4() ) # generate a random UUID


def getArnetIpAddr( ip ):
   return Tac.Value( 'Arnet::IpGenAddrWithMask', str( ip ) ) if ip else NullIpAddr


def threadName():
   return threading.currentThread().name


def getDeviceMonitorInstanceName( deviceSetName, deviceIp ):
   return f'dmi:set={deviceSetName}:dev={deviceIp}'


def getServicePolicyName( devicePolicy, nearLinks, farLinks,
                          nearSwitchLinks=None, farSwitchLinks=None ):
   def switchLinkStr( links ):
      #Lib.mssMultiVwireEnabled:
      return '+'.join( [ '{}_{}'.format(
         Lib.shortenIntfName( switchIntfPair.intf ),
         compressMacAddress( switchIntfPair.switchId, keepAristaOUI=False ) )
         for switchIntfPair in links ] )

   def linkStr( links ):
      return '+'.join( [ '{}_{}'.format(
         link.switchIntf.replace( 'Ethernet', 'Et' ),
         compressMacAddress( link.switchId, keepAristaOUI=False ) )
         for link in links.values() ] )

   vlanStr =  ",".join( list( farLinks.values() )[ 0 ].allowedVlan )
   nearLinkStr = ''
   farLinkStr = ''
   if Lib.isMssMultiVwireEnabled():
      nearLinkStr = switchLinkStr( nearSwitchLinks )
      farLinkStr = switchLinkStr( farSwitchLinks )
   else:
      nearLinkStr = linkStr( nearLinks )
      farLinkStr = linkStr( farLinks )
   policyName = '{}:{}_N:{}_F:{}_V:{}'.format(
      devicePolicy.serviceDeviceType, devicePolicy.serviceDeviceId,
      nearLinkStr, farLinkStr, vlanStr )
   policyName = Lib.shortenPluginName( policyName )
   b1( 'policyName: ', policyName )
   return policyName


def genNewServicePolicy( devicePolicy,
                         nearSwitchLinks, nearLinks, nearIntfUuid,
                         farSwitchLinks, farLinks, farIntfUuid,
                         mssSvcPolicyCfg, mpmStatus ):
   # caller must already have tac activity lock
   svcPolicyName = getServicePolicyName( devicePolicy, nearLinks, farLinks,
                                         nearSwitchLinks, farSwitchLinks )
   b1( v( svcPolicyName ), 'from DevPol', v( devicePolicy.policyName ) )
   servicePolicy = mssSvcPolicyCfg.newServicePolicy( svcPolicyName )
   servicePolicy.policyName[ devicePolicy.policyName ] = True
   servicePolicy.serviceDeviceId = devicePolicy.serviceDeviceId
   servicePolicy.configSource = getConfigSource( devicePolicy )
   servicePolicy.nearIntfUuid = nearIntfUuid
   servicePolicy.farIntfUuid = farIntfUuid
   deviceStatus = mpmStatus.serviceDeviceStatus.get( devicePolicy.serviceDeviceId )
   mssState = Lib.MSSPM_STATE_TO_MSS_STATE.get( deviceStatus.state )
   if mssState:  # only set states that map
      servicePolicy.state = mssState
   return servicePolicy


def invertZoneClass( zoneClass ):
   if zoneClass == INTERCEPT_ZONE:
      return NON_INTERCEPT_ZONE
   elif zoneClass == NON_INTERCEPT_ZONE:
      return INTERCEPT_ZONE
   else:
      return UNCLASSIFIED_ZONE


def compressMacAddress( mac, keepAristaOUI=True ):
   mac = mac.replace( '.', '' )
   mac = mac.replace( ':', '' )
   if not keepAristaOUI:
      for oui in Lib.ARISTA_OUI:
         mac = mac.lower().replace( oui, '' )
   return mac


def getConfigSource( devicePolicy ):
   return devicePolicy.serviceDeviceType  # in future may include additional info


def isActiveState( monitoringState ):
   # pylint: disable-next=consider-using-in
   return monitoringState == ACTIVE or monitoringState == DRY_RUN


def doUpdatesInDeviceState( monitoringState ):
   return isActiveState( monitoringState )


def isDeviceSetCompleteAndActiveUsingLock( deviceSet ):
   with ActivityLock():
      return isActiveState( deviceSet.state ) and deviceSetComplete( deviceSet )


def deviceSetComplete( deviceSet ):
   ''' Verify that all necessary attributes have been set.
   '''
   complete = bool(
      deviceSet.state != Lib.INITIALIZING and
      deviceSet.policySourceType != Lib.UNASSIGNED and
      deviceSet.serviceDevice and  # at least one service device
      ( deviceSet.policyTag or deviceSet.offloadTag ) and  # at least one tag
      deviceSet.queryInterval and
      deviceSet.timeout and
      # deviceSet.retries and  # retries can be 0
      deviceSet.exceptionHandling )
   b4( 'DeviceSet:', v( deviceSet.name ), 'complete:', v( complete ) )
   return complete


def updatePolicyNames( devicePolicy, servicePolicy ):
   if devicePolicy.policyName not in servicePolicy.policyName:
      servicePolicy.policyName[ devicePolicy.policyName ] = True

   if servicePolicy.name not in devicePolicy.childServicePolicy:
      devicePolicy.childServicePolicy[ servicePolicy.name ] = True


def populateIntfNeighbors( rawPolicy, lldpNeighbors, deviceId, deviceSet ):
   ''' Populate service device interface to switch interface neighbors.
   '''
   neighbors = Lib.mergeNeighborsWithIntfMap( lldpNeighbors, deviceId, deviceSet )
   for zoneIntfs in [ rawPolicy.srcZoneInterfaces, rawPolicy.dstZoneInterfaces ]:
      for sdIntf in zoneIntfs:
         for physIntf in sdIntf.physicalIntfs:
            if physIntf.name in neighbors:
               rawPolicy.intfNeighbors[ physIntf.name ] = neighbors[ physIntf.name ]
            else:
               b3( 'No neighbor for:', v( deviceId ), 'physIntf:', 
                   v( physIntf.name ) )


def isL2AndMultiVwireEnabled( policy ):
   return Lib.isL2RawPolicy( policy ) and Lib.isMssMultiVwireEnabled()


def isValidRawPolicy( policy, deviceId, vinst, skippedPolicyLogger ):
   ''' validate raw policy from service device
   '''
   def log( msg ):
      skippedPolicyLogger.log( policy.name, deviceId, vinst, msg )

   if not policy.name:
      b1( 'skip policy with no name, deviceId', v( deviceId ) )
      log( "name is missing" )
      return False

   if policy.name.startswith( MssGlobals.internalNamePrefix ):
      b1( 'skip policy ', v( policy.name ),
          ' starting with reserved internal prefix ',
          v( MssGlobals.internalNamePrefix ) )
      log( "name starts with a reserved prefix (%)" )
      return False

   if Lib.isL2RawPolicy( policy ):
      if policy.srcZoneType != policy.dstZoneType:
         b1( 'skip policy', v( policy.name ), 'src and dest zones not same type' )
         return False

      if not policy.srcIpAddrList and not policy.dstIpAddrList:
         b1( 'skip policy', v( policy.name ), 'no intercepts found' )
         return False

      if not policy.srcZoneInterfaces or not policy.dstZoneInterfaces:
         b1( 'skip policy', v( policy.name ), 'no src and/or dest zone interfaces' )
         return False

      if not ( { i.name for i in policy.srcZoneInterfaces } ^
               { i.name for i in policy.dstZoneInterfaces } ):
         b1( 'skip policy', v( policy.name ), 'src and dest intf sets equivalent' )
         return False

      symmDiff = (
            { vl for i in policy.srcZoneInterfaces for vl in i.vlans } ^
            { vl for i in policy.dstZoneInterfaces for vl in i.vlans } )
      if symmDiff:
         b1( 'skip policy', v( policy.name ), 'invalid vwire, src and dest '
             'interfaces carry different vlans:', v( symmDiff ) )
         return False

   else:
      if ( not all( i.ipAddr for i in policy.srcZoneInterfaces ) or
           not all( i.ipAddr for i in policy.dstZoneInterfaces ) ):
         b1( 'skip policy', v( policy.name ), 'L3 zone intf missing IP address' )
         log( "source and/or destination zone interface's IP address is missing" )
         return False

      if ( policy.isForwardOnly and not policy.isVerbatim ):
         b1( 'skip forward only non-verbatim policy', v( policy.name ) )
         log( "policy for forward-only enforcement must have a verbatim tag" )
         return False

      if ( not policy.isOffloadPolicy and policy.isVerbatim
            and not policy.isForwardOnly and
           ( ( policy.srcZoneName == 'any' and not policy.srcIpAddrList ) or
             ( policy.dstZoneName == 'any' and not policy.dstIpAddrList ) ) ):
         b1( 'skip redirect verbatim policy', v( policy.name ), 
             'cannot resolve any src or dst' )
         log( "verbatim bi-directional redirect policy must define a source and "
               "a destination zone or IP address" )
         return False

      if ( not policy.isOffloadPolicy and policy.isVerbatim
            and policy.isForwardOnly and policy.srcZoneName == 'any'
            and not policy.srcIpAddrList ):
         b1( 'skip redirect forward-only verbatim policy', v( policy.name ),
             'cannot resolve any src' )
         log( "verbatim forward-only redirect policy must define a source zone or "
               "IP address" )
         return False

      vrfSet = { i.vrf for i in policy.srcZoneInterfaces if i.vrf }
      vrfSet |= { i.vrf for i in policy.dstZoneInterfaces if i.vrf }
      if len( vrfSet ) > 1:
         b1( 'skip policy', v( policy.name ), 'not all zone intfs in the same vrf' )
         log( "must apply to a single virtual-router domain" )
         return False

   # Maintain a list of invalid IPv4 subnets if any
   invalidIpv4Subnets = []
   for zoneIpAddrList in [ policy.srcIpAddrList, policy.dstIpAddrList ]:
      for ipAddr in zoneIpAddrList:
         if not Lib.isIpV4AddrOrRangeOrSubnet( ipAddr ):
            b1( 'skip policy', v( policy.name ), 'invalid IPv4 intercept:',
                v( ipAddr ) )
            log( "source/destination must be either IPv4 addresses or 'any'" )
            return False

         # Check for invalid IPv4 subnets, if the ip address is not an ip range
         # and not a valid Arnet subnet, flag the ip address
         if not ( Lib.isIpRange( ipAddr ) or
               Lib.isValidArnetSubnet( Lib.convertIpMaskSyntax( ipAddr ),
                  strictCheck=True ) ):
            invalidIpv4Subnets.append( ipAddr )

   # Check if we detected any invalid IPv4 subnets
   if invalidIpv4Subnets:
      # Print the top 5 invalid IPv4 subnets
      subnets = ", ".join( invalidIpv4Subnets[ : 5 ] )
      # If more than 5 invalid subnets, append ellipses to indicate
      # that there more invalid subnets than displayed.
      if len( invalidIpv4Subnets ) > 5:
         subnets += " ..."
      b1( 'skip policy', v( policy.name ), 'invalid IPv4 subnet(s): ',
            v( subnets ) )
      log( "Invalid IPv4 source and/or destination subnet(s): " + subnets )
      return False

   if ( ( policy.srcZoneInterfaces and
          all( intf.state != Lib.LINK_STATE_UP
               for intf in policy.srcZoneInterfaces ) ) or
        ( policy.dstZoneInterfaces and
          all( intf.state != Lib.LINK_STATE_UP
               for intf in policy.dstZoneInterfaces ) ) ):
      b1( 'skip policy', v( policy.name ),
            'all interfaces in source or destination zones are down:\n',
            v( str( policy ) ) )
      log( "source and/or destination interfaces are down" )
      return False
      
   vwireMap = defaultdict( dict ) # dict of vwire to intf map per zone
   zoneNo = 0
   for zoneIntfs, zoneType in [ ( policy.srcZoneInterfaces, policy.srcZoneType ),
                                ( policy.dstZoneInterfaces, policy.dstZoneType ) ]:
      vwirePhysicalIntfHasNbor = False
      for sdIntf in zoneIntfs:
         if sdIntf.state != Lib.LINK_STATE_UP:
            if isL2AndMultiVwireEnabled( policy ):
               b2( v( sdIntf ) )
               continue
         elif ( Lib.isL2RawPolicy( policy ) and
                ( not sdIntf.vlans or sdIntf.vlans == [ '0' ] ) ):
            b1( 'skip policy', v( policy.name ), 'link:', v( sdIntf.name ),
                'vwire must carry at least one tagged VLAN:', v( sdIntf.vlans ) )
            return False
         lagPhysicalIntfHasNbor = False
         for physIntf in sdIntf.physicalIntfs:
            # pylint: disable-next=no-else-continue
            if physIntf.state != Lib.LINK_STATE_UP:
               b2( v( sdIntf ), v( physIntf ) )
               continue
            elif not Lib.isL2RawPolicy( policy ) or zoneType == Lib.LAYER3:
               # don't need to verify neighbor for L3 intfs
               continue
            hasNbor = bool(
               physIntf.name in policy.intfNeighbors and
               'switchChassisId' in policy.intfNeighbors[ physIntf.name ] and
               'switchIntf' in policy.intfNeighbors[ physIntf.name ] and
               policy.intfNeighbors[ physIntf.name ][ 'switchChassisId' ] and
               policy.intfNeighbors[ physIntf.name ][ 'switchIntf' ] )
            if hasNbor:
               if sdIntf.isLag:
                  lagPhysicalIntfHasNbor = True
               elif isL2AndMultiVwireEnabled( policy ):
                  vwirePhysicalIntfHasNbor = True
                  if sdIntf.attribs is not None and 'vwire' in sdIntf.attribs:
                     vwire = sdIntf.attribs[ 'vwire' ]
                     vwireMap[ vwire ][ zoneNo ] = physIntf.name
                     b2( 'vwireMap: ', v( vwireMap ), ' for:', v( physIntf.name ) )
               neighbor = 'nbor={} {}'.format(
                  policy.intfNeighbors[ physIntf.name ][ 'switchIntf' ],
                  policy.intfNeighbors[ physIntf.name ][ 'switchChassisId' ] )
            else:
               neighbor = 'nbor=None'
            b2( v( sdIntf ), v( physIntf if sdIntf.isLag else '' ), v( neighbor ) )
            if not ( sdIntf.isLag or isL2AndMultiVwireEnabled( policy ) ) and \
               not hasNbor:
               b1( 'skip policy', v( policy.name ), 'no neighbor for:',
                   v( physIntf.name ) )
               return False
         if ( Lib.isL2RawPolicy( policy ) and
              ( sdIntf.isLag and not lagPhysicalIntfHasNbor ) ):
            b1( 'skip policy', v( policy.name ),
                'no physical intfs up with neighbor:', v( sdIntf.name ) )
            return False
      if ( isL2AndMultiVwireEnabled( policy ) and not vwirePhysicalIntfHasNbor ):
         b1( 'skip policy', v( policy.name ),
             'no physical intfs up with neighbor' )
         return False
      zoneNo += 1 # increment zoneNo

   if isL2AndMultiVwireEnabled( policy ):
      # Check that link state on both ends of the firewall vwire are UP
      b2( 'vwire to intf map: ', v( vwireMap ) )
      bothEndsOfVwireUp = False
      for vwire, zoneIntf in vwireMap.items():
         vwireIntf0 = zoneIntf.get( 0, None )
         vwireIntf1 = zoneIntf.get( 1, None )
         if vwireIntf0 and vwireIntf1:
            bothEndsOfVwireUp = True
         elif vwireIntf0 != vwireIntf1:
            # only one intf is UP, bring down the other end of vwire
            policy.intfNeighbors.pop( vwireIntf0, None )
            policy.intfNeighbors.pop( vwireIntf1, None )
            b1( 'skip interfaces on both ends of vwire:', v( vwire ),
                'as only one of those were UP' )

      if not bothEndsOfVwireUp:
         b1( 'skip policy', v( policy.name ),
             'no vwire with physical intfs up on both ends' )
         return False

   return True


def updateSDLinkMembership( sdZoneLink, sdLinks ):
   ''' Since can't just replace ServiceDeviceZone.link instantiating 
       collection with dict of current ServcieDeviceLinks, compute
       differences then update ServiceDeviceZone.link collection as needed.
       Since tacc-Python bindings don't call the entities own hash()
       function we must use the algorithm below.
   '''
   oldLinks = { ( linkName, serviceDeviceLink.hash() )
                for ( linkName, serviceDeviceLink ) in sdZoneLink.items() }
   newLinks = { ( linkName, serviceDeviceLink.hash() )
                for ( linkName, serviceDeviceLink ) in sdLinks.items() }
   removedLinks = oldLinks - newLinks
   addedLinks = newLinks - oldLinks
   for linkName, _ in removedLinks:
      #b3( 'updateSDLinkMembership removing link:', v( linkName ) )
      del sdZoneLink[ linkName ]
   for linkName, _ in addedLinks:
      #b3( 'updateSDLinkMembership adding link:', v( linkName ) )
      sdZoneLink.addMember( sdLinks[ linkName ] )


def updateSwitchLinkMembership( sdZone ):
   sdZone.switchLink.clear()
   for linkName, sdLink in sdZone.link.items():
      switchIntfPair = getSwitchIntfPair( sdLink )
      if switchIntfPair in sdZone.switchLink:
         switchLink = sdZone.switchLink[ switchIntfPair ]
         switchLink.serviceDeviceLinkName.add( linkName )
      else:
         sdZone.switchLink.newMember( switchIntfPair )
         sdZone.switchLink[ switchIntfPair ].\
               serviceDeviceLinkName.add( linkName )


def equivalentAllowedVlans( av1, av2 ):
   return not bool( set( av1.keys() ) ^ set( av2.keys() ) )


def equivalentSwitchIntfs( serviceDeviceLinks, mssServiceIntfs ):
   sdLinksCopy = list( serviceDeviceLinks.values() )[ : ]
   for svcIntf in mssServiceIntfs:
      for sdLink in serviceDeviceLinks.values():
         # b3( 'SDLink:', v( sdLink.switchId ), v( sdLink.switchIntf ),
         #     v( sdLink.portChannel ), v( sdLink.allowedVlan.keys() ), 'SvcIntf:',
         #     v( svcIntf.switchId ), v( svcIntf.intf ),
         #     v( svcIntf.allowedVlan.keys() ) )
         if ( ( ( sdLink.mlagSystemId and sdLink.mlagSystemId == svcIntf.switchId and
                  sdLink.mlag == svcIntf.intf )
                 or
                 ( sdLink.switchId == svcIntf.switchId and
                   ( ( sdLink.portChannel and sdLink.portChannel == svcIntf.intf ) or
                     ( not sdLink.portChannel and sdLink.switchIntf == svcIntf.intf )
                   ) ) )
              and
              equivalentAllowedVlans( sdLink.allowedVlan, svcIntf.allowedVlan ) ):
            sdLinksCopy.remove( sdLink )  # remove matches
   if not sdLinksCopy:  # will be empty if both lists are equivalent
      return True
   return False


def allLinksAreLagsOrMlags( devicePolicy ):
   return (
      devicePolicy.zoneA.link and devicePolicy.zoneB.link and
      all( link.portChannel or link.mlag
           for link in devicePolicy.zoneA.link.values() ) and
      all( link.portChannel or link.mlag
           for link in devicePolicy.zoneB.link.values() ) )


def incPolicyCounters( mpmStatus, deviceStatus ):
   with ActivityLock():
      mpmStatus.numPoliciesProcessed += 1
      deviceStatus.numPoliciesProcessed += 1


def validateMonitoringThread( monThreadName, deviceStatus ):
   if monThreadName != deviceStatus.monitorThreadName:
      b0( 'WARNING threadName:', v( monThreadName ),
          '!= deviceStatus.monitorThreadName:', v( deviceStatus.monitorThreadName ) )
      return False
   return True


def getInterceptZoneAttribs( devicePolicy, interceptZone ):
   if interceptZone == ZONE_A:  # determine near and far service device links
      nearZoneName = devicePolicy.zoneA.zoneName
      farZoneName = devicePolicy.zoneB.zoneName
      nearLinks = devicePolicy.zoneA.link
      farLinks = devicePolicy.zoneB.link
      updateSwitchLinkMembership( devicePolicy.zoneA )
      updateSwitchLinkMembership( devicePolicy.zoneB )
      nearSwitchLinks = devicePolicy.zoneA.switchLink
      farSwitchLinks = devicePolicy.zoneB.switchLink
   else:  # interceptZone == ZONE_B:
      nearZoneName = devicePolicy.zoneB.zoneName
      farZoneName = devicePolicy.zoneA.zoneName
      nearLinks = devicePolicy.zoneB.link
      farLinks = devicePolicy.zoneA.link
      updateSwitchLinkMembership( devicePolicy.zoneA )
      updateSwitchLinkMembership( devicePolicy.zoneB )
      nearSwitchLinks = devicePolicy.zoneB.switchLink
      farSwitchLinks = devicePolicy.zoneA.switchLink
   return nearZoneName, nearLinks, nearSwitchLinks,\
          farZoneName, farLinks, farSwitchLinks


def genServicePolicyCacheKey( nearZoneName, nearLinks, farZoneName, farLinks ):
   cacheKey = 'N={}_{}_F={}_{}'.format(
      nearZoneName,
      ':'.join( [ str( link.hash() ) for link in nearLinks.values() ] ),
      farZoneName,
      ':'.join( [ str( link.hash() ) for link in farLinks.values() ] ) )
   return cacheKey


def getSwitchIntfPair( link ):
   ''' Get switchIntfPair from serviceDeviceLink
   '''
   switchId = link.switchId
   # TODO: uncomment when Mss agent supports MLAG ServiceInterfaces
   # if link.mlagSystemId and link.mlag:
   #    switchId = link.mlagSystemId
   #    intf = link.mlag
   # elif link.portChannel:
   if link.portChannel:
      intf = link.portChannel
   else:
      intf = link.switchIntf
   return SwitchIntfPairType( switchId, intf )


def traceInstances( monitorInstances, msg='' ):
   ''' trace output for
       self.monitorInstances[ deviceSet.name ][ instanceName ] = instance
   '''
   for deviceSetName, instanceMap in monitorInstances.items():
      for instName, inst in instanceMap.items():
         b2( v( msg ), 'devSet', deviceSetName, v( instName ), v( inst ) )


def setConvergenceComplete( sysdbMgr ):
   b0( 'setting mss/servicePolicySourceState.convergenceComplete=True' )
   sysdbMgr.mssSvcPolicySrcState.convergenceComplete = True
   sysdbMgr.purgeExpiredDevicePolicies()


def getDeviceSetUsingLock( deviceSetName, mpmConfig ):
   with ActivityLock():
      return mpmConfig.deviceSet.get( deviceSetName )


def getServiceDevicesUsingLock( deviceSet ):
   with ActivityLock():
      return list( deviceSet.serviceDevice.values() )


def getSvcDeviceStatusUsingLock( deviceId, mpmStatus, createIfNeeded=False ):
   with ActivityLock():
      return ( mpmStatus.serviceDeviceStatus.get( deviceId ) or
               ( createIfNeeded and mpmStatus.newServiceDeviceStatus( deviceId ) ) )


def getSvcDeviceStatusItemsUsingLock( mpmStatus ):
   with ActivityLock():
      return list( mpmStatus.serviceDeviceStatus.items() )


def getServicePolicyUsingLock( servicePolicyName, mssSvcPolicyCfg ):
   with ActivityLock():
      return mssSvcPolicyCfg.servicePolicy.get( servicePolicyName )


def genMutexAsNeededFor( deviceId ):
   ''' Generate a mutex for the passed deviceId if it doesn't already exist.
   '''
   if deviceId not in deviceStateMutexes:
      deviceStateMutexes[ deviceId ] = threading.RLock()  # reentrant lock


def deviceStateMutexExistsFor( deviceId ):
   if deviceId in deviceStateMutexes:
      return True
   else:
      b4( v( deviceId ), 'not in deviceStateMutexes' )
      return False


def deviceStateMutex( deviceId ):
   if Reactors.activityLockHolder == threading.currentThread().name:
      raise Exception( '%s should NOT be holding ActivityLock here!' % threadName() )
   return deviceStateMutexes.get( deviceId )

@contextmanager
def deviceStateLocksGrabbed( deviceIds ):
   def grabDeviceStateLocks():
      ''' This grabs the locks of the device status in a
          deterministic order. This ensures that there is no
          deadlock where two threads grabs the locks in different order.
      '''
      grabbedLocks = []
      for deviceId in sorted( deviceIds ):
         devStateMutex = deviceStateMutex( deviceId )
         if devStateMutex:
            devStateMutex.acquire()
            grabbedLocks.append( devStateMutex )
         else:
            # Couldn't find device state mutex for some devices
            # exit now.
            b1( 'no deviceStatusMutexExistsFor:', v( deviceId ) )
            return grabbedLocks, False
      return grabbedLocks, True

   def releaseDeviceStateLocks( grabbedLocks ):
      ''' Release all the grabbed locks in reverse order
      '''
      for lock in reversed( grabbedLocks ):
         lock.release()

   locks, status = grabDeviceStateLocks()
   if not status:
      releaseDeviceStateLocks( locks )
      raise DeviceStateMutexError( 'Cannot grab all device state locks' )
   try:
      yield
   finally:
      releaseDeviceStateLocks( locks )

####################################################################################
class SysdbMgr:
   ''' Handles updates to Sysdb tac models at multiple mount points.
   '''
   def __init__( self, mpmConfig, mpmStatus, mssConfig,
                 mssSvcIntfCfg, mssSvcPolicyCfg, mssSvcPolicySrcState, svcDevSrcCfg,
                 policySrcCfg, cdbTopology, policyMonitorInstances, sslStatus ):
      self.mpmConfig = mpmConfig
      self.mpmStatus  = mpmStatus
      self.mssConfig = mssConfig
      self.mssSvcIntfCfg = mssSvcIntfCfg
      self.mssSvcPolicyCfg = mssSvcPolicyCfg
      self.mssSvcPolicySrcState = mssSvcPolicySrcState
      self.svcDevSrcCfg = svcDevSrcCfg
      self.policySrcCfg = policySrcCfg
      self.monitorInstances = policyMonitorInstances
      self.cdbTopology = cdbTopology
      self.zoneClassMap = {}
      self.sslStatus = sslStatus
      self.maxInterceptsPerSvcPolicy = mpmConfig.maxInterceptsPerServicePolicy
      # HA peers share the deviceId to avoid churn when a switch over happens
      self.haDeviceIdMap = {}
      self.fwZoneLogger = Logger.MssZoneInterfaceLogger()
      self.skippedPolicyLogger = Logger.MssSkippingPolicyLogger()
      self.firewallLogger = Logger.MssFirewallLogger()
      self.aggrMgrLogger = Logger.MssFirewallLogger()
      self.hasAggrMgrPair = {} # This is a dictionary which indicates if there is
                               # an aggregation manager pair in a device set
      b2( 'maxInterceptsPerSvcPolicy:', v( self.maxInterceptsPerSvcPolicy ) )

   def ceEnabled( self ):
      return self.mssConfig.policyEnforcementConsistency == 'strict'

   def buildMssDeviceName( self, devName, vinstName, fwType ):
      return MssDeviceName( devName, vinstName, fwType, self.ceEnabled() )

   def getDeviceName( self, mssDeviceId ):
      return MssDeviceName.fromString( mssDeviceId, self.ceEnabled() ).devName

   # Monitoring plugin and instance mgmt
   #--------------------------------------------------------------------------------
   def isDeviceAvailableUsingLock( self, device, deviceSet ):
      with ActivityLock():
         return self.isDeviceAvailable( device, deviceSet )

   def isDeviceAvailable( self, device, deviceSet ):
      ''' ServiceDeviceStatus object can only be used by one device-set
          at a time.  Only one device-set at a time can "own" the device
          by moving it out of shutdown state.
          Caller should be holding ActivityLock
      '''
      devStatus = self.mpmStatus.serviceDeviceStatus.get( device.ipAddrOrDnsName )
      if ( devStatus and
           devStatus.deviceSetName != deviceSet.name and
           devStatus.state != SHUTDOWN ):
         return False
      else:
         return True

   def isDeviceStateActive( self, serviceDeviceId ):
      deviceStatus = getSvcDeviceStatusUsingLock( serviceDeviceId, self.mpmStatus )
      return bool( deviceStatus and isActiveState( deviceStatus.state ) )

   def isMemberDevice( self, deviceId, deviceSet ):
      ''' If the device set has an aggregation manager and if the instance name
          corresponding to the ( deviceSet, deviceId ) is not present in the
          monitoring instances, then this is a member device.
      '''
      instanceName = getDeviceMonitorInstanceName( deviceSet.name, deviceId )
      return ( deviceSet.isAggregationMgr and
          instanceName not in self.monitorInstances[ deviceSet.name ] )

   def isMonitoringInstanceActive( self, deviceId, deviceSet ):
      ''' Check if a monitoring instance has been created for a service device.
          This can be applied to aggregation managers or firewalls.
      '''
      instanceName = getDeviceMonitorInstanceName( deviceSet.name, deviceId )
      return instanceName in self.monitorInstances[ deviceSet.name ]

   def cleanupStalePolicyAndDeviceStatus( self, deviceSet ):
      ''' This is an audit function that cleans up orphan policies and device
          status. If there are no more monitoring instances left that means all
          service devices have been cleaned up. Cleanup any stale policy and
          device status.
      '''
      if not self.monitorInstances[ deviceSet.name ]:
         for deviceId, devStatus in \
             getSvcDeviceStatusItemsUsingLock( self.mpmStatus ):
            if devStatus.deviceSetName == deviceSet.name:
               b2( 'cleanup stale policy for DeviceSet:', v( deviceSet.name ),
                   'Device:', deviceId )
               self.deletePoliciesForDevice( deviceId )
               devStatus.monitorThreadName = ''
         self.deleteDeviceStatusForDeviceSet( deviceSet.name )

   def isAggrMgrHAPairActive( self, deviceSet ):
      # If the device set is an Aggregation Manager and has 2 monitoring
      # instances, then we have an Aggregation Manager HA pair
      return ( deviceSet.isAggregationMgr and
          len( self.monitorInstances[ deviceSet.name ] ) == 2 )

   def restartMonitoringDeviceSetAsNeeded( self, deviceSet, cleanup=False ):
      ''' Start/Restart policy monitoring for all devices in set
      '''
      if isDeviceSetCompleteAndActiveUsingLock( deviceSet ):
         b2( '(re)startMonitoringDeviceSet:', v( deviceSet.name ), 'cleanup:',
             v( cleanup ) )
         self.stopMonitoringDeviceSet( deviceSet )
         if cleanup:
            self.deleteAllPoliciesForDeviceSet( deviceSet )
            self.deleteDeviceStatusForDeviceSet( deviceSet.name )

         for serviceDevice in getServiceDevicesUsingLock( deviceSet ):
            self.restartMonitoringDeviceAsNeeded( serviceDevice, deviceSet )

   def getDeviceConfig( self, deviceSetName, serviceDeviceName ):
      deviceSet = self.mpmConfig.deviceSet.get( deviceSetName )
      if not deviceSet:
         return {}

      serviceDevice = deviceSet.serviceDevice.get( serviceDeviceName )
      if not serviceDevice:
         return {}

      return Lib.getDeviceConfig( deviceSet, serviceDevice )

   def getVirtualInstanceConfig( self, deviceSet, serviceDeviceName, vinstName ):
      serviceDevice = deviceSet.serviceDevice.get( serviceDeviceName )
      if not serviceDevice:
         return None
      return serviceDevice.virtualInstance.get( vinstName )

   def restartMonitoringDeviceAsNeeded( self, serviceDevice, deviceSet ):
      ''' start/restart device policy monitoring thread if appropriate
      '''
      with ActivityLock():
         isStartable = ( deviceSetComplete( deviceSet ) and
                         not serviceDevice.isAccessedViaAggrMgr and
                         isActiveState( deviceSet.state ) and
                         deviceConfigComplete( serviceDevice, deviceSet ) and
                         self.isDeviceAvailable( serviceDevice, deviceSet ) )
         self.hasAggrMgrPair[ deviceSet.name ] = ( isStartable and
             Lib.checkAggrMgrPair( deviceSet ) )

      if isStartable:
         b1( 'startMonitoringSvcDevice:', v( serviceDevice.ipAddrOrDnsName ),
             v( threadName() ) )
         genMutexAsNeededFor( serviceDevice.ipAddrOrDnsName )
         instanceName = getDeviceMonitorInstanceName(
            deviceSet.name, serviceDevice.ipAddrOrDnsName )
         self.stopMonitoringInstance( deviceSet.name, instanceName ) # stop any old

         deviceConfig = Lib.getDeviceConfig( deviceSet, serviceDevice )
         b3( 'creating monitoring instance for', v( serviceDevice.ipAddrOrDnsName ) )
         monitorInstance = PluginController.genDeviceMonitor(
            deviceSet.policySourceType, deviceConfig, self )
         self.monitorInstances[ deviceSet.name ][ instanceName ] = monitorInstance
         self.updateDeviceStatus( serviceDevice, deviceSet,
                                  monitorInstance.instanceThreadName )
         PluginController.startMonitoringPolicies( monitorInstance )
         traceInstances( self.monitorInstances, msg='startMonitoring')

   def stopMonitoringDeviceSet( self, deviceSet ):
      for device in getServiceDevicesUsingLock( deviceSet ):
         if self.isDeviceAvailableUsingLock( device, deviceSet ):
            self.stopMonitoringServiceDevice( device.ipAddrOrDnsName, deviceSet )

   def stopMonitoringServiceDevice( self, deviceId, deviceSet, wait=False ):
      ''' Stop monitoring thread for serviceDevice if it exists.
      '''
      instanceName = getDeviceMonitorInstanceName( deviceSet.name, deviceId )
      self.stopMonitoringInstance( deviceSet.name, instanceName, wait=wait )
      serviceDevice = deviceSet.serviceDevice.get( deviceId )
      if serviceDevice:
         self.updateDeviceStatus( serviceDevice, deviceSet )

   def stopMonitoringInstance( self, deviceSetName, instanceName, wait=False ):
      instances = self.monitorInstances.get( deviceSetName )
      if instances:
         instance = instances.get( instanceName )
         if instance:
            PluginController.stopMonitoringPolicies( instance,
                                                     waitForCompletion=wait )
            b2( v( instanceName ), v( instance.instanceThreadName ) )
            del self.monitorInstances[ deviceSetName ][ instanceName ]

   def cleanupForDeletedDeviceSetName( self, deviceSetName ):
      ''' deviceSet already deleted from sysdb (e.g. by config cli)
          stop any monitoring instances, delete status and policies.
      '''
      instances = self.monitorInstances.get( deviceSetName )
      if instances:
         for instanceName in list( instances ): # TODO: Improve; dup code.
            self.stopMonitoringInstance( deviceSetName, instanceName )
         del self.monitorInstances[ deviceSetName ]

      for deviceId, devStatus in getSvcDeviceStatusItemsUsingLock( self.mpmStatus ):
         if devStatus.deviceSetName == deviceSetName:
            self.deletePoliciesForDevice( deviceId )
            devStatus.monitorThreadName = ''
      self.deleteDeviceStatusForDeviceSet( deviceSetName )

      # Cleanup the entry if present
      self.hasAggrMgrPair.pop( deviceSetName, None )

   def cleanupForDeletedVInst( self, serviceDevice, vinst, deviceSet ):
      '''
      Cleanup MssL3 input and MssPM status after removal of a vinst from a service
      device. Monitoring is stopped and must be restarted.
      '''
      deviceId = serviceDevice.ipAddrOrDnsName
      b2( 'cleanup deleted vinst:', v( deviceId ), v( vinst ) )
      mssDeviceId = self.buildMssDeviceName( deviceId, vinst,
                                             deviceSet.policySourceType )
      deviceOrPairId = self.haDeviceIdMap.get( deviceId, deviceId )
      mssDeviceOrPairId = self.buildMssDeviceName( deviceOrPairId, vinst,
                                                   deviceSet.policySourceType )

      self.stopMonitoringServiceDevice( deviceId, deviceSet, wait=True )
      with ActivityLock():
         # policies must be cleanup before the routing table, so MSS agent
         # won't try to resolve policies against non-existing next hop.
         if mssDeviceOrPairId.name() in self.policySrcCfg.policySet:
            b4( 'remove ', v( mssDeviceOrPairId ),
                ' from policySrcCfg.policySet' )
            del self.policySrcCfg.policySet[ mssDeviceOrPairId.name() ]
         if mssDeviceOrPairId.name() in self.svcDevSrcCfg.serviceDevice:
            b4( 'remove ', v( mssDeviceOrPairId ),
                ' from svcDevSrcCfg.serviceDevice' )
            del self.svcDevSrcCfg.serviceDevice[ mssDeviceOrPairId.name() ]

      if not deviceStateMutexExistsFor( deviceId ):
         return
      with deviceStateMutex( deviceId ):
         with ActivityLock():
            if mssDeviceId.name() in self.mpmStatus.devicePolicyList:
               b4( 'remove ', v( mssDeviceId ), ' from mpmStatus.devicePolicyList' )
               del self.mpmStatus.devicePolicyList[ mssDeviceId.name() ]

   # deviceStatus mgmt
   #--------------------------------------------------------------------------------
   def updateDeviceStatus( self, serviceDevice, deviceSet, monitorThreadName='' ):
      ''' Update device status.
          Create a status entity if doesn't already exist.
          Caller can't have ActivityLock since grabbing deviceMutex here.
      '''
      deviceId = serviceDevice.ipAddrOrDnsName
      if not deviceStateMutexExistsFor( deviceId ):
         return
      with deviceStateMutex( deviceId ):
         b3( 'updateDeviceStatus, mutex', v( deviceId ), v( threadName() ) )
         devStatus = getSvcDeviceStatusUsingLock( deviceId, self. mpmStatus,
                                                  createIfNeeded=True )
         if devStatus.deviceSetName and devStatus.deviceSetName != deviceSet.name:
            b1( v( deviceId ), 'devStatus owner deviceSet change from',
                v( devStatus.deviceSetName ), 'to', v( deviceSet.name ) )

         devStatus.deviceSetName = deviceSet.name
         devStatus.state = deviceSet.state
         devStatus.policySourceType = deviceSet.policySourceType
         if monitorThreadName:
            devStatus.monitorThreadName = monitorThreadName
         if not serviceDevice.isAccessedViaAggrMgr:
            devStatus.isAggregationMgr = deviceSet.isAggregationMgr
         # propagate state to children
         for childDeviceId in devStatus.groupMember:
            childDevStatus = getSvcDeviceStatusUsingLock( childDeviceId,
                                                          self.mpmStatus )
            if childDevStatus:
               b2( 'updating serviceDeviceStatus for child:', v( childDeviceId ) )
               if childDevStatus.deviceSetName != deviceSet.name:
                  b1( v( deviceId ), 'childDevStatus owner change from',
                      v( childDevStatus.deviceSetName ), 'to', v( deviceSet.name ) )
                  childDevStatus.deviceSetName = deviceSet.name
               childDevStatus.state = devStatus.state
            else:
               b2( 'Warn: no deviceStatus for child:', v( childDeviceId ) )

   def updateStatus( self, deviceId, monitorThreadName, mgmtIp=None, error=False ):
      ''' Called from PluginController to update device timeLastSeen timestamp etc.
      '''
      b4( 'updateStatus for:', v( deviceId ), v( threadName() ) )
      if not deviceStateMutexExistsFor( deviceId ):
         return
      with deviceStateMutex( deviceId ):
         deviceStatus = getSvcDeviceStatusUsingLock( deviceId, self.mpmStatus )
         if ( not deviceStatus or
              not validateMonitoringThread( monitorThreadName, deviceStatus ) or
              not doUpdatesInDeviceState( deviceStatus.state ) ):
            return

         if not error:
            deviceStatus.timeLastSeen = Lib.getTimestamp()
         if mgmtIp:
            deviceStatus.mgmtIp = getArnetIpAddr( mgmtIp )  # used for aggMgrs
         if not deviceStatus.isAggregationMgr:
            deviceStatus.policyReadAttempted = True
            if not self.mssSvcPolicySrcState.convergenceComplete:
               self.handleInitialReadNotify()

   def updateHaStatus( self, deviceId, haState ):
      ''' Called from PluginController to update HA Status
      '''
      b4( v( deviceId ) )
      if not deviceStateMutexExistsFor( deviceId ):
         return
      with deviceStateMutex( deviceId ):
         deviceStatus = getSvcDeviceStatusUsingLock( deviceId, self.mpmStatus )
         if ( not deviceStatus or
              not doUpdatesInDeviceState( deviceStatus.state ) ):
            return
         deviceStatus.isHaEnabled = haState.isHaEnabled()
         deviceStatus.haState = haState.getHaState()
         deviceStatus.haPeerMgmtIp = getArnetIpAddr( haState.getPeerManagementIp() )

   def cleanupChildrenDeviceStatus( self, deviceStatus ):
      if deviceStatus:
         for childDevId in deviceStatus.groupMember:
            if childDevId in self.mpmStatus.serviceDeviceStatus:
               b2( 'deleting serviceDeviceStatus for:', v( childDevId ) )
               del self.mpmStatus.serviceDeviceStatus[ childDevId ]

   def deleteDeviceStatus( self, deviceId, childrenCleanup=True ):
      # caller must NOT be holding ActivityLock
      if not deviceStateMutexExistsFor( deviceId ):
         return
      with deviceStateMutex( deviceId ):
         b3( 'deleteDeviceStatus, mutex', v( deviceId ), v( threadName() ) )
         with ActivityLock():
            devStatus = self.mpmStatus.serviceDeviceStatus.get( deviceId )
            # first delete status for any children
            if childrenCleanup:
               self.cleanupChildrenDeviceStatus( devStatus )
            b3( 'deleting deviceStatus for:', v( devStatus.serviceDeviceId ) )
            del self.mpmStatus.serviceDeviceStatus[ devStatus.serviceDeviceId ]

   def deleteDeviceStatusForDeviceSet( self, deviceSetName ):
      for deviceId, devStatus in getSvcDeviceStatusItemsUsingLock( self.mpmStatus ):
         if devStatus.deviceSetName == deviceSetName:
            self.deleteDeviceStatus( deviceId )

   def getAggMgrChildDeviceIds( self, aggMgrDeviceId ):
      deviceStatus = getSvcDeviceStatusUsingLock( aggMgrDeviceId, self.mpmStatus )
      if deviceStatus:
         return list( deviceStatus.groupMember )
      return []

   def addGroupMember( self, memberDeviceId, memberDeviceThreadName, aggrMgrDeviceId,
                       aggMgrThreadName ):
      ''' Called from PluginController
      '''
      b2( 'addGroupMember:', v( memberDeviceId ), 'thread:',
          v( memberDeviceThreadName ), 'by:', v( aggMgrThreadName ) )
      if not deviceStateMutexExistsFor( aggrMgrDeviceId ):
         return
      with deviceStateMutex( aggrMgrDeviceId ):
         with ActivityLock():
            aggmgrDevStat = self.mpmStatus.serviceDeviceStatus.get( aggrMgrDeviceId )
         if ( not aggmgrDevStat or
              not doUpdatesInDeviceState( aggmgrDevStat.state ) or
              not validateMonitoringThread( aggMgrThreadName, aggmgrDevStat ) ):
            return
         genMutexAsNeededFor( memberDeviceId )
         with deviceStateMutex( memberDeviceId ):
            aggmgrDevStat.groupMember[ memberDeviceId ] = True  # add
            devStatus = getSvcDeviceStatusUsingLock( memberDeviceId, self.mpmStatus,
                                                     createIfNeeded=True )
            devStatus.accessedViaAggrMgr = aggrMgrDeviceId
            devStatus.policySourceType = aggmgrDevStat.policySourceType
            devStatus.deviceSetName = aggmgrDevStat.deviceSetName
            devStatus.monitorThreadName = memberDeviceThreadName
            devStatus.state = aggmgrDevStat.state

   def deleteGroupMember( self, memberDeviceId,  aggrMgrDeviceId ):
      ''' Called from MssPolicyMonitor.PluginController
      '''
      if not deviceStateMutexExistsFor( aggrMgrDeviceId ):
         return
      with deviceStateMutex( aggrMgrDeviceId ):
         #b3( 'delDevGrpMem, mutex', v( aggrMgrDeviceId ), v( threadName() ) )
         with ActivityLock():
            aggmgrDevStat = self.mpmStatus.serviceDeviceStatus.get( aggrMgrDeviceId )
         if ( aggmgrDevStat and doUpdatesInDeviceState( aggmgrDevStat.state ) and
              deviceStateMutexExistsFor( memberDeviceId ) ):
            with deviceStateMutex( memberDeviceId ):
               b2( 'delGroupMember, mutex', v( memberDeviceId ), 
                   v( threadName() ) )
               if memberDeviceId in aggmgrDevStat.groupMember:
                  del aggmgrDevStat.groupMember[ memberDeviceId ]
               with ActivityLock():
                  if memberDeviceId in self.mpmStatus.serviceDeviceStatus:
                     b1( 'deleting svcDeviceStatus for device', v( memberDeviceId ) )
                     del self.mpmStatus.serviceDeviceStatus[ memberDeviceId ]

   # link and interface mgmt
   #--------------------------------------------------------------------------------
   def getLogicalIntfs( self, switchChassisId, intf ):
      ''' Returns port-channel, mlag and mlagSystemId (if any) for the
          passed switch interface.  Uses CVX Network Topology Service
          data in Controllerdb to determine these relationships.
      ''' 
      portChannel, mlag, mlagSystemId = '', '', ''
      try:
         with ActivityLock():
            host = self.cdbTopology.host.get( switchChassisId )
            if host:
               if intf.startswith( PORT_CHANNEL ):
                  portChannel = intf
                  # check if this portChannel is a member of an existing MLAG
                  if ( portChannel in host.portGroup and
                       host.portGroup[ portChannel ].mlag ):
                     mlag = host.portGroup[ portChannel ].mlag.name
               else:  # ethernet intf lookup
                  port = host.port.get( intf )
                  if port and port.portGroup:
                     portChannel = port.portGroup.name
                     if port.portGroup.mlag:
                        mlag = port.portGroup.mlag.name

               if mlag and host.mlagSystemId:
                  mlagSystemId = host.mlagSystemId

         b2( 'topology service lookup:', v( switchChassisId ), v( intf ), 'lag:',
             v( portChannel ), 'mlag:', v( mlag ), 'mlagSysId:', v( mlagSystemId ) )
      except Exception as ex:  # pylint: disable-msg=W0703
         b0( 'Controllerdb Topology lookup failed: %s' % ex )
         traceback.print_exc()
      return ( portChannel, mlag, mlagSystemId )

   def genServiceDeviceLink( self, serviceDeviceIntf, intfStr, sdIntf, neighbor ):
      if neighbor:
         switchIntf = neighbor[ 'switchIntf' ]
         switchId = neighbor[ 'switchChassisId' ]
         if switchIntf.startswith( MLAG ):  # handle cli cfg'd "map device-interface"
            portChannel, mlag, mlagSysId = '', switchIntf, switchId
         else:
            portChannel, mlag, mlagSysId = self.getLogicalIntfs( switchId,
                                                                 switchIntf )
         linkName = '{}--{}:{}:SW={}:VL={}'.format(
            serviceDeviceIntf, switchIntf.replace( 'Ethernet', 'Eth' ),
            portChannel.replace( PORT_CHANNEL, 'Po' ),
            compressMacAddress( switchId, keepAristaOUI=False ),
            ','.join( sdIntf.vlans ) )
      else:  # L3 policy link
         linkName = f'{serviceDeviceIntf}={sdIntf.ipAddr}'

      link = ServiceDeviceLinkType( linkName )
      link.serviceDeviceIntf = serviceDeviceIntf
      link.serviceDeviceIntfStr = intfStr
      link.serviceDeviceLagName = sdIntf.name if sdIntf.isLag else ''
      if neighbor:
         link.switchId = switchId
         link.switchIntf = switchIntf
         link.portChannel = portChannel
         link.mlag = mlag
         link.mlagSystemId = mlagSysId
         for vlan in sdIntf.vlans:
            link.allowedVlan[ str( vlan ) ] = True
      else:
         link.ipAddr = IpGenAddrWithMask( sdIntf.ipAddr ).ipGenAddr
         link.vrf = Tac.Value( 'L3::VrfName', sdIntf.vrf )
      return link

   def updateServiceDeviceLinks( self, newLinks, oldLinks, zoneName,
                                 isInterceptZone, devicePolicy ):
      newSvcIntfUuid = self.getServiceIntfSetUuid( newLinks, zoneName, devicePolicy )
      # update ServiceIntf in each ServicePolicy associated with this DevicePolicy
      for servicePolicy in self.getChildServicePoliciesFor( devicePolicy ):
         if servicePolicy:
            if isInterceptZone:
               b1( 'updating SvcPolicy:', v( servicePolicy.name ), 
                   'NEAR SvcIntf to:', v( newSvcIntfUuid ) )
               servicePolicy.nearIntfUuid = newSvcIntfUuid
            else:
               b1( 'updating SvcPolicy:', v( servicePolicy.name ), 
                   'FAR SvcIntf to:', v( newSvcIntfUuid ) )
               servicePolicy.farIntfUuid = newSvcIntfUuid

      self.cleanupOldServiceIntfsandPolicies( oldLinks, zoneName, devicePolicy )

   def cleanupOldServiceIntfsandPolicies( self, oldLinks, zoneName, devicePolicy ):
      # cleanup any old intfSet UUIDs
      oldSvcIntfUuid = self.getServiceIntfSetUuid(
         oldLinks, zoneName, devicePolicy, existingOnly=True )
      if oldSvcIntfUuid:
         self.deleteServiceIntfAsNeeded( oldSvcIntfUuid )

      # cleanup ServicePolicy without near and/or far links
      for servicePolicy in self.getChildServicePoliciesFor( devicePolicy ):
         if ( servicePolicy and (
              not servicePolicy.nearIntfUuid or not servicePolicy.farIntfUuid ) ):
            del devicePolicy.childServicePolicy[ servicePolicy.name ]
            self.deleteServicePolicy( servicePolicy )

   def updateDevPolicyLinks( self, rawPolicy, devicePolicy ):
      ''' Populate/update MssPolicyMonitor::DevicePolicy service device to
          switch links.
      '''
      for zoneIntfs, zoneType, zoneId in [
         ( rawPolicy.srcZoneInterfaces, rawPolicy.srcZoneType, ZONE_A ),
         ( rawPolicy.dstZoneInterfaces, rawPolicy.dstZoneType, ZONE_B ) ]:
         sdLinks = {}
         for sdIntf in zoneIntfs:
            for physIntf in sdIntf.physicalIntfs:
               if Lib.isL2RawPolicy( rawPolicy ):
                  if physIntf.name not in rawPolicy.intfNeighbors:
                     b2( 'no neighbor switch intf found for:', v( physIntf.name ) )
                     continue  # could be a down physical intf in LAG
                  intfNbor = rawPolicy.intfNeighbors[ physIntf.name ]
                  if ( 'switchIntf' in intfNbor and
                        ( intfNbor[ 'switchIntf' ].startswith( MLAG ) or
                          intfNbor[ 'switchIntf' ].startswith( PORT_CHANNEL ) ) ):
                     b2( v( physIntf.name ), 'has (M)LAG neighbor from map '
                        'device-interface config, ignoring physical intf state' )
                  elif physIntf.state != Lib.LINK_STATE_UP:
                     b3( 'updateDevPolicyLinks, switch intf not up:', v( physIntf ) )
                     continue
                  neighbor = rawPolicy.intfNeighbors[ physIntf.name ]
               else:
                  neighbor = None

               intfStr = ( physIntf.name if physIntf.name == sdIntf.name
                           else f'{sdIntf.name}({physIntf.name})' )
               sdlink = self.genServiceDeviceLink( physIntf.name, intfStr, sdIntf,
                                                   neighbor )
               sdLinks[ sdlink.linkName ] = sdlink

            b4( 'policy', v( rawPolicy.name ), ': interface', v( sdIntf.name ),
                'is', v( sdIntf.state ) )
            if devicePolicy.serviceDeviceType == Lib.FORTIMGR_PLUGIN:
               # Fortinet doesn't support zone
               zoneName = 'N/A'
            else:
               zoneName = ( rawPolicy.srcZoneName if zoneId == ZONE_A
                            else rawPolicy.dstZoneName )
            self.fwZoneLogger.log( sdIntf.name, devicePolicy.serviceDeviceId,
                                   zoneName, sdIntf.state == Lib.LINK_STATE_UP )

         if zoneId == ZONE_A:
            if not devicePolicy.zoneA:
               devicePolicy.zoneA = ( ( ZONE_A, ) )  # instantiating attribute
            assignZoneType( devicePolicy.zoneA, zoneType )
            updateSDLinkMembership( devicePolicy.zoneA.link, sdLinks )
            updateSwitchLinkMembership( devicePolicy.zoneA )
            devicePolicy.zoneA.zoneName = rawPolicy.srcZoneName
         elif zoneId == ZONE_B:
            if not devicePolicy.zoneB:
               devicePolicy.zoneB = ( ( ZONE_B, ) )
            assignZoneType( devicePolicy.zoneB, zoneType )
            updateSDLinkMembership( devicePolicy.zoneB.link, sdLinks )
            updateSwitchLinkMembership( devicePolicy.zoneB )
            devicePolicy.zoneB.zoneName = rawPolicy.dstZoneName
         # b2( v( Lib.dumpServiceDeviceLink( sdLinks, 'ServiceDeviceLinks' ) ) )

   def updateZoneName( self, oldZoneName, newZoneName, serviceDeviceLinks ):
      ''' update zone name in existing ServiceIntfInfo object
          (i.e. don't create a new uuid)
      '''
      # use first link's switch & intf to find serviceIntf; ae will share svcIntfInfo
      link = next( iter( serviceDeviceLinks.values() ) )
      switchIntfPair = getSwitchIntfPair( link )
      with ActivityLock():
         for svcIntfInfo in self.mssSvcIntfCfg.intfSet.values():
            svcIntf = svcIntfInfo.intf.get( switchIntfPair )
            if ( svcIntf and svcIntfInfo.zone == oldZoneName and
                 equivalentAllowedVlans( link.allowedVlan, svcIntf.allowedVlan ) ):
               b2( 'update SvcIntf zone name:', v( svcIntfInfo.zone ), 'to:',
                   v( newZoneName ) )
               svcIntfInfo.zone = newZoneName
               return
      b2( 'no serviceIntf found for:', v( switchIntfPair ) )

   def getServiceIntfSetUuid( self, serviceDeviceLinks, zoneName, devicePolicy,
                              existingOnly=False ):
      ''' Returns a UUID for the passed set of serviceDeviceLinks.  If a
          UUID does not already exist a new UUID will be generated.
      ''' 
      if not serviceDeviceLinks:
         return ''
      with ActivityLock():
         for svcIntfInfo in self.mssSvcIntfCfg.intfSet.values():
            # pylint: disable-next=consider-using-in
            if ( ( svcIntfInfo.serviceDeviceId != devicePolicy.serviceDeviceId and
                   svcIntfInfo.serviceDeviceId != devicePolicy.haPeerDeviceId ) or
                 svcIntfInfo.configSource != getConfigSource( devicePolicy ) or
                 ( zoneName and zoneName != svcIntfInfo.zone ) ):
               continue
            serviceIntfs = list( svcIntfInfo.intf.values() )
            if equivalentSwitchIntfs( serviceDeviceLinks, serviceIntfs ):
               b2( v( svcIntfInfo.serviceDeviceId ), 'use existing',
                   v( svcIntfInfo.intf ), v( list( serviceDeviceLinks ) ) )
               if ( devicePolicy.haPeerDeviceId and
                    devicePolicy.haPeerDeviceId == svcIntfInfo.serviceDeviceId ):
                  b1( 'new HA primary', v( devicePolicy.serviceDeviceId ),
                      'taking over', v( svcIntfInfo.uuid ) )
                  svcIntfInfo.serviceDeviceId = devicePolicy.serviceDeviceId
               return svcIntfInfo.uuid  # found an exact match
         if existingOnly:
            return ''  # didn't find an existing service intf

         # create a new MSS::ServiceInterface
         svcIntfUuid = genServiceInterfaceUuid()
         svcIntfInfo = self.mssSvcIntfCfg.newIntfSet( svcIntfUuid )
         svcIntfInfo.zone = zoneName
         svcIntfInfo.configSource = getConfigSource( devicePolicy )
         svcIntfInfo.serviceDeviceId = devicePolicy.serviceDeviceId
         for link in serviceDeviceLinks.values():
            switchIntfPair = getSwitchIntfPair( link )
            if switchIntfPair not in svcIntfInfo.intf:
               b2( v( devicePolicy.policyName ), 'gen/upd svcIntf', v( svcIntfUuid ),
                   'for', v( switchIntfPair.switchId ), v( switchIntfPair.intf ) )
               serviceIntf = ServiceInterfaceType( switchIntfPair )
               for vlan in link.allowedVlan:
                  serviceIntf.allowedVlan[ vlan ] = True
               svcIntfInfo.intf.addMember( serviceIntf )
         return svcIntfInfo.uuid

   def deleteServiceIntfAsNeeded( self, intfUuid, force=False ):
      ''' Delete service interface if not in use by any service policies unless
          force is true in which case the service interface will be deleted
          regardless.
      '''
      if not intfUuid:
         return
      b4( 'deleteServiceIntfAsNeeded uuid:', v( intfUuid ), 'force:', v( force ) )
      with ActivityLock():
         if not force:
            # check if any other policies are using this intfSet
            # pylint: disable-next=consider-using-in
            if any( sPol.nearIntfUuid == intfUuid or sPol.farIntfUuid == intfUuid
                    for sPol in self.mssSvcPolicyCfg.servicePolicy.values() ):
               b2( 'intf uuid', v( intfUuid ), 'still in use, not deleting' )
               return
         if intfUuid in self.mssSvcIntfCfg.intfSet:
            b2( 'deleting SvcIntf set:', v( intfUuid ) )
            del self.mssSvcIntfCfg.intfSet[ intfUuid ]

   # intercept mgmt
   #--------------------------------------------------------------------------------
   def classifyDevPolicyZones( self, rawPolicy, devPolicy ):
      ''' self.zoneClassMap = {
             key = deviceId,
             val = dict{ key = zoneName, val = zoneInterceptClass } }
          rawPolicy has been validated and has intercepts in at least one zone.
          Classifications are cleared at the beginning of each policy read cycle.
      '''
      deviceId = devPolicy.serviceDeviceId
      policyName = devPolicy.policyName
      zoneAName = devPolicy.zoneA.zoneName
      zoneBName = devPolicy.zoneB.zoneName
      if deviceId not in self.zoneClassMap:
         self.zoneClassMap[ deviceId ] = {}  # initialize if necessary

      # get any current zone classifications (i.e. from this policy read cycle)
      zoneAClass = self.zoneClassMap[ deviceId ].get( zoneAName, UNCLASSIFIED_ZONE )
      zoneBClass = self.zoneClassMap[ deviceId ].get( zoneBName, UNCLASSIFIED_ZONE )
      # pylint: disable-next=consider-using-in
      if zoneAClass != UNCLASSIFIED_ZONE and zoneBClass != UNCLASSIFIED_ZONE:
         # detect policy mis-config w/ both intercept or nonIntercept zones
         if zoneAClass == zoneBClass and zoneAClass == INTERCEPT_ZONE:
            b0( 'Notice: both zones in %s are INTERCEPT zones' % v( policyName ) )
            
            return False
         elif zoneAClass == zoneBClass and zoneAClass == NON_INTERCEPT_ZONE:
            b0( 'Notice: both zones in %s are NON-INTERCEPT' % v( policyName ) )
            return False
         else:
            devPolicy.zoneA.zoneClass = zoneAClass
            devPolicy.zoneB.zoneClass = zoneBClass
            b2( v( policyName ), 'using existing zone classifications:', 
                v( zoneAName ), v( devPolicy.zoneA.zoneClass ), v( zoneBName ),
                v( devPolicy.zoneB.zoneClass ) )
            return True

      elif zoneAClass == UNCLASSIFIED_ZONE and zoneBClass == UNCLASSIFIED_ZONE:
         # when both zones unclassified a dest zone with intercept(s) has precedence
         if rawPolicy.dstIpAddrList:
            devPolicy.zoneB.zoneClass = INTERCEPT_ZONE
            devPolicy.zoneA.zoneClass = NON_INTERCEPT_ZONE
         elif rawPolicy.srcIpAddrList:
            devPolicy.zoneA.zoneClass = INTERCEPT_ZONE
            devPolicy.zoneB.zoneClass = NON_INTERCEPT_ZONE

      elif zoneAClass != UNCLASSIFIED_ZONE:
         b2( v( devPolicy.zoneA.zoneName ),
             'prior classification forces dest zone class' )
         devPolicy.zoneA.zoneClass = zoneAClass
         devPolicy.zoneB.zoneClass = invertZoneClass( zoneAClass )

      elif zoneBClass != UNCLASSIFIED_ZONE:
         b2( v( devPolicy.zoneB.zoneName ), 
             'prior classification forces src zone class' )
         devPolicy.zoneB.zoneClass = zoneBClass
         devPolicy.zoneA.zoneClass = invertZoneClass( zoneBClass )

      b2( v( policyName ), 'zones classified as:', v( zoneAName ), 
          v( devPolicy.zoneA.zoneClass ), v( zoneBName ), 
          v( devPolicy.zoneB.zoneClass ) )
      self.zoneClassMap[ deviceId ][ zoneAName ] = devPolicy.zoneA.zoneClass
      self.zoneClassMap[ deviceId ][ zoneBName ] = devPolicy.zoneB.zoneClass
      return True

   def updateDevPolicyIntercepts( self, rawPolicy, devicePolicy ):
      ''' Populate/update/delete MssPolicyMonitor::DevicePolicy intercepts
          from latest policy retrieved from service device
      '''
      if Lib.isMssL3EnabledAndL3Policy( devicePolicy ):
         self.populateRawIntercepts( devicePolicy.zoneA, rawPolicy.srcIpAddrList )
         self.populateRawIntercepts( devicePolicy.zoneB, rawPolicy.dstIpAddrList )

      elif devicePolicy.zoneA.zoneClass == INTERCEPT_ZONE:  # MssL2
         devicePolicy.zoneB.intercept.clear()  # ensure no icepts in nonIcept zone
         self.populateRawIntercepts( devicePolicy.zoneA, rawPolicy.srcIpAddrList )
         self.populateIntercepts( devicePolicy, rawPolicy, SOURCE_ZONE )

      elif devicePolicy.zoneB.zoneClass == INTERCEPT_ZONE:  # MssL2
         devicePolicy.zoneA.intercept.clear()
         self.populateRawIntercepts( devicePolicy.zoneB, rawPolicy.dstIpAddrList )
         self.populateIntercepts( devicePolicy, rawPolicy, DEST_ZONE )

   def populateRawIntercepts( self, devicePolicyZone, ipFieldsList ):
      devicePolicyZone.rawIntercept.clear()
      for ipField in ipFieldsList:
         devicePolicyZone.rawIntercept[ ipField ] = True

   def populateIntercepts( self, devicePolicy, rawPolicy, zoneSelector ):
      if zoneSelector == SOURCE_ZONE:
         zoneIntercepts = devicePolicy.zoneA.intercept
         ipFieldsList = rawPolicy.srcIpAddrList
      elif zoneSelector == DEST_ZONE:
         zoneIntercepts = devicePolicy.zoneB.intercept
         ipFieldsList = rawPolicy.dstIpAddrList
      else:
         return
      newIntercepts = set()  # use set to dedup overlapping IP address ranges
      for ipField in ipFieldsList:
         newIntercepts.update( Lib.expandIpRange( ipField ) )
      b2( v( devicePolicy.policyName ), zoneSelector, 'has',
          v( len( newIntercepts ) ), 'IP intercepts' )
      # list of strings much faster than list of Tac IPs for operations done here
      prevIntercepts = [ i.stringValue for i in zoneIntercepts ]
      for ip in newIntercepts:
         tacIp = Tac.Value( 'Arnet::IpGenAddrWithMask', ip )
         if tacIp not in zoneIntercepts:
            zoneIntercepts.addMember( InterceptSpecType( tacIp ) )
         else:
            prevIntercepts.remove( tacIp.stringValue )

      for ip in prevIntercepts:
         del zoneIntercepts[ Tac.Value( 'Arnet::IpGenAddrWithMask', ip ) ]

   def populateSvcPolicyIntercepts( self, ipIntercepts, servicePolicy ):
      ''' populate Mss::ServicePolicy host traffic intercepts
      '''
      for tacIp in ipIntercepts:
         self.addServicePolicyIntercept( tacIp, servicePolicy )

   def addServicePolicyIntercept( self, tacIp, servicePolicy ):
      if tacIp not in servicePolicy.interceptFlow:
         if len( servicePolicy.interceptFlow ) <= self.maxInterceptsPerSvcPolicy:
            b2( v( servicePolicy.name ), 'adding intercept:', v( tacIp ) )
            servicePolicy.interceptFlow.addMember( InterceptFlowType( tacIp ) )

   def deleteServicePolicyIntercept( self, intercept, interceptZone, devicePolicy,
                                     numIntercepts ):
      servicePolicy = self.getMssServicePolicy( devicePolicy, interceptZone,
                                                existingOnly=True )
      if not servicePolicy:
         return
      if numIntercepts == 0:
         b2( 'no intercepts remain in DevicePolicy', v( devicePolicy.policyName ), 
             v( interceptZone ), 'removing name from ServicePolicy' )
         if devicePolicy.policyName in servicePolicy.policyName:
            with ActivityLock():
               del servicePolicy.policyName[ devicePolicy.policyName ]

      if ( intercept in servicePolicy.interceptFlow and
           intercept not in self.interceptsFromOtherParents( servicePolicy,
                                                             devicePolicy ) ):
         b2( v( servicePolicy.name ), 'remove intercept:', v( intercept ) )
         with ActivityLock():
            del servicePolicy.interceptFlow[ intercept ]
         if not servicePolicy.interceptFlow:
            b1( 'deleting SvcPolicy', v( servicePolicy.name ), 'has no intercepts' )
            self.deleteServicePolicy( servicePolicy )

   def getDevicePolicyIntercepts( self, devicePolicy ):
      ''' Returns DevicePolicy IP intercepts for Intercept Zone.
      '''
      if devicePolicy.zoneA.zoneClass == INTERCEPT_ZONE:
         return list( devicePolicy.zoneA.intercept )
      elif devicePolicy.zoneB.zoneClass == INTERCEPT_ZONE:
         return list( devicePolicy.zoneB.intercept )
      else:
         b0( 'Warning: No intercept zone in policy:', v( devicePolicy.policyName ) )
         return []

   # policy management
   #--------------------------------------------------------------------------------
   def updateDeviceVrf( self, policyList, virtualInstanceConfig, fwType ):
      defaultFirewallVrf = Lib.getDefaultVrfName( fwType )
      if not virtualInstanceConfig or not virtualInstanceConfig.firewallVrf:
         # Vrf has been removed from config, reset to default state
         for vrfName in policyList.firewallVrf:
            if vrfName != defaultFirewallVrf:
               del policyList.firewallVrf[ vrfName ]
         policyList.firewallVrf[ defaultFirewallVrf ] = 'default'
         return

      # cleanup old entries
      for vrfName in policyList.firewallVrf:
         if vrfName not in virtualInstanceConfig.firewallVrf:
            del policyList.firewallVrf[ vrfName ]

      # Add new entires
      for vrfName, netVrf in virtualInstanceConfig.firewallVrf.items():
         policyList.firewallVrf[ vrfName ] = netVrf

   def updateTrafficInspection( self, policyList, deviceSet, virtualInstanceConfig ):
      if virtualInstanceConfig and virtualInstanceConfig.trafficInspection:
         policyList.trafficInspection = virtualInstanceConfig.trafficInspection
      else:
         policyList.trafficInspection = deviceSet.trafficInspection

   def updateDevicePolicy( self, rawPolicy, vinst, lldpNeighbors,
                           device, deviceSet ):
      ''' Creates or updates DevicePolicy objects in sysdb.
          Caller should already have deviceStateMutex for deviceId.
      '''
      deviceId = device.deviceId
      mssDeviceId = self.buildMssDeviceName( device.deviceId, vinst,
                                             deviceSet.policySourceType )
      isL2Policy = Lib.isL2RawPolicy( rawPolicy )
      if isL2Policy:
         populateIntfNeighbors( rawPolicy, lldpNeighbors, deviceId, deviceSet )
      rawPolicy.isOffloadPolicy = bool( set( rawPolicy.tags ) &
                                        set( deviceSet.offloadTag.keys() ) )
      rawPolicy.isVerbatim = VERBATIM in deviceSet.modifierTag and \
                                bool( set( rawPolicy.tags ) &
                                      set( deviceSet.modifierTag[ VERBATIM ]
                                             .tag.keys() ) )
      rawPolicy.isForwardOnly = FORWARD_ONLY in deviceSet.modifierTag and \
                                bool( set( rawPolicy.tags ) &
                                      set( deviceSet.modifierTag[ FORWARD_ONLY ]
                                             .tag.keys() ) )
      b1( v( rawPolicy ) )
      # for multiVwire mode, isValidRawPolicy can modify rawPolicy based
      # on neighbor UP/DOWN state
      validRawPolicy = isValidRawPolicy( rawPolicy, deviceId, vinst,
                                         self.skippedPolicyLogger )
      with ActivityLock():
         policyList = ( self.mpmStatus.devicePolicyList.get( mssDeviceId.name() ) or
                        self.mpmStatus.newDevicePolicyList( mssDeviceId.name() ) )
         # TODO vinstConfig may be None is some situation:
         # a) An aggregation manager is used and the device member isn't in the CLI;
         # b) Device's ID (deviceId), when controlled by an aggregation manager,
         #    may not match the identifier used in the CLI.
         # While it is okay to not fix a) since we need the VRF information to be
         # provided, b) is more problematic. We would require the user to make sure
         # that the CLI uses the following:
         # - serial number for members under Panorama
         # - fortigate device's name for FortiManager
         # - ip address for CheckpointServerManager
         #
         # This limitation shoud be addressed.
         vinstConfig = self.getVirtualInstanceConfig( deviceSet, deviceId, vinst )
         self.updateDeviceVrf( policyList, vinstConfig, deviceSet.policySourceType )
         self.updateTrafficInspection( policyList, deviceSet, vinstConfig )
         if validRawPolicy and not Lib.isRawPolicyVrfValid( rawPolicy, policyList ):
            validRawPolicy = False
            self.skippedPolicyLogger.log( rawPolicy.name, deviceId, vinst,
                             "since multiple VRFs are configured, " +
                             "source or destination zone must be defined" )

      # Verify that group policies aren't programmed when running
      # MSS in verbatim mode.
      if ( validRawPolicy and not rawPolicy.isVerbatim
           and not self.mssConfig.policyEnforceRule.group ):
         validRawPolicy = False
         self.skippedPolicyLogger.log( rawPolicy.name, deviceId, vinst,
               "since policy enforcement rule configuration is 'verbatim', " +
               "verbatim tag must be added" )

      devicePolicy = policyList.devicePolicy.get( rawPolicy.name )
      if not validRawPolicy and not devicePolicy:
         return
      elif not validRawPolicy and devicePolicy:
         self.deleteDeviceAndServicePolicies( devicePolicy, policyList )
         return
      elif validRawPolicy and not devicePolicy:
         b1( 'gen NEW DevicePolicy:', v( rawPolicy.name ), 'device:', v( deviceId ) )
         devicePolicy = policyList.newDevicePolicy( rawPolicy.name, deviceId,
                                                    deviceSet.policySourceType )
         devicePolicy.timeFirstSeen = Lib.getTimestamp()
      elif validRawPolicy and devicePolicy:
         b1( 'updating existing DevicePolicy:', v( rawPolicy.name ), 'device:',
             v( deviceId ) )

      devicePolicy.logSessionStart = rawPolicy.logSessionStart
      devicePolicy.virtualInstance = vinst
      devicePolicy.isL2Policy = isL2Policy
      devicePolicy.timeLastSeen = Lib.getTimestamp()
      devicePolicy.haPeerDeviceId = self.getDeviceIdForMgmtIp( device.haPeerMgmtIp )
      self.updateHADeviceIdMap( deviceId, devicePolicy.haPeerDeviceId )
      devicePolicy.action = rawPolicy.action
      devicePolicy.number = rawPolicy.number
      populatePolicyL4Services( rawPolicy, devicePolicy )
      devicePolicy.tag.clear()
      for tag in rawPolicy.tags:
         devicePolicy.tag[ tag ] = True
      devicePolicy.isOffloadPolicy = rawPolicy.isOffloadPolicy
      devicePolicy.isVerbatim = VERBATIM in deviceSet.modifierTag and \
                                         not set( rawPolicy.tags ).isdisjoint(
                                                  set( deviceSet.modifierTag[
                                                       VERBATIM ].tag ) )
      devicePolicy.isForwardOnly = FORWARD_ONLY in deviceSet.modifierTag and \
                                         not set( rawPolicy.tags ).isdisjoint(
                                                  set( deviceSet.modifierTag[
                                                       FORWARD_ONLY ].tag ) )
      b4( 'mssDeviceId:', mssDeviceId, 'policyList:', policyList,
          'devicePolicy:', devicePolicy )
      self.updateDevPolicyLinks( rawPolicy, devicePolicy )
      if Lib.isMssL3EnabledAndL3Policy( devicePolicy ):
         devicePolicy.zoneA.zoneClass = INTERCEPT_ZONE
         devicePolicy.zoneB.zoneClass = INTERCEPT_ZONE
      else:
         classifySuccess = self.classifyDevPolicyZones( rawPolicy, devicePolicy )
         if not classifySuccess:
            b3( 'L2 zone classification failed, deleting devPolicy',
                v( devicePolicy.policyName ) )
            self.deleteDeviceAndServicePolicies( devicePolicy, policyList )
            return
      self.updateDevPolicyIntercepts( rawPolicy, devicePolicy )
      if ( isL2Policy and
            not ( devicePolicy.zoneA.intercept or devicePolicy.zoneB.intercept ) ):
         b1( 'No hosts in DevPolicy:', v( devicePolicy.policyName ), 'deleting')
         self.deleteDeviceAndServicePolicies( devicePolicy, policyList )
         return
      if self.isDeviceStateActive( deviceId ):
         devicePolicy.isCurrent = True

   def updatePolicies( self, rawPolicies, interfaces, neighbors, routingTables,
                       device, haPrimary=True ):
      ''' Called from PluginController with latest policies from a service device.
          device arg is a PluginController.ServiceDevice object
      '''
      start = time.time()
      deviceId = device.deviceId
      b2( 'deviceId:', v( deviceId ), 'thread:', v( device.threadName ),
          'policies:', v( { vinst : [ p.name for p in policyList ]
             for vinst, policyList in rawPolicies.items() } ),
          '\nrouteTables:', v( routingTables ) )
      if not deviceStateMutexExistsFor( deviceId ):
         b1( 'no deviceStateMutexExistsFor:', v( deviceId ) )
         return
      with deviceStateMutex( deviceId ):
         try:
            deviceSet = getDeviceSetUsingLock( device.deviceSetName, self.mpmConfig )
            deviceStatus = getSvcDeviceStatusUsingLock( deviceId, self.mpmStatus )
            if ( not deviceSet or not deviceStatus or
                 not doUpdatesInDeviceState( deviceSet.state ) or
                 not validateMonitoringThread( device.threadName, deviceStatus ) ):
               b1( 'unable to process policies for:', v( deviceId ) )
               return

            deviceStatus.mgmtIp = getArnetIpAddr( device.mgmtIp )
            deviceStatus.haPeerMgmtIp = getArnetIpAddr( device.haPeerMgmtIp )
            self.clearDevicePolicyCurrentFlags( deviceId )
            if deviceId in self.zoneClassMap:  # reset zone classifications
               self.zoneClassMap[ deviceId ].clear()

            # cleanup old virtual instance in skippedPolicyLogger
            self.skippedPolicyLogger.cleanupLoggedVInst(
                  deviceId, list( rawPolicies ) )

            for vinst, policyList in rawPolicies.items():
               # cleanup old policies in skippedPolicyLogger
               self.skippedPolicyLogger.cleanupLoggedPolicies( deviceId, vinst,
                     [ pol.name for pol in policyList ] )

               for policy in policyList:
                  self.updateDevicePolicy( policy, vinst, neighbors,
                                           device, deviceSet )
                  incPolicyCounters( self.mpmStatus, deviceStatus )
                  if ( not isActiveState( deviceSet.state ) or
                       not isActiveState( deviceStatus.state ) ):
                     break

            if ( isActiveState( deviceSet.state ) and
                 isActiveState( deviceStatus.state ) ):
               self.deletePoliciesForDevice( deviceId, nonCurrentOnly=True,
                  hasLiveHaPeer=(
                     not device.isSingleLogicalDeviceHaModel and
                     not haPrimary and self.hasLiveHaPeer( device ) ) )
               if ( routingTables and routingTables.featureSupported ):
                  for vinst in rawPolicies:
                     self.genMssL3Policies( deviceId, vinst, deviceSet, interfaces,
                                            routingTables.routingTables )
            servicePolicyCache[ deviceId ] = {}  # clear cache for this device
            with ActivityLock():
               b2( v( deviceId ), 'raw', v( len( rawPolicies ) ),
                   'proc', v( deviceStatus.numPoliciesProcessed ),
                   'totProc', v( self.mpmStatus.numPoliciesProcessed ),
                   'svcPols', v( len( self.mssSvcPolicyCfg.servicePolicy ) ),
                   'svcIntfs', v( len( self.mssSvcIntfCfg.intfSet ) ),
                   'clusterHaModel:', v( device.isSingleLogicalDeviceHaModel ),
                   v( delta( start ) ) )
         except Exception as ex:  # pylint: disable-msg=W0703
            b0( 'Policy update error: %s' % ex )
            traceback.print_exc()

   def updateMonitoringThreadNames( self, deviceThreads, aggrMgrThreadName ):
      ''' Called from PluginController to update the thread name in
          the service device status. A dictionary having the deviceId
          and thread information is passed to this. A lock for the
          service device status of all member devices is taken before
          updating the thread name.
      '''
      try:
         with deviceStateLocksGrabbed( deviceThreads ):
            for deviceId, thread in deviceThreads.items():
               # Update the thread name
               deviceStatus = getSvcDeviceStatusUsingLock( deviceId, 
                     self.mpmStatus )
               if deviceStatus:
                  b4( 'deviceId:', v( deviceId ), 'thread:', v( thread.name ) )
                  deviceStatus.monitorThreadName = thread.name
      except DeviceStateMutexError:
         # Couldn't grab all locks, update the thread names in next cycle
         b1( v( aggrMgrThreadName ), 'could not update thread names for devices:',
               v( deviceThreads.keys() ) )

   def cleanupPoliciesForDevice( self, device ):
      ''' Called from PluginController
      '''
      deviceId = device.deviceId
      b2( v( deviceId ), 'haPeer:', v( device.haPeerMgmtIp ),
          'isSingleLogicalDeviceHaModel:', v( device.isSingleLogicalDeviceHaModel ) )
      if not deviceStateMutexExistsFor( deviceId ):
         b1( 'no deviceStateMutexExistsFor:', v( deviceId ) )
         return
      with deviceStateMutex( deviceId ):
         self.deletePoliciesForDevice(
            deviceId, hasLiveHaPeer=( self.hasLiveHaPeer( device ) and
                                      not device.isSingleLogicalDeviceHaModel ) )
   def hasLiveHaPeer( self, device ):
      ''' Returns True if device has a responsive/alive HA Peer
      '''
      maxIntervalsNotSeen = 3
      if not device.haPeerMgmtIp:
         return False
      deviceSet = getDeviceSetUsingLock( device.deviceSetName, self.mpmConfig )
      peerStatus = self.getDeviceStatusForMgmtIp( device.haPeerMgmtIp )
      if deviceSet and peerStatus:
         secondsSinceSeen = int( Lib.getTimestamp() - peerStatus.timeLastSeen )
         maxSecondsNotSeen = maxIntervalsNotSeen * deviceSet.queryInterval
         isAlive = secondsSinceSeen < maxSecondsNotSeen
         b3( v( device.deviceId ), 'HA peer', v( device.haPeerMgmtIp ), 'isAlive:',
             v( isAlive ), '(secondsSinceLastSeen', v( secondsSinceSeen ), 'max is',
             v( maxSecondsNotSeen ), ',', v( maxIntervalsNotSeen ), 'intervals)' )
         if isAlive:
            return True
      return False

   def updateHADeviceIdMap( self, deviceId, peerDeviceId ):
      ''' Update the haDeviceIdMap which will be used to create the MssL3 entities
          HA peers share the deviceId to avoid churn when an HA failover happens.
          Also, handle rare cases where a new peer device is deployed or a device
          no longer has an HA peer.
      '''
      if not peerDeviceId and deviceId in self.haDeviceIdMap:
         oldHAPairName = self.haDeviceIdMap[ deviceId ]
         for devId, haPairName in self.haDeviceIdMap.items():
            if haPairName == oldHAPairName:  # purge both deviceId and old HA peer id
               b2( v( devId ), 'no longer has an HA peer, purge:',
                   v( oldHAPairName ) )
               del self.haDeviceIdMap[ devId ]

      elif ( peerDeviceId and ( deviceId not in self.haDeviceIdMap or
           peerDeviceId in self.haDeviceIdMap and
           self.haDeviceIdMap[ peerDeviceId ] != self.haDeviceIdMap[ deviceId ] ) ):
         haDeviceId = HAPairDeviceId.encode( deviceId, peerDeviceId )
         self.haDeviceIdMap[ deviceId ] = haDeviceId
         self.haDeviceIdMap[ peerDeviceId ] = haDeviceId
      b3( 'haDeviceIdMap:', v( self.haDeviceIdMap ) )

   def getDeviceStatusForMgmtIp( self, ipAddr ):
      with ActivityLock():
         for deviceStatus in self.mpmStatus.serviceDeviceStatus.values():
            if deviceStatus.mgmtIp.v4Addr == ipAddr:
               return deviceStatus
      return None

   def getDeviceIdForMgmtIp( self, ipAddr ):
      with ActivityLock():
         for deviceStatus in self.mpmStatus.serviceDeviceStatus.values():
            if deviceStatus.mgmtIp.v4Addr == ipAddr:
               return deviceStatus.serviceDeviceId
      return ''

   def handleInitialReadNotify( self ):
      ''' Notify MSS Agent when first pass of policies are read from all
          devices in active DeviceSets.
      '''
      with ActivityLock():
         atLeastOneActiveDeviceSet = False
         for deviceSet in self.mpmConfig.deviceSet.values():
            if deviceSetComplete( deviceSet ) and isActiveState( deviceSet.state ):
               atLeastOneActiveDeviceSet = True
               if deviceSet.isAggregationMgr:
                  if not self.initialReadDoneForAllChildren( deviceSet ):
                     return
               elif self.ceEnabled():
                  if not all( self.initialReadDoneForDeviceVsys( deviceId )
                              for deviceId, device in deviceSet.serviceDevice.items()
                              if deviceConfigComplete( device, deviceSet ) ):
                     return
               else:
                  if not all( self.initialReadDoneForDevice( deviceId )
                              for deviceId, device in deviceSet.serviceDevice.items()
                              if deviceConfigComplete( device, deviceSet ) ):
                     return
         if atLeastOneActiveDeviceSet:
            setConvergenceComplete( self )

   def initialReadDoneForDevice( self, deviceId ):
      deviceStatus = self.mpmStatus.serviceDeviceStatus.get( deviceId )
      if not deviceStatus:
         b2( 'check policyReadAttempt failed, no deviceStatus for:', v( deviceId ) )
         return False

      b2( v( deviceId ), 'policyReadAttempted:',
          v( deviceStatus.policyReadAttempted ) )
      return deviceStatus.policyReadAttempted

   def initialReadDoneForDeviceVsys( self, deviceId ):
      deviceOrPairId = self.haDeviceIdMap.get( deviceId, deviceId )
      devicePolicyList = { devStatusId : devStatus
                           for devStatusId,
                           devStatus in self.mpmStatus.devicePolicyList.items()
                           if self.getDeviceName( devStatusId ) == deviceOrPairId }

      for devStatusId, devStatus in devicePolicyList.items():
         b2( v( devStatusId ), ' policyConverge:', v( devStatus.policyConverge ),
             ' serviceDeviceConverge:', v( devStatus.serviceDeviceConverge ) )
         if not ( devStatus.policyConverge and devStatus.serviceDeviceConverge ):
            return False
      return True

   def initialReadDoneForAllChildren( self, deviceSet ):
      foundAggMgr = False
      for device in deviceSet.serviceDevice.values():
         if not device.isAccessedViaAggrMgr:
            foundAggMgr = True
            aggMgrDevStat = self.mpmStatus.serviceDeviceStatus.get(
               device.ipAddrOrDnsName )
            if self.ceEnabled():
               if ( not aggMgrDevStat or
                    not all( self.initialReadDoneForDeviceVsys( childId )
                             for childId in aggMgrDevStat.groupMember ) ):
                  return False
            else:
               if ( not aggMgrDevStat or
                    not all( self.initialReadDoneForDevice( childId )
                             for childId in aggMgrDevStat.groupMember ) ):
                  return False
      return foundAggMgr

   def getMssServicePolicy( self, devicePolicy, interceptZone, existingOnly=False ):
      ''' Returns an existing Mss::ServicePolicy if one already exists with same
          near and far interfaces or if one intf is empty otherwise returns new one.
      '''
      deviceId = devicePolicy.serviceDeviceId
      nearZoneName, nearLinks, nearSwitchLinks,\
         farZoneName, farLinks, farSwitchLinks = getInterceptZoneAttribs(
         devicePolicy, interceptZone )
      cacheKey = genServicePolicyCacheKey( nearZoneName, nearLinks,
                                           farZoneName, farLinks )
      if ( deviceId in servicePolicyCache and
           cacheKey in servicePolicyCache[ deviceId ] ):
         servicePolicy = servicePolicyCache[ deviceId ][ cacheKey ]
         updatePolicyNames( devicePolicy, servicePolicy )
         return servicePolicy

      nearIntfUuid = self.getServiceIntfSetUuid(
         nearLinks, nearZoneName, devicePolicy, existingOnly=existingOnly )
      farIntfUuid =  self.getServiceIntfSetUuid(
         farLinks, farZoneName, devicePolicy, existingOnly=existingOnly )
      if existingOnly and ( not nearIntfUuid or not farIntfUuid ):
         return None

      with ActivityLock():
         for servicePolicy in self.mssSvcPolicyCfg.servicePolicy.values():
            # use existing serivcePolicy if has both intf or one and other is empty
            if ( ( servicePolicy.nearIntfUuid == nearIntfUuid  and
                   servicePolicy.farIntfUuid == farIntfUuid ) or
                 ( not servicePolicy.nearIntfUuid and
                   servicePolicy.farIntfUuid == farIntfUuid ) or
                 ( not servicePolicy.farIntfUuid and
                   servicePolicy.nearIntfUuid == nearIntfUuid ) ):

               if not servicePolicy.nearIntfUuid:
                  servicePolicy.nearIntfUuid = nearIntfUuid
               elif not servicePolicy.farIntfUuid:
                  servicePolicy.farIntfUuid = farIntfUuid

               if ( devicePolicy.haPeerDeviceId and
                    devicePolicy.haPeerDeviceId == servicePolicy.serviceDeviceId ):
                  b1( 'new HA primary', v( deviceId ), 'taking over SvcPolicy:',
                      v( servicePolicy.name ) )
                  servicePolicy.serviceDeviceId = deviceId

               updatePolicyNames( devicePolicy, servicePolicy )
               servicePolicyCache[ deviceId ] = { cacheKey : servicePolicy }
               b2( 'use existing', v( servicePolicy.name ) )
               return servicePolicy

         if existingOnly:
            return None  # didn't find an existing service policy
         servicePolicy = genNewServicePolicy(
            devicePolicy, nearSwitchLinks, nearLinks, nearIntfUuid,
            farSwitchLinks, farLinks, farIntfUuid,
            self.mssSvcPolicyCfg, self.mpmStatus )
         devicePolicy.childServicePolicy[ servicePolicy.name ] = True
         servicePolicyCache[ deviceId ] = { cacheKey : servicePolicy }
         return servicePolicy

   def deleteServicePolicyNamed( self, servicePolicyName ):
      servicePolicy = getServicePolicyUsingLock( servicePolicyName,
                                                 self.mssSvcPolicyCfg )
      if servicePolicy:
         self.deleteServicePolicy( servicePolicy )

   def deleteServicePolicy( self, servicePolicy ):
      nearIntfUuid = servicePolicy.nearIntfUuid
      farIntfUuid = servicePolicy.farIntfUuid
      b2( 'deleting ServicePolicy:', v( servicePolicy.name ) )
      with ActivityLock():
         del self.mssSvcPolicyCfg.servicePolicy[ servicePolicy.name ]
      self.deleteServiceIntfAsNeeded( nearIntfUuid )
      self.deleteServiceIntfAsNeeded( farIntfUuid )

   def deleteDeviceAndServicePolicies( self, devicePolicy, policyList,
                                       waitForHaPeerToTakeoverSvcPolicy=False ):
      ''' Delete the MssPolicyMonitor::DevicePolicy and any associated
          Mss::ServicePolicyConfig and Mss::ServiceIntfConfig objects if
          ServicePolicy has no other interceptFlow IP addresses.
          Caller must NOT be holding ActivityLock.
      '''
      deviceId = devicePolicy.serviceDeviceId
      if not deviceStateMutexExistsFor( deviceId ):
         return
      with deviceStateMutex( deviceId ):
         for svcPolicyName in devicePolicy.childServicePolicy:
            servicePolicy = getServicePolicyUsingLock( svcPolicyName,
                                                       self.mssSvcPolicyCfg )
            if servicePolicy:
               if waitForHaPeerToTakeoverSvcPolicy:
                  if servicePolicy.serviceDeviceId == devicePolicy.serviceDeviceId:
                     b2( 'wait for SvcPolicy owner chg',
                         v( devicePolicy.serviceDeviceId ), 'to',
                         v( devicePolicy.haPeerDeviceId ) )
                     return
                  else:
                     break  # HA peer owns svcPol now so just delete devicePolicy
               with ActivityLock():
                  del servicePolicy.policyName[ devicePolicy.policyName ]
               intercepts = set( self.getDevicePolicyIntercepts( devicePolicy ) )
               interceptsInUse = self.interceptsFromOtherParents( servicePolicy,
                                                                  devicePolicy )
               interceptsNotInUse = intercepts - interceptsInUse
               with ActivityLock():
                  for ip in interceptsNotInUse:
                     b2( v( servicePolicy.name ), 'removing intercept:', v( ip ) )
                     del servicePolicy.interceptFlow[ ip ]

               if not servicePolicy.interceptFlow:  # remove policy if no intercepts
                  b1( 'no intercepts in SvcPolicy:', v( servicePolicy.name ) )
                  # remove policy first, then interfaces if not used elsewhere
                  self.deleteServicePolicy( servicePolicy )
                  self.deleteServiceIntfAsNeeded( servicePolicy.nearIntfUuid )
                  self.deleteServiceIntfAsNeeded( servicePolicy.farIntfUuid )
         b2( 'deleting', v( deviceId ), 'DevPolicy:', v( devicePolicy.policyName ) )
         with ActivityLock():
            del policyList.devicePolicy[ devicePolicy.policyName ]

   def deletePoliciesForDevice( self, deviceId, aggMgr=False, nonCurrentOnly=False,
                                hasLiveHaPeer=False, childrenCleanup=True ):
      ''' Remove DevicePolicy and associated ServicePolicy and ServiceInterface
          objects for this deviceId.
          If nonCurrentOnly=True only delete when DevicePolicy.isCurrent=False.
      '''
      deviceIdList = []
      if aggMgr:  # get child devices from status object
         if childrenCleanup:
            deviceIdList = self.getAggMgrChildDeviceIds( deviceId )
      else:
         deviceIdList = [ deviceId ]

      for devId in deviceIdList:
         with ActivityLock():
            policyList = [ devicePolicy for mssDevId, devicePolicy
                           in self.mpmStatus.devicePolicyList.items()
                           if self.getDeviceName( mssDevId ) == devId ]

         if policyList:
            for pl in policyList:
               for devicePolicy in pl.devicePolicy.values():
                  if ( not nonCurrentOnly or
                       ( nonCurrentOnly and not devicePolicy.isCurrent ) ):
                     self.deleteDeviceAndServicePolicies(
                        devicePolicy, pl, waitForHaPeerToTakeoverSvcPolicy=(
                           devicePolicy.isL2Policy and hasLiveHaPeer and
                           allLinksAreLagsOrMlags( devicePolicy ) ) )
         if not nonCurrentOnly and not hasLiveHaPeer:
            b1( 'delete all L3 SvcDeviceCfg & PolicySrcCfg for device:', v( devId ) )
            with ActivityLock():
               deviceOrHaPairId = self.haDeviceIdMap.get( devId, devId )
               b3( 'deviceOrHaPairId:', v( deviceOrHaPairId ),
                   '\nhaDeviceIdMap:', v( self.haDeviceIdMap ),
                   '\nsvcDevSrcCfg:', v( list( self.svcDevSrcCfg.serviceDevice ) ),
                   '\npolicySrcCfg:', v( list( self.policySrcCfg.policySet ) ) )
               # policies must be cleanup before the routing table, so MSS agent
               # won't try to resolve policies against non-existing next hop.
               for sd in self.policySrcCfg.policySet:
                  if self.getDeviceName( sd ) == deviceOrHaPairId:
                     del self.policySrcCfg.policySet[ sd ]
               for sd in self.svcDevSrcCfg.serviceDevice:
                  if self.getDeviceName( sd ) == deviceOrHaPairId:
                     del self.svcDevSrcCfg.serviceDevice[ sd ]
               if devId in self.haDeviceIdMap:
                  b2( 'purge haDeviceIdMap deviceId:', v( deviceId ) )
                  del self.haDeviceIdMap[ deviceId ]

   def deleteAllPoliciesForDeviceSet( self, deviceSet ):
      b1( 'removing policies for devices in DeviceSet:', v( deviceSet.name ) )
      for device in getServiceDevicesUsingLock( deviceSet ):
         if self.isDeviceAvailableUsingLock( device, deviceSet ):
            if not deviceSet.isAggregationMgr:
               self.deletePoliciesForDevice( device.ipAddrOrDnsName )
            else:
               for childId in self.getAggMgrChildDeviceIds( device.ipAddrOrDnsName ):
                  self.deletePoliciesForDevice( childId )

   def clearDevicePolicyCurrentFlags( self, deviceId ):
      with ActivityLock():
         for mssDeviceId, policyList in \
      self.mpmStatus.devicePolicyList.items():
            if self.getDeviceName( mssDeviceId ) == deviceId:
               for devicePolicy in policyList.devicePolicy.values():
                  devicePolicy.isCurrent = False

   def interceptsFromOtherParents( self, servicePolicy, devicePolicy ):
      ''' Get intercept IP addresses in a ServicePolicy that were contributed
          by other parent DevicePolicies.
      '''
      policyList = devicePolicy.parent
      intercepts = set()
      for parentDevPolicyName in servicePolicy.policyName:
         if parentDevPolicyName == devicePolicy.policyName:  # exclude this parent
            continue
         parentDevPolicy = policyList.devicePolicy.get( parentDevPolicyName )
         if parentDevPolicy:
            intercepts.update( self.getDevicePolicyIntercepts( parentDevPolicy ) )
      return intercepts

   def isSharingChildServicePolicy( self, devicePolicy ):
      ''' Return True if other device policies are sharing child service
          policy(ies) referred to by the passed DevicePolicy.
      '''
      devicePoliciesUsingServicePolicy = []
      for svcPolicy in self.getChildServicePoliciesFor( devicePolicy ):
         if svcPolicy:
            devicePoliciesUsingServicePolicy.extend( list( svcPolicy.policyName ) )
      return ( set( devicePoliciesUsingServicePolicy ) -
               { devicePolicy.policyName } )

   def getChildServicePoliciesFor( self, devicePolicy ):
      servicePolicies = []
      for svcPolicyName in devicePolicy.childServicePolicy:
         servicePolicy = getServicePolicyUsingLock( svcPolicyName,
                                                    self.mssSvcPolicyCfg )
         if servicePolicy:
            servicePolicies.append( servicePolicy )
      return servicePolicies

   def removePolicyNameFromChildServicePolicies( self, devicePolicy ):
      for svcPolicy in self.getChildServicePoliciesFor( devicePolicy ):
         if devicePolicy.policyName in svcPolicy.policyName:
            b2( 'removing policy name:', v( devicePolicy.policyName ), 
                'from SvcPolicy:', v( svcPolicy.name ) )
            with ActivityLock():
               del svcPolicy.policyName[ devicePolicy.policyName ]

   def removeInterceptsFromChildServicePolicies( self, devicePolicy, intercepts ):
      for svcPolicy in self.getChildServicePoliciesFor( devicePolicy ):
         for ipAddr in intercepts:
            if ipAddr in svcPolicy.interceptFlow:
               b2( 'removing', v( ipAddr.stringValue ), 'from svcPolicy', 
                   v( svcPolicy.name ) )
               with ActivityLock():
                  del svcPolicy.interceptFlow[ ipAddr ]

   def doShutdownCleanup( self ):
      ''' Cleanup sysdb before shutting down agent
          remove all:
            devicePolicies
            serviceDevice status
            spm/ServicePolicy
            spm/ServiceIntf
         reset convergence complete flag
         Caller should have Tac activity lock
      '''
      b0( 'doShutdownCleanup, deleting all spm/ServicePolicy and spm/ServiceIntf' )
      self.mpmStatus.serviceDeviceStatus.clear()
      self.mpmStatus.devicePolicyList.clear()
      self.mssSvcPolicyCfg.servicePolicy.clear()
      self.mssSvcIntfCfg.intfSet.clear()
      b1( 'deleting all L3 SvcDeviceCfg and PolicySrcCfg' )
      self.svcDevSrcCfg.serviceDevice.clear()
      self.policySrcCfg.policySet.clear()
      self.mssSvcPolicySrcState.convergenceComplete = False

   # startup sysdb validation and consistency checks
   #--------------------------------------------------------------------------------
   def validateAndSanitizeSysdbState( self ):
      ''' MPM Agent Sysdb consistency checks and sanitization for
          MssPolicyMonitor::Status and Mss::ServicePolicyConfig models
      '''
      b1( 'doing sysdb consistency checks and sanitization' )
      self.sanitizeDeviceStatus()
      self.sanitizeDevicePolicy()
      self.sanitizeServicePolicy()
      self.sanitizeServiceInterface()
      self.cleanupL3SvcDevicesAndPolicies()

   def sanitizeDeviceStatus( self ):
      ''' MssPolicyMonitor::Status ServiceDeviceStatus consistency checks:
          . Ensure each entity is associated with a configured ServiceDevice
          . Ensure monitoring state matches ServiceDevice.state; if shutdown
            remove status entity and any associated policies
          . Purge old thread name
      '''
      b3( 'sysdb consistency checks and sanitization for ServiceDeviceStatus' )
      for deviceId, devStat in self.mpmStatus.serviceDeviceStatus.items():
         b2( 'checking ServiceDeviceStatus for deviceId', v( deviceId ) )
         devSet = self.mpmConfig.deviceSet.get( devStat.deviceSetName )
         if not devSet:
            b0( '!!deviceSetName:', v( devStat.deviceSetName ), 
                'not found, deleting obj' )
            del self.mpmStatus.serviceDeviceStatus[ deviceId ]
            continue 

         if devStat.state != devSet.state:
            b0( '!!state mismatch between deviceSet & devStatus, deleting status' )
            try:
               genMutexAsNeededFor( deviceId )
               self.deletePoliciesForDevice( deviceId, devStat.isAggregationMgr )
            except Exception as ex:  # pylint: disable-msg=W0703
               b0( 'Error during policy cleanup:', v( ex ) )
            del self.mpmStatus.serviceDeviceStatus[ deviceId ]
            continue

         if devStat.accessedViaAggrMgr:  # device accessed via an aggregationMgr
            # pylint: disable-next=no-else-continue
            if devStat.accessedViaAggrMgr not in devSet.serviceDevice:
               b0( '!!parent aggrMgr deviceId not found in serviceDevices, ' 
                   'deleting' )
               del self.mpmStatus.serviceDeviceStatus[ deviceId ]
               continue
            else:
               mgrId = devStat.accessedViaAggrMgr
               aggrMgrDevStat = self.mpmStatus.serviceDeviceStatus.get( mgrId )
               if deviceId not in aggrMgrDevStat.groupMember:
                  b0( '!!device is not a group member for aggreagtionMgr, deleting' )
                  del self.mpmStatus.serviceDeviceStatus[ deviceId ]
                  continue
         elif deviceId not in devSet.serviceDevice:
            # device is an aggrMgr or a device w/ policies accessed directly
            b0( '!!deviceId not found in configured serviceDevices, deleting' ) 
            del self.mpmStatus.serviceDeviceStatus[ deviceId ]
            continue
         devStat.monitorThreadName = ''
      # ensure a deviceStateMutex entry exists for current devices
      # and that HA deviceId map is updated
      for deviceId, sdStatus in self.mpmStatus.serviceDeviceStatus.items():
         genMutexAsNeededFor( deviceId )
         haDevicePeerId = self.getDeviceIdForMgmtIp( sdStatus.haPeerMgmtIp.v4Addr )
         self.updateHADeviceIdMap( deviceId, haDevicePeerId )

   def sanitizeDevicePolicy( self ):
      ''' MssPolicyMonitor::Status DevicePolicy consistency checks:
          . Purge any non-current
          . Ensure each is associated with a device in ServiceDeviceStatus entity
      '''
      b3( 'sysdb consistency checks and sanitization for DevicePolicy' )
      for devPolList in self.mpmStatus.devicePolicyList.values():
         for devPol in devPolList.devicePolicy.values():
            deletePolicies = False
            b2( 'DevicePolicy:', v( devPol.policyName ), 'current:',
                v( devPol.isCurrent ) )
            if not devPol.isCurrent:
               b2( '!!purging nonCurrent DevicePolicy:', v( devPol.policyName ) )
               deletePolicies = True
            elif not devPol.serviceDeviceId in self.mpmStatus.serviceDeviceStatus:
               b2( '!!DevicePolicy.serviceDeviceId:', v( devPol.serviceDeviceId ),
                   'not found in serviceDeviceStatus, deleting')
               deletePolicies = True
            if deletePolicies:
               genMutexAsNeededFor( devPol.serviceDeviceId )
               self.deleteDeviceAndServicePolicies( devPol, devPolList )

   def purgeExpiredDevicePolicies( self ):
      ''' Purge old MssPolicyMonitor::Status DevicePolicy entities
      '''
      b2( 'sanitizing any expired DevicePolices' )
      maxAge = self.mpmConfig.purgeDevicePoliciesNotSeenIn
      for devPolList in self.mpmStatus.devicePolicyList.values():
         for devPolicy in devPolList.devicePolicy.values():
            age = int( Lib.getTimestamp() - devPolicy.timeLastSeen )
            b2( 'DevicePolicy', v( devPolicy.policyName ), 'age', v( age ),
                'maxAge', v( maxAge ) )
            if age > maxAge:
               b2( '!!purging expired DevicePolicy:', v( devPolicy.policyName ) )
               # first delete associated servicePolicies
               for svcPolicyName in devPolicy.childServicePolicy:
                  svcPolicy = self.mssSvcPolicyCfg.servicePolicy.get( svcPolicyName )
                  if svcPolicy:
                     del svcPolicy.policyName[ devPolicy.policyName ]
                     intercepts = self.getDevicePolicyIntercepts( devPolicy )
                     inUse = self.interceptsFromOtherParents( svcPolicy, devPolicy )
                     for ip in intercepts:
                        if ip not in inUse:
                           del svcPolicy.interceptFlow[ ip ]

                     if not svcPolicy.interceptFlow:  # remove svcPol if no i'cepts
                        self.deleteServicePolicy( svcPolicy )
               del devPolList.devicePolicy[ devPolicy.policyName ]

   def sanitizeServicePolicy( self ):
      ''' ServicePolicy consistency checks:
          . Ensure each ServicePolicy is referenced by at least one DevicePolicy.
            Note that more than one DevicePolicy can generate the same ServicePolicy.
          . Ensure that a ServicePolicy actually exists for each DevicePolicy
            reference.
          Note: self.mssSvcPolicyCfg mount point is a directory only for spm
                so don't need to filter by ServicePolicy.configSource
      '''
      b3( 'sysdb consistency checks and sanitization for ServicePolicy' )
      allSvcPols = list( self.mssSvcPolicyCfg.servicePolicy )
      svcPolRefsFromDevPols = []
      for devPolList in self.mpmStatus.devicePolicyList.values():
         for devPol in devPolList.devicePolicy.values():
            svcPolRefsFromDevPols.extend( list( devPol.childServicePolicy ) )

      b3( 'allSvcPols=', v( allSvcPols ), 'svcPolRefs=', v( svcPolRefsFromDevPols ) )
      svcPolsWithoutDevPolRef = set( allSvcPols ) - set( svcPolRefsFromDevPols )
      for svcPol in svcPolsWithoutDevPolRef:
         b0( '!!removing ServicePolicy:', v( svcPol ),
             'as it has no referencing DevicePolicy' )
         del self.mssSvcPolicyCfg.servicePolicy[ svcPol ]

      allSvcPols = list( self.mssSvcPolicyCfg.servicePolicy )  # refresh
      devPolRefWithNoSvcPol = set( svcPolRefsFromDevPols ) - set( allSvcPols )
      b3( 'svcPolsWithoutDevPolRef=',  v( svcPolsWithoutDevPolRef ),
          'devPolRefWithNoSvcPol=',  v( devPolRefWithNoSvcPol ) )
      for svcPol in devPolRefWithNoSvcPol:
         b0( 'Warning: DevicePolicy reference but no ServicePolicy:', v( svcPol ),
             'it should get regenerated when DevicePolicySM initializes' )

   def sanitizeServiceInterface( self ):
      ''' ServiceInterface consistency checks:
          . Ensure each ServiceInterface is referenced by at least
            one ServicePolicy.
          . Ensure ServiceIntfConfig.intfSet key is same as value
            ServiceIntfInfo.uuid attribute.
          Note: self.mssSvcIntfCfg mount point is a directory only for spm
                so don't need to filter by ServiceIntfInfo.configSource
      '''
      b3( 'sysdb consistency checks and sanitization for ServiceInterface' )
      svcIntfUuidsInSvcpolicies = set()
      for svcPol in self.mssSvcPolicyCfg.servicePolicy.values():
         svcIntfUuidsInSvcpolicies.add( svcPol.nearIntfUuid )
         svcIntfUuidsInSvcpolicies.add( svcPol.farIntfUuid )
      b3( 'svcIntfUuidsInSvcpolicies:', v( svcIntfUuidsInSvcpolicies ) ) 
      for intfUuid in self.mssSvcIntfCfg.intfSet:
         if intfUuid not in svcIntfUuidsInSvcpolicies:
            self.deleteServiceIntfAsNeeded( intfUuid, force=True )

   # MssL3 methods
   #--------------------------------------------------------------------------------
   def cleanupL3SvcDevicesAndPolicies( self ):
      ''' Purge any SvcDeviceCfg and PolicySrcCfg entities not referenced by a
          current device or HAPairName.  Used for startup sanitization and to
          handle rare cases where deviceIds have changed or when a partially
          configured device-set is active and polling has started for a device
          with an unknown or not yet configured firewall HA peer.
      '''
      validL3DeviceIds = set()
      # first get deviceIds that have current DevicePolicies
      for mssDeviceId, devicePolicyList in self.mpmStatus.devicePolicyList.items():
         if len( devicePolicyList.devicePolicy ) > 0:  # pylint: disable-msg=C1801
            validL3DeviceIds.add( self.getDeviceName( mssDeviceId ) )
      # now adjust for devices that are in an HA Pair
      for deviceId, haPairName in self.haDeviceIdMap.items():
         if deviceId in validL3DeviceIds:
            validL3DeviceIds.remove( deviceId )
            validL3DeviceIds.add( haPairName )
      b3( 'validL3DeviceIds:', v( validL3DeviceIds ) )

      # cleanup loggers
      self.fwZoneLogger.cleanupLoggedDevice( validL3DeviceIds )
      self.skippedPolicyLogger.cleanupLoggedDevice( validL3DeviceIds )
      self.firewallLogger.cleanupLoggedDevice( validL3DeviceIds )

      # policies must be cleanup before the routing table, so MSS agent
      # won't try to resolve policies against non-existing next hop.
      for mssDeviceName in self.policySrcCfg.policySet:
         devName = self.getDeviceName( mssDeviceName )
         if devName not in validL3DeviceIds:
            b2( 'purging device', v( mssDeviceName ), 'from MssPolicySourceConfig' )
            del self.policySrcCfg.policySet[ mssDeviceName ]

      for mssDeviceName in self.svcDevSrcCfg.serviceDevice:
         devName = self.getDeviceName( mssDeviceName )
         if devName not in validL3DeviceIds:
            b2( 'purging device', v( mssDeviceName ),
                'from ServiceDeviceSourceConfig' )
            del self.svcDevSrcCfg.serviceDevice[ mssDeviceName ]

   def getReachableSubnets( self, vrfName, deviceId ):
      reachableSubnets = {}
      l3ServiceDevice = self.svcDevSrcCfg.serviceDevice.get( deviceId.name() )
      if l3ServiceDevice:
         vrf = l3ServiceDevice.vrf.get( vrfName )
         if vrf:
            for ip, l3IntfConfig in vrf.l3Intf.items():
               reachableSubnets[ ip ] = list( l3IntfConfig.reachableSubnet )
      b2( vrfName, deviceId, 'subnets:\n', v(
          [ ( ip.stringValue, [ s.stringValue for s in subnets ] )
            for ip, subnets in reachableSubnets.items() ] ) )
      return reachableSubnets

   def triggerPollingComplete( self, mssDeviceId, vrfName ):
      '''
      Flip pollingComplete flag in serviceDeviceV2 and policySetV2.
      ActivityLock must be grabbed.
      '''
      l3ServiceDevice = self.svcDevSrcCfg.serviceDevice[ mssDeviceId.name() ]
      l3ServiceDeviceVrf = l3ServiceDevice.vrf[ vrfName ]
      l3ServiceDeviceVrf.pollingComplete = not l3ServiceDeviceVrf.pollingComplete
      policySet = self.policySrcCfg.policySet[ mssDeviceId.name() ]
      policyVrf = policySet.policy[ vrfName ]
      policyVrf.pollingComplete = not policyVrf.pollingComplete

   def genL3ServiceDevice( self, deviceId, vinst, deviceSet,
                           interfaces, routingTables ):
      '''
      Generate MssL3 service device input. ActivityLock must be grabbed.
      '''
      intfNameToIp, ipToIntfName = getIntfNameToIpAddrMap( interfaces[ vinst ] )
      routes = mergeCliConfigRoutes( deviceId, deviceSet, routingTables,
                                     ipToIntfName )
      b4( 'intfNameToIp:', v( intfNameToIp ), 'routes:', v( routes ) )
      deviceOrHaPairId = self.haDeviceIdMap.get( deviceId, deviceId )
      mssDeviceId = self.buildMssDeviceName( deviceOrHaPairId, vinst,
                                             deviceSet.policySourceType )
      l3ServiceDevice = (
            self.svcDevSrcCfg.serviceDevice.get( mssDeviceId.name() ) or
            self.svcDevSrcCfg.newServiceDevice( mssDeviceId.name() ) )
      currentVrfs = []
      if not self.ceEnabled():
         # If fortinet is used, FW VRF should be set to "default" (instead of '0')
         Lib.setDefaultVrf( routes, deviceSet.policySourceType )

      for vrfName, vrfRoutes in routes.items():
         vrf = ( l3ServiceDevice.vrf.get( vrfName ) or
                 l3ServiceDevice.newVrf( vrfName ) )
         currentVrfs.append( vrf.vrfName )
         currentL3Intfs = []
         reachableSubnets = {}
         for subnet, intf, _ in vrfRoutes:
            if intf and isinstance( intf, str ) and intf in intfNameToIp:
               intf = intfNameToIp[ intf ]
            elif not intf or not isinstance( intf, IpGenAddrType ):
               continue
            if isinstance( subnet, str ):
               subnet = Lib.convertIpMaskSyntax( subnet )
               # We have verified that Palo Alto and Fortinet does not allow
               # to specify a static route with an invalid IPv4 subnet
               # hence we are not checking if the subnet is valid here.
               subnet = Tac.Value( 'Arnet::IpGenPrefix', subnet )
            l3Intf = ( vrf.l3Intf.get( intf ) or vrf.newL3Intf( intf ) )
            currentL3Intfs.append( l3Intf.ip )
            if l3Intf not in reachableSubnets:
               reachableSubnets[ l3Intf ] = []
            reachableSubnets[ l3Intf ].append( subnet )

         for l3Intf, subnets in reachableSubnets.items():
            replaceSet( l3Intf.reachableSubnet, subnets )
         Lib.purgeNonCurrentKeys( vrf.l3Intf, currentL3Intfs )
      Lib.purgeNonCurrentKeys( l3ServiceDevice.vrf, currentVrfs )
      b2( v( Lib.dumpMssL3ServiceDevice( l3ServiceDevice ) ) )
      return list( routes )

   def genMssL3Policies( self, deviceId, vinst, deviceSet, interfaces, routes ):
      ''' Process all L3 service device policies and generate
          MssL3::serviceDeviceSourceConfig and MssL3::PolicySetConfig entities.
      '''
      deviceOrHaPairId = self.haDeviceIdMap.get( deviceId, deviceId )
      mssDeviceId = self.buildMssDeviceName( deviceId, vinst,
                                             deviceSet.policySourceType )
      mssDeviceOrHaPairId = self.buildMssDeviceName( deviceOrHaPairId, vinst,
                                                     deviceSet.policySourceType )
      with ActivityLock():
         policyList = self.mpmStatus.devicePolicyList.get( mssDeviceId.name() )
         b4( 'mssDeviceId:', mssDeviceId, 'policyList:', policyList )
         if ( not policyList or not routes or not
              any( not p.isL2Policy for p in policyList.devicePolicy.values() ) ):
            # policies must be cleanup before the routing table, so MSS agent
            # won't try to resolve policies against non-existing next hop.
            deletePolicySetConfigFor( mssDeviceOrHaPairId, self.policySrcCfg )
            deleteServiceDeviceConfigFor( mssDeviceOrHaPairId, self.svcDevSrcCfg )
            return

      with ActivityLock():
         self.genL3ServiceDevice( deviceId, vinst, deviceSet, interfaces, routes )
         policySet = (
               self.policySrcCfg.policySet.get( mssDeviceOrHaPairId.name() ) or
               self.policySrcCfg.newPolicySet( mssDeviceOrHaPairId.name() ) )

         perVrfPolicies = Lib.getDevicePolicyVrf( policyList, self.ceEnabled() )
         # Cleanup unused VRFs in policy collection
         Lib.purgeNonCurrentKeys( policySet.policy, list( perVrfPolicies ) )
         for vrfName, policies in perVrfPolicies.items():
            allIpAddrs = []
            allIpRanges = []
            exactMatchRulePriority = Lib.MssL3MssPriority.max
            reachSubnets = self.getReachableSubnets( vrfName, mssDeviceOrHaPairId )

            policyCfg = ( policySet.policy.get( vrfName ) or
                          policySet.newPolicy( vrfName ) )
            # create or update the one aggregate redirect rule for all L3 policies
            if ( Lib.MssL3MssPriority.min not in policyCfg.rule and
                 any( not p.isVerbatim for p in policies ) ):
               redirectRule = policyCfg.newRule( Lib.MssL3MssPriority.min )
               redirectRule.action = Lib.MSSL3_ACTION_REDIRECT
               redirectRule.match = ()
            else:
               redirectRule = policyCfg.rule.get( Lib.MssL3MssPriority.min )

            # Clear origin for any policy that has been removed
            allPolNames = [ devPolicy.policyName for devPolicy in policies ]
            for ruleConfig in policyCfg.rule.values():
               for polName in ruleConfig.origin :
                  if polName not in allPolNames:
                     del ruleConfig.origin[ polName ]

            policies.sort( key=lambda pol: pol.number )
            for devPolicy in policies:
               if devPolicy.isL2Policy:
                  b3( 'skip L2 policy:', v( devPolicy.policyName ) )
                  continue

               b2( 'processing L3 policy:', v( Lib.dumpDevicePolicy( devPolicy ) ) )
               srcIps, srcIpRanges = genIpTypesForMssL3( devPolicy.zoneA )
               dstIps, dstIpRanges = genIpTypesForMssL3( devPolicy.zoneB )

               # We don't support the external zone to subnets
               # expansion except for regular redirect.
               if ( devPolicy.isOffloadPolicy or devPolicy.isVerbatim ) and \
                  hasExternalZoneToExpand( srcIps, srcIpRanges, dstIps, dstIpRanges,
                                           devPolicy, reachSubnets ):
                  b1( 'skip L3 policy', v( devPolicy.policyName ),
                      'external zone expansion is not supported for '
                      'exact match rule' )
                  self.skippedPolicyLogger.log( devPolicy.policyName,
                                   deviceId, vinst,
                                   "offload and verbatim policies must not have " +
                                   "any external zone as source or destination" )
                  continue

               expandQualifier( devPolicy, reachSubnets, srcIps, dstIps,
                                srcIpRanges, dstIpRanges)

               # for both verbatim redirect and verbatim offload, we don't add
               # the ips to the implicit redirect group, and we don't add external
               # ip to the group.
               if redirectRule and not devPolicy.isVerbatim:
                  if not updateRedirectRule( allIpAddrs, allIpRanges, srcIps,
                                      dstIps, srcIpRanges, dstIpRanges, devPolicy,
                                      redirectRule, reachSubnets ):
                     b1( 'skip L3 policy', v( devPolicy.policyName ),
                         'next hop not available for redirect policy',
                         v( Lib.dumpIpAndZone( srcIps, srcIpRanges, dstIps,
                                               dstIpRanges, devPolicy ) ) )
                     self.skippedPolicyLogger.log( devPolicy.policyName,
                                   deviceId, vinst,
                                   "No IP address in redirect policy is " +
                                   "resolvable." )
                     continue

               if devPolicy.isOffloadPolicy or devPolicy.isVerbatim:
                  # this branch is for the the rule that need to match the qualifiers
                  # exactly -- offload, verbatim offload and verbatim redirect
                  updateExactMatchRule( exactMatchRulePriority,
                                        policyCfg, devPolicy, srcIps,
                                        srcIpRanges, dstIps,
                                        dstIpRanges, reachSubnets )
                  exactMatchRulePriority -= 1

               # Policy is correctly configured and is added to Mss input.
               # Clear the policy skippedPolicyLogger.
               self.skippedPolicyLogger.clearLog( devPolicy.policyName,
                                                  deviceId, vinst )

            if redirectRule:
               replaceSet( redirectRule.match.srcIp, allIpAddrs )
               replaceSet( redirectRule.match.srcIpRange, allIpRanges )
               redirectRule.seqNo += 1
            cleanupUnusedRules( policyCfg, exactMatchRulePriority, redirectRule )
            
            # Trigger end polling cycle for translate SMs
            self.triggerPollingComplete( mssDeviceOrHaPairId, vrfName )

         b2( v( Lib.dumpMssL3PolicySet( policySet ) ) )
         self.cleanupL3SvcDevicesAndPolicies()

def filterExternalIp( ips, ipRanges, localSubnets ):
   redirectIps = [ ip for ip in ips
                   if not ip.isDefaultRoutePrefix and 
                      any ( subnet.overlaps( ip ) for subnet in localSubnets ) ]
   redirectIpRanges = [ ipRange for ipRange in ipRanges
                        if any( subnet.contains( ipRange.startIp ) and
                                subnet.contains( ipRange.endIp )
                                for subnet in localSubnets ) ]
   return redirectIps, redirectIpRanges

def hasExternalZoneToExpand( srcIps, srcIpRanges, dstIps, dstIpRanges, 
                             devPolicy, reachSubnets ):
   return  ( not srcIps and not srcIpRanges and
             isExternalZone( devPolicy.zoneA, reachSubnets ) ) or \
           ( not dstIps and not dstIpRanges and
             isExternalZone( devPolicy.zoneB, reachSubnets ) )

def isExternalZone( devPolicyZone, reachSubnets ):
   if devPolicyZone.zoneType == Lib.LAYER3:
      for link in devPolicyZone.link.values():
         for subnet in reachSubnets.get( link.ipAddr, [] ):
            if subnet.isDefaultRoutePrefix:
               return True
   return False

def getLocalSubnets( reachSubnets ):
   return [ subnet for subnetList in reachSubnets.values()
            for subnet in subnetList if not subnet.isDefaultRoutePrefix ]

def expandQualifier( devPolicy, reachSubnets, srcIps, dstIps, 
                     srcIpRanges, dstIpRanges ):
   if not srcIps and not srcIpRanges and \
      not dstIps and not dstIpRanges: 
      if devPolicy.zoneA.zoneType == Lib.UNKNOWN_ZONE_TYPE and \
         devPolicy.zoneB.zoneType == Lib.UNKNOWN_ZONE_TYPE:
         # special case for 'any' for both zones, any for src and dst ip, 
         # expand all the ips in local subnets
         srcIps.extend( getLocalSubnets( reachSubnets ) )
      else:
         expandZoneToSubnets( srcIps, srcIpRanges, devPolicy.zoneA, 
                              reachSubnets )
         expandZoneToSubnets( dstIps, dstIpRanges, devPolicy.zoneB, 
                              reachSubnets )

def expandZoneToSubnets( ipAddrs, ipRanges, devPolicyZone, reachSubnets ):
   # if the zone is a L3 zone and there is no ip specified, then we expand the zone
   # qualifier to all reachable subnets ips qualifier
   if devPolicyZone.zoneType == Lib.LAYER3 and not ipAddrs and not ipRanges:
      ipAddrs.extend( getReachableSubnetsForZone( devPolicyZone, reachSubnets ) )

def updateRedirectRule( allIpAddrs, allIpRanges, srcIps, dstIps, srcIpRanges, 
                        dstIpRanges, devPolicy, redirectRule, reachSubnets ):
   srcIps, srcIpRanges = filterExternalIp( srcIps, srcIpRanges,
                                           getLocalSubnets( reachSubnets ) )
   dstIps, dstIpRanges = filterExternalIp( dstIps, dstIpRanges,
                                           getLocalSubnets( reachSubnets ) )
   allIpAddrs.extend( srcIps )
   allIpAddrs.extend( dstIps )
   allIpRanges.extend( srcIpRanges )
   allIpRanges.extend( dstIpRanges )

   # If policy has no valid IP to redirect, origin must not be populated
   if not srcIps and not srcIpRanges and not dstIps and not dstIpRanges:
      if devPolicy.policyName in redirectRule.origin:
         b4( 'Delete origin rule ', v( devPolicy.policyName ) )
         del redirectRule.origin[ devPolicy.policyName ]
      return False

   if devPolicy.policyName in redirectRule.origin:
      origin = redirectRule.origin[ devPolicy.policyName ]
   else:
      b4( 'Add new origin ', v( devPolicy.policyName ) )
      origin = redirectRule.newOrigin( devPolicy.policyName )
      origin.match = ()
   origin.action = Lib.getMssAction( devPolicy )
   replaceSet( origin.match.srcIp, srcIps )
   replaceSet( origin.match.dstIp, dstIps )
   replaceSet( origin.match.srcIpRange, srcIpRanges )
   replaceSet( origin.match.dstIpRange, dstIpRanges )
   l3Apps, l4Apps, l4Ranges = genIpProtocolTypes( devPolicy )
   replaceSet( origin.match.l3App, l3Apps )
   replaceSet( origin.match.dstL4App, l4Apps )
   replaceSet( origin.match.dstL4AppRange, l4Ranges )
   origin.source = getConfigSource( devPolicy )
   origin.tags = ','.join( devPolicy.tag.keys() )
   origin.logEnabled = devPolicy.logSessionStart
   return True

# Check and add multiple modifier tags. In the future, we can add
# more if's to determine if we have more tags to add.
def getPolicyModifiers( ruleOrigin, devicePolicy ):
   if devicePolicy.isVerbatim:
      ruleOrigin.policyModifierSet.add( MssL3MssPolicyModifier.verbatim )
   else:
      ruleOrigin.policyModifierSet.remove( MssL3MssPolicyModifier.verbatim )

   if devicePolicy.isForwardOnly:
      ruleOrigin.policyModifierSet.add( MssL3MssPolicyModifier.forwardOnly )
   else:
      ruleOrigin.policyModifierSet.remove( MssL3MssPolicyModifier.forwardOnly )

def updateExactMatchRule( priority, policyCfg, devPolicy, srcIpAddrs,
                          srcIpRanges, dstIpAddrs, dstIpRanges, reachSubnets ):
   ''' caller must be holding Tac.ActivityLock '''
   def populateRule( rule ):
      rule.action = Lib.getMssAction( devPolicy )
      updateExactMatchRuleAddrs( rule.match.srcIp, rule.match.srcIpRange,srcIpAddrs,
                                 srcIpRanges, reachSubnets, devPolicy.zoneA )
      updateExactMatchRuleAddrs( rule.match.dstIp, rule.match.dstIpRange, dstIpAddrs,
                                 dstIpRanges, reachSubnets, devPolicy.zoneB )

      l3Apps, l4Apps, l4Ranges = genIpProtocolTypes( devPolicy )
      replaceSet( rule.match.l3App, l3Apps )
      replaceSet( rule.match.dstL4App, l4Apps )
      replaceSet( rule.match.dstL4AppRange, l4Ranges )

   if priority not in policyCfg.rule:
      rule = policyCfg.newRule( priority )
      rule.match = ()
   else:
      rule = policyCfg.rule.get( priority )
   
   populateRule( rule )
   rule.seqNo += 1
   
   # update the raw rule
   if devPolicy.policyName in rule.origin:
      origin = rule.origin[ devPolicy.policyName ]
   else:
      rule.origin.clear()
      origin = rule.newOrigin( devPolicy.policyName )
      origin.match = ()

   origin.logEnabled = devPolicy.logSessionStart

   populateRule( origin )
   origin.source = getConfigSource( devPolicy )
   origin.tags = ','.join( devPolicy.tag.keys() )
   getPolicyModifiers( origin, devPolicy )

   return rule

def updateExactMatchRuleAddrs( ruleMatchIp, ruleMatchIpRange, ipAddrs, ipRanges,
                               reachSubnets, devicePolicyZone ):
   # if not an 'any' zone and is 'any' IP addr, use reachable subnets from zone intfs
   if devicePolicyZone.zoneType == Lib.LAYER3 and not ipAddrs and not ipRanges:
      replaceSet( ruleMatchIp,
                  getReachableSubnetsForZone( devicePolicyZone, reachSubnets ) )
   else:
      replaceSet( ruleMatchIp, ipAddrs )
      replaceSet( ruleMatchIpRange, ipRanges )

def getReachableSubnetsForZone( devicePolicyZone, reachableSubnets ):
   subnets = []
   for link in devicePolicyZone.link.values():
      if link.ipAddr in reachableSubnets:
         subnets.extend( reachableSubnets[ link.ipAddr ] )
   return subnets


def mergeCliConfigRoutes( deviceId, deviceSet, routingTables, ipToIntfName ):
   ''' merge CLI configured routes '''
   b2( "mergeCliConfigRoutes ipToIntfName", v( ipToIntfName ) )
   dumpRoutes( 'Firewall routingTables:', routingTables )
   cliRoutingTables = ServiceDeviceRoutingTables()
   with ActivityLock():  # gather cfg routes then process later not holding lock
      svcDevice = deviceSet.serviceDevice.get( deviceId )
      if not svcDevice:  # for aggMgr deviceSet without device member config
         return routingTables
      for vrf, vrfCfg in svcDevice.vrfConfig.items():
         for l3IntfIp, routes in vrfCfg.routingTable.items():
            for dest in routes.route:
               intfIp = l3IntfIp.stringValue
               intfName = ipToIntfName.get( intfIp )
               b2( "mergeCliConfigRoutes add cli route", v( vrf ),
                   v( dest.stringValue ), v( intfName ), v( intfIp ) )
               cliRoutingTables.addRoute( vrf, dest.stringValue, intfName, intfIp )

   dumpRoutes( 'CLI routingTables:', cliRoutingTables.routingTables )
   if cliRoutingTables.routingTables:
      # If we have a routing table configured in CLI, use it and ignore the one
      # retrieved from the firewall
      b0( 'Using CLI routing table instead of firewall routes' )
      return cliRoutingTables.routingTables

   return routingTables


def genIpTypesForMssL3( devicePolicyZone ):
   ipAddrs = []
   ipRanges = []
   for ip in devicePolicyZone.rawIntercept:
      if Lib.isIpRange( ip ):
         ipRanges.append( genMssL3IpRange( ip ) )
      else:
         ip = Lib.convertIpMaskSyntax( ip )
         ipAddrs.append( Tac.Value( 'Arnet::IpGenPrefix', ip ) )
   return ipAddrs, ipRanges


def genIpProtocolTypes( devPolicy ):
   l3Apps = []
   l4Apps = []
   l4Ranges = []
   for l4 in devPolicy.dstL4App.values():
      if l4.protocol == 'ICMP':
         l3Apps.append( Lib.IP_PROTOCOL_TAC_VALUE.get( 'ICMP' ) )
      else:
         l4Proto = Lib.IP_PROTOCOL_TAC_VALUE.get( l4.protocol.upper() )
         if not l4Proto:
            b3( 'Invalid protocol:', v( l4.protocol ) )
            continue
         for port in l4.port:
            if ':' in port:
               # TODO: need to add a feature to support the source port as it is
               # supported by MSS and DF. Syntaxes are:
               # <dstPort>
               # <startDstPort>-<endDstPort>
               # <dstPort>:<srcPort>
               # <dstPort>:<startSrcPort>-<endSrcPort>
               # <startDstPort>-<endDstPort>:<srcPort>
               # <startDstPort>-<endDstPort>:<startSrcPort>-<endSrcPort>
               b0( 'Removing Source Port. Not supported:', v( port ) )
               port = port.split( ':' )[ 0 ]
            if '-' not in port:
               l4Apps.append( Tac.Value( 'MssL3::MssL4App', l4Proto,
                                         int( port ) ) )
            else:
               startPort, endPort = port.split( '-' )
               l4Ranges.append( Tac.Value( 'MssL3::MssL4AppRange', l4Proto,
                                           int( startPort ), int( endPort ) ) )
   return l3Apps, l4Apps, l4Ranges

def cleanupUnusedRules( policyCfg, exactMatchRulePriority, redirectRule ):
   if ( redirectRule and
        not redirectRule.match.srcIp and not redirectRule.match.srcIpRange ):
      del policyCfg.rule[ Lib.MssL3MssPriority.min ]

   # remove any remaining unused ExactMatchRules
   while exactMatchRulePriority in policyCfg.rule:
      b2( 'deleting old rule priority:', v( exactMatchRulePriority ) )
      del policyCfg.rule[ exactMatchRulePriority ]
      exactMatchRulePriority -= 1

def replaceSet( tacSet, newElementList ):
   ''' The tac void set type doesn't allow directly assigning a new set
       so this function individually adds new elements and purges old ones
       that don't exist in the newElementList.  Also handles deduplicating
       newElementList.
   '''
   if ( newElementList and
         isinstance( newElementList[ 0 ], Tac.Type( 'Arnet::IpProto' ) ) ):
      # workaround for broken tac set that doesn't allow retrieving original type
      origSetKeys = [ Tac.Value( 'Arnet::IpProto', i ) for i in tacSet ]
   else:
      origSetKeys = list( tacSet )
   if origSetKeys != newElementList:
      for element in newElementList:
         if element not in tacSet:
            tacSet.add( element )
         elif element in origSetKeys:
            origSetKeys.remove( element )
         else:
            b4( 'deduping element:', v( element ) )
      for removedElement in origSetKeys:
         tacSet.remove( removedElement )
      b4( 'replaceSet exit tacSet:', v( list( tacSet ) ) )


def genMssL3IpRange( addrField ):
   startIp, endIp = addrField.split( '-' )
   return Tac.Value( 'MssL3::MssIpRange',
                     Tac.Value( 'Arnet::IpGenAddr', startIp ),
                     Tac.Value( 'Arnet::IpGenAddr', endIp ) )


def getIntfNameToIpAddrMap( interfaces ):
   intfNameToIp = {}
   ipToIntfName = {}
   for intf in interfaces.values():
      if intf.state == Lib.LINK_STATE_UP and intf.ipAddr:
         intfNameToIp[ intf.name ] = IpGenAddrWithMask( intf.ipAddr ).ipGenAddr
         ipToIntfName = { ip.stringValue:intf for intf, ip in intfNameToIp.items() }
   b3( 'intfNameToIp:',
       v( [ ( intf, ip.stringValue ) for intf, ip in intfNameToIp.items() ] ) )
   return intfNameToIp, ipToIntfName


def deleteServiceDeviceConfigFor( mssDeviceId, svcDevSrcCfg ):
   if mssDeviceId.name() in svcDevSrcCfg.serviceDevice:
      del svcDevSrcCfg.serviceDevice[ mssDeviceId.name() ]


def deletePolicySetConfigFor( mssDeviceId, policySrcCfg ):
   if mssDeviceId.name() in policySrcCfg.policySet:
      del policySrcCfg.policySet[ mssDeviceId.name() ]


def dumpRoutes( msg, routes ):
   for vrf in routes:
      for dest, intf, _ in routes[ vrf ]:
         if intf and isinstance( intf, str ):
            b2( v( msg ), v( vrf ), v( dest ), v( intf ) )
         elif intf:
            b2( v( msg ), v( vrf ), v( dest ), v( intf.stringValue ) )

####################################################################################
# This pointer allows debugging live MssPolicyMonitor Agent state on Acons
agentPtr = None

# The following method can be used when MssPolicyMonitor is instantiated in a cohab
# breadth test environment. Before removing the local reference to MssPolicyMonitor
# object in the test cleanup, the breadth test can call this method. Thus, as soon as
# the local reference of the MssPolicyMonitor object held by the breadth test is
# removed, the corresponding finalizer will be called immediately.
def clearAgentReference():
   global agentPtr
   agentPtr = None

class MssPolicyMonitor( Agent.Agent ):

   def __init__( self, entityMgr, agentName='MssPolicyMonitor', blocking=False ):
      global agentPtr
      agentPtr = self
      Agent.Agent.__init__( self, entityMgr, agentName=agentName )
      b0( 'initializing, sysname:', v( entityMgr.sysname() ), 'on thread:',
          v( threadName() ), 'pid:', v( os.getpid() ) )
      self.blocking = blocking
      self.warm_ = False
      self.controllerdbMountsComplete = False
      self.sysdbMountsComplete = False
      self.sysdbMgr = None
      self.ctrldbStatus = None
      self.mssConfig = None
      self.mssStatus = None
      self.mssL3Status = None
      self.mpmConfig = None
      self.mpmStatus = None
      self.mssSvcIntfCfg = None
      self.mssSvcPolicyCfg = None
      self.mssSvcPolicySrcState = None
      self.mssL2PreReleaseCfg = None  # MssL2
      self.svcDevSrcCfg = None  # MssL3
      self.policySrcCfg = None  # MssL3
      self.svcDevSrcCfgV2 = None # MssL3V2
      self.policySrcCfgV2 = None # MssL3V2
      self.runnabilityHandler = None
      self.mssStatusReactor = None
      self.mssL3StatusReactor = None
      self.mpmConfigReactor = None
      self.topoConvergenceSM = None
      self.deviceSetCollectionReactor = None
      self.devicePolicyListCollectionReactor = None
      self.serviceDeviceStatusCollectionReactor = None
      self.policyTranslateSm = None
      self.svcDeviceTranslateSm = None
      self.monitorInstances = {}  # key=DeviceSet name
      self.cdbTopology = None
      self.sslStatus = None
      cdbSocket = os.environ.get(
         'CONTROLLERDBSOCKNAME',
         Tac.Value('Controller::Constants').controllerdbDefaultSockname )
      b1( 'using Controllerdb socket name:', v( cdbSocket ) )
      self.cdbEm = Controllerdb(
         entityMgr.sysname(), controllerdbSockname_=cdbSocket, mountRoot=False )

   def doControllerdbMounts( self ):
      def _onControllerdbMountsComplete():
         b2( 'controllerdb mounts complete' )
         self.controllerdbMountsComplete = True
         self.doMaybeFinishInit()

      cdbMg = self.cdbEm.mountGroup()
      cdbMg.mount( '', 'Tac::Dir', 'rt' )  # t=toplevel rootMount, must do this first
      self.cdbTopology = cdbMg.mount( 'topology/version3/global/status',
                                      'NetworkTopologyAggregatorV3::Status', 'r' )
      if self.blocking:
         b2( 'doing Controllerdb mounts: blocking' )
         cdbMg.close( blocking=True )
      else:
         b2( 'doing Controllerdb mounts: non-blocking' )
         cdbMg.close( callback=_onControllerdbMountsComplete )

   def doSysdbMounts( self, entityMgr ):
      def _onMountsComplete():
         b2( 'sysdb mounts complete' )
         self.sysdbMountsComplete = True
         self.doMaybeFinishInit()

      mg = entityMgr.mountGroup()
      self.ctrldbStatus = mg.mount( 'controller/status',
                                 'Controllerdb::Status', 'r' )
      self.mssConfig = mg.mount( 'mss/config',
                                 'Mss::Config', 'r' )
      self.mssStatus = mg.mount( 'mss/status',
                                 'Mss::Status', 'r' )
      self.mssL3Status = mg.mount( 'mssl3/status',
                                   'MssL3::Status', 'r' )
      self.mpmConfig = mg.mount( 'msspolicymonitor/config',
                                 'MssPolicyMonitor::Config', 'r' )
      self.mpmStatus = mg.mount( 'msspolicymonitor/status',
                                 'MssPolicyMonitor::Status', 'w' )
      self.mssSvcIntfCfg = mg.mount( 'mss/serviceIntfConfig/spm',
                                     'Mss::ServiceIntfConfig', 'w' )
      self.mssSvcPolicyCfg = mg.mount( 'mss/servicePolicyConfig/spm',
                                       'Mss::ServicePolicyConfig', 'w' )
      self.mssSvcPolicySrcState = mg.mount( 'mss/servicePolicySourceState/spm',
                                            'Mss::MssPolicySourceState', 'w' )
      self.mssL2PreReleaseCfg = mg.mount( 'mss/preReleaseConfig',
                                          'Mss::PreReleaseConfig', 'r' )
      self.sslStatus = mg.mount( 'mgmt/security/ssl/status',
                                 'Mgmt::Security::Ssl::Status', 'r' )
      self.svcDevSrcCfg = mg.mount( 'mssl3/serviceDeviceSourceConfig/spm',
                                    'MssL3::ServiceDeviceSourceConfig', 'w' )
      self.policySrcCfg = mg.mount( 'mssl3/policySourceConfig/spm',
                                    'MssL3::MssPolicySourceConfig', 'w' )
      self.svcDevSrcCfgV2 = mg.mount( 'mssl3/serviceDeviceSourceConfigV2/spm',
                                      'MssL3V2::ServiceDeviceSourceConfig', 'w' )
      self.policySrcCfgV2 = mg.mount( 'mssl3/policySourceConfigV2/spm',
                                      'MssL3V2::MssPolicySourceConfig', 'w' )
      if self.blocking:
         b2( 'doing Sysdb mounts: blocking' )
         mg.close( blocking=True )  # synchronous mounting
         _onMountsComplete()
      else:
         b2( 'doing Sysdb mounts: non-blocking' )
         mg.close( callback=_onMountsComplete )  # asynchronous mounting

   def doMaybeFinishInit( self ):
      if self.sysdbMountsComplete and self.controllerdbMountsComplete:
         if os.environ.get( 'A4_CHROOT' ):
            self.finishAgentInit()  # don't need to wait for topo convergence
         else:
            self.topoConvergenceSM = Reactors.TopologyConvergenceSM(
               self.cdbTopology, self )
      elif ( self.sysdbMountsComplete and not os.environ.get( 'A4_CHROOT' ) and
             ( ( not self.mssStatus.running and not self.mssL3Status.running ) or
               not self.ctrldbStatus.enabled ) ):
         b0( 'MssPm shutting down, Mss, MssL3 or Controllerdb agent '
             'not running at startup!' )
         with ActivityLock():
            self.mssSvcIntfCfg.intfSet.clear()
            self.mssSvcPolicyCfg.servicePolicy.clear()
            self.mssSvcPolicySrcState.convergenceComplete = False
            self.svcDevSrcCfg.serviceDevice.clear()
            self.policySrcCfg.policySet.clear()
            self.svcDevSrcCfgV2.serviceDevice.clear()
            self.policySrcCfgV2.policySet.clear()
            self.mpmStatus.devicePolicyList.clear()
            self.mpmStatus.serviceDeviceStatus.clear()
            self.mpmStatus.running = False  # do last, Launcher will now kill agent

   def finishAgentInit( self ):
      b2( 'finishAgentInit on thread:', v( threading.current_thread().name ) )
      with ActivityLock():
         self.mpmStatus.startTime = Lib.getTimestamp()
         Lib.mssMultiVwireEnabled = self.mssL2PreReleaseCfg.multiVwireEnabled
      b0( 'MssL2 multiVwire mode enabled:', v( Lib.isMssMultiVwireEnabled() ) )
      self.sysdbMgr = SysdbMgr(
         self.mpmConfig, self.mpmStatus, self.mssConfig,
         self.mssSvcIntfCfg, self.mssSvcPolicyCfg,
         self.mssSvcPolicySrcState, self.svcDevSrcCfg, self.policySrcCfg,
         self.cdbTopology, self.monitorInstances, self.sslStatus )
      self.sysdbMgr.validateAndSanitizeSysdbState()

      b1( 'initializing state machines' )
      self.devicePolicyListCollectionReactor = Tac.collectionChangeReactor(
         self.mpmStatus.devicePolicyList, Reactors.DevicePolicyListSM,
         reactorArgs=( self.sysdbMgr, ) )
      b1( 'initializing config reactors' )
      self.deviceSetCollectionReactor = Tac.collectionChangeReactor(
         self.mpmConfig.deviceSet, Reactors.DeviceSetReactor,
         reactorArgs=( self.sysdbMgr, self.monitorInstances ) )
      self.mpmConfigReactor = Reactors.MPMConfigReactor( self.sysdbMgr,
                                                         self.monitorInstances )
      self.runnabilityHandler = Reactors.RunnabilityHandler( self.mssStatus,
                                                             self.mssL3Status,
                                                             self.sysdbMgr )
      self.mssStatusReactor = Reactors.MssStatusReactor( self.mssStatus,
                                                         self.runnabilityHandler )
      self.mssL3StatusReactor = Reactors.MssL3StatusReactor( self.mssL3Status,
                                                            self.runnabilityHandler )

      if self.mssConfig.policyEnforcementConsistency == 'strict':
         self.policyTranslateSm = Tac.newInstance(
               'MssPolicyMonitor::PolicyTranslateRootSm',
               self.policySrcCfg, self.policySrcCfgV2, self.mpmStatus )
         self.svcDeviceTranslateSm = Tac.newInstance(
               'MssPolicyMonitor::ServiceDeviceTranslateRootSm',
               self.svcDevSrcCfg, self.svcDevSrcCfgV2, self.mpmStatus )

      with ActivityLock():
         if ( not self.mssSvcPolicySrcState.convergenceComplete and
              not any( deviceSetComplete( devSet ) and isActiveState( devSet.state )
                       for devSet in self.mpmConfig.deviceSet.values() ) ):
            # We set the convergence flag if no valid configuration is provided
            setConvergenceComplete( self.sysdbMgr )
      self.warm_ = True
      b0( 'MssPolicyMonitor agent initialization complete' )

   def doInit( self, entityManager ):
      # Sysdb mounts must be completed first to make sure ControllerDb is ready
      self.doSysdbMounts( entityManager )
      self.doControllerdbMounts()
      b3( 'doInit() exiting' )

   def warm( self ):
      ''' Def: Agent is warm when behaving in accordance with its configuration '''
      return self.warm_


def name():
   return Lib.AGENT_NAME


def main():
   Lib.bothTraceInit( 'MssPolicyMonitor' )
   b2( 'Creating MssPolicyMonitor Agent container and calling runAgents()' )
   try:
      container = Agent.AgentContainer( [ MssPolicyMonitor ] )
      container.runAgents()
   except Exception as ex:  # pylint: disable-msg=W0703
      b0( 'Exception in agent main(): %s' % ex )
      traceback.print_exc()


if __name__ == "__main__":
   main()
