#!/usr/bin/env python3
#
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

''' API access to a Fortinet FortiManager Centralized Security Management platform
    Initial JSON-RPC Request API developed and tested with FortiManager 5.6.2

'''
# pylint: disable=E1101

import re
import json
import time
import requests
from requests.adapters import HTTPAdapter
from copy import deepcopy
from MssPolicyMonitor import Lib
from MssPolicyMonitor.Error import ServiceDeviceError, FirewallAPIError
from MssPolicyMonitor.Lib import ( t0, t2, t4, t6,
                                   LINK_STATE_UNKNOWN, LINK_STATE_UP,
                                   LINK_STATE_DOWN, HA_ACTIVE, HA_PASSIVE,
                                   HA_ACTIVE_PASSIVE, HA_ACTIVE_ACTIVE )
from MssPolicyMonitor.PluginLib import ( ServiceDeviceHAState, ServiceDevicePolicy,
                                         NetworkInterface,
                                         ServiceDeviceRoutingTables )
import ReversibleSecretCli

# TODO: suppress "Unverified HTTPS request is being made" warning message
# until we add support for TLS/SSL certificate validation
import urllib3
urllib3.disable_warnings( urllib3.exceptions.InsecureRequestWarning )

t4( 'imported python requests version:', requests.__version__ )

POLICY_TAGS_REGEX = re.compile( r'tags[[(](.+?)[])]', re.IGNORECASE | re.DOTALL )
DEFAULT_VLAN_RANGE = '1-4094'
HA_STRING_MAP = {
   'a-p': HA_ACTIVE_PASSIVE,
   'a-a': HA_ACTIVE_ACTIVE,
   # 'active': HA_ACTIVE,
   # 'passive': HA_PASSIVE,
   # 'active-primary': HA_ACTIVE_PRIMARY,
   # 'active-secondary': HA_ACTIVE_SECONDARY,
}

####################################################################################
class FortiManager:

   def __init__( self, config ):
      self.config = config
      self.ipAddr = config[ 'ipAddress' ]
      protocol = config[ 'protocol' ]
      portNum = config[ 'protocolPortNum' ]
      self.username = config[ 'username' ]
      self.password = config[ 'password' ]
      self.timeout = config[ 'timeout' ]
      self.retries = config[ 'retries' ]
      self.adom = config[ 'adminDomain' ]
      self.vdom = config[ 'virtualDomain' ]
      # The opmode for a vdom can be 'transparent' (L2) or 'nat' (L3)
      self.opmodeDict = {}
      self.baseUrl = f'{protocol}://{self.ipAddr}:{portNum}/jsonrpc'
      t2('FortiManager plugin initializing, API baseUrl:', self.baseUrl,
         'ADOM:', self.adom )
      self.requestHeaders = { 'content-Type': 'application/json',
                              'accept':       'application/json' }
      self.deviceInfo = {}
      self.fmgrApi = None
      self.sessionId = ''
      self.sslProfileName = config[ 'sslProfileName' ]
      self.trustedCertsPath = config.get( 'trustedCertsPath', '' )
      self.setupSession()

   def setupSession( self ):
      self.fmgrApi = requests.session()  # use HTTP 1.1 persistent TCP connection
      self.fmgrApi.mount( 'http://', HTTPAdapter( max_retries=self.retries ) )
      self.fmgrApi.mount( 'https://', HTTPAdapter( max_retries=self.retries ) )
      self.fmgrApi.verify = self.config[ 'verifyCertificate' ]  # also see getUrl

   def closeApiConnection( self ):
      if self.sessionId:
         t4('closing API connection for sessionId:', self.sessionId )
         self.getUrl( 'exec', '/sys/logout' )
         self.fmgrApi.close()
         self.sessionId = ''

   def resetApiConnection( self ):
      t4('resetting API connection to FortiMgr:', self.ipAddr )
      self.closeApiConnection()
      time.sleep( 1 )
      self.setupSession()
      self.loginAsNeededAndConfirm()

   def loginAsNeededAndConfirm( self ):
      """ Login to FortiManager Server.  Return True on success.
      """
      if self.sessionId:
         return True
      credentials = { 'user': self.username,
                      'passwd': self.password.getClearText() }
      #t2('login to FortiManager with:', credentials )
      resp = self.getUrl( 'exec', '/sys/login/user', loginRequest=True,
                          addParams={ 'data': credentials } )
      if not resp or 'session' not in resp:
         t4( '%s FortiManager API login failed, check device address, '
              'username, password and network availability' % self.ipAddr )
         return False
      else:
         self.sessionId = resp[ 'session' ]
         t4('login successful, sessionId:', self.sessionId )
         return True

   def getUrl( self, method, urlParam, addParams=None, loginRequest=False ):
      """ Make REST API calls to a FortiManager
          method: get, add, set, update, delete, move, clone, replace, and execute
      """
      if not loginRequest and not self.loginAsNeededAndConfirm():
         raise ServiceDeviceError( 'login failed' )
      params = { 'url': urlParam }
      if addParams:
         params.update( addParams )
      requestBody = { 'method': method, 'params': [ params ], 'id': 1, 'skip' : 0,
                      'session': self.sessionId, 'verbose': 1, }  #'jsonrpc': '2.0' }
      url = self.baseUrl
      if loginRequest:
         t4( 'API REQ LOGIN' )
      else:
         t4('API REQ URL', url, 'BODY:', requestBody )
      if self.sslProfileName and not self.trustedCertsPath:
         raise ServiceDeviceError( Lib.SSL_ERROR_MSG )

      connectionFailed = False
      jsonParseFailed = False
      statusFailed = False
      for attempt in range( 1, self.retries + 2 ):
         resp = None
         respJson = None

         # Query firewall's API
         try:
            resp = self.fmgrApi.post(
               url, data=json.dumps( requestBody ), headers=self.requestHeaders,
               verify=( self.trustedCertsPath if self.sslProfileName else False ),
               timeout=self.timeout )
            connectionFailed = False
         except requests.exceptions.SSLError:
            # pylint: disable-next=raise-missing-from
            raise ServiceDeviceError( Lib.SSL_ERROR_MSG )
         except Exception as ex:  # pylint: disable=W0703
            if loginRequest:
               t4( '{} FortiManager API login access attempt {}, {}'.format(
                    self.ipAddr, attempt, type( ex ) ) )
            else:
               t4( '{} FortiManager API access attempt {}, {}'.format(
                    self.ipAddr, attempt, ex ) )
            connectionFailed = True
            continue # retry

         # Get json response
         try:
            respJson = resp.json()
            jsonParseFailed = False
         except ValueError as err:
            t4( 'API RESP: json decode error ', err )
            jsonParseFailed = True
            continue # retry

         # Get response status
         t4('API RESP', resp.status_code,
             json.dumps( respJson, indent=3, sort_keys=1 ))
         # pylint: disable-next=no-else-break
         if ( resp.status_code == requests.codes.ok  # pylint: disable=E1101
              and respJson[ 'result' ][ 0 ][ 'status' ][ 'code' ] == 0 ):
            statusFailed = False
            break # success
         else:
            statusFailed = True

      if connectionFailed: # pylint: disable=no-else-raise
         # Connection failed after max retries
         raise ServiceDeviceError( 'Connection error' )
      elif jsonParseFailed:
         raise FirewallAPIError( resp.status_code, '' )
      elif statusFailed and not loginRequest:
         raise FirewallAPIError( resp.status_code,
                                 respJson[ 'result' ][ 0 ][ 'status' ][ 'code' ] )

      return respJson if loginRequest else respJson[ 'result' ][ 0 ]

   def getDeviceInfo( self, cachedOk=True ):
      if cachedOk and self.deviceInfo:
         t4('get device info, cached')
         return self.deviceInfo

      resp = self.getUrl( 'get', '/sys/status' )
      if not resp:
         return {}

      devInfo = {}
      info = resp[ 'data' ]
      devInfo[ 'model' ] = info[ 'Platform Full Name' ]
      devInfo[ 'name' ] = info[ 'Hostname' ]
      t2( info[ 'Hostname' ], info[ 'Platform Full Name' ], 'version:',
          info[ 'Version' ] )

      resp = self.getUrl( 'get', '/cli/global/system/interface' )
      if not resp:
         devInfo[ 'ipAddr' ] = devInfo[ 'name' ]  # best guess dns name
         return devInfo

      devInfo[ 'ipAddr' ] = resp[ 'data' ][ 0 ][ 'ip' ][ 0 ]
      self.deviceInfo = devInfo  # update cache
      return devInfo

   def getGroupMembers( self, groupName ):
      resp = self.getUrl(
         'get', f'/dvmdb/adom/{self.adom}/group/{groupName}',
         { 'expand member': [
            { 'url': '/device', 'fields': [ 'name', 'ip' ] } ] } )
      if not resp or 'data' not in resp or 'expand member' not in resp[ 'data' ]:
         return []

      members = []
      for device in resp[ 'data' ][ 'expand member' ][ '/device' ]:
         members.append( device[ 'name' ] )
         if 'vdom' in device:
            for vdom in device[ 'vdom' ]:
               self.opmodeDict[ vdom[ 'name' ] ] = vdom[ 'opmode' ]
      return members

   def getInterfacesVirtualWires( self ):
      resp = self.getUrl(
         'get', '/pm/config/adom/%s/obj/system/virtual-wire-pair' % self.adom )
      if not resp:
         return {}

      intfs = {}
      for vw in resp[ 'data' ]:
         vwireName = vw[ 'name' ]
         vlans = [ DEFAULT_VLAN_RANGE if vw[ 'wildcard-vlan' ] == 'enable' else '0' ]
         for intf in vw[ 'member' ]:
            intfs[ intf ] = { 'vwire': vwireName, 'allowedVlans': vlans }
      return intfs

   def getVirtualWires( self ):
      resp = self.getUrl(
         'get', '/pm/config/adom/%s/obj/system/virtual-wire-pair' % self.adom )
      if not resp:
         return {}

      vwires = {}
      for vw in resp[ 'data' ]:
         name = vw[ 'name' ]
         # pylint: disable-next=unnecessary-comprehension
         intfs = [ intf for intf in vw[ 'member' ] ]
         vlans = [ DEFAULT_VLAN_RANGE if vw[ 'wildcard-vlan' ] == 'enable' else '0' ]
         vwires[ name ] = { 'intfs': intfs, 'allowedVlans': vlans }
      return vwires


####################################################################################
class FortiGate:
   ''' Represents a FortiGate firewall
       Uses parent FortiManager proxy API to access firewall objects
   '''

   def __init__( self, config, deviceName ):
      self.config = config
      self.deviceName = deviceName
      self.ipAddr = '?'
      self.adom = config[ 'adminDomain' ]
      # If no 'virtual instance' CLI config was provided, we will pull data only
      # from the default VDOM configured from 'virtualDomain' in legacy code.
      self.vdom = config[ 'virtualInstance' ] or [ config[ 'virtualDomain' ] ]
      self.vrouters = config.get( 'vrouters', [ '0' ] )
      self.mgmtIntfVdom = config[ 'mgmtIntfVdom' ]
      self.deviceInfo = {}
      self.parentUrl = '/sys/proxy/json'
      t2('FortiGate:', deviceName, 'ADOM:', self.adom, 'VDOM:', self.vdom,
         'mgmtIntfVdom:', self.mgmtIntfVdom, 'initializing parent FMgr proxy API' )
      self.parentFortiManager = FortiManager( config )
      self.parentFortiManager.getGroupMembers( config[ 'group' ] )
      # Only LAYER3 device mode is supported.
      self.deviceMode = Lib.LAYER3

   def closeApiConnection( self ):
      self.parentFortiManager.closeApiConnection()

   def resetApiConnection( self ):
      t2('reset API connection deviceInfo:', self.deviceInfo )
      self.parentFortiManager.resetApiConnection()
      self.getDeviceInfo( cachedOk=False )
      t2('done resetting conn. deviceInfo:', self.deviceInfo )

   def getUrl( self, resource, resultsOnly=True ):
      params = {
         'data': {
            'action': 'get',
            'resource': resource,
            'target': [ f'adom/{self.adom}/device/{self.deviceName}' ] } }
      resp = self.parentFortiManager.getUrl( 'exec', self.parentUrl, params )
      if resp[ 'data' ][ 0 ][ 'status' ][ 'code' ] != 0:
         warnMsg = '%s FortiGate API url %s returned status %s target %s' % \
                   ( self.deviceName, resource, resp[ 'data' ][ 0 ][ 'status' ],
                     params[ 'data' ][ 'target' ] )
         t0( warnMsg )
         raise FirewallAPIError( requests.codes.ok,
                                 resp[ 'data' ][ 0 ][ 'status' ][ 'code' ] )
      if resultsOnly:
         return resp[ 'data' ][ 0 ][ 'response' ][ 'results' ]
      else:
         return resp[ 'data' ][ 0 ][ 'response' ]

   def getDeviceInfo( self, cachedOk=True ):
      if cachedOk and self.deviceInfo:
         t4('get device info, cached')
         return self.deviceInfo
      resp = self.getUrl( '/api/v2/monitor/web-ui/state', resultsOnly=False )
      if not resp or 'results' not in resp:
         return {}
      devInfo = {
         'model': '{} {}'.format( resp[ 'results' ][ 'model_name' ],
                              resp[ 'results' ][ 'model_number' ] ),
         'name': resp[ 'results' ][ 'hostname' ],
         'serialNum': resp[ 'serial' ],
         'ipAddr': '' }
      t0( resp[ 'results' ][ 'hostname' ], resp[ 'results' ][ 'model_name' ],
          resp[ 'results' ][ 'model_number' ], 'version:', resp[ 'version' ] )
      self.deviceInfo = devInfo
      resp = self.getUrl( '/api/v2/monitor/system/interface?vdom=%s' %
                          self.mgmtIntfVdom )
      if not resp:
         return devInfo
      for mgmtIntf in [ 'mgmt', 'mgmt1', 'mgmt2' ]:  # search intfs in this order
         if ( mgmtIntf in resp and 'ip' in resp[ mgmtIntf ] and
              resp[ mgmtIntf ][ 'ip' ] and resp[ mgmtIntf ][ 'link' ] ):
            ip = resp[ mgmtIntf ][ 'ip' ]
            devInfo[ 'ipAddr' ] = ip
            self.ipAddr = ip
      return devInfo

   def getHighAvailabilityState( self ):
      haState = ServiceDeviceHAState()
      haState.mgmtIp = self.ipAddr
      resp = self.getUrl( '/api/v2/cmdb/system/ha' )
      if not resp:
         return haState
      mode = resp[ 'mode' ]
      haState.enabled = mode != 'standalone'
      haState.mode = HA_STRING_MAP[ mode ] if mode in HA_STRING_MAP else ''
      if not haState.enabled:
         return haState

      resp = self.getUrl( '/api/v2/monitor/system/ha-checksums', resultsOnly=False )
      if not resp or 'results' not in resp:
         return haState
      for haDevice in resp[ 'results' ]:
         state = HA_ACTIVE if haDevice[ 'is_root_master' ] else HA_PASSIVE
         serialNum = haDevice[ 'serial_no' ]
         t2('FortiGate serialNumber:', serialNum, 'haState:', state )
         if serialNum == resp[ 'serial' ]:
            haState.state = state

      if ( 'serialNum' in self.deviceInfo and
           self.deviceInfo[ 'serialNum' ] != resp[ 'serial' ] ):
         t2('possible HA master change, updating deviceInfo, old:', self.deviceInfo )
         self.getDeviceInfo( cachedOk=False )
         t2('new deviceInfo:', self.deviceInfo )
      return haState

   def _updateServiceMapWithServices( self, respJson, vdom, svcMap ):
      protocolToPort = {
          'TCP'   : 'tcp-portrange',
          'UDP'   : 'udp-portrange',
          'SCTP'  : 'sctp-portrange',
          'IP'    : 'protocol-number',
          'ICMP'  : 'icmptype',
          'ICMP6' : 'icmptype',
          'ALL'   : ''
          }
      for serviceJson in respJson:
         serviceStr = serviceJson[ 'name' ]
         svcMap[ vdom ][ serviceStr ] = {}
         protocols = serviceJson[ 'protocol' ]
         for protocol in protocols.split( '/' ) :
            protocolStr = protocol
            try:
               portIndex = protocolToPort[ protocolStr ]
               if portIndex:
                  portsStr = str( serviceJson[ portIndex ] )
                  svcMap[ vdom ][ serviceStr ][ protocolStr ] = portsStr.split()
            except KeyError:
               t2( 'Unsupported Protocol:{} in Service:{}'.format(
                   protocolStr, serviceStr ) )

   def _updateServiceMapWithServiceGroups( self, respJson, vdom, svcMap ):
      for serviceGroupJson in respJson:
         # Build a dictionary of { protocol, [ ports ] }
         # corresponding to all services in the service group
         serviceDict = {}
         serviceGroup = serviceGroupJson[ 'name' ]
         for member in serviceGroupJson[ 'member' ]:
            service = member[ 'name' ]
            if service in svcMap[ vdom ]:
               for protocol, ports in svcMap[ vdom ][ service ].items():
                  serviceDict.setdefault( protocol, [] ).extend( ports )
            else:
               t4( 'Service in ServiceGroup is not yet read from firewall' )

         # Add the built dictionary to svcMap
         svcMap[ vdom ][ serviceGroup ] = serviceDict

   def getServiceProtocolPortMap( self ):
      svcMap = { vdom : {} for vdom in self.vdom }
      for vdom in self.vdom:
         # Update service map with services
         resp = self.getUrl(
            '/api/v2/cmdb/firewall.service/custom?vdom=%s' % vdom )
         if not resp:
            continue
         self._updateServiceMapWithServices( resp, vdom, svcMap )

         # Update service map with service groups
         resp = self.getUrl(
            '/api/v2/cmdb/firewall.service/group?vdom=%s' % vdom )
         if not resp:
            continue
         self._updateServiceMapWithServiceGroups( resp, vdom, svcMap )

      t4( 'L4 services map:', svcMap )
      return svcMap

   def resolveL4Ports( self, policies ):
      serviceProtocolPortMap = self.getServiceProtocolPortMap()

      for vdom, policyList in policies.items():
         for policy in policyList:
            l4Services = {}
            for service in policy.services:
               if service == 'application-default':
                  continue
               if service in serviceProtocolPortMap[ vdom ]:
                  t6( 'serviceProtocolPortMap:', serviceProtocolPortMap[ vdom ] )
                  l4Services[ service ] = deepcopy(
                        serviceProtocolPortMap[ vdom ][ service ] )
            t6( 'l4Services after processing policy.services:', l4Services )

            # aggregate TCP ports, UDP ports write policy.dstL4Services:
            for protocolPorts in l4Services.values():
               for protocol, ports in protocolPorts.items():
                  if protocol == 'IP':
                     protocol = 'IPv4'
                  if protocol not in Lib.IP_PROTOCOL_TAC_VALUE:
                     t0( 'Unknown protocol: %s' % protocol )
                     continue
                  if protocol not in policy.dstL4Services:
                     policy.dstL4Services[ protocol ] = ports
                  else:
                     policy.dstL4Services[ protocol ].extend( ports )
            t4( 'policy:', policy.name, 'apps:', policy.applications, 'services:',
                policy.services, 'L4Ports:', policy.dstL4Services )

   def getPolicies( self, mssTags=None ):
      # alt: /api/v2/monitor/firewall/policy?vdom=
      mssTags = set( mssTags )
      intfInfo = self.getInterfacesInfo()
      policies = { vdom : [] for vdom in self.vdom }

      for vdom in self.vdom:
         addressObjects = self.getAddressObjects( vdom )
         resp = self.getUrl(
            '/api/v2/cmdb/firewall/policy?vdom=%s&filter=status==enable'
            '&filter=comments=@tags(,comments=@tags['
            '&format=name|comments|srcaddr|dstaddr|srcintf|dstintf|action|status'
            '|service|logtraffic'
            % vdom )

         if not resp:
            continue

         policyOrder = 1
         for polJson in resp:
            match = POLICY_TAGS_REGEX.search( polJson[ 'comments' ] )
            if not match:
               continue
            tags = [ t.strip() for t in match.group( 1 ).split( ',' ) ]
            tags = { t for t in tags if t } # filter empty strings, make set
            t4( polJson[ 'name' ], 'extracted policy tags:', tags )
            if tags.isdisjoint( mssTags ):
               continue
            policy = ServiceDevicePolicy( polJson[ 'name' ],
                                          managementIp=self.ipAddr,
                                          number=policyOrder )
            t4( 'Processed Policy:', polJson[ 'name' ],
                'policyOrder:', policyOrder )

            policy.logSessionStart = polJson[ 'logtraffic' ] == 'all'

            policyOrder += 1
            policy.tags = list( tags )
            policy.action = polJson[ 'action' ]
            policy.srcZoneName, policy.srcZoneInterfaces, \
               policy.srcIpAddrList, policy.srcZoneType = \
               getZoneInfo( polJson, intfInfo[ vdom ], addressObjects,
                            'srcintf', 'srcaddr', self.deviceMode )
            policy.dstZoneName, policy.dstZoneInterfaces, \
               policy.dstIpAddrList, policy.dstZoneType = \
               getZoneInfo( polJson, intfInfo[ vdom ], addressObjects,
                            'dstintf', 'dstaddr', self.deviceMode )
            policy.services = [ service[ 'name' ]
                                for service in polJson[ 'service' ] ]
            policy.applications = []
            policies[ vdom ].append( policy )
      self.resolveL4Ports( policies )
      return policies

   def getInterfacesInfo( self ):
      ''' Get all necessary interface information for service devices.
          Returns a list of NetworkInterface objects.
      '''
      intfVwires = self.parentFortiManager.getInterfacesVirtualWires()
      interfaces = { vdom : {} for vdom in self.vdom }

      for vdom in self.vdom:
         resp = self.getUrl( '/api/v2/monitor/system/available-interfaces'
                             '?vdom=%s&view_type=zone' % vdom )
         if not resp:
            continue

         for intf in resp:
            if 'name' not in intf or 'link' not in intf:
               continue
            intfName = intf[ 'name' ]
            linkState = translateIntfState( intf[ 'link' ], intfName )
            if intf[ 'type' ] == 'physical':
               netIntf = NetworkInterface( intfName, state=linkState,
                                           isEthernet=True )
            elif intf[ 'type' ] == 'vlan':
               vlan = intf[ 'vlan_id' ]
               netIntf = NetworkInterface( intfName, state=linkState, isSubIntf=True,
                                           vlans=[ vlan ] )
               physIntfName = intf[ 'vlan_interface' ]
               netIntf.addPhysicalIntf( physIntfName, linkState )
            elif intf[ 'type' ] == 'aggregate':
               netIntf = NetworkInterface( intfName, state=linkState, isLag=True )
               for physIntfName in intf[ 'members' ]:
                  netIntf.addPhysicalIntf( physIntfName )
            else:
               continue  # unsupported intf type

            interfaces[ vdom ][ intfName ] = netIntf
            if 'virtual_wire_pair' in intf:
               vwireName = intf[ 'virtual_wire_pair' ]
               netIntf.attribs[ 'vwire' ] = vwireName
               netIntf.zone = f'{vwireName}_{intfName}'
               if intfName not in intfVwires:
                  t2('intf:', intfName, 'not found in:', intfVwires )
                  continue
               if 'allowedVlans' in intfVwires[ intfName ]:
                  netIntf.vlans = intfVwires[ intfName ][ 'allowedVlans' ]
               if ( 'vwire' in intfVwires[ intfName ] and
                     intfVwires[ intfName ][ 'vwire' ] != vwireName ):
                  t2('vwire names do not match:', vwireName,
                     intfVwires[ intfName ][ 'vwire' ] )
            elif 'is_routable' in intf and intf[ 'is_routable' ] and \
                 'ipv4_addresses' in intf:
               netIntf.zone = f'L3_{intfName}'
               netIntf.ipAddr = intf[ 'ipv4_addresses' ][ 0 ][ 'ip' ]
               # Fortinet supports only VRF 0 at this time
               netIntf.vrf = '0'

      # now set link state for LAG member intfs
      for vdom, intfs in interfaces.items():
         for netIntf in intfs.values():
            if netIntf.isLag:
               for lagMemberIntf in netIntf.physicalIntfs:
                  if lagMemberIntf.name in intfs:
                     lagMemberIntf.state = intfs[ lagMemberIntf.name ].state

      return interfaces

   def getInterfaceNeighbors( self ):
      neighbors = {}
      try:
         resp = self.getUrl( '/api/v2/monitor/network/lldprx/neighbors?vdom=%s'
                             % self.vdom[ 0 ] )
         if not resp:
            return {}
         intfIndexMap = self.getInterfaceIndexMap()
         for nbr in resp:
            fwIntfIndex = nbr[ 'port' ]
            if fwIntfIndex in intfIndexMap:
               chassis = nbr[ 'chassis_id' ] if nbr[ 'chassis_id' ] else ''
               intf = nbr[ 'port_id' ] if nbr[ 'port_id' ] else ''
               desc = nbr[ 'system_desc' ] if nbr[ 'system_desc' ] else ''
               sysName = nbr[ 'system_name' ] if nbr[ 'system_name' ] else ''
               neighbors[ intfIndexMap[ fwIntfIndex ] ] = {
                  'switchChassisId': chassis, 'switchIntf': intf, 'nborDesc': desc,
                  'nborSysName': sysName }
      except Exception:  # pylint: disable=W0703
         t2('unable to get LLDP data, some FortiOS versions do not support LLDP' )
      t4('LLDP neighbors:', neighbors )
      return neighbors

   def getInterfaceIndexMap( self ):
      intfIndexMap = {}
      resp = self.getUrl(
         '/api/v2/cmdb/system/interface?format=name|devindex&vdom=%s' %
         self.vdom[ 0 ] )
      if not resp:
         return {}
      for intf in resp:
         intfIndexMap[ intf[ 'devindex' ] ] = intf[ 'name' ]
      return intfIndexMap

   def getDeviceResources( self ):
      # URL below is used by FortiGate webUI for system resources view
      # The first cpu entry is average utilization over last 5 minutes (per Fortinet)
      resp = self.getUrl(
         '/api/v2/monitor/system/resource/usage?scope=global&interval=1-min' )
      if not resp:
         return {}
      pfmt = '%-20s %3s%%\n'
      fmt =  '%-20s %3s\n'
      info =  pfmt % ( 'CPU Utilization:', resp[ 'cpu' ][ 0 ][ 'current' ] )
      info += pfmt % ( 'Memory Utilization:', resp[ 'mem' ][ 0 ][ 'current' ] )
      info += pfmt % ( 'Disk Utilization: ', resp[ 'disk' ][ 0 ][ 'current' ] )
      info += fmt % ( 'Number of Sessions:', resp[ 'session' ][ 0 ][ 'current' ] )
      info += '\n'
      return { 'resourceInfo' : info }

   def getDeviceRoutingTables( self ):
      ''' Returns a ServiceDeviceRoutingTables object
      '''
      routingTables = ServiceDeviceRoutingTables()
      for vdom in self.vdom:
         resp = self.getUrl( '/api/v2/monitor/router/ipv4?vdom=%s&type=3' % vdom )
         for route in resp:
            # Fortinet supports only VRF 0 at this time
            vrfName = '0'
            destination = route[ 'ip_mask' ].replace( "\\", "" )
            interface = route[ 'interface' ]
            nexthop = route[ 'gateway' ]
            routingTables.addRoute( vrfName, destination, interface, nexthop )
      routingTables.featureSupported = True
      return routingTables

   def getAddressObjects( self, vdom ):
      resp = self.getUrl( '/api/v2/cmdb/firewall/address?vdom=%s'
                          '&format=name|type|ipmask|start-ip|end-ip|subnet' % vdom )
      if not resp:
         return {}

      objs = {}
      for addr in resp:
         if 'start-ip' in addr and addr[ 'start-ip' ] == '0.0.0.0':
            continue
         if addr[ 'type' ] == 'ipmask':
            if 'subnet' in addr:
               # SW version >= 6.2.0
               ip, mask = addr[ 'subnet' ].split()
               if ip == '0.0.0.0':
                  continue
               if mask == '255.255.255.255':
                  objs[ addr[ 'name' ] ] = ip
               else:
                  objs[ addr[ 'name' ] ] = f'{ip}|{mask}'
            elif 'end-ip' in addr and 'start-ip' in addr:
               # SW version < 6.2.0
               if addr[ 'end-ip' ] == '255.255.255.255':
                  objs[ addr[ 'name' ] ] = addr[ 'start-ip' ]
               else:
                  objs[ addr[ 'name' ] ] = '{}|{}'.format(
                     addr[ 'start-ip' ], addr[ 'end-ip' ] )
            else:
               t0( 'Invalid address reply: %s' % addr )
         elif addr[ 'type' ] == 'iprange':
            objs[ addr[ 'name' ] ] = '{}-{}'.format( addr[ 'start-ip' ],
                                                     addr[ 'end-ip' ] )
      resp = self.getUrl( '/api/v2/cmdb/firewall/addrgrp?vdom=%s' % vdom )
      if not resp:
         return objs

      for agp in resp:
         groupName = agp[ 'name' ]
         addrList = []
         for member in agp[ 'member' ]:
            memberName = member[ 'name' ]
            if memberName in objs:
               addrList.append( objs[ memberName ] )
         objs[ groupName ] = addrList
      return objs

# helper functions
def translateIntfState( state, intfName ):
   ''' Translate FortiOS intf state to MSS intf state
   '''
   if state.lower() == 'up':
      return LINK_STATE_UP
   elif state.lower() == 'down':
      return LINK_STATE_DOWN
   else:
      t4('intf:', intfName, 'unrecognized link state value:', state )
      return LINK_STATE_UNKNOWN

def getZoneInfo( polJson, intfInfo, addressObjects, intfKey, addrKey, opmode ):
   zoneName = ''
   zoneType = opmode
   intfs = []
   addrs = []
   for intfName in [ i[ 'name' ] for i in polJson[ intfKey ] ]:
      if intfName == 'any':
         zoneName = 'any'
         zoneType = '' # reset zone type for "any" interface
      elif intfName in intfInfo:
         intfs.append( intfInfo[ intfName ] )
         zoneName = intfInfo[ intfName ].zone
         if zoneName.startswith( 'L3' ):
            zoneType = Lib.LAYER3

   for addr in [ a[ 'name' ] for a in polJson[ addrKey ] ]:
      if addr in addressObjects:
         if addr == 'all': # Not supported
            continue
         Lib.appendOrExtend( addrs, addressObjects[ addr ] )
   t6( 'zone: %s, intfs: %s, addrs: %s, zoneType: %s' %
         ( zoneName, intfs, addrs, zoneType ) )
   return zoneName, intfs, addrs, zoneType


####################################################################################
# test cases
def testRetiresAndTimeouts( deviceDict ):
   print( '\n\nTEST BAD FMGR ADDR' )
   orig = deviceDict[ 'ipAddress' ]
   deviceDict[ 'ipAddress' ] = 'BAD_FMGR_ADDR'
   testFMgrApi( deviceDict )
   testFGateApi( deviceDict, 'bizdev-fnet' )
   deviceDict[ 'ipAddress' ] = orig

   print( '\n\nTEST BAD FMGR PASSWORD' )
   orig = deviceDict[ 'password' ]
   deviceDict[ 'password' ] = 'BAD_PASSWORD'
   testFMgrApi( deviceDict )
   testFGateApi( deviceDict, 'bizdev-fnet' )
   deviceDict[ 'password' ] = orig

   print( '\n\nTEST BAD FGATE IP' )
   testFGateApi( deviceDict, 'BAD_FGATE_IP' )

   print( '\n\nTEST TIMEOUT' )
   orig = deviceDict[ 'timeout' ]
   deviceDict[ 'timeout' ] = 0.001
   testFMgrApi( deviceDict )
   testFGateApi( deviceDict, 'bizdev-fnet' )
   deviceDict[ 'timeout' ] = orig

   print( '\n\nTEST ALL GOOD' )
   testFMgrApi( deviceDict )
   testFGateApi( deviceDict, 'bizdev-fnet' )

def testFMgrApi( config ):
   print( '\nTest FortiManager API' )
   fm = FortiManager( config )

   info = fm.getDeviceInfo()
   print( 'DeviceInfo:', info )

   for group in [ 'MssFortinet', 'MssFortinetHA' ]:
      gm = fm.getGroupMembers( groupName=group )
      print( 'GroupMembers:', group, '=', gm )

   # vw = fm.getVirtualWires()
   # print '\nVirtual Wires: %s \n' % vw

   # intfVw = fm.getInterfacesVirtualWires()
   # print '\nIntf Vwires: %s \n' % intfVw

   fm.closeApiConnection()


def testFGateApi( config, deviceName ):
   print( '\nTest FortiGate API access via FortiManager' )
   fg = FortiGate( config, deviceName )

   info = fg.getDeviceInfo()
   print( deviceName, 'DeviceInfo:', info, '\n' )

   res = fg.getDeviceResources()
   print( '\nRESOURCES:\n', res[ 'resourceInfo' ] if res else '' )

   ha = fg.getHighAvailabilityState()
   print( 'HA State:', ha )

   intfs = fg.getInterfacesInfo()  # getPolicies also calls getInterfacesInfo
   print( '\nIntfInfo:' )
   for vdom, intfList in intfs.items():
      print( f"{vdom}: {intfList}" )

   nbors = fg.getInterfaceNeighbors()
   print( 'LLDP Neighbors:' )
   for p, n in nbors.items():
      print( p, n )

   pols = fg.getPolicies( mssTags=[ 'Arista_MSS', 'mss1', 'mss2' ] )
   print( '\nPolicies:' )
   for vdom, p in pols.items():
      print( f'{vdom}: {p}', '\n' )

   # ado = fg.getAddressObjects()
   # print 'AddressObjects:', ado

   fg.closeApiConnection()

if __name__ == "__main__":
   cfg = {
      'ipAddress': 'mss-fortimgr-v56', 'username': 'admin',
      'password': ReversibleSecretCli.generateSecretEntity( 'Arastra123!' ),
      'protocol': 'https', 'protocolPortNum': 443,
      'method': 'tls', 'verifyCertificate': False, 'timeout': 5, 'retries': 1,
      'exceptionMode': 'bypass', 'group': 'MssFortinet', 'mgmtIntfVdom': 'root',
      'adminDomain': 'root', 'virtualDomain': 'L2_Firewall', 'sslProfileName' : '',
      'interfaceMap': {
         'port17': {
            'switchIntf': 'Port-Channel70', 'switchChassisId': '001c.737e.2811' },
         'port18': {
            'switchIntf': 'Port-Channel70', 'switchChassisId': '001c.737e.2811' },
         'port19': {
            'switchIntf': 'Port-Channel75', 'switchChassisId': '001c.737e.2811' },
         'port20': {
            'switchIntf': 'Port-Channel75', 'switchChassisId': '001c.737e.2811' },
         'port29': {
            'switchIntf': 'Ethernet39', 'switchChassisId': '001c.737e.2811' },
         'port30' : {
            'switchIntf' : 'Ethernet40', 'switchChassisId' : '001c.737e.2811' },
         'port31' : {
            'switchIntf' : 'Ethernet41', 'switchChassisId' : '001c.737e.2811' },
         'port32' : {
            'switchIntf' : 'Ethernet42', 'switchChassisId' : '001c.737e.2811' }, },
      'virtualInstance' : [ 'fwfortvd6', 'fwfortvd7', 'fwfortvd8' ] }

   testFMgrApi( cfg )
   testFGateApi( cfg, 'fwfort102' )
