#!/usr/bin/env python3
#
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
#
# pylint: disable=W0231


import sys
from MssPolicyMonitor.Plugin import FortinetApi
from MssPolicyMonitor.PluginLib import ( IServiceDevicePlugin, IAggregationMgrPlugin,
                                         IHAStatePlugin, IPolicyPlugin )
from MssPolicyMonitor.Error import ServiceDeviceError
from MssPolicyMonitor.Lib import registerPlugin, FORTIMGR_PLUGIN, t0


registerPlugin( FORTIMGR_PLUGIN, sys.modules[ __name__ ] )


def getAggMgrPluginObj( deviceConfig ):
   return FortiManagerPlugin( deviceConfig )


def getPluginObj( deviceConfig, aggMgrMemberId=None ):
   return FortiGatePlugin( deviceConfig, aggMgrMemberId )


####################################################################################
class FortiManagerPlugin( IAggregationMgrPlugin ):
   ''' MSS Service Policy Monitor Plugin for Fortinet FortiManager
   '''
   def __init__( self, deviceConfig ):
      self.deviceApi = FortinetApi.FortiManager( deviceConfig )
      self.deviceConfig = deviceConfig

   def getDeviceInfo( self ):
      ''' Return dict with at least these keys: 'ipAddr', 'name', 'model'
      '''
      return self.deviceApi.getDeviceInfo()

   def getAggMgrGroupMembers( self, groupName=None  ):
      ''' Return a list of firewalls accessible from this FortiManager
      '''
      if not groupName and 'group' in self.deviceConfig:
         groupName = self.deviceConfig[ 'group' ]
      return self.deviceApi.getGroupMembers( groupName )

   def closeApiConnection( self ):
      ''' Close any open connections to the service device
      '''
      return self.deviceApi.closeApiConnection()


####################################################################################
class FortiGatePlugin( IServiceDevicePlugin, IHAStatePlugin, IPolicyPlugin ):
   ''' MSS Service Policy Monitor Plugin for Fortinet FortiGate firewalls
   '''
   def __init__( self, deviceConfig, deviceName ):
      self.deviceName = deviceName
      self.deviceConfig = deviceConfig
      self.deviceApi = FortinetApi.FortiGate( deviceConfig, deviceName )

   def getDeviceInfo( self ):
      ''' Return dict with at least these keys: 'ipAddr', 'name', 'model'
      '''
      return self.deviceApi.getDeviceInfo()

   def getHighAvailabilityState( self ):
      ''' Returns a ServiceDeviceHAState object with current
          High Availability State for the service device.
      '''
      haState = self.deviceApi.getHighAvailabilityState()
      haState.isSingleLogicalDeviceHaModel = self.isSingleLogicalDeviceHaModel()
      return haState

   # pylint: disable-next=invalid-overridden-method
   def isSingleLogicalDeviceHaModel( self ):
      return True

   def getPolicies( self, mssTags=None ):
      ''' Returns a list of ServiceDevicePolicy objects
      '''
      return self.deviceApi.getPolicies( mssTags=mssTags )

   def getInterfacesInfo( self, resolveZoneNames=True ):
      ''' Returns a list of NetworkInterface objects
      '''
      return self.deviceApi.getInterfacesInfo()

   def getInterfaceNeighbors( self ):
      ''' Returns a dict of service device neighbor links
      '''
      return self.deviceApi.getInterfaceNeighbors()

   def getDeviceResources( self ):
      ''' Returns a dict with device resource info
      '''
      return self.deviceApi.getDeviceResources()

   def getDeviceRoutingTables( self ):
      ''' Returns a ServiceDeviceRoutingTables object
      '''
      return self.deviceApi.getDeviceRoutingTables()

   def closeApiConnection( self ):
      ''' Close any open connections to the service device
      '''
      return self.deviceApi.closeApiConnection()

#-----------------------------------------------------------------------------------
# tests
def test():
   SW1 = '001c.737e.2811'
   deviceDict = {
      'ipAddress': 'bizdev-fortimgr', 'username': 'admin', 'password': 'arista',
      'protocol': 'https', 'protocolPortNum': 443, 'method': 'tls',
      'verifyCertificate': False, 'timeout': 15, 'retries': 1,
      'exceptionMode': 'bypass', 'group': 'mssGroup',
      'adminDomain': 'root', 'virtualDomain': 'L2_Firewall', 'mgmtIntfVdom': 'root',
      'interfaceMap': {
         'port17': { 'switchIntf': 'Port-Channel70', 'switchChassisId': SW1 },
         'port18': { 'switchIntf': 'Port-Channel70', 'switchChassisId': SW1 },
         'port19': { 'switchIntf': 'Port-Channel75', 'switchChassisId': SW1 },
         'port20': { 'switchIntf': 'Port-Channel75', 'switchChassisId': SW1 },
         'port29': { 'switchIntf': 'Ethernet39', 'switchChassisId': SW1 },
         'port30': { 'switchIntf': 'Ethernet40', 'switchChassisId': SW1 },
         'port31': { 'switchIntf': 'Ethernet41', 'switchChassisId': SW1 },
         'port32': { 'switchIntf': 'Ethernet42', 'switchChassisId': SW1 } }
      }

   print( '\nTEST FORTIMANAGER API CALLS' )
   fmgr = getAggMgrPluginObj( deviceDict )

   info = fmgr.getDeviceInfo()
   print( 'FortiManager DeviceInfo:', info, '\n' )

   aggMembers = fmgr.getAggMgrGroupMembers( 'mssGroup' )
   print( 'FortiManager AggMgrGroupMembers:', aggMembers )
   firewall = aggMembers[ 0 ]

   fmgr.closeApiConnection()

   print( '\nTEST FORTIGATE API CALLS for:', firewall )
   fw = getPluginObj( deviceDict, firewall )

   info = fw.getDeviceInfo()
   print( 'FortiGate DeviceInfo:', info, '\n' )

   ha = fw.getHighAvailabilityState()
   print( '\n' +'HAState:', ha, 'isHaPassiveOrSecondary:',
          ha.isHaPassiveOrSecondary() )

   r = fw.getDeviceResources()
   print( 'Resources:\n', r[ 'resourceInfo' ] )

   intfs = fw.getInterfacesInfo()
   print( 'Interfaces:' )
   for intf in intfs:
      print( intf )

   nbors = fw.getInterfaceNeighbors()
   print( '\nLLDP_Neighbors:', nbors )

   pols = fw.getPolicies( mssTags=[ 'Arista_MSS', 'mss1', 'mss2' ] )
   print( '\nPolicies:' )
   for p in pols:
      print( p, '\n' )

   fw.closeApiConnection()


def testHA():
   import time # pylint: disable=import-outside-toplevel
   SW1 = '00:1c:73:7e:21:e1'
   SW2 = '00:1c:73:7e:28:11'
   cfg = {
      'ipAddress': 'bizdev-fortimgr', 'username': 'admin', 'password': 'arista',
      'protocol': 'https', 'protocolPortNum': 443, 'method': 'tls',
      'verifyCertificate': False, 'timeout': 5, 'retries': 1,
      'exceptionMode': 'bypass', 'group': 'mssHA',
      'adminDomain': 'root', 'virtualDomain': 'L2_FW', 'mgmtIntfVdom': 'root',
      'interfaceMap': {
         'port13': { 'switchIntf': 'Port-Channel60', 'switchChassisId': SW1 },
         'port14': { 'switchIntf': 'Port-Channel60', 'switchChassisId': SW2 },
         'port15': { 'switchIntf': 'Port-Channel65', 'switchChassisId': SW1 },
         'port16': { 'switchIntf': 'Port-Channel65', 'switchChassisId': SW2 } }
      }

   print( '\nTEST FORTIMANAGER API CALLS' )
   fmgr = getAggMgrPluginObj( cfg )
   info = fmgr.getDeviceInfo()
   print( 'FortiManager DeviceInfo:', info, '\n' )
   aggMembers = fmgr.getAggMgrGroupMembers()
   print( 'FortiManager AggMgrGroupMembers:', aggMembers )
   fmgr.closeApiConnection()

   firewall = aggMembers[ 0 ]
   print( '\nTEST FORTIGATE API CALLS for:', firewall )
   fw = getPluginObj( cfg, firewall )
   info = fw.getDeviceInfo()
   print( 'FortiGate DeviceInfo:', info, '\n' )
   for _ in range( 300 ):
      try:
         ha = fw.getHighAvailabilityState()
         t0( 'HAState:', ha )
      except ServiceDeviceError:
         t0('ignoring ServiceDeviceError exception')
      time.sleep( 2 )
   fw.closeApiConnection()


if __name__ == "__main__":
   # test()
   testHA()
