#!/usr/bin/env python3
#
# Copyright (c) 2015-2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
#
# pylint: disable-msg=W0231

import sys
from MssPolicyMonitor.Plugin import PaloAltoApi
from MssPolicyMonitor.PluginLib import ( IServiceDevicePlugin, IAggregationMgrPlugin,
                                         IHAStatePlugin, IPolicyPlugin )
from MssPolicyMonitor.Lib import registerPlugin, PAN_FW_PLUGIN, PANORAMA_PLUGIN


registerPlugin( PAN_FW_PLUGIN, sys.modules[ __name__ ] )
registerPlugin( PANORAMA_PLUGIN, sys.modules[ __name__ ] )


def getAggMgrPluginObj( deviceConfig ):
   return PanoramaPlugin( deviceConfig )


def getPluginObj( deviceConfig, aggMgrMemberId=None ):
   ''' Return a plugin object.
       If firewall is accessed via Panorama the aggMgrMemberId is the
       firewall's serial number.
   '''
   if aggMgrMemberId:
      return PanFirewallViaPanoramaPlugin( deviceConfig, aggMgrMemberId )
   else:
      return PanFirewallPlugin( deviceConfig )


####################################################################################
class PanoramaPlugin( IAggregationMgrPlugin, IHAStatePlugin ):
   ''' MSS Service Policy Monitor Plugin for Palo Alto Networks Panorama
   '''
   def __init__( self, deviceConfig ):
      self.deviceApi = PaloAltoApi.PanoramaMgmtServer( deviceConfig )
      self.deviceConfig = deviceConfig

   def getAggMgrGroupMembers( self, groupName ):
      ''' Return a list of device group members, from Panorama
      '''
      if not groupName and 'group' in self.deviceConfig:
         groupName = self.deviceConfig[ 'group' ]
      return self.deviceApi.getGroupMembers( groupName )

   def getDeviceInfo( self ):
      ''' Returns selected device info
      '''
      return self.deviceApi.getDeviceInfo()

   def getHighAvailabilityState( self ):
      ''' Returns a ServiceDeviceHAState object with current
          High Availability State for the service device.
      '''
      return self.deviceApi.getHighAvailabilityState()

   # pylint: disable-next=invalid-overridden-method
   def isSingleLogicalDeviceHaModel( self ):
      return False

   def closeApiConnection( self ):
      ''' Close any open connections to the service device
      '''
      return self.deviceApi.closeConnection()


####################################################################################
class PanFirewallPlugin( IHAStatePlugin, IPolicyPlugin, IServiceDevicePlugin ):
   ''' MSS Service Policy Monitor Plugin for Palo Alto Networks Firewalls
   '''
   def __init__( self, deviceConfig ):
      self.deviceApi = PaloAltoApi.PanDevice( deviceConfig )
      self.deviceConfig = deviceConfig

   def getDeviceInfo( self ):
      ''' Returns selected device info
      '''
      return self.deviceApi.getDeviceInfo()

   def getHighAvailabilityState( self ):
      ''' Returns a ServiceDeviceHAState object with current
          High Availability State for the service device.
      '''
      return self.deviceApi.getHighAvailabilityState()

   # pylint: disable-next=invalid-overridden-method
   def isSingleLogicalDeviceHaModel( self ):
      return False

   def getPolicies( self, mssTags=None ):
      ''' Returns a list of ServiceDevicePolicy objects
      '''
      return self.deviceApi.getPolicies( mssTags )

   def getInterfacesInfo( self, resolveZoneNames=True ):
      ''' Returns a list of NetworkInterface objects
      '''
      return self.deviceApi.getInterfacesInfo()

   def getInterfaceNeighbors( self ):
      ''' Returns a dict of service device neighbor links
      '''
      return self.deviceApi.getInterfaceNeighbors()

   def getDeviceResources( self ):
      ''' Returns a dict with device resource info
      '''
      return self.deviceApi.showSystemResources()

   def getDeviceRoutingTables( self ):
      ''' Returns a ServiceDeviceRoutingTables object
      '''
      return self.deviceApi.getDeviceRoutingTables()

   def closeApiConnection( self ):
      ''' Close any open connections to the service device
      '''
      return self.deviceApi.closeConnection()

# ---------------------------------------------------------------------------------
class PanFirewallViaPanoramaPlugin( PanFirewallPlugin ):
   ''' MssPolicyMonitorPlugin object for a device group member.
       Member identifier is a PAN firewall serial number.
   '''
   def __init__( self, deviceConfig, serialNum ):
      # do not call __init__() on superclass
      self.deviceApi = PaloAltoApi.FirewallViaPanorama( deviceConfig, serialNum )
      self.deviceConfig = deviceConfig
