#!/usr/bin/env python3
# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Cell
import Tac
import Tracing

t0 = Tracing.t0

class MultiSliceEthPhyIntfCreator:
   '''This SM reacts to slices getting added to

      interface/archer/status/init/eth/phy/slice/<name>/

   and creates EthPhyIntfCreatorSm for each Cell-id slice.
   Forwarding agents take care of slices such as
   "Linecard3" or "FixedSystem". The slicenames "1" and "2"
   for example are used for management and internal interfaces.

   These SMs should only run on the active supervisor
   which is why we need to react to redundancyStatus.
   '''
   ethPhyIntfDescDirType = Tac.Type( "Interface::EthPhyIntfDescDir" )
   ethPhyIntfConfigDirType = Tac.Type( "Interface::EthPhyIntfConfigDir" )
   ethPhyIntfStatusDirType = Tac.Type( "Interface::EthPhyIntfStatusDir" )
   tacDirType = Tac.Type( "Tac::Dir" )
   agentName = "PhyEthtool"

   def __init__( self, allSliceDescDir, defaultConfigDir,
                 allSliceEthPhyIntfConfigDir,
                 allSliceArcherIntfStatusDir,
                 allSliceArcherIntfStatusLocalDir,
                 allSliceEthPhyIntfStatusDir,
                 redundancyStatus,
                 managedCreatorSlices,
                 managedReplicatorSlices,
                 ethIntfPortIdDir,
                 managedIntfList,
                 master ):
      # Arguments
      self.allSliceDescDir_ = allSliceDescDir
      self.defaultConfigDir_ = defaultConfigDir
      self.ethIntfPortIdDir_ = ethIntfPortIdDir
      self.allSliceEthPhyIntfConfigDir = allSliceEthPhyIntfConfigDir
      self.allSliceArcherIntfStatusDir_ = allSliceArcherIntfStatusDir
      self.allSliceArcherIntfStatusLocalDir_ = allSliceArcherIntfStatusLocalDir
      self.allSliceEthPhyIntfStatusDir_ = allSliceEthPhyIntfStatusDir
      self.redundancyStatus_ = redundancyStatus
      self.managedCreatorSlices_ = managedCreatorSlices.ethPhyIntfCreatorSm
      self.managedReplicatorSlices_ = managedReplicatorSlices.replicatorSm
      self.fwdModelReplicatorSlices_ = {}
      self.managedIntfList_ = managedIntfList
      self.mySupeName_ = str( Cell.cellId() )
      self.master_ = master

      # create a pair of descReactors and ethPhyIntfConfigDirReactors
      # one reacts on changes in "this" supe in both active+stby mode
      # another reacts on changes in "other" supe, but only in active mode
      t0( "Creating AllSliceEthPhyIntfDescDir local reactor" )
      self.descReactorForThisSupe_ = DescReactor_( self.allSliceDescDir_, self,
                                             managingOtherSupeIntfs=False )
      self.ethPhyIntfConfigDirReactorForThisSupe_ = EthPhyIntfConfigDirReactor_(
                                             self.allSliceEthPhyIntfConfigDir,
                                             self, managingOtherSupeIntfs=False )
      self.descReactorForOtherSupe_ = None
      self.ethPhyIntfConfigDirReactorForOtherSupe_ = None

      self.redStatusReactor = RedundancyStatusReactor_( self.redundancyStatus_,
                                                        self )
      self.oldEthPhyIntfStatusReactor = OldIntfStatus_(
                                 self.allSliceEthPhyIntfStatusDir_, self )

   def ignorable( self, sliceName ):
      '''If the slice name isn't just a number, then ignore it'''
      try:
         _ = int( sliceName )
      except ValueError:
         return True
      return False

   ####################################
   # Redundancy mode change
   def handleRedMode( self, mode ):
      '''Called when redundancyStatus::mode changes'''
      if mode == "active":
         if not self.descReactorForOtherSupe_:
            t0( "Creating AllSliceEthPhyIntfDescDir reactors for other Supe" )
            self.descReactorForOtherSupe_ = DescReactor_( self.allSliceDescDir_,
                                                self,
                                                managingOtherSupeIntfs=True )
            self.ethPhyIntfConfigDirReactorForOtherSupe_ = \
                                                EthPhyIntfConfigDirReactor_(
                                                self.allSliceEthPhyIntfConfigDir,
                                                self, managingOtherSupeIntfs=True )
      else:
         # Delete all SMs/state for the other supe since we are in standby mode
         t0( "Deleting AllSliceEthPhyIntfDescDir reactors for other Supe" )
         for sliceName in self.managedCreatorSlices_:
            if sliceName != self.mySupeName_:
               t0( "removing %s from managedCreatorSlices " % sliceName )
               del self.managedCreatorSlices_[ sliceName ]
               t0( "removing %s from managedReplicatorSlices " % sliceName )
               # with tacc, it's always ok to use del even if the collection is
               # empty. However with a pure python dict, indexing with a
               # non-existent key will raise error. This case it is possible
               # on modular systems for these collections/dict to be empty,
               # and hence need to use pop for pure python dict.
               del self.managedReplicatorSlices_[ sliceName ]
               self.fwdModelReplicatorSlices_.pop( sliceName, None )
         self.descReactorForOtherSupe_ = None
         self.ethPhyIntfConfigDirReactorForOtherSupe_ = None

   ####################################
   # Slice add/removal reactor
   def addSlice( self, sliceName ):
      '''Helper method to add a slice'''
      if self.ignorable( sliceName ):
         t0( "handleEntity ignoring slice", sliceName )
         return

      descDir = self.allSliceDescDir_.get( sliceName )
      epicDir = self.allSliceEthPhyIntfConfigDir.get( sliceName )

      t0( "handleEntity sliceName=%s descDir=%s epicDir=%s" %
          ( sliceName, descDir, epicDir ) )
      if ( descDir and isinstance( descDir, self.ethPhyIntfDescDirType) and
           epicDir and isinstance( epicDir, self.ethPhyIntfConfigDirType ) ):
         if sliceName not in self.managedCreatorSlices_:
            subsliceEpisDir = self.allSliceArcherIntfStatusDir_.newEntity(
               "Tac::Dir", sliceName )
            episDir = subsliceEpisDir.newEntity( "Interface::EthPhyIntfStatusDir",
                                                 self.agentName )
            subsliceEpislDir = self.allSliceArcherIntfStatusLocalDir_.newEntity(
               'Tac::Dir', sliceName )
            epislDir = subsliceEpislDir.newEntity(
               'Interface::EthPhyIntfStatusLocalDir', self.agentName )
            epiCreatorSm = Tac.newInstance(
               "Interface::EthPhyIntfCreatorSm",
               descDir, self.defaultConfigDir_,
               epicDir, episDir, epislDir,
               self.ethIntfPortIdDir_,
               Tac.newInstance( "Interface::DescTypeFilter", "ethPciPort" ),
               self.managedIntfList_ )
            self.managedCreatorSlices_[ sliceName ] = epiCreatorSm
            t0( "handleEntity: epiCreatorSm for slice", sliceName, "added" )
            # ReplicatorSm
            self.maybeAddReplicatorSm( sliceName )
         else:
            episDir = self.allSliceArcherIntfStatusDir_[ sliceName ].get(
               self.agentName )

         if sliceName == self.mySupeName_:
            # start reactors depending on episDir
            self.master_.handleEpisArcherDir( episDir )
            self.master_.maybeStartInterfacePoller()

   def removeSlice( self, sliceName ):
      '''Helper method to remove a slice'''
      if self.ignorable( sliceName ):
         t0( "handleEntity ignoring slice", sliceName )
         return

      if sliceName in self.managedCreatorSlices_:
         t0( "handleEntity %s removed" % sliceName )
         del self.managedCreatorSlices_[ sliceName ]
         self.delReplicatorSm( sliceName )

         self.allSliceArcherIntfStatusDir_.deleteEntity( sliceName )

   ####################################
   # Replicator SM reactors
   def maybeAddReplicatorSm( self, sliceName ):
      if self.ignorable( sliceName ):
         t0( "maybeAddReplicatorSm ignoring slice", sliceName )
         return

      srcEpisDir = self.allSliceEthPhyIntfStatusDir_.get( sliceName )
      subsliceEpisDir = self.allSliceArcherIntfStatusDir_.get( sliceName )

      if not ( subsliceEpisDir and isinstance( subsliceEpisDir, self.tacDirType ) ):
         t0( "maybeAddReplicatorSm subslice dir not ready" )
         return

      archerEpisDir = subsliceEpisDir.get( self.agentName )
      t0( "maybeAddReplicatorSm slice=%s srcEpisDir=%s archerEpisDir=%s" %
          ( sliceName, srcEpisDir, archerEpisDir ) )

      if ( srcEpisDir and archerEpisDir and
           isinstance( srcEpisDir, self.ethPhyIntfStatusDirType ) and
           isinstance( archerEpisDir, self.ethPhyIntfStatusDirType ) ):

         repliSm = Tac.newInstance( "Interface::EthPhyIntfStatusDirReplicatorSm",
                                    srcEpisDir, archerEpisDir )
         forwardingModelRepliSm = Tac.newInstance(
            "Interface::EthPhyIntfForwardingModelReplicatorSm",
            archerEpisDir, srcEpisDir )
         self.managedReplicatorSlices_[ sliceName ] = repliSm
         self.fwdModelReplicatorSlices_[ sliceName ] = forwardingModelRepliSm
         t0( "replicator SM for", sliceName, "added" )

   def delReplicatorSm( self, sliceName ):
      if self.ignorable( sliceName ):
         t0( "handleEntity ignoring slice", sliceName )
         return

      t0( "delReplicatorSm slice", sliceName )
      # with tacc, it's always ok to use del even if the collection is
      # empty. However with a pure python dict, indexing with a
      # non-existent key will raise error. This case it is possible
      # on modular systems for these collections/dict to be empty,
      # and hence need to use pop for pure python dict.
      del self.managedReplicatorSlices_[ sliceName ]
      self.fwdModelReplicatorSlices_.pop( sliceName, None )

class RedundancyStatusReactor_( Tac.Notifiee ):
   '''Reacts to change in redundancyStatus::mode.'''
   notifierTypeName = "Redundancy::RedundancyStatus"

   def __init__( self, redStatus, msCreator ):
      t0( "RedundancyStatusReactor __init__" )
      Tac.Notifiee.__init__( self, redStatus )
      self.parent_ = msCreator

      self.parent_.handleRedMode( redStatus.mode )

   @Tac.handler( "mode" )
   def handleMode( self ):
      self.parent_.handleRedMode( self.notifier_.mode )

class OldIntfStatus_( Tac.Notifiee ):
   '''Reacts to slices getting added to
      interface/status/eth/phy/slice
      Used to add EthPhyIntfStatusReplicatorSm
   '''
   notifierTypeName = "Tac::Dir"

   def __init__( self, episDir, msCreator ):
      t0( "OldIntfStatus __init__" )
      self.episDir_ = episDir
      Tac.Notifiee.__init__( self, episDir )
      self.parent_ = msCreator

      t0( "Handle all existing (%d) entries" % len( self.episDir_ ) )
      for sliceName in self.episDir_:
         self.parent_.maybeAddReplicatorSm( sliceName )

   @Tac.handler( "entityPtr" )
   def handleEntity( self, sliceName ):
      t0( "OldIntfStatus::handleEntity", sliceName )

      if sliceName in self.episDir_:
         self.parent_.maybeAddReplicatorSm( sliceName )
      else:
         self.parent_.delReplicatorSm( sliceName )

class DescReactor_( Tac.Notifiee ):
   '''Reacts to change in
      interface/archer/status/init/eth/phy/slice
   '''
   notifierTypeName = "Tac::Dir"

   def __init__( self, tacDirEntity, msCreator, managingOtherSupeIntfs=False ):
      t0( "DescReactor __init__" )
      self.mountPath_ = tacDirEntity
      Tac.Notifiee.__init__( self, tacDirEntity )
      self.parent_ = msCreator
      # indicates whether this reactor is managing this or other supe changes
      self.managingOtherSupeIntfs_ = managingOtherSupeIntfs
      self.mySupeName_ = str( Cell.cellId() )

      t0( "Handle all existing (%d) entries" % len( self.mountPath_ ) )
      for sliceName in self.mountPath_:
         if self.managingOtherSupeIntfs_ == ( sliceName != self.mySupeName_ ):
            self.parent_.addSlice( sliceName )

   @Tac.handler( "entityPtr" )
   def handleEntity( self, sliceName ):
      t0( "DescReactor::handleEntity", sliceName )

      # only handling changes on this or other supe,
      # depending on flag managingOtherSupeIntfs
      if self.managingOtherSupeIntfs_ == ( sliceName != self.mySupeName_ ):
         if sliceName in self.mountPath_:
            self.parent_.addSlice( sliceName )
         else:
            self.parent_.removeSlice( sliceName )

class EthPhyIntfConfigDirReactor_( Tac.Notifiee ):
   """Reacts to changes in
      interface/config/eth/phy/slice
   """
   notifierTypeName = "Tac::Dir"

   def __init__( self, configDir, msCreator, managingOtherSupeIntfs=False ):
      t0( "EthPhyIntfConfigDirReactor __init__" )
      self.mountPath_ = configDir
      Tac.Notifiee.__init__( self, configDir )
      self.parent_ = msCreator
      self.managingOtherSupeIntfs_ = managingOtherSupeIntfs
      self.mySupeName_ = str( Cell.cellId() )

      t0( "Handle all existing (%d) entries" % len( self.mountPath_ ) )
      for sliceName in self.mountPath_:
         if self.managingOtherSupeIntfs_ == ( sliceName != self.mySupeName_ ):
            self.parent_.addSlice( sliceName )

   @Tac.handler( "entityPtr" )
   def handleEntity( self, sliceName ):
      t0( "EthPhyIntfConfigDirReactor::handleEntity", sliceName )

      # only handling changes on this or other supe,
      # depending on flag managingOtherSupeIntfs
      if self.managingOtherSupeIntfs_ == ( sliceName != self.mySupeName_ ):
         if sliceName in self.mountPath_:
            self.parent_.addSlice( sliceName )
         else:
            self.parent_.removeSlice( sliceName )
