#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os, sys
import Tac

reloadConstants = Tac.Value( "ReloadCause::ReloadCauseConstants" )

def reloadHistoryPreProcessing():
   reloadHistoryFile = f"{reloadConstants.debugDir()}reload_history"
   if os.path.exists( reloadHistoryFile ):
      with open( reloadHistoryFile, "r" ) as f:
         history = f.readlines()
      with open( reloadHistoryFile, "w" ) as f:
         for line in history:
            tokens = line.split( "," )
            if len( tokens ) > 3 and tokens[ 0 ].isdigit():
               reloadIdx = int( tokens[ 0 ] )
               if reloadIdx < 9:
                  tokens[ 0 ] = str( reloadIdx + 1 )
                  f.write( ','.join( tokens ) )
            else:
               f.write( line )
   else:
      print( f"Reload history file {reloadHistoryFile} not found", file=sys.stderr )


def reloadCookiePreProcessing():
   """Converting reload cause cookie files' rebootIsImminent flag to preReboot flag.
   rebootIsImminent flag indicates the cause of a reload is about to happen.
   After the reload, this function converts the flag to preReboot
   which indicates the cookie file has a high priority for reload cause agent."""
   if not os.path.exists( reloadConstants.debugDir() ):
      return
   reloadCookieFiles = []
   for cookieFileName in os.listdir( reloadConstants.debugDir() ):
      if cookieFileName.startswith( 'local_reload_cause' ):
         reloadCookieFiles.append( f"{reloadConstants.debugDir()}{cookieFileName}" )
   if os.path.exists( reloadConstants.oldLocalReloadCauseFile ):
      reloadCookieFiles.append( reloadConstants.oldLocalReloadCauseFile )

   for fileName in reloadCookieFiles:
      stInfo = os.stat( fileName )
      with open( fileName, "r" ) as f:
         cookieContent = f.readlines()
      with open( fileName, "w" ) as f:
         for line in cookieContent:
            if reloadConstants.rebootImminentString() in line:
               f.write( reloadConstants.preRebootString() )
               print( f"Converting {fileName} to pre-reboot reload cause file",
                      file=sys.stdout )
            else:
               f.write( line )
      os.utime( fileName, ( stInfo.st_atime, stInfo.st_mtime ) )

def reloadCausePreProcessing( context ):
   try:
      reloadHistoryPreProcessing()
      reloadCookiePreProcessing()
   except Exception:  # pylint: disable=broad-except
      print( "Error initializing reload cause agent", file=sys.stderr )
      sys.excepthook( *sys.exc_info() )


def Plugin( context ):
   context.registerPostFpgaUpgradeFn( reloadCausePreProcessing,
                                      lambda sid: True )
