# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import ctypes

libNaming = ctypes.cdll.LoadLibrary( "libPassiveMountNaming.so" )

# Everything about agent naming for passive mount should be here

# Return the ProcTitle of the agent that should be used to connect to Sysdb
# We parse argv and append the binary name with
#    -XXX if there's an option with an argument --xxx=XXX or --xxx XXX (e.g.,sliceId)
#    -YYY if there's an option without argument --YYY (e.g., system)
# where XXX and YYY are supported options defined in passiveMountNamingLib.cpp
# The order of substitution is also defined in createAgentProcTitle() in there.
# For example: if
#      argv = [ "/usr/bin/testAgent", "argv1", "--system", "--sliceId=Linecard0" ]
# The function returns "testAgent-system-Linecard0".
def createAgentProcTitle( argv, processName=None ):

   cArgv = ( ctypes.c_char_p * len( argv ) )()
   cArgv[ : ] = [ arg.encode() for arg in argv ]
   # pylint: disable-msg=E1101
   cTitle = ctypes.create_string_buffer( ctypes.c_uint.in_dll( libNaming,
               "passiveMountNamingTitleMaxLenCopy" ).value )

   libNaming.createAgentProcTitle( len( argv ), cArgv, cTitle )
   return cTitle.value.decode()

# Return agentName from the procTitleLine read from mount profile
# Current logic is to discards all template keys (start with %)
# pylint: disable-next=inconsistent-return-statements
def mountProfileParseAgentName( AGENTNAME_TAG, procTitleLine ):
   if procTitleLine.lower().startswith( AGENTNAME_TAG ):
      agentStringL = procTitleLine.split( ":" )
      # pylint: disable-next=simplifiable-condition
      assert len( agentStringL ) > 1 and "agentName not found"
      agentString = agentStringL[ -1 ].strip()
      agentNameL = agentString.split( "-" )

      # sanity check
      templateSet = False
      for tok in agentNameL:
         if tok.startswith( '%' ):
            templateSet = True
            continue
         if templateSet:
            print( "agent title ", procTitleLine, " has regular tokens ", tok,
                   " past template keys" )
            assert False

      # agent name without template keys
      return "-".join( tok for tok in agentNameL if not tok.startswith( '%' ) )

