# Copyright (c) 2006, 2007 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""Utility functions for userspace access to PCI resources.

This module provides utility functions for direct read, write and memory-map access
to PCI resources.
"""

import Pci, mmap

def strToInt( s ):
   try:
      i = int( s )
   except ValueError:
      if s.startswith( "0b" ) or s.startswith( "0B" ):
         i = int( s[2:], 2 )
      else:
         i = int( s, 16 )
   return i

def roundToPageSize( addr ):
   start = ( addr // mmap.PAGESIZE ) * mmap.PAGESIZE
   return start


def read8( device, resource, addr ):
   """Reads an 8-bit value from the specified address in the specified PCI
   resource."""
   if resource == "config":
      res = Pci.Device( device ).config()
   else:
      start = roundToPageSize( addr )
      res = Pci.Device( device ).resource( resource, readOnly=True,
               startOffset=start, endOffset=addr+1)
   if res:
      return res.read8( addr )
   else:
      return 0

def read16( device, resource, addr ):
   """Reads a 16-bit value from the specified address in the specified PCI
   resource."""
   if resource == "config":
      res = Pci.Device( device ).config()
   else:
      start = roundToPageSize( addr )
      res = Pci.Device( device ).resource( resource, readOnly=True,
               startOffset=start, endOffset=addr+2 )
   if res:
      return res.read16( addr )
   else:
      return 0

def read32( device, resource, addr ):
   """Reads a 32-bit value from the specified address in the specified PCI
   resource."""
   if resource == "config":
      res = Pci.Device( device ).config()
   else:
      start = roundToPageSize( addr )
      res = Pci.Device( device ).resource( resource, readOnly=True,
               startOffset=start, endOffset=addr+4 )
   if res:
      return res.read32( addr )
   else:
      return 0

def write8( device, resource, addr, value ):
   """Writes an 8-bit value to the specified address in the specified PCI
   resource."""
   if resource == "config":
      res = Pci.Device( device ).config()
   else:
      start = roundToPageSize( addr )
      res = Pci.Device( device ).resource( resource, readOnly=False,
               startOffset=start, endOffset=addr+1 )
   if res:
      res.write8( addr, value )

def write16( device, resource, addr, value ):
   """Writes a 16-bit value to the specified address in the specified PCI
   resource."""
   if resource == "config":
      res = Pci.Device( device ).config()
   else:
      start = roundToPageSize( addr )
      res = Pci.Device( device ).resource( resource, readOnly=False,
               startOffset=start, endOffset=addr+2 )
   if res:
      res.write16( addr, value )

def write32( device, resource, addr, value ):
   """Writes a 32-bit value to the specified address in the specified PCI
   resource."""
   if resource == "config":
      res = Pci.Device( device ).config()
   else:
      start = roundToPageSize( addr )
      res = Pci.Device( device ).resource( resource, readOnly=False,
               startOffset=start, endOffset=addr+4 )
   if res:
      res.write32( addr, value )
