#!/usr/bin/env python3
# Copyright (c) 2006-2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string
# pylint: disable=c-extension-no-member

# ------------------------------------------------------------------------------
#
#   PhyEthtool: manage linux kernel ethernet interfaces
#
# Per AID166:
#
#   1.  PhyEthtool pushes physical-layer configuration from
#   Sysdb/interface/config/ethernet/ into the Linux kernel for all
#   interfaces listed in Sysdb/hardware/<cellId>/phy/ethtool (where
#   physical-layer configuration include autonegotiation settings,
#   flow control settings, and so on). These interfaces are only
#   management interfaces.
#
#   2. PhyEthtool pulls physical-layer status from the Linux kernel into
#   Sysdb/interface/status/ethernet/ (where physical-layer status includes
#   operating link state, speed, duplex, flow control, etc.)
#
#   3. PhyEthtool sets the MAC address for linux kernel interfaces listed in
#   Sysdb/kernel/interface. These interfaces include management interfaces,
#   vlan interfaces, and router interfaces.
#
#   4. PhyEthtool pulls MAC-level statistics (packet and byte counters) for
#   linux kernel interfaces listed in Sysdb/kernel/interface.
#
# Once an interface stops being PhyEthtool-managed, PhyEthtool does not touch
# the kernel device.  In particular, it does not reset the device in any way.
# The rationale is that as soon as the interface is no longer
# PhyEthtool-managed, we assume that some other agent is managing it and
# therefore PhyEthtool must not interfere.
#
# In order to handle failures while setting the interface configuration,
# PhyEthtool periodically checks each PhyEthtool-managed interface to see if the
# Linux kernel devices have the correct configuration, and if not, it reapplies
# the correct configuration.
# ------------------------------------------------------------------------------

import Tac
import Tracing
import Logging
import Agent
import Cell
import EntityManager
import SharedMem
import Smash
import re
import sys
import os
from Intf.SyncMtu import syncMtu
import ethtool
from Arnet.NsLib import DEFAULT_NS, runMaybeInNetNs
import weakref

from SysConstants.if_h import IFF_RUNNING
from MultiSliceEthPhyIntfCreator import MultiSliceEthPhyIntfCreator
import QuickTrace
import VeosHypervisor
import KernelVersion
from StandbyPoller import RprStandbyMgmtStatusSync
from TypeFuture import TacLazyType
from collections import defaultdict
from GenericReactor import GenericReactor
import StageMgr
from CEosHelper import isCeosLab

tracePeriodic = Tracing.trace9
traceDetail = Tracing.trace8
traceNormal = Tracing.trace7
traceAlert = Tracing.trace1
traceError = Tracing.trace0

qv = QuickTrace.Var
qtError = QuickTrace.trace1
qtDetail = QuickTrace.trace0

# Below drivers don't support any  of the ethtool commands to
# inspect the properties of the interface.
# BUG537028: remove ixgbe from the list once the bug is understood and fixed.
limitedDrivers = [ "virtio_net", "tulip", "ixgbevf", "BESS", "hv_netvsc",
                   "vmxnet3", "ena", "vif", "e1000", "mlx4_en", "veth",
                   "macvlan", "r8152", "ixgbe", "apm" ]

ethLinkMode = TacLazyType( 'Interface::EthLinkMode' )
ethLaneCount = TacLazyType( 'Interface::EthLaneCount' )
EthTypesApi = TacLazyType( 'Interface::EthTypesApi' )
RedundancyMode = TacLazyType( 'Redundancy::RedundancyMode' )

ssoStageMaIntfBringUp = "ManagementIntfBringUp"

def isSsoMaIntfBringUpStage( redundancyStatus ):
   return ( redundancyStatus.mode == RedundancyMode.switchover and
   ssoStageMaIntfBringUp in redundancyStatus.switchoverStage )


# Cpu interfaces are internal NICs, that can be used in a variety of ways:
# - control plane (pktdma) for FPGA applications (e.g. switchapp)
# - directly connected to front panel via crosspoint to provide extra management
#   ports
# - other specialized applications
#   * PTP snooping in MetaWatch
#   * TapAgg capture/analytics
# These NICs support backplane ethernet enabling high speed links over PCBs
# connecting internal components to one another.
# They support one or more of backplane ethernet modes defined in 802.3ap
def isCpuIntf( intfName ):
   return Tac.Value( 'Arnet::CpuIntfId' ).isCpuIntfId( intfName )

backplaneAdvertisedModes = [
   ( 'mode1GbpsFull', ethtool.ADVERTISED_1000baseKX_Full ),
   ( 'mode10GbpsFull', ethtool.ADVERTISED_10000baseKR_Full )
]
backplaneSupportedModes = [
   ( 'mode1GbpsFull', ethtool.SUPPORTED_1000baseKX_Full ),
   ( 'mode10GbpsFull', ethtool.SUPPORTED_10000baseKR_Full )
]
advertisedModes = [
   ( 'mode10MbpsHalf', ethtool.ADVERTISED_10baseT_Half ),
   ( 'mode10MbpsFull', ethtool.ADVERTISED_10baseT_Full ),
   ( 'mode100MbpsHalf', ethtool.ADVERTISED_100baseT_Half ),
   ( 'mode100MbpsFull', ethtool.ADVERTISED_100baseT_Full ),
   ( 'mode1GbpsHalf', ethtool.ADVERTISED_1000baseT_Half ),
   ( 'mode1GbpsFull', ethtool.ADVERTISED_1000baseT_Full ),
   ( 'mode10GbpsFull', ethtool.ADVERTISED_10000baseT_Full ),
   # BUG18016: ethtool doesn't support 40G/100G yet.
   # (modes.mode40GbpsFull, ethtool.ADVERTISED_????),
   # (modes.mode100GbpsFull, ethtool.ADVERTISED_????),
]
supportedModes = [
   ('mode10MbpsHalf', ethtool.SUPPORTED_10baseT_Half),
   ('mode10MbpsFull', ethtool.SUPPORTED_10baseT_Full),
   ('mode100MbpsHalf', ethtool.SUPPORTED_100baseT_Half),
   ('mode100MbpsFull', ethtool.SUPPORTED_100baseT_Full),
   ('mode1GbpsHalf', ethtool.SUPPORTED_1000baseT_Half),
   ('mode1GbpsFull', ethtool.SUPPORTED_1000baseT_Full),
   ('mode10GbpsFull', ethtool.SUPPORTED_10000baseT_Full)
   # BUG18016: ethtool doesn't support 40G/100G yet.
   # (modes.mode40GbpsFull, ethtool.SUPPORTED_????),
   # (modes.mode100GbpsFull, ethtool.SUPORTED_????),
]
def validateModes( modeList ):
   # Verify that a mode key is defined only once in the (mode, Y) tuple list.
   if len( modeList ) != len( dict( modeList ) ):
      modeMap = defaultdict( list )
      for x, y in modeList:
         modeMap[ x ].append( y )
      badModes = [ x for x in modeMap if len( modeMap[ x ] ) > 1 ]
      traceError( 'Mode(s) present multiple times in list:', ','.join( badModes ) )
      assert False
# Statically validate all mode lists to ensure that a linkMode only appears once in
# the list
for ml in [ advertisedModes, supportedModes,
            backplaneAdvertisedModes, backplaneSupportedModes ]:
   validateModes( ml )

# i40e driver does not enable link state reporting by default - yes really.
# It needs to be told to monitor and report link state by setting a special
# driver flag called LinkPolling. This flag is not present in python's ethtool
# implementation and hence we have to resort to invoking ethtool bash call
def handlei40eQuirks( intf, ns=DEFAULT_NS ):
   try:
      runMaybeInNetNs( ns, [ 'ethtool',
                       '--set-priv-flags', intf, 'LinkPolling', 'on'  ],
                       stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )
   except Tac.SystemCommandError:
      traceDetail( " Could not enable LinkPolling for i40e %s in ns %s" %
                   ( intf, ns ) )
   if KernelVersion.supports( KernelVersion.I40E_DIS_SRC_PR ):
      try:
         runMaybeInNetNs( ns, [ 'ethtool',
                          '--set-priv-flags', intf,
                          'disable-source-pruning', 'on'  ],
                          stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )
      except Tac.SystemCommandError:
         traceDetail( " Could not enable disable-source-pruning"
                      " for i40e %s in ns %s" % ( intf, ns ) )

# Drivers which need special handling i.e. are quirky. We will handle special
# quirks using a dispatch table
qurikyDrivers = { "i40e" : handlei40eQuirks }

def handleDriverQuirks( driverName, intf, ns=DEFAULT_NS ):
   if driverName not in qurikyDrivers:
      return
   qurikyDrivers[ driverName ]( intf, ns )

def isModular():
   return Cell.cellType() == "supervisor"

# Regular expression obj for searching mac address
macRex = re.compile( r'link/\S+\s+([a-f\d:]+)' )

# Customizable by tests, to run wrapped versions of helper commands
ipCommand = [ 'ip' ]
sysClassNet = "/sys/class/net"

def ipLinkSet():
   return ipCommand + [ 'link', 'set' ]

def ipLinkShow( intf ):
   return ipCommand + [ '-o', 'link', 'show', intf ]

def ipLinkShowDetail( intf ):
   return ipCommand + [ '-d', 'link', 'show', intf ]

def ipLinkSetStatus( intf, status ):
   # Some NIC drivers can disable bottom-halfs for up to 1 sec when changing
   # link status (tg3_poll_fw), run on CPU0 to avoid interfering with BFD rx/tx
   # See BUG932159 for details
   return [ 'taskset', '-c', '0' ] + ipLinkSet() + [ intf, status ] 

def ipLinkSetAddr( intf, addr):
   return ipLinkSet() + [ intf, 'addr', addr ]

# ------------------------------------------------------------------------------
# Declarations of log messages.
# ------------------------------------------------------------------------------

# If we hit an error condition where we need to log a message, it is
# likely that we will hit the same condition again when we retry to
# sync the config or poll the status. We therefore want these messages
# to be rate limited with a fairly high interval. For now, we're using
# an interval of 1 minute. Really, this interval should be larger, but
# we should also have a more severe log message that is logged when
# PhyEthtool detects that things are really messed up.

ETH_VRF_ASSIGNMENT_FAILED = Logging.LogHandle(
   "ETH_VRF_ASSIGNMENT_FAILED",
   severity=Logging.logWarning,
   fmt="Failed to assign interface %s to %s",
   explanation="An attempt to assign the interface to a VRF failed.",
   recommendedAction=Logging.CALL_SUPPORT_IF_PERSISTS,
   minInterval=300 )

ETH_SETADDRFAIL = Logging.LogHandle(
   "ETH_SETADDRFAIL",
   severity=Logging.logWarning,
   fmt="Failed to set the MAC address of interface %s to %s",
   explanation="An attempt to set the MAC address of the interface failed " +
   "for an unknown reason.",
   recommendedAction=Logging.CALL_SUPPORT_IF_PERSISTS,
   minInterval=300 )

ETH_SETENFAIL = Logging.LogHandle(
   "ETH_SETENFAIL",
   severity=Logging.logWarning,
   fmt="Failed to enable interface %s",
   explanation="An attempt to enable the interface failed for an unknown " +
   "reason.",
   recommendedAction=Logging.CALL_SUPPORT_IF_PERSISTS,
   minInterval=300 )

ETH_SETDISFAIL = Logging.LogHandle(
   "ETH_SETDISFAIL",
   severity=Logging.logWarning,
   fmt="Failed to disable interface %s",
   explanation="An attempt to disable the interface failed for an unknown " +
   "reason.",
   recommendedAction=Logging.CALL_SUPPORT_IF_PERSISTS,
   minInterval=300 )

ETH_SETLINKMODEFAIL = Logging.LogHandle(
   "ETH_SETLINKMODEFAIL",
   severity=Logging.logWarning,
   fmt="Failed to set speed/duplex on interface %s to %s/%s (%s)",
   explanation="An attempt to set the interface's speed/duplex mode failed.",
   recommendedAction=Logging.CALL_SUPPORT_IF_PERSISTS,
   minInterval=300 )

ETH_SETPAUSEPARAMFAIL = Logging.LogHandle(
   "ETH_SETPAUSEPARAMFAIL",
   severity=Logging.logWarning,
   fmt="Failed to set pause parameters on interface %s (%s)",
   explanation="An attempt to set the interface's pause parameters failed.",
   recommendedAction=Logging.CALL_SUPPORT_IF_PERSISTS,
   minInterval=300 )

ETH_POLLLINKSTATUSFAIL = Logging.LogHandle(
   "ETH_POLLLINKSTATUSFAIL",
   severity=Logging.logWarning,
   fmt="Failed to poll link status on interface %s (%s)",
   explanation="An attempt to poll the interface's status failed.",
   recommendedAction=Logging.CALL_SUPPORT_IF_PERSISTS,
   minInterval=600 )

ETH_LINKMODEUNCONFIGURED = Logging.LogHandle(
   "ETH_LINKMODEUNCONFIGURED",
   severity=Logging.logWarning,
   fmt="Interface %s link mode is unconfigured",
   explanation="The interface's link mode has not been configured.  " +
   "The interface lacks a proper default configuration.",
   recommendedAction=Logging.CALL_SUPPORT_IF_PERSISTS,
   minInterval=300 )

ETH_SETMTUFAIL = Logging.LogHandle(
   "ETH_SETMTUFAIL",
   severity=Logging.logWarning,
   fmt="Failed to set interface %s mtu",
   explanation="An attempt to set the interface mtu failed for an unknown " +
   "reason.",
   recommendedAction=Logging.CALL_SUPPORT_IF_PERSISTS,
   minInterval=300 )

# ------------------------------------------------------------------------------
# The PhyEthtool agent and Reactors:
#
#   o The ConfigReactor reacts to the Hardware::Phy::EthtoolConfig
#     'phy' collection and creates a PhyReactor for each Phy that
#     shows up in the collection.
#
#   o The PhyReactor creates an EthIntfConfigReactor for the
#     corresponding EthIntfStatus.
#
#   o The EthIntfConfigReactor propagates configuration events into
#     the hardware using 'ethtool'.
#
#   o Instantiates the EthPhyIntfCreatorSM to create EthPhyIntfStatus on
#     EthPhyIntfDesc requests (from Fru)
# ------------------------------------------------------------------------------
class PhyEthtool( Agent.Agent ):

   # The interval (in seconds) at which to update the Ethernet status
   # and statistics.

   # BUG220: polling EthIntf status is painfully inefficient. Rather
   #         than disabling it, run it less frequently.
   # statusUpdateInterval_ = 0.5
   statusUpdateInterval_ = 2.0
   syncConfigInterval_ = 0.5
   netCleanupYieldDuration_ = 1

   def __init__( self, entityManager ):
      # On new modular systems, it takes longer time for PhyEthtool to warmup,
      # We have to increase the warmup warning timeout ( default=60sec ) to
      # stop the Agent base class from complaining.
      self.config_ = None
      self.status_ = None
      self.configReactor_ = None
      self.kernelInterfaceManager_ = None
      self.episDir_ = None
      self.epicDir_ = None
      self.episDirReactor_ = None
      self.epicDirReactor_ = None
      self.ethIntfXcvrStatusDir = None
      self.epiDescDir_ = None
      self.episArcherDir_ = None
      self.rprStandbyStatusTracker = None
      self.rprStandbyMgmtStatusSync = None
      self.allSliceEthPhyIntfCreator_ = None
      self.globalIntfConfig_ = None
      self.intfHwCapability_ = None
      self.intfNsConfigDir_ = None
      self.intfStatusLocalDir_ = None
      self.ethIntfCounterWriterStatus = None
      self.allEthIntfCounters = None
      self.numStatusUpdates_ = 0
      self.statusUpdateClockNotifiee_ = None
      self.nsConfigReactor_ = None
      self.intfStatusLocalDirReactor_ = None
      self.redundancyStatus_ = None
      self.redundancyStatusActive_ = False
      self.redundancyStatusReactor_ = None
      self.switchoverAgentStatus_ = None
      self.maIntfBringUpHelper_ = None

      Agent.Agent.__init__( self, entityManager, warmupWarningInterval=120 )
      if "QUICKTRACE_DISABLE" not in os.environ:
         qtfile = "%s%s.qt" % ( self.agentName, "" if "QUICKTRACEDIR" in os.environ
                                                else "-%d" )
         QuickTrace.initialize( qtfile )

   def doInit( self, entityManager ):
      mountGroup = entityManager.mountGroup()
      need = mountGroup.mount
      need( "interface/config/eth/phy/slice",
            "Tac::Dir", "ri" )
      need( "interface/status/eth/phy/slice",
            "Tac::Dir", "wfi" )
      need( Cell.path( "stageAgentStatus/switchover/PhyEthtool" ),
            "Stage::AgentStatus", "wcf" )

      # For EthPhyIntfCreatorSm
      need( "interface/archer/status/init/eth/phy/slice",
            "Tac::Dir", "ri" )
      need( "interface/config/eth/phy/default",
            "Interface::EthPhyIntfDefaultConfigDir", "r" )
      need( "interface/eth/portid",
            "Interface::EthIntfPortIdDir", "r" )
      need( "interface/archer/status/eth/phy/slice",
            "Tac::Dir", "wfi" )
      need( Cell.path( "interface/archer/status/eth/phy/slice/local" ),
            "Tac::Dir", "wfi" )

      self.createLocalEntity( "ForwardingModelResolverState",
                              "Interface::LocalForwardingModelResolverState",
                              "interface/status/forwardingModelResolverState" )
      self.createLocalEntity( "ManagedIntfList",
                              "Interface::IntfStatusPtrDir",
                              "interface/status/managedIntfList" )
      self.localEntitiesCreatedIs( True )

      mountGroup.close( self._mountConfig )

   def _mountConfig( self ):
      # The interface creation task should only be done on the active supe.
      # React to redundancy status for this reason.
      em = self.entityManager
      allSliceDescDir = em.lookup( "interface/archer/status/init/eth/phy/slice" )
      self.epiDescDir_ = allSliceDescDir
      defConfigDir = em.lookup( "interface/config/eth/phy/default" )
      allSliceEthPhyIntfConfigDir = em.lookup( "interface/config/eth/phy/slice" )
      allSliceArcherEpisDir = em.lookup( "interface/archer/status/eth/phy/slice" )
      allSliceArcherEpislDir = em.lookup(
         Cell.path( "interface/archer/status/eth/phy/slice/local" ) )
      ethIntfPortIdDir = em.lookup( "interface/eth/portid" )
      # redundancy/status is already mounted through CAgent layer
      self.redundancyStatus_ = em.lookup( Cell.path( "redundancy/status" ) )
      managedIntfList = em.getLocalEntity( "interface/status/managedIntfList" )

      # For the replicator SM
      allSliceEpisDir = em.lookup( "interface/status/eth/phy/slice" )
      sliceName = str( Cell.cellId() )
      
      # Until we are ready for the archer transition, we
      # will need to work with original mount paths
      self.episDir_ = em.lookup( "interface/status/eth/phy/slice/%s" % sliceName )
      self.epicDir_ = em.lookup( "interface/config/eth/phy/slice/%s" % sliceName )
      # Note: self.episDirReactor_ is created by MultiSliceEthPhyIntfCreator
      self.epicDirReactor_ = EthPhyIntfConfigDirReactor( self.epicDir_, self )
      if isModular():
         qtDetail( "Rpr Standby Poll feature is enabled " )
         traceDetail( "Rpr Standby Poll feature is enabled " )
         peerSliceName = str( Cell.peerCellId() )
         self.rprStandbyStatusTracker = \
               self.agentRoot_.newEntity( "PhyEthtool::RprStandbyStatusTracker",
                                       "rprStandbyStatusTracker" )
         self.rprStandbyMgmtStatusSync = \
               RprStandbyMgmtStatusSync( peerSliceName, self.redundancyStatus_,
                                      allSliceEpisDir, allSliceArcherEpisDir,
                                      self.rprStandbyStatusTracker )

      managedCreatorSlices = self.agentRoot_.newEntity(
            "Interface::EthPhyIntfCreatorSmList",
            "managedCreatorSlices" )
      managedReplicatorSlices = self.agentRoot_.newEntity(
            "Interface::EthPhyIntfStatusDirReplicatorSmList",
            "managedReplicatorSlices" )

      self.allSliceEthPhyIntfCreator_ = MultiSliceEthPhyIntfCreator( allSliceDescDir,
                                             defConfigDir,
                                             allSliceEthPhyIntfConfigDir,
                                             allSliceArcherEpisDir,
                                             allSliceArcherEpislDir,
                                             allSliceEpisDir,
                                             self.redundancyStatus_,
                                             managedCreatorSlices,
                                             managedReplicatorSlices,
                                             ethIntfPortIdDir,
                                             managedIntfList,
                                             self )

      mg = self.entityManager.mountGroup()
      cellId = str( Cell.cellId() )
      configName = "hardware/cell/%s/phy/ethtool/config" % cellId
      try:
         self.config_ = mg.mount( configName,
                                  "Hardware::Phy::EthtoolConfig", "r" )
      except EntityManager.MountError:
         # This case arises when we could not even make the mountpoint.
         sys.stderr.write(
            "PhyEthtool: no Hardware::Phy::EthtoolConfig found at {}/{}\n".format(
               self.entityManager.root(), configName ) )
         sys.exit( 1 )

      try:
         self.status_ = mg.mount( "cell/%s/phyethtool/status" % cellId,
                                  "PhyEthtool::Status", "w" )
      except EntityManager.MountError:
         # This case arises when we could not even make the mountpoint.
         sys.stderr.write(
            "PhyEthtool: no PhyEthtool::Status found at {}/{}\n".format(
               self.entityManager.root(), "cell/%s/phyethtool/status" % cellId ) )
         sys.exit( 1 )
      
      try:
         self.globalIntfConfig_ = mg.mount( "interface/config/global",
                                            "Interface::GlobalIntfConfig", "r" )
      except EntityManager.MountError:
         # This case arises when we could not even make the mountpoint.
         sys.stderr.write(
            "PhyEthtool: no Interface::GlobalIntfConfig found at {}/{}\n".format(
               self.entityManager.root(), "interface/config/global" ) )
         sys.exit( 1 )

      try:
         self.intfHwCapability_ = mg.mount( "interface/hardware/capability",
                                            "Interface::Hardware::Capability", "r" )
      except EntityManager.MountError:
         # This case arises when we could not even make the mountpoint.
         sys.stderr.write(
            "PhyEthtool: no Interface::Hardware::Capability found at {}/{}\n".format(
               self.entityManager.root(), "interface/hardware/capability" ) )
         sys.exit( 1 )

      self.intfNsConfigDir_ = mg.mount( Cell.path( "interface/nsconfig" ),
                                        "Interface::AllIntfNamespaceConfigDir", "r" )
      self.intfStatusLocalDir_ = mg.mount(
         Cell.path( "interface/status/eth/phy/local" ),
         "Interface::EthPhyIntfStatusLocalDir", "wf" )
      self.ethIntfCounterWriterStatus = mg.mount(
         f"interface/ethIntfCounter/writerStatus/PhyEthtool-{cellId}",
         "Interface::EthIntfCounterWriterStatus", "wf" )
      self.ethIntfXcvrStatusDir = mg.mount(
         "interface/archer/eth/xcvr/slice/%s/phyethtool" % cellId,
         "Interface::EthIntfXcvrStatusDirEntry", "wc" )

      mg.close( self._finishMountingConfig )

   def _finishMountingConfig( self ):
      # Mounting smash collections
      maxNumIntf = 10  # We shouldn't have more than a couple Management interfaces
      smashEm = SharedMem.entityManager( self.entityManager.sysname(), False )
      mountHelper = Tac.newInstance( "Interface::EthIntfCounterMountHelper",
                                     smashEm )
      self.allEthIntfCounters = mountHelper.doMountWrite(
         self.ethIntfCounterWriterStatus.writerMountName, "current", maxNumIntf )

      # Phew! Everything's mounted. Create the reactors, activities,
      # etc. that handle all these things.
      traceDetail( "Finished all mounts -- creating ConfigReactor" )
      qtDetail( "Finished all mounts -- creating ConfigReactor" )
      self.configReactor_ = ConfigReactor( self.config_, self )
      self.kernelInterfaceManager_ = self.agentRoot_.newEntity(
         "PhyEthtool::KernelInterfaceManager",
         "kernelInterfaceManager" )
      # create self.kernelInterfaceManager_.kernelInterfacePoller
      self.maybeStartInterfacePoller()

      self.numStatusUpdates_ = 0
      self.statusUpdateClockNotifiee_ = Tac.ClockNotifiee(
         handler=self._updateEthernetStatus )
      self.statusUpdateClockNotifiee_.timeMin = 0

      self.nsConfigReactor_ = Tac.collectionChangeReactor(
         self.intfNsConfigDir_.intfNamespaceConfig, IntfNsConfigReactor,
         reactorArgs=( self.configReactor_, ) )

      self.intfStatusLocalDirReactor_ = Tac.collectionChangeReactor(
         self.intfStatusLocalDir_.intfStatusLocal, IntfStatusLocalReactor,
         reactorArgs=( self.configReactor_, ) )
      
      self.switchoverAgentStatus_ = self.entityManager.lookup(
         Cell.path( "stageAgentStatus/switchover/PhyEthtool" ) )
      
      traceDetail( "Creating redundancyStatusReactor" )
      self.redundancyStatusReactor_ = GenericReactor( 
         self.redundancyStatus_, [ "mode", "switchoverStage" ],
         self._handleRedundancyStatus, callBackNow=True ) 
   def _updateEthernetStatus( self ):
      tracePeriodic( 'Updating status of all Ethernet interfaces: %r' %
                     list( self.config_.phy ) )

      self.numStatusUpdates_ += 1

      nextTime = Tac.now() + self.statusUpdateInterval_
      self.statusUpdateClockNotifiee_.timeMin = nextTime

      if self.episDir_ is None:
         Tracing.t0( "_updateEthernetStatus: episDir_ is not initialized (yet)" )
         return

      for phy in self.config_.phy.values():
         intfId = phy.intfId
         if intfId in self.episDir_.intfStatus:
            status = self.episDir_.intfStatus[intfId]
            ethIntfXcvrStatus = \
                     self.ethIntfXcvrStatusDir.intfXcvrStatus.get( status.intfId )
            updateEthIntfStatus( status, ethIntfXcvrStatus )
         else:
            Tracing.t0( "_updateEthernetStatus: intfStatus %s doesn't exist" %
                        intfId )

   def handleEpisArcherDir( self, episArcherDir ):
      if not self.episDirReactor_:
         self.episArcherDir_ = episArcherDir
         # for now, watch collection under episDir_
         self.episDirReactor_ = EthPhyIntfStatusDirReactor( self.episDir_, self )
         qtDetail( "EthPhyIntfStatusDirReactor created" )

   def maybeStartInterfacePoller( self ):
      if self.episArcherDir_ and self.kernelInterfaceManager_:
         if not self.kernelInterfaceManager_.kernelInterfacePoller:
            self.kernelInterfaceManager_.kernelInterfacePoller = (
               Tac.activityManager.clock,
               self.config_, self.episDir_, self.epicDir_, self.globalIntfConfig_,
               self.allEthIntfCounters, self.ethIntfCounterWriterStatus )
            qtDetail( "kernelInterfacePoller created" )
      else:
         if self.episArcherDir_ is None:
            Tracing.t0( "Error: episDir_ is None" )
         if self.kernelInterfaceManager_ is None:
            Tracing.t0( "Error: kernelInterfaceManager_ is None" )

   def handleEthPhyIntf( self, intfId, master ):

      def getPhyId( intfId ):
         if self.config_ is None:
            return None
         if not hasattr( self.config_, 'phy' ):
            # Rarely the config_ mount may not be fully complete. See BUG672142
            traceError( "Error: getPhyId: phy is None for", intfId )
            qtError( "Error: getPhyId: phy is None for", qv( intfId ) )
            return None
         for phy in self.config_.phy.values():
            if phy.intfId == intfId:
               return phy
         return None

      def doMaybeRemoveOldReactors( intfId ):
         # Handle phy removal case only for cEOSR platform and is NO-OP for other
         # platforms.
         # Datapath interfaces are inserted and removed online on cEOSR
         # whenever the associated application container is created/deleted.
         # We need to delete stale reactors on interface removal,
         # as interfaces are inserted and removed very frequently and number
         # of the stale reactors can keep increasing.
         if not VeosHypervisor.platformCeosR():
            return

         if ( self.episDirReactor_ and
              intfId in self.episDirReactor_.ethPhyIntfStatusReactors_ ):
            traceDetail( "handleEthPhyIntf: " +
               "deleting ethPhyIntfStatusReactor for %s " % intfId )
            del self.episDirReactor_.ethPhyIntfStatusReactors_[ intfId ]
         if ( self.epicDirReactor_ and
              intfId in self.epicDirReactor_.ethPhyIntfConfigReactors_ ):
            traceDetail( "handleEthPhyIntf: " +
               "deleting ethPhyIntfConfigReactor for %s " % intfId )
            del self.epicDirReactor_.ethPhyIntfConfigReactors_[ intfId ]

      # Create EthPhyIntfStatus and EthPhyIntfConfig reactors here
      # Once we have all 3 entities: Config, Status and Phy
      # Note: Phy removal case is deliberately not being handled( except for cEOSR )
      # because we do not expect Management phys to ever get removed
      # and because handling the removal has wider implications on
      # dependent code that we decided not to deal with at this time.
      phy = getPhyId( intfId )
      if phy is None:
         traceDetail( "handleEthPhyIntf: can't get phy for %s " % intfId )
         doMaybeRemoveOldReactors( intfId )
         return
      if self.episDir_ is None or intfId not in self.episDir_.intfStatus:
         traceDetail( "handleEthPhyIntf: no episDir entry for intf %s " % intfId )
         doMaybeRemoveOldReactors( intfId )
         return
      if self.epicDir_ is None or intfId not in self.epicDir_.intfConfig:
         traceDetail( "handleEthPhyIntf: no epicDir entry for intf %s " % intfId )
         doMaybeRemoveOldReactors( intfId )
         return

      eis = self.episDir_.intfStatus[ intfId ]
      eis.maxMtu = phy.maxMtu
      eixs = self.ethIntfXcvrStatusDir.newIntfXcvrStatus( intfId )
      eic = self.epicDir_.intfConfig[ intfId ]

      assert master
      # for now, it is possible that episDir exists, but episDirReactor
      # is not created yet
      if self.episDirReactor_ is None:
         traceDetail( "handleEthPhyIntf: no episDirReactor_ for intf %s" % intfId )
         return
      if intfId not in self.episDirReactor_.ethPhyIntfStatusReactors_:
         self.episDirReactor_.ethPhyIntfStatusReactors_[ intfId ] = \
                              EthIntfStatusReactor( eis, eixs, master, phy )
      if intfId not in self.epicDirReactor_.ethPhyIntfConfigReactors_:
         self.epicDirReactor_.ethPhyIntfConfigReactors_[ intfId ] = \
                              EthIntfConfigReactor( eic, eis, eixs, master, phy,
                                                    self.globalIntfConfig_,
                                                    self.intfHwCapability_,
                                                    self.redundancyStatusActive_,
                                                    self.status_,
                                                    self.maIntfBringUpHelper_,
                                                    self.entityManager )

   def numStatusUpdates( self ):
      '''Return the number of times we have polled for status updates.  For use by
      tests.'''
      return getattr( self, 'numStatusUpdates_', 0 )

   def numConfigSyncs( self, intf ):
      '''Return the number of times we have reacted to configuration changes.  For
      use by tests.'''
      return getattr( self.epicDirReactor_.ethPhyIntfConfigReactors_[ intf ],
                      'numConfigSyncs_',
                      0 )

   def warm( self ):
      # Because the EthIntfConfigReactor uses an activity to defer syncing
      # the sysdb configuration to the kernel, the default wait-for-warmup
      # implementation is not sufficient. PhyEthtool is only considered
      # warm when there are no pending sync activities to push configuration
      # to the kernel.
      if not self.configReactor_:
         # Haven't finished mounting my state yet
         return False
      for phyReactor in self.configReactor_.phyReactors_.values():
         if not phyReactor.warm():
            return False
      return True

   @staticmethod
   def statusUpdateInterval():
      return PhyEthtool.statusUpdateInterval_

   @staticmethod
   def statusUpdateIntervalIs( secs ):
      PhyEthtool.statusUpdateInterval_ = secs

   @staticmethod
   def syncConfigInterval():
      return PhyEthtool.syncConfigInterval_

   def _handleRedundancyStatus( self, notifiee=None, switchoverStage=None ):
      
      if not isModular():
         traceDetail( "Not modular, return" )
         qtDetail( "Not modular, return" )
         return
         
      if self.redundancyStatusActive_ or self.status_.maIntfBringUpCompleteOnActive:
         traceDetail( "Active already determined or intf bring up done, return" )
         qtDetail( "Active already determined or intf bring up done, return" )
         if self.status_.maIntfBringUpCompleteOnActive:
            self.maIntfBringUpHelper_ = None
         return

      traceDetail( "handleRedundancyStatus redundancyMode:",
                   self.redundancyStatus_.mode )
      qtDetail( "handleRedundancyStatus redundancyMode:",
               qv( self.redundancyStatus_.mode ) )

      ssoMaIntfBringUpStage = isSsoMaIntfBringUpStage( self.redundancyStatus_ )
      
      traceDetail( "SSO Mgmt intf bringup stage is",
                   ssoMaIntfBringUpStage )
      qtDetail( "SSO Mgmt intf bringup stage is",
                qv( ssoMaIntfBringUpStage ) )
      
      if ( self.redundancyStatus_.mode == RedundancyMode.active or
           ssoMaIntfBringUpStage ):
         traceDetail( "Set redundancyStatusActive to True" )
         self.redundancyStatusActive_ = True
         self.maIntfBringUpHelper_ = MaIntfBringUpHelper(
               self.redundancyStatus_, self.status_,
               self.agentName, self.switchoverAgentStatus_ ) 
         if self.epicDirReactor_:
            for reactor in self.epicDirReactor_.ethPhyIntfConfigReactors_.values():
               reactor.handleRedundancyStatusActive( self.maIntfBringUpHelper_ )
            self.maIntfBringUpHelper_.maybeSetCompleted()

class MaIntfBringUpHelper:
   def __init__( self, redundancyStatus, phyEthtoolStatus,
                 agentName, switchoverAgentStatus ):
      traceDetail( "MaIntfBringUpHelper.__init__ start" )
      self.redundancyStatus_ = redundancyStatus
      self.phyEthtoolStatus_ = phyEthtoolStatus
      self.agentName_ = agentName
      self.switchoverAgentStatus_ = switchoverAgentStatus
      self.timeOutClockNotifiee_ = Tac.ClockNotifiee(
            handler=self._handleTimeout )
      self.timeOutClockNotifiee_.timeMin = Tac.now() + 30
      self.maIntfBringUpDevices = {}
      self.operstateReactors = {}
      self.kniReactors = {}
      traceDetail( "MaIntfBringUpHelper.__init__ end" )

   def _updateStatusAndSsoStage( self ):
      ssoMaIntfBringUpStage = isSsoMaIntfBringUpStage( self.redundancyStatus_ )
      traceDetail( "SSO Mgmt intf bringup stage is ", 
                   ssoMaIntfBringUpStage )
      traceDetail( "Mgmt intf bringup complete is ", 
                   self.phyEthtoolStatus_.maIntfBringUpCompleteOnActive )
      if ssoMaIntfBringUpStage:
         traceDetail( "Set %s stage complete" % ssoStageMaIntfBringUp )
         qtDetail( "Set", qv( ssoStageMaIntfBringUp ), "stage complete" )
         agentStatusKey = Tac.Value( 
            "Stage::AgentStatusKey",
            self.agentName_, 
            ssoStageMaIntfBringUp, 
            StageMgr.defaultStageInstance )
         self.switchoverAgentStatus_.complete[ agentStatusKey ] = True
      traceDetail( "Set maIntfBringUp complete" )
      qtDetail( "Set maIntfBringUp complete" )
      self.phyEthtoolStatus_.maIntfBringUpCompleteOnActive = True
      self.maIntfBringUpDevices.clear()
      self.kniReactors.clear()
      self.operstateReactors.clear()
      self.timeOutClockNotifiee_.timeMin = Tac.endOfTime

   def _handleTimeout( self ):
      traceDetail( "Mgmt intf bringup timed out" )
      qtDetail( "Mgmt intf bringup timed out")
      self._updateStatusAndSsoStage()

   def maybeSetCompleted( self ):
      if len( self.maIntfBringUpDevices ) == 0:
         self._updateStatusAndSsoStage()

   def handleDeviceReady( self, deviceName ):
      assert not self.phyEthtoolStatus_.maIntfBringUpCompleteOnActive
      if deviceName in self.maIntfBringUpDevices and \
         self.maIntfBringUpDevices[ deviceName ]:
         del self.maIntfBringUpDevices[ deviceName ]
         self.maybeSetCompleted()

class KniReactor( Tac.Notifiee ):
   notifierTypeName = "KernelNetInfo::Status"

   def __init__( self, netNs, kniStatus, operstateReactor ):
      self.netNs_ = netNs
      self.kniStatus_ = kniStatus
      self.operstateReactor_ = operstateReactor
      Tac.Notifiee.__init__( self, self.kniStatus_ )

      for k in self.kniStatus_.interface:
         self.handleKniIntf( k )

   @Tac.handler( 'interface' )
   def handleKniIntf( self, key ):
      interface = self.kniStatus_.interface.get( key )
      if not interface:
         return

      self.operstateReactor_.handleInterface( interface )

class OperstateReactor:
   def __init__( self, deviceName, maIntfBringUpHelper, netNs, em ):
      traceDetail( "OperstateReactor.__init__ start" )
      self.em_ = em
      self.netNs_ = netNs
      self.deviceName_ = deviceName
      self.helper_ = maIntfBringUpHelper
      self.kniReactor_ = None
      self.kniStatus_ = None
      self.createKniReactor()
      traceDetail( "OperstateReactor.__init__ end" )

   def __del__( self ):
      traceDetail( "OperstateReactor.__del__ start" )
      self.deleteKniReactor()

   def createKniReactor( self ):
      traceDetail( "OperstateReactor.createKniReactor start" )
      shmemEm = SharedMem.entityManager( sysdbEm =self.em_ )
      self.kniStatus_ = shmemEm.doMount( f"kni/ns/{self.netNs_}/status",
                                   "KernelNetInfo::Status",
                                   Smash.mountInfo( "keyshadow" ) )
      if self.netNs_ not in self.helper_.kniReactors:
         self.kniReactor_ = GenericReactor( self.kniStatus_, [ "interface" ],
               self.handleInterface )
         self.helper_.kniReactors[ self.netNs_ ] = self.kniReactor_
      traceDetail( "OperstateReactor.createKniReactor end" )

   def deleteKniReactor( self ):
      traceDetail( "OperstateReactor.deleteKniReactor start" )
      if self.netNs_ in self.helper_.kniReactors:
         del self.helper_.kniReactors[ self.netNs_ ]
         self.kniReactor_ = None
         shmemEm = SharedMem.entityManager( sysdbEm =self.em_ )
         shmemEm.doUnmount( f"kni/ns/{self.netNs_}/status" ) 
      traceDetail( "OperstateReactor.deleteKniReactor end" )

   def handleInterface( self, notifiee=None, key=None ):
      traceDetail( "OperstateReactor.handleInterface start for: ", key )

      interface = self.kniStatus_.interface.get( key )
      if not interface:
         return

      if interface.deviceName != self.deviceName_:
         traceDetail( "Ignoring ", interface.deviceName )
         return

      if interface.deviceName not in self.helper_.maIntfBringUpDevices:
         traceDetail( "Active already determined or intf bring up done, return" )
         qtDetail( "Active already determined or intf bring up done, return" )
         return

      if interface.flags & IFF_RUNNING:
         traceDetail( "Device has operational state UP or UNKNOWN" )
         qtDetail( "Device has operational state UP or UNKNOWN" )
         isDeviceReady = True
      else:
         traceDetail( "Device has operational state DOWN" )
         qtDetail( "Device has operational state DOWN" )
         isDeviceReady = False

      if not isDeviceReady:
         self.helper_.maIntfBringUpDevices[ self.deviceName_ ] = True
         traceDetail( "OperstateReactor.handleInterface end for: ",
                      interface.deviceName )
         return
   
      self.helper_.handleDeviceReady( self.deviceName_ )
      traceDetail( "OperstateReactor.handleInterface end for: ",
                   interface.deviceName )

class IntfNsConfigReactor( Tac.Notifiee ):
   notifierTypeName = "Interface::IntfNamespaceConfig"

   def __init__( self, nsConfig, master ):
      self.intfId_ = nsConfig.intfId
      traceDetail( "Creating IntfNsConfigReactor for '%s'" % self.intfId_ )
      self.master_ = master
      Tac.Notifiee.__init__( self, nsConfig )
      self.master_.handleNetNsName( self.intfId_ )

   @Tac.handler( 'netNsName' )
   def handleNetNsName( self ):
      traceDetail( "handleNetNsName for IntfNsConfigReactor( %s )" % self.intfId_ )
      self.master_.handleNetNsName( self.intfId_ )

   def close( self ):
      traceDetail( "Deleting IntfNsConfigReactor for '%s'" % self.intfId_ )
      self.master_.handleNetNsName( self.intfId_ )
      Tac.Notifiee.close( self )

class IntfStatusLocalReactor( Tac.Notifiee ):
   notifierTypeName = 'Interface::EthPhyIntfStatusLocal'

   def __init__( self, episl, master ):
      self.intfId_ = episl.intfId
      traceDetail( "Creating IntfStatusLocalReactor for \'%s\'" % self.intfId_ )
      self.master_ = master
      Tac.Notifiee.__init__( self, episl )
      self.master_.handleNetNsName( self.intfId_ )

   def close( self ):
      traceDetail( "Deleting IntfStatusLocalReactor for \'%s\'" % self.intfId_ )
      self.master_.handleNetNsName( self.intfId_ )
      Tac.Notifiee.close( self )

class ConfigReactor( Tac.Notifiee ):

   notifierTypeName = "Hardware::Phy::EthtoolConfig"

   def __init__( self, entity, phyEthtool ):
      Tac.Notifiee.__init__( self, entity )
      traceDetail( 'Constructing PhyEthtool.ConfigReactor (', entity, ')' )
      self.phyReactors_ = {}
      self.phyReactorsFromKernelDev_ = {}
      self.phyEthtool_ = phyEthtool
      self.deviceFromIntfId_ = {}
      self.intfIdFromDevice_ = {}
      self.handlePhy( None )
      traceDetail( 'Finished constructing PhyEthtool.ConfigReactor' )

   @Tac.handler( 'phy' )
   def handlePhy( self, key ):
      traceDetail( "phy '%s' has changed" % str( key ) )
      Tac.handleCollectionChange(
         PhyReactor,
         key,
         self.phyReactors_,
         self.notifier_.phy,
         reactorArgs=( self, ) )

      # Del the interface from the stats poller
      if key is not None and key not in self.phyReactors_:
         traceDetail( "phy '%s' being deleted" % str( key ) )
         intfMgr = self.phyEthtool_.kernelInterfaceManager_
         if intfMgr and intfMgr.kernelInterfacePoller:
            intfMgr.kernelInterfacePoller.delIntf( key )

      phys = [ key ]
      if not key:
         phys = list( self.phyReactors_ )
      for phy in phys:
         if phy in self.phyReactors_:
            intfId = self.phyReactors_[ phy ].notifier_.intfId
            sliceName = str( Cell.cellId() )
            kdev = self.phyEthtool_.epiDescDir_.get( sliceName ).ethPhyIntfDesc[
                  intfId ].deviceName
            self.phyReactorsFromKernelDev_[ kdev ] = self.phyReactors_[ phy ]
            self.deviceFromIntfId_[ intfId ] = kdev
            self.intfIdFromDevice_[ kdev ] = intfId

      traceDetail( "finished handling change of phy '%s'" % str( key ) )

   def intfNsConfig( self, intfId ):
      return self.phyEthtool_.intfNsConfigDir_.intfNamespaceConfig.get( intfId )

   def intfStatusLocal( self, intfId ):
      return self.phyEthtool_.intfStatusLocalDir_.intfStatusLocal.get( intfId )

   def handleNewEthIntfConfigReactor( self, eicr ):
      device = eicr.ethIntfStatus_.deviceName
      intfId = eicr.ethIntfConfig_.intfId
      self.deviceFromIntfId_[ intfId ] = device
      self.intfIdFromDevice_[ device ] = intfId

   def handleDelEthIntfConfigReactor( self, eicr, intfId ):
      device = self.deviceFromIntfId_[ intfId ]
      assert self.intfIdFromDevice_[ device ] == intfId
      del self.deviceFromIntfId_[ intfId ]
      del self.intfIdFromDevice_[ device ]

   def handleNetNsName( self, intfId ):
      traceDetail( "handleNetNsName for ConfigReactor( %s )" % ( intfId ) )
      configReactors = self.phyEthtool_.epicDirReactor_.ethPhyIntfConfigReactors_
      # If the config reactor has been set up for this interface, call it as it is
      # the one that actually sets configDirty to true and thereby processes the
      # network namespace. Note that if intfId exists in configReactors than so
      # must ethIntfConfig_ and ethIntfStatus_ in EthIntfConfigReactor which implies
      # that it is safe to call _syncConfig().
      if intfId in configReactors:
         configReactors[intfId].handleNetNsName()

class PhyReactor( Tac.Notifiee ):

   notifierTypeName = "Hardware::Phy::PhyEthtool"

   def __init__( self, phy, master ):
      Tac.Notifiee.__init__( self, phy )
      traceDetail( "Creating PhyEthtool.PhyReactor for '%s'" % phy.name )
      self.grandmaster_ = master.phyEthtool_
      self.intfId_ = phy.intfId
      self.ethIntfStatusReactor_ = None
      self.grandmaster_.handleEthPhyIntf( self.intfId_, master )
      # Add the interface to the stats poller
      intfMgr = master.phyEthtool_.kernelInterfaceManager_
      if intfMgr and intfMgr.kernelInterfacePoller:
         intfMgr.kernelInterfacePoller.addIntf( phy.name )

   def handleEthIntfStatus( self ):
      traceDetail( "PhyReactor '%s': handleEthIntfStatus() is disabled"
                   % ( self.notifier_.name ) )

   def warm( self ):
      # Return whether the associated EthIntfStatusReactor is warm
      if self.ethIntfStatusReactor_ is None:
         if self.grandmaster_.episDirReactor_ is None:
            return False
         if self.intfId_ not in ( self.grandmaster_.episDirReactor_.
                                  ethPhyIntfStatusReactors_.keys() ):
            return False
         self.ethIntfStatusReactor_ = ( self.grandmaster_.episDirReactor_.
                               ethPhyIntfStatusReactors_[ self.intfId_ ] )
      return self.ethIntfStatusReactor_.warm()

   def __del__( self ):
      traceDetail( "Deleting PhyEthtool.PhyReactor '%s'" % self.notifier_.name )
      del self.ethIntfStatusReactor_
      Tac.Notifiee.__del__( self )

class EthPhyIntfStatusDirReactor( Tac.Notifiee ):
   """ Top level reactor that tracks EthPhyIntfStatus instances in an
   EthPhyIntfStatusDir """

   notifierTypeName = "Interface::EthPhyIntfStatusDir"

   def __init__( self, episDir, master ):
      self.master_ = master
      self.ethPhyIntfStatusDir_ = episDir
      self.ethPhyIntfStatusReactors_ = {}
      Tac.Notifiee.__init__( self, episDir )
      self.handleIntfStatus( None )
      traceDetail( "created EthPhyIntfStatusDirReactor" )

   @Tac.handler( 'intfStatus' )
   def handleIntfStatus( self, intfId ):
      if not intfId:
         # Loop through all intfIds
         for intfId_ in self.notifier_.intfStatus:
            self.master_.handleEthPhyIntf( intfId_, self.master_.configReactor_ )
         return
      self.master_.handleEthPhyIntf( intfId, self.master_.configReactor_ )

class EthPhyIntfConfigDirReactor( Tac.Notifiee ):
   """ Top level reactor that tracks EthPhyIntfConfig instances in an
   EthPhyIntfConfigDir """

   notifierTypeName = "Interface::EthPhyIntfConfigDir"

   def __init__( self, epicDir, master ):
      self.master_ = master
      self.ethPhyIntfConfigDir_ = epicDir
      self.ethPhyIntfConfigReactors_ = {}
      Tac.Notifiee.__init__( self, epicDir )
      self.handleIntfConfig( None )
      traceDetail( "created EthPhyIntfConfigDirReactor" )

   @Tac.handler( 'intfConfig' )
   def handleIntfConfig( self, intfId ):
      if not intfId:
         # Loop through all intfIds
         for intfId_ in self.notifier_.intfConfig:
            self.master_.handleEthPhyIntf( intfId_, self.master_.configReactor_ )
         return
      self.master_.handleEthPhyIntf( intfId, self.master_.configReactor_ )

class EthIntfStatusReactor( Tac.Notifiee ):
   """ Waits for EthIntfStatus.intfConfig to be set and creates an
   EthIntfConfigReactor for it. """

   notifierTypeName = "Interface::EthIntfStatus"

   def __init__( self, eis, eixs, master, phy ):
      self.master_ = master
      self.phy = phy
      self.intfId_ = phy.intfId
      self.ethIntfXcvrStatus_ = eixs
      self.ethIntfConfigReactor_ = None
      Tac.Notifiee.__init__( self, eis )
      traceDetail( "Created EthIntfStatusReactor for interface  %s " % eis.intfId )
      qtDetail( "Created EthIntfStatusReactor for", qv( eis.intfId ) )

   @Tac.handler( 'forwardingModel' )
   def handleForwardingModel( self ):
      if self.ethIntfConfigReactor_ is not None:
         self.ethIntfConfigReactor_.handleForwardingModel()

   @Tac.handler( 'deviceName' )
   def handleDevName( self ):
      traceDetail( "deviceName handler called for %s " % self.phy.name )
      if self.notifier_.deviceName:
         traceDetail( "DeviceName added %s" % self.notifier_.deviceName )
         intfMgr = self.master_.phyEthtool_.kernelInterfaceManager_
         if intfMgr and intfMgr.kernelInterfacePoller:
            intfMgr.kernelInterfacePoller.addIntf( self.phy.name )

   def warm( self ):
      # Return whether the associated EthIntfConfigReactor is warm.
      if self.ethIntfConfigReactor_ is None:
         if self.master_.phyEthtool_.epicDirReactor_ is None:
            return False
         ethPhyIntfConfigReactors = ( self.master_.phyEthtool_.epicDirReactor_.
                                      ethPhyIntfConfigReactors_ )
         if self.intfId_ not in ethPhyIntfConfigReactors:
            return False
         self.ethIntfConfigReactor_ = ethPhyIntfConfigReactors[ self.intfId_ ]
      # We don't consider ourselves warm
      # until the ethIntfConfigReactor exists
      return self.ethIntfConfigReactor_.warm()

class GlobalIntfConfigReactor( Tac.Notifiee ):

   notifierTypeName = "Interface::GlobalIntfConfig"

   def __init__( self, globalIntfConfig, epicReactor ):
      traceDetail( "Creating GlobalIntfConfigReactor for '%s'" %
                   epicReactor.ethIntfConfig_.intfId )
      self.epicReactor = weakref.proxy( epicReactor )
      Tac.Notifiee.__init__( self, globalIntfConfig )

   @Tac.handler( 'l3Mtu' )
   def handleL3Mtu( self ):
      traceDetail( "handleL3Mtu configured global L3 MTU is ",
                   self.notifier_.l3Mtu )
      qtDetail( "handleL3Mtu", qv( self.notifier_.l3Mtu ) )
      self.epicReactor._configDirtyIs( True )  # pylint: disable-msg=protected-access

class EthIntfConfigReactor( Tac.Notifiee ):

   notifierTypeName = "Interface::EthPhyIntfConfig"

   def __init__( self, ethIntfConfig, ethIntfStatus,
                 ethIntfXcvrStatus, master, phy, globalIntfConfig,
                 intfHwCapability, redundancyStatusActive, phyEthtoolStatus,
                 maIntfBringUpHelper, em ):
      traceDetail( "Creating EthIntfConfigReactor for '%s'" % ethIntfConfig.intfId )
      assert ethIntfConfig.intfId == ethIntfStatus.intfId
      self.ethIntfConfig_ = ethIntfConfig
      self.ethIntfStatus_ = ethIntfStatus
      self.ethIntfXcvrStatus_ = ethIntfXcvrStatus
      self.name_ = ethIntfConfig.intfId
      self.master_ = master
      self.phy = phy
      self.globalIntfConfig_ = globalIntfConfig
      self.intfHwCapability_ = intfHwCapability
      self.redundancyStatusActive_ = redundancyStatusActive
      self.phyEthtoolStatus_ = phyEthtoolStatus
      self.maIntfBringUpHelper_ = maIntfBringUpHelper
      self.entityManager_ = em

      # OperStatusReactor manages the intf operStatus.
      self.operStatusReactor_ = Tac.newInstance(
         "EthIntf::OperStatusReactor", ethIntfStatus.intfId,
         ethIntfStatus, ethIntfXcvrStatus )

      # Default and max intervals for syncing the config to
      # the kernel devices
      self.syncConfigInterval_ = PhyEthtool.syncConfigInterval()
      self.maxSyncConfigInterval_ = 120

      self.master_.handleNewEthIntfConfigReactor( self )
      # If this is an interface which supports the global mtu config,
      # create the necessary reactors.
      # We only do this for switched interfaces (i.e. eth ports in Sfa), not
      # for management or internal interfaces.
      if self.phy.role == 'Switched':
         self.effectiveL3Mtu_ = Tac.newInstance( "Interface::EffectiveL3Mtu" )
         self.l3MtuSm = Tac.newInstance(
            "Interface::L3MtuSm",
            self.ethIntfConfig_,
            self.globalIntfConfig_,
            self.effectiveL3Mtu_,
            self.intfHwCapability_ )
         self.gicReactor = GlobalIntfConfigReactor( self.globalIntfConfig_, self )

      # The syncConfigActivity synchronizes the kernel and tac model state
      self.numConfigSyncs_ = 0
      self.syncConfigClockNotifiee_ = Tac.ClockNotifiee(
         handler=self._syncConfig )
      self.syncConfigClockNotifiee_.timeMin = Tac.endOfTime
      self.configDirty_ = False

      Tac.Notifiee.__init__( self, ethIntfConfig )
      qtDetail( "Created EthIntfConfigReactor for", qv( self.name_ ) )
      self._configDirtyIs( True )

      # XXX_LWR: fill in the EthIntfStatus's default values here
      #          (capabilities)?

      # We need to loop through all the configured namespaces and delete
      # stale devices from the previous run. With a scale of 1K VRFs we need
      # to do this in a timer, because a single call to delL3IntfWithNoException
      # (which deletes a device from one namespace) takes approximately 200ms.
      root = Tac.newInstance( "KernelNetNsInfo::Root", "root" )
      self.knim = Tac.newInstance( 'KernelNetNsInfo::Manager', root )

      # Backlog from which we will consume items in deferredNetCleanup().
      self.netCleanupBacklog_ = list( root.netNsByName )
      self.netCleanupActivity_ = Tac.ClockNotifiee(
         self.deferredNetCleanup )
      self.scheduleCleanupActivity()

   def scheduleCleanupActivity( self, delay=0 ):
      self.netCleanupActivity_.timeMin = Tac.now() + delay

   def deferredNetCleanup( self ):
      status = self.ethIntfStatus_
      statusLocal = self.master_.intfStatusLocal( self.ethIntfConfig_.intfId )

      traceDetail( "deferredNetCleanup() Items in cleanup backlog = '%s'"
         % len( self.netCleanupBacklog_ ) )

      # We can clear the timer here. Instead set the delay so that the timer never
      # fires. This allows us to debug the timer state.
      if not self.netCleanupBacklog_:
         self.scheduleCleanupActivity( delay=Tac.endOfTime )
         return

      timeTaken = 0
      index = 0
      for index, ns in enumerate( self.netCleanupBacklog_ ):
         startTime = Tac.now()
         self.delL3IntfWithNoException( ns, 'ma-vrf' )
         if ns not in ( DEFAULT_NS, statusLocal.netNsName ):
            self.delL3IntfWithNoException( ns, status.deviceName )
         timeTaken += ( Tac.now() - startTime )
         if timeTaken > PhyEthtool.netCleanupYieldDuration_:
            break
      traceDetail( "deferredNetCleanup() Processed '%s' items" % ( index + 1 ) )
      self.netCleanupBacklog_ = self.netCleanupBacklog_[ index + 1 : ]
      self.scheduleCleanupActivity()

   def __del__( self ):
      traceDetail( "Deleting EthIntfConfigReactor '%s'" % self.notifier_.intfId )
      self.master_.handleDelEthIntfConfigReactor( self, self.notifier_.intfId )
      Tac.Notifiee.__del__( self )

   def warm( self ):
      # The EthIntfConfigReactor is warm once all configuration has been synced
      # to the kernel.
      return not self.configDirty_

   def delL3IntfWithNoException( self, ns, intf ):
      try:
         runMaybeInNetNs( ns, [ 'ip', 'link', 'delete', intf ],
                          stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )
      except Tac.SystemCommandError:
         traceDetail( "( %s ) Failed removal of device %s in ns %s" %
                      ( self.name_, intf, ns ) )

   def isMacvlanVepaMode( self, ns, intf ):
      assert ns != DEFAULT_NS
      traceDetail( f"isMacvlanVepaMode( {self.name_}, {ns} )" )
      qtDetail( "isMacvlanVepaMode(", qv( self.name_ ), ",", qv( ns ), ")" )
      output = runMaybeInNetNs( ns, ipLinkShowDetail( intf ), stdout=Tac.CAPTURE )
      return 'macvlan mode vepa' in output

   def createL3Intf( self, ns, intf ):
      traceDetail( f"createL3Intf( {ns}, {self.name_} )" )

      try:
         # Create the logical layered interface with a temp name
         # and move into the namespace of the vrf
         driver = 'macvlan'
         # Only one passthru macvlan can be created. So, on active
         # supe in mudular dut, we create in VEPA mode because the
         # Ma0 interface also creates another VEPA mode macvlan.
         mode = 'vepa' if self.redundancyStatusActive_ else 'passthru'
         if isCeosLab():
            kernelDriver = ethtool.gdrvinfo( intf ).driver
            # Some cEOS-lab instances use a macvlan interface for the management
            # interface, and you can't stack a macvlan above a macvlan in this
            # way, so in that case we use ipvlan.  ipvlan works similarly to
            # macvlan, but can not support DHCP.
            if kernelDriver == 'macvlan':
               driver = 'ipvlan'
               mode = 'l2'
         traceDetail( "Setting interface to ", driver, " ", mode,
               "mode ( ", self.name_, " )" )
         qtDetail( "Setting interface to mode", qv( driver ), qv( mode ),
                                                qv( self.name_ ) )

         Tac.run( [ 'ip', 'link', 'add', 'ma-vrf', 'link', intf,
                    'netns', ns, 'type', driver, 'mode', mode ],
                  asRoot=True, stdout=Tac.CAPTURE )

         # rename interface to intfStatus.deviceName in the vrf
         runMaybeInNetNs( ns, ipLinkSet() + [ 'dev', 'ma-vrf', 'name',
                                intf ], stdout=Tac.CAPTURE )
         runMaybeInNetNs( ns, ipLinkSetStatus( intf, 'up' ), stdout=Tac.CAPTURE )
      except Tac.SystemCommandError as e:
         # Log this error
         traceError( self.name_, " :Failed to add device", intf, "in ns ", ns )
         traceError( "%s", e )
         qtError( "(", qv( self.name_ ), ") Failed to add device", qv( intf ),
                  "in ns", qv( ns ) )
         qtError( qv( e ) )
         ETH_VRF_ASSIGNMENT_FAILED( intf, ns )

         # Cleanup just in case
         self.delL3IntfWithNoException( ns, 'ma-vrf' )
         self.delL3IntfWithNoException( ns, intf )
         return False
      return True

   def handleForwardingModel( self ):
      traceDetail( "handleForwardingModel( %s )" % self.name_ )
      qtDetail( "handleForwardingModel", qv( self.name_ ) )
      self._configDirtyIs( True )

   def handleNetNsName( self ):
      traceDetail( "handleNetNsName( %s )" % self.name_ )
      qtDetail( "handleNetNsName", qv( self.name_ ) )
      self._configDirtyIs( True )

   def getSrcAndDstNs( self ):
      statusLocal = self.master_.intfStatusLocal( self.name_ )
      nsConfig = self.master_.intfNsConfig( self.name_ )
      dstNs = DEFAULT_NS
      srcNs = DEFAULT_NS
      if statusLocal:
         assert statusLocal.netNsName != ""
         srcNs = statusLocal.netNsName
      if statusLocal and nsConfig:
         assert nsConfig.netNsName != ""
         dstNs = nsConfig.netNsName
      return ( srcNs, dstNs )

   def handleL3Intf( self ):
      status = self.ethIntfStatus_
      statusLocal = self.master_.intfStatusLocal( self.name_ )
      ( srcNs, dstNs ) = self.getSrcAndDstNs()
      traceDetail( "handleL3Intf( {} ) srcNs={} dstNs={}".format(
         self.name_, srcNs, dstNs ) )
      success = True
      if srcNs == dstNs:
         # In case of modulars, since active to standby transition
         # happens only by reboot, we dont have to deal with changing
         # macvlan from VEPA to passthru mode. We simply return
         # success if we are not active as we would have already
         # created macvlan in passthru mode (non-default NS)
         if not self.redundancyStatusActive_:
            traceDetail( "Redundancy status not active. return sucess" )
            qtDetail( "Redundancy status not active. return sucess" )
            return success
         
         if dstNs == DEFAULT_NS:
            traceDetail( "Interface %s is in default ns. "
                         "Return success" % self.name_ )
            qtDetail( "Interface", qv( self.name_ ), "is in default ns. "
                      "Return success" )
            return success

         if self.isMacvlanVepaMode( dstNs, status.deviceName ):
            traceDetail( "Macvlan in VEPA mode. Return sucess" )
            qtDetail( "Macvlan in VEPA mode. Return sucess" )
            return success

      if srcNs != DEFAULT_NS:
         self.delL3IntfWithNoException( srcNs, status.deviceName )
         status.routedAddr = status.addr

      if dstNs != DEFAULT_NS:
         # Remove any left out ma-vrf/ma devices
         self.delL3IntfWithNoException( dstNs, 'ma-vrf' )
         self.delL3IntfWithNoException( dstNs, status.deviceName )

         success = self.createL3Intf( dstNs, status.deviceName )

      # set the netNsName
      if statusLocal:
         statusLocal.netNsName = dstNs if success else DEFAULT_NS
      return success

   @Tac.handler( 'defaultConfig' )
   def handleDefaultConfig( self ):
      # When the default config is changed, we need to do a complete
      # resync, too.
      traceDetail( "handleDefaultConfig( %s )" % self.name_ )
      qtDetail( "handleDefaultConfig", qv( self.name_ ) )
      self._configDirtyIs( True )

   @Tac.handler( 'addr' )
   def handleAddr( self ):
      traceDetail( "handleAddr( %s )" % self.name_ )
      qtDetail( "handleAddr", qv( self.name_ ) )
      self._configDirtyIs( True )

   @Tac.handler( 'enabledStateLocal' )
   def handleEnabledStateLocal( self ):
      traceDetail( "handleEnabledStateLocal( %s )" % self.name_ )
      qtDetail( "handleEnabledStateLocal", qv( self.name_ ) )
      self._configDirtyIs( True )

   @Tac.handler( 'advertisedModesLocal' )
   def handleAdvertisedModesLocal( self ):
      traceDetail( "handleAdvertisedModes( %s )" % self.name_ )
      qtDetail( "handleAdvertisedModes", qv( self.name_ ) )
      self._configDirtyIs( True )

   @Tac.handler( 'txFlowcontrolLocal' )
   def handleTxFlowcontrolLocal( self ):
      traceDetail( "handleTxFlowcontrol( %s )" % self.name_ )
      qtDetail( "handleTxFlowControl", qv( self.name_ ) )
      self._configDirtyIs( True )

   @Tac.handler( 'rxFlowcontrolLocal' )
   def handleRxFlowcontrolLocal( self ):
      traceDetail( "handleRxFlowcontrol( %s )" % self.name_ )
      qtDetail( "handleRxFlowControl", qv( self.name_ ) )
      self._configDirtyIs( True )

   @Tac.handler( 'linkModeLocal' )
   def handleLinkModeLocal( self ):
      traceDetail( "handleLinkMode( %s )" % self.name_ )
      qtDetail( "handleLinkMode", qv( self.name_ ) )
      self._configDirtyIs( True )

   @Tac.handler( 'mtu' )
   def handleMtu( self ):
      traceDetail( "handleMtu(", self.name_,
          ") configured mtu is", self.notifier_.mtu )
      qtDetail( "handleMtu", qv( self.name_ ) )
      self._configDirtyIs( True )

   @Tac.handler( 'l3MtuConfigured' )
   def handleL3MtuConfigured( self ):
      traceDetail( "handleL3MtuConfigured(", self.name_,
          ") l3MtuConfigured is set to", self.notifier_.l3MtuConfigured )
      qtDetail( "handleL3MtuConfigured", qv( self.name_ ) )
      self._configDirtyIs( True )

   def handleRedundancyStatusActive( self, maIntfBringUpHelper ):
      traceDetail( "handleRedundancyStatusActive(", self.name_, ")" )
      qtDetail( "handleRedundancyStatusActive(", qv( self.name_ ), ")" )
     
      assert not self.phyEthtoolStatus_.maIntfBringUpCompleteOnActive,\
            "Management Intf bring up is complete"

      self.maIntfBringUpHelper_ = maIntfBringUpHelper
      if not self.redundancyStatusActive_:
         self.redundancyStatusActive_ = True
         ( srcNs, dstNs ) = self.getSrcAndDstNs()
         if srcNs == dstNs and dstNs == DEFAULT_NS:
            return
         status = self.ethIntfStatus_
         deviceName = status.deviceName
         enabled = status.linkStatus != 'linkDown'
         if enabled:
            traceDetail(
                  "Adding %s to mgmt intf bring up devices" % deviceName )
            qtDetail(
                  "Adding", qv( deviceName ), "to mgmt intf bring up devices" )
            self.maIntfBringUpHelper_.maIntfBringUpDevices[ deviceName ] = False
            self.maIntfBringUpHelper_.operstateReactors[ deviceName ] = \
                  OperstateReactor( deviceName, self.maIntfBringUpHelper_, dstNs,
                                    self.entityManager_ )

      self._configDirtyIs( True )

   def _configDirtyIs( self, configDirty ):
      # Always reset our config interval back to the default, since
      # something has changed and we want to address it "soon"
      self.syncConfigInterval_ = PhyEthtool.syncConfigInterval()

      if self.configDirty_ != configDirty:
         qtDetail( "change configDirty flag for", qv( self.name_ ), "to",
                   qv( configDirty ) )

      self.configDirty_ = configDirty
      if configDirty:
         resTime = Tac.now() + self.syncConfigInterval_
      else:
         resTime = Tac.endOfTime
      self.syncConfigClockNotifiee_.timeMin = resTime

   # Set the mac address of the physical interface to the mac address
   # of the interface present in the vrf. Program the mac address of the
   # interface in the vrf to the config mac address.
   def _setPhyVirtMacAddr( self, ns, intf, configMacAddr):

      Tac.run( ipLinkSetStatus( intf, 'down' ), stdout=Tac.CAPTURE )
      runMaybeInNetNs( ns, ipLinkSetStatus( intf, 'down' ),
            stdout=Tac.CAPTURE )

      line = runMaybeInNetNs( ns, ipLinkShow( intf ), stdout=Tac.CAPTURE )
      m = macRex.search( line )
      if m:
         virtualMacAddr = m.group( 1 )
         Tac.run( ipLinkSetAddr( intf, virtualMacAddr ),
            asRoot=True, stdout=Tac.CAPTURE )
         runMaybeInNetNs( ns, ipLinkSetAddr( intf, configMacAddr ),
            asRoot=True, stdout=Tac.CAPTURE )
         traceDetail( "Set default namespace "
               "mac to ", virtualMacAddr, ", Set ", ns, "namespace ",
               "mac to ", configMacAddr )
         qtDetail( "Set default namespace mac to",
               qv( virtualMacAddr ), ", Set", qv( ns ), "namespace ",
               "mac to", qv( configMacAddr ) )
      else:
         traceError( "Error getting management interface mac address"
               " in namespace:", ns, intf )
         qtError( "Error getting management interface mac address "
               " in namespace:", qv( ns ), qv( intf ) )

      Tac.run( ipLinkSetStatus( intf, 'up' ), stdout=Tac.CAPTURE )
      runMaybeInNetNs( ns, ipLinkSetStatus( intf, 'up' ),
            stdout=Tac.CAPTURE )

   def _syncConfig( self ):
      """
      Push the interface's phy-layer configuration into the hardware:
        o MAC address
        o mtu
        o 'enabled' state
        o link mode (enable autoneg or force the configured speed/duplex)
        o disable WOL (Wake-on-LAN)

      On failure, a log is written, and the synchronization is retried by calling
      _configDirtyIs """

      self.numConfigSyncs_ += 1

      # Handle any namespace changes
      if not self.handleL3Intf():
         # Something went wrong and the interface wasn't created
         self._configDirtyIs( True )
         return

      # This should only get run when the config is dirty
      assert self.configDirty_

      config = self.ethIntfConfig_
      status = self.ethIntfStatus_

      traceDetail( "syncConfig of interface", config.intfId )

      # Some sanity checks
      assert config
      assert status
      assert config.intfId == status.intfId

      if not status.deviceName:
         # If we don't know the kernel device name we're associated with,
         # reschedule ourselves.
         traceDetail( status.intfId,
                      "device name not set. Rescheduling sync config" )
         self._configDirtyIs( True )
         return

      # Go ahead and poll status.  This isn't really needed except in the
      # special case that status has never been polled for this interface.
      # It's actually somewhat hard to tell for sure, and config updates
      # are rare, so the cost of status polling seems like an acceptable
      # price to pay to close the race.  (Config syncing needs the status
      # to have been polled at least once to retrieve the supported link
      # modes).
      updateEthIntfStatus( status, self.ethIntfXcvrStatus_ )

      syncSuccessful = True
      
      intfName = status.intfId
      deviceName = status.deviceName

      try:
         driver = ethtool.gdrvinfo( deviceName ).driver
         limitedDriverMode = ( driver in limitedDrivers )
      except OSError:
         limitedDriverMode = False

      mustSetAddr = True

      if config.addr == '00:00:00:00:00:00':
         # If the configured address is zero, the interface should use
         # the burned-in address.
         addr = status.burnedInAddr
         if addr == '00:00:00:00:00:00':
            # This should not happen. But I don't control the fdl.
            traceError( 'burnedInAddr is 00:00:00:00:00:00. Unusable.' )
            mustSetAddr = False
      else:
         addr = config.addr

      enabled = config.enabled
      ( _, dstNs ) = self.getSrcAndDstNs()

      # Look for the deviceName first in the non-default namespace and then in
      # the default namespace and return the output and the mac address string
      # if the device is found.
      def _findDeviceName():
         line = runMaybeInNetNs( dstNs, ipLinkShow( deviceName ),
               stdout=Tac.CAPTURE )
         m = macRex.search( line )
         if not m:
            traceDetail( "Unable to get MAC addr for intf %s " 
                         "in ns %s, looking in default NS" % (
                        ( deviceName,  dstNs ) ) )
            qtDetail( "Unable to get MAC addr for intf", qv( deviceName ),
                      "in ns", qv( dstNs ), ", looking in default NS" )
            line = Tac.run( ipLinkShow( deviceName ), stdout=Tac.CAPTURE )
            # Look for the following info in the output line
            #  ... link/ether 00:11:ee:22:ff:33 ...
            m = macRex.search( line )
         return line, m

      # Find out which MAC address the device is currently using.  If
      # it's already using the required address, there's nothing to do.
      # Check if the device is present in a non-default vrf. If it is,
      # use that mac address. In case the device is in pass-thru mode,
      # the mac address will be same as the physical mac address so it
      # doesn't change the behaviour. However, when in vepa mode, we will
      # consider the mac address inside the vrf since the mac address will
      # be swapped with the physical management device and that is what
      # we're interested in.
      try:
         line, m = _findDeviceName()

         if m:
            # Update the EthIntfStatus object with the old MAC address.
            # This way, if we don't need to change the MAC address, or if
            # we fail to set the new MAC address, the status object will
            # correctly reflect the current status.
            oldAddr = m.group( 1 )
            status.addr = oldAddr
            status.routedAddr = oldAddr

            if addr == oldAddr:
               mustSetAddr = False
         else:
            # This is an unexpected error.  We'll just ignore it silently
            # and if we fail to set the MAC address as well then the code
            # below will log.
            traceError( "Surprising output of 'ip -o link show %s': '%s'" %
                        ( deviceName, line ) )
            qtError( "Surprising output of 'ip -o link show ",
                  qv( deviceName ), "'", qv( line ) )
      except Tac.SystemCommandError as e:
         # The most likely cause of this failure is that the kernel
         # device has just gone away.  We ignore this failure silently
         # and attempt to set the MAC address anyway - if that fails too,
         # the code below will log.
         traceError( 'Error getting the MAC address: %s' % e )
         syncSuccessful = False

      if mustSetAddr:
         try:
            # Set the new MAC address.  Take the link down first, as most
            # linux network drivers won't allow us to change the MAC address
            # of an active link.  (We'll re-enable it below if necessary.)
            traceNormal( "syncConfig: ip link set %s addr %s"
                        % ( deviceName, addr ) )

            def setMacAddrinNs( intf, ns ):
               if ns == DEFAULT_NS:
                  Tac.run( ipLinkSetStatus( intf, 'down' ),
                        stdout=Tac.CAPTURE )
                  Tac.run( ipLinkSetAddr( intf, addr ),
                        stdout=Tac.CAPTURE )
                  Tac.run( ipLinkSetStatus( intf, 'up' ),
                        stdout=Tac.CAPTURE )
               else:
                  runMaybeInNetNs( ns, ipLinkSetStatus( intf, 'down' ),
                        stdout=Tac.CAPTURE )
                  runMaybeInNetNs( ns, ipLinkSetAddr( intf, addr ),
                        stdout=Tac.CAPTURE )
                  runMaybeInNetNs( ns, ipLinkSetStatus( intf, 'up' ),
                        stdout=Tac.CAPTURE )

            if self.redundancyStatusActive_ and dstNs != DEFAULT_NS:
               # In modular systems, on active supe, configure the mac 
               # address differently if the interface is in non-default
               # vrf. This is possible since  we've set it up in vepa mode
               traceDetail( "Swapping mac addr between VEPA macvlan and "
                            "parent interface" )
               self._setPhyVirtMacAddr( dstNs, deviceName, addr )
            else:
               traceDetail( "Set mac addr for parent intf in default" )
               setMacAddrinNs( deviceName, DEFAULT_NS )
               if dstNs != DEFAULT_NS:
                  traceDetail( "Set mac addr for macvlan intf in netns" )
                  setMacAddrinNs( deviceName, dstNs )

            # Update the EthIntfStatus object with the new MAC address.
            status.addr = addr
            status.routedAddr = addr

         except Tac.SystemCommandError as e:
            # The most likely cause of this failure is that the kernel
            # device has just gone away.  If the device has not gone
            # away, we'll retry the next time the RetrySyncActivity is
            # scheduled.
            traceError( 'Error setting the MAC address: %s' % e )
            ETH_SETADDRFAIL( intfName, addr )
            syncSuccessful = False

      # Update the mtu status based on the configuration
      isl = self.master_.intfStatusLocal( self.ethIntfStatus_.intfId )
      if self.phy.role == 'Switched':
         configOrEffectiveMtu = self.effectiveL3Mtu_.mtu
      else:
         configOrEffectiveMtu = config
      if not syncMtu( status, isl, configOrEffectiveMtu ):
         # The most likely cause of this failure is that the kernel
         # device has just gone away.  If the device has not gone away,
         # we'll retry the next time the RetrySyncActivity is scheduled.
         traceError( 'Error setting the link mtu: %s' % intfName )
         ETH_SETMTUFAIL( intfName )
         syncSuccessful = False

      try:
         # Set the 'enabled' state.
         updown = "up" if enabled else "down"
         traceNormal( "syncConfig mustSetAddr: ip link set %s %s"
                      % ( deviceName, updown ) )
         Tac.run( ipLinkSetStatus( deviceName, updown ), stdout=Tac.CAPTURE )

         if dstNs != DEFAULT_NS:
            runMaybeInNetNs( dstNs, ipLinkSetStatus( deviceName, updown ),
                  stdout=Tac.CAPTURE )
      except Tac.SystemCommandError as e:
         # The most likely cause of this failure is that the kernel
         # device has just gone away.  If the device has not gone away,
         # we'll retry the next time the RetrySyncActivity is scheduled.
         traceError( 'Error setting the link enabled status: %s' % e )
         if enabled:
            ETH_SETENFAIL( intfName )
         else:
            ETH_SETDISFAIL( intfName )
         syncSuccessful = False

      # Update MTU in the layer-3 device if present
      # ns = status.netNsName
      # if( ns and ns != DEFAULT_NS ):
      #    configuredMtu = config.mtu if config.mtu < status.maxMtu else \
      #                                  status.maxMtu
      #    strMtu = str( configuredMtu )
      #    try:
      #       runMaybeInNetNs( ns, [ 'ip', 'link', 'set',  deviceName,
      #                              'mtu', strMtu ],
      #                        stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )
      #    except Tac.SystemCommandError, e:
      #       t0( "Error running: ", "ip link set", deviceName, "mtu", strMtu )

      # Don't attempt to configure phy parameters on disabled device or
      # devices where ethtool is not allowed to configure the phy.
      if not enabled or not self.phy.ethtoolConfigEnabled:
         self._configDirtyIs( not syncSuccessful )
         if syncSuccessful:
            if not self.phyEthtoolStatus_.maIntfBringUpCompleteOnActive and \
                  self.maIntfBringUpHelper_:
               self.maIntfBringUpHelper_.maIntfBringUpDevices[ deviceName ] = True
               self.maIntfBringUpHelper_.handleDeviceReady( deviceName )
         return

      # Configure the phy parameters.

      if config.linkMode == 'linkModeUnknown':
         ETH_LINKMODEUNCONFIGURED( intfName )
         self._configDirtyIs( not syncSuccessful )
         if syncSuccessful:
            if not self.phyEthtoolStatus_.maIntfBringUpCompleteOnActive and \
                  self.maIntfBringUpHelper_:
               self.maIntfBringUpHelper_.maIntfBringUpDevices[ deviceName ] = True
               self.maIntfBringUpHelper_.handleDeviceReady( deviceName )
         return

      # Disable Wake-on-LAN
      try:
         ethtool.swol( deviceName, wolopts=0 )
      except OSError:
         traceError( 'Error disabling Wake-on-LAN' )

      if config.advertisedModesConfigured:
         modes = config.advertisedModes
      else:
         modes = status.linkModeCapabilities

      # 'supported', 'advertising', 'lp_advertising' are 3 u32 words each
      # in linux kernel 5.10. The bit for 2500Mbps is in second u32 field.
      # So to advertise or not the 2500Mbps speed, user would have to 
      # use the second u32 field. Hence the code below uses advertising_0
      # as the first u32 field and advertising_1 as the second u32 field.
      # As of now, none of the agent code is using the bits in third u32
      # field of 'advertising', hence that u32 field is ignored below
      advertising_0 = 0
      intfAdvertisedModes = []
      if isCpuIntf( intfName ):
         # Only advertise backplane speeds on CPU NICs
         intfAdvertisedModes = backplaneAdvertisedModes
      else:
         intfAdvertisedModes = advertisedModes
      for x, y in intfAdvertisedModes:
         if getattr( modes, x ):
            advertising_0 |= y

      advertising_1 = 0
      for x, y in [ (modes.mode2p5GbpsFull, ethtool.SUPPORTED_2500baseT_Full),
                  ]:
         if x:
            advertising_1 |= y
 
      supportedLinkModes = dict(
         linkMode100MbpsFull=( 100, ethtool.DUPLEX_FULL ),
         linkModeForced10MbpsHalf=( 10, ethtool.DUPLEX_HALF ),
         linkModeForced10MbpsFull=( 10, ethtool.DUPLEX_FULL ),
         linkModeForced100MbpsHalf=( 100, ethtool.DUPLEX_HALF ),
         linkModeForced100MbpsFull=( 100, ethtool.DUPLEX_FULL ),
         linkModeForced1GbpsHalf=( 1000, ethtool.DUPLEX_HALF ),
         linkModeForced1GbpsFull=( 1000, ethtool.DUPLEX_FULL ),
         linkModeForced2p5GbpsFull=( 2500, ethtool.DUPLEX_FULL ),
         linkModeForced10GbpsFull=( 10000, ethtool.DUPLEX_FULL ),
      )

      # use autoneg mode as a default one, if unsupported mode is configured
      if config.linkMode == 'linkModeAutoneg' or \
         supportedLinkModes.get( config.linkMode ) is None:
         try:
            ethtool.slinksettings( deviceName,
                                   autoneg=ethtool.AUTONEG_ENABLE,
                                   advertising_0=advertising_0,
                                   advertising_1=advertising_1 )
         except OSError as e:
            traceError( 'Error setting speed/duplex to auto/auto: %s' % e )
            if not limitedDriverMode:
               ETH_SETLINKMODEFAIL( intfName, 'auto', 'auto', e )
               syncSuccessful = False
      else:
         ( s, d ) = supportedLinkModes.get( config.linkMode )
         try:
            ethtool.slinksettings( deviceName,
                                   autoneg=ethtool.AUTONEG_DISABLE,
                                   speed=s,
                                   advertising_0=advertising_0,
                                   advertising_1=advertising_1,
                                   duplex=d )
         except OSError as e:
            traceError( f'Error forcing speed/duplex to {s}/{d}: {e}' )
            if not limitedDriverMode:
               ETH_SETLINKMODEFAIL( intfName, s, d, e )
               syncSuccessful = False

      try:
         ethtool.spauseparam(
            deviceName,
            autoneg=(config.linkMode == 'linkModeAutoneg'),
            tx_pause=(config.txFlowcontrol != 'flowControlConfigOff'),
            rx_pause=(config.rxFlowcontrol != 'flowControlConfigOff') )
      except OSError as e:
         traceError( 'Error setting pause parameters: %s' % e )
         if not limitedDriverMode:
            ETH_SETPAUSEPARAMFAIL( intfName, e )
            syncSuccessful = False

      if syncSuccessful:
         # Update configDirty and reset timeMin
         self._configDirtyIs( False )
      else:
         assert self.configDirty_

         # Double the current sync interval, up to the max, and
         # reschedule.
         # configDirty better still be False (it was above, and
         # we shouldn't have reset it in between).
         self.syncConfigInterval_ = self.syncConfigInterval_ * 2
         if self.syncConfigInterval_ > self.maxSyncConfigInterval_:
            self.syncConfigInterval_ = self.maxSyncConfigInterval_
         self.syncConfigClockNotifiee_.timeMin = \
             Tac.now() + self.syncConfigInterval_

# ------------------------------------------------------------------------------
# Update the EthIntfStatus's capabilities with the device's capabilities.
# ------------------------------------------------------------------------------
# "unknown" means "up"; see linux/netdevice.h netif_oper_up().
_sysfsOperstateStrings = { 'unknown' : 'linkUp',  # tap interfaces.
                           'notpresent' : 'linkDown',
                           'down' : 'linkDown',
                           'lowerlayerdown' : 'linkDown',
                           'testing' : 'linkDown',
                           'dormant' : 'linkDown',
                           'up' : 'linkUp' }

# ------------------------------------------------------------------------------
# Updates the contents of an EthIntfStatus object by invoking ethtool
# ------------------------------------------------------------------------------

# Generic Helper to update linkModeStatus in ethIntfStatus for 1G or 10G.
# laneCount should be updated to laneCount1 when auto-neg is ON.
# linkModeStatus should be updated to linkModeAutoneg if auto-neg is ON.
# linkModeStatus should be updated to corresponding ethLinkMode when linkMode
# is forced.
def updateOneGigOrTenGigLinkModeStatus( status, anegStatus, ethToolEthIntfStatus,
                                        supportedSpeeds ):
   if status[ 'speed' ] in supportedSpeeds:
      anegStatus.laneCount = ethLaneCount.laneCountUnknown
      if ethToolEthIntfStatus and ethToolEthIntfStatus.autoneg:
         anegStatus.laneCount = ethLaneCount.laneCount1
         lms = ethLinkMode.linkModeAutoneg
      else:
         if status[ 'duplex' ] in [ 'duplexFull', 'duplexHalf' ]:
            lms = EthTypesApi.ethSpeedLaneToLinkMode(
               status[ 'speed' ], ethLaneCount.laneCount1, status[ 'duplex' ] )
         else:
            # Invalid duplex, do not set 'linkModeStatus' and return early
            traceError( "Invalid duplex: %s", status[ 'duplex' ] )
            qtError( "Invalid duplex:", qv( status[ 'duplex' ] ) )
            return
      status[ 'linkModeStatus' ] = lms

# Helper to update linkModeStatus in ethIntfStatus for management or backplane
# interfaces. Use the correct supported speed modes to constrain updating
# linkModeStatus
def updateLinkModeStatus( intfId, status, anegStatus, ethToolEthIntfStatus ):
   supportedSpeeds = ( [ 'speed1Gbps', 'speed10Gbps' ]
                       if isCpuIntf( intfId ) else [ 'speed1Gbps' ] )
   if status[ 'speed' ] in supportedSpeeds:
      updateOneGigOrTenGigLinkModeStatus( status, anegStatus, ethToolEthIntfStatus,
                                          supportedSpeeds )

def updateAutonegCapabilities( intfId, status, ethToolLinkStatus ):
   if not isCpuIntf( intfId ):
      # For now, we only claim support for IEEE Clause 28 (10/100/1000BASE-T) on
      # management or front panel ports
      if ( ( ethToolLinkStatus.supported_0 & ethtool.SUPPORTED_Autoneg ) and
           ( ethToolLinkStatus.supported_0 & ethtool.SUPPORTED_TP or
             ethToolLinkStatus.supported_0 & ethtool.SUPPORTED_MII ) ):
         status[ 'autonegCapabilities' ] = Tac.Value(
            "Interface::AutonegCapabilities",
            mode='anegModeClause28', rxPause=False, txPause=False,
            linkModes=status['linkModeCapabilities'] )
   else:
      # For now, we only claim support for IEEE Clause 73 (10GBASE-KR)
      # for backplane interfaces. We may in the future support IEEE Clause 28 for
      # 1GBASE-KX
      if ( ( ethToolLinkStatus.supported_0 & ethtool.SUPPORTED_Autoneg ) and
             ( ethToolLinkStatus.supported_0 &
                ethtool.SUPPORTED_10000baseKR_Full ) ):
         status[ 'autonegCapabilities' ] = Tac.Value(
            "Interface::AutonegCapabilities",
            mode='anegModeClause73', rxPause=False, txPause=False,
            linkModes=status['linkModeCapabilities'] )
   if 'autonegCapabilities' not in status:
      status[ 'autonegCapabilities' ] = Tac.Value(
         "Interface::AutonegCapabilities",
         mode='anegModeUnknown', rxPause=False, txPause=False,
         linkModes=Tac.Value( "Interface::EthLinkModeSet" ) )

def updateEthIntfStatus( ethIntfStatus, ethIntfXcvrStatus ):
   tracePeriodic( "updateEthIntfStatus( %s, dev '%s' )" %
                  ( ethIntfStatus.intfId, ethIntfStatus.deviceName ) )

   if ethIntfStatus.deviceName == '':
      traceDetail( "Not updating status for '%s' yet -- device name unknown" %
                   ethIntfStatus.intfId )
      return

   status = {}
   driver = None

   try:
      driver = ethtool.gdrvinfo( ethIntfStatus.deviceName ).driver
      limitedDriverMode = ( driver in limitedDrivers )
   except OSError:
      limitedDriverMode = False

   handleDriverQuirks( driver, ethIntfStatus.deviceName )

   try:
      ethtoolLinkStatus = None
      ethtoolLinkStatus = ethtool.glinksettings( ethIntfStatus.deviceName )

      if ( limitedDriverMode and
         not ethtoolLinkStatus.supported_0 & ethtool.SUPPORTED_Autoneg ):
         raise OSError( 95, 'Operation not supported' )

      status['duplex'] = {
         ethtool.DUPLEX_HALF: 'duplexHalf',
         ethtool.DUPLEX_FULL: 'duplexFull' }.get( ethtoolLinkStatus.duplex,
               'duplexUnknown' )

      status['speed'] = {
         10: 'speed10Mbps',
         100: 'speed100Mbps',
         1000: 'speed1Gbps',
         2500: 'speed2p5Gbps',
         10000: 'speed10Gbps' }.get( ethtoolLinkStatus.speed, 'speedUnknown' )

      # Currently almost no Linux network drivers set SUPPORTED_Pause, so
      # we punt and assume the interface supports flow control
      # FIXME: should be ethtoolLinkStatus.supported & ethtool.SUPPORTED_Pause
      # pylint: disable=using-constant-test
      if True:
         status['rxFlowcontrolCapabilities'] = 'flowControlCapable'
         status['txFlowcontrolCapabilities'] = 'flowControlCapable'
      else:
         status['rxFlowcontrolCapabilities'] = 'flowControlNotCapable'
         status['txFlowcontrolCapabilities'] = 'flowControlNotCapable'
      # FIXME: what about SUPPORTED_Asym_Pause?

      intfSupportedModes = []
      if isCpuIntf( ethIntfStatus.intfId ):
         intfSupportedModes = backplaneSupportedModes
      else:
         intfSupportedModes = supportedModes

      supportedModesDict = { x : bool(ethtoolLinkStatus.supported_0 & y)
                             for x, y in  intfSupportedModes }

      if not isCpuIntf(ethIntfStatus.intfId):
         # 2500Mbps bit is in bit47.
         supportedModesDict[ 'mode2p5GbpsFull' ] =\
            bool( ethtoolLinkStatus.supported_1 & ethtool.SUPPORTED_2500baseT_Full )
      status['phyLinkModeCapabilities'] = Tac.Value( "Interface::EthLinkModeSet",
                                                     **supportedModesDict )

      # Assume that 'ethtool' already tells us the intersection of the
      # PHY and xcvr capabilities.
      status['linkModeCapabilities'] = status['phyLinkModeCapabilities']

      # Export the autoneg capabilities based on what ethtool claims.
      updateAutonegCapabilities( ethIntfStatus.intfId, status, ethtoolLinkStatus )

      anegStatus = Tac.Value( "Interface::AutonegStatus" )
      if ( ethtoolLinkStatus.autoneg and
            status[ 'autonegCapabilities'].mode != 'anegModeUnknown' ):
         anegStatus.mode = status[ 'autonegCapabilities'].mode
         # Ethtool doesn't tell us much about the outcome of autoneg, so
         # we simply say it is successful.
         anegStatus.state = 'anegStateSuccessful'

         intfAdvertisedModes = []
         if isCpuIntf( ethIntfStatus.intfId ):
            intfAdvertisedModes = backplaneAdvertisedModes
         else:
            intfAdvertisedModes = advertisedModes

         advertisedModes_ = { x : bool( ethtoolLinkStatus.advertising_0 & y )
                              for x, y in intfAdvertisedModes }

         if not isCpuIntf( ethIntfStatus.intfId ):
            advertisedModes_[ 'mode2p5GbpsFull' ] = bool(
               ethtoolLinkStatus.advertising_1 & ethtool.ADVERTISED_2500baseT_Full )
         linkModes = Tac.Value( "Interface::EthLinkModeSet", **advertisedModes_ )

         pause = 'pauseDisabled'
         anegStatus.advertisement = Tac.Value(
            "Interface::AutonegAdvertisement", linkModes=linkModes, pause=pause )
         linkPartnerModes_ = { x : bool( ethtoolLinkStatus.lp_advertising_0 & y )
                               for x, y in intfAdvertisedModes }

         if not isCpuIntf( ethIntfStatus.intfId ):
            linkPartnerModes_[ 'mode2p5GbpsFull' ] = bool( 
               ethtoolLinkStatus.lp_advertising_1 & 
               ethtool.ADVERTISED_2500baseT_Full )

         linkPartnerModes = Tac.Value( "Interface::EthLinkModeSet", 
                                       **linkPartnerModes_ )
         anegStatus.partnerAdvertisement = Tac.Value(
            "Interface::AutonegAdvertisement",
            linkModes=linkPartnerModes )

         anegStatus.speed = status[ 'speed' ]
         anegStatus.duplex = status[ 'duplex' ]

      # Update linkModeStatus related attributes.
      updateLinkModeStatus( ethIntfStatus.intfId, status, anegStatus,
                            ethtoolLinkStatus )
      status[ 'autonegStatus' ] = anegStatus

      # The updateEthIntfStatus function is called periodically from the PhyEthtool
      # class. There is a possibility that this function gets called before
      # the ConfigReactor recreates EthIntfXcvrStatuses.
      if ethIntfXcvrStatus:
         ethIntfXcvrStatus.xcvrPresence = 'xcvrPresent'
         ethIntfXcvrStatus.xcvrType = '10/100/1000'
   except OSError as e:
      if limitedDriverMode:
         # The virtio/tulip drivers unfortunately don't support any
         # of the ethtool commands to inspect the properties of the
         # interface.  We fill in a sensible default.

         # Lie and say we're at 1Gbps/full
         status['speed'] = 'speed1Gbps'
         status['duplex'] = 'duplexFull'
         # Don't claim flow control
         status['rxFlowcontrolCapabilities'] = 'flowControlNotCapable'
         status['txFlowcontrolCapabilities'] = 'flowControlNotCapable'
         # Hard to know what to say here... we'll only claim to support 1Gbps/full
         # so that you can't change the speed at the CLI
         status['phyLinkModeCapabilities'] = Tac.Value( "Interface::EthLinkModeSet",
            **dict( [ ('mode10MbpsHalf', False),
                      ('mode10MbpsFull', False),
                      ('mode100MbpsHalf', False),
                      ('mode100MbpsFull', False),
                      ('mode1GbpsHalf', False),
                      ('mode1GbpsFull', True),
                      ('mode10GbpsFull', False) ] ) )
         status['linkModeCapabilities'] = status['phyLinkModeCapabilities']
         # Claim autoneg support
         status[ 'autonegCapabilities' ] = Tac.Value(
            "Interface::AutonegCapabilities",
            mode='anegModeClause28', rxPause=False, txPause=False,
            linkModes=status['linkModeCapabilities'] )
         # Claim autoneg was successful
         anegStatus = Tac.Value( "Interface::AutonegStatus" )
         anegStatus.mode = status[ 'autonegCapabilities'].mode
         anegStatus.state = 'anegStateSuccessful'
         # Lie and say we received the same capabilities we support
         linkModes = status['phyLinkModeCapabilities']
         pause = 'pauseDisabled'
         anegStatus.advertisement = Tac.Value(
            "Interface::AutonegAdvertisement", linkModes=linkModes, pause=pause )
         anegStatus.partnerAdvertisement = Tac.Value(
            "Interface::AutonegAdvertisement" )
         anegStatus.speed = status[ 'speed' ]
         anegStatus.duplex = status[ 'duplex' ]
         # Update linkModeStatus related attributes.
         updateLinkModeStatus( ethIntfStatus.intfId, status, anegStatus,
                               ethtoolLinkStatus )
         status[ 'autonegStatus' ] = anegStatus

         if ethIntfXcvrStatus:
            ethIntfXcvrStatus.xcvrPresence = 'xcvrPresent'
            ethIntfXcvrStatus.xcvrType = '10/100/1000'
      else:
         ETH_POLLLINKSTATUSFAIL( ethIntfStatus.intfId, e )

   try:
      ethtoolLinkStatus = ethtool.gpauseparam( ethIntfStatus.deviceName )

      status['rxFlowcontrol'] = 'flowControlStatusOn' if ethtoolLinkStatus.rx_pause \
                                else 'flowControlStatusOff'
      status['txFlowcontrol'] = 'flowControlStatusOn' if ethtoolLinkStatus.tx_pause \
                                else 'flowControlStatusOff'
   except OSError as e:
      if limitedDriverMode:
         # The virtio/tulip drivers unfortunately don't support any of
         # the ethtool commands to inspect the properties of the
         # interface.  We fill in a sensible default.
         status['rxFlowcontrol'] = 'flowControlStatusOff'
         status['txFlowcontrol'] = 'flowControlStatusOff'
      else:
         ETH_POLLLINKSTATUSFAIL( ethIntfStatus.intfId, e )

   # Extract the operational status.
   # We do not use the ethtool glink command, because this returns the
   # netif_carrier_ok() result, which (according to net-sysfs.c's
   # show_carrier()) is valid only when netif_running().  However,
   # ethtool exposes it regardless.
   try:
      with open( "%s/%s/operstate" %
            ( sysClassNet, ethIntfStatus.deviceName ) ) as f:
         operstate = f.read().strip()
   except OSError:
      operstate = "down"
   traceDetail( "operstate is %s for %s" % ( operstate, ethIntfStatus.deviceName ) )
   linkVal = _sysfsOperstateStrings.get( operstate )
   if linkVal:
      status[ 'linkStatus' ] = linkVal
      if ethIntfStatus.linkStatus != linkVal:
         status[ 'linkStatusChanges' ] = ethIntfStatus.linkStatusChanges + 1
         status[ 'lastLinkStatusChangeTime' ] = Tac.now()

   # Update the status values and statistics.
   for ( attrName, value ) in status.items():
      setattr( ethIntfStatus, attrName, value )

   tracePeriodic( "updateEthIntfStatus( %s, dev '%s' ) complete" %
                  ( ethIntfStatus.intfId, ethIntfStatus.deviceName ) )
   
def main():
   container = Agent.AgentContainer( [ PhyEthtool ] )
   container.runAgents()

if __name__ == "__main__":
   main()
