#!/usr/bin/env python3
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
from TypeFuture import TacLazyType

EthIntfId = TacLazyType( 'Arnet::EthIntfId' )
FabricIntfId = TacLazyType( 'Arnet::FabricIntfId' )

# EthIntf is an ugly place to have this file, but PhyEee depends on EthIntf and both
# packages use functions in this file.
def phyConfigStatusAtPos( sysdbRoot, phyTopology, intfId, phyPos,
                          firstLaneOnly=False ):
   '''Given the root phy topology object, returns all phy configs and statuses that
   are associated with the intfId at the specified phy position. There can be
   multiple phy objects due to the organization of some 40G-only ports where each
   lane has its own phy object.

   firstLaneOnly: Whether or not only the first lane's phy config and status object
   should be returned for 40G-only ports with multiple phy statuses.

   Returns a list of ( phyIntfConfig, phyIntfStatus ) pairs.'''
   phyData = []

   phyIntfStatuses = phyTopology.phyIntfStatuses.get( intfId )
   # Entry has to exist for the specified interface and position.
   if phyIntfStatuses and phyPos in phyIntfStatuses.phyIntfData:
      phyIntfConfig, phyIntfStatus = phyTopologyConfigStatus(
         sysdbRoot, phyIntfStatuses.phyIntfData[ phyPos ] )
      # If we either have both a phyConfig and phyStatus or a phyStatus and
      # the phyStatus tells us it doesn't have a config, add what we have
      # to the results list. Note that phyConfig may be returned as None.
      if ( phyIntfConfig and phyIntfStatus ) or \
         ( phyIntfStatus and getattr( phyIntfStatus, 'ignoreConfig', False ) ):
         phyData.append( ( phyIntfConfig, phyIntfStatus ) )

   # If none are found, it may be a 40G-only interface that has multiple phy
   # statuses. Assume that a 40G-only Ethernet interface EthernetX may have phy
   # topology entries named EthernetX/1, EthernetX/2, EthernetX/3, EthernetX/4. This
   # organization only appears for 40G-only ports. When 40G-only interfaces without
   # a "/1" are deprecated we can remove everything within this if statement and
   # everything in it.
   if not phyData and ( EthIntfId.isEthIntfId( intfId ) or
                        FabricIntfId.isFabricIntfId( intfId ) ):
      laneIntfs = [ f"{intfId}/{lane}"
                    for lane in range( 1, 2 if firstLaneOnly else 5 ) ]
      for possibleIntf in laneIntfs:
         # laneIntfs may include invalid IntfIds with too many parts.
         try:
            Tac.Value( "Arnet::IntfId", possibleIntf )
         except IndexError:
            continue
         phyIntfStatuses = phyTopology.phyIntfStatuses.get( possibleIntf )
         if phyIntfStatuses and phyPos in phyIntfStatuses.phyIntfData:
            phyIntfConfig, phyIntfStatus = phyTopologyConfigStatus(
               sysdbRoot, phyIntfStatuses.phyIntfData[ phyPos ] )
            if ( phyIntfConfig and phyIntfStatus ) or \
               ( phyIntfStatus and getattr( phyIntfStatus, 'ignoreConfig', False ) ):
               phyData.append( ( phyIntfConfig, phyIntfStatus ) )

   return phyData

def phyTopologyConfigStatus( sysdbRoot, phyIntfData ):
   '''Given a Hardware::PhyTopology::PhyIntfData object, returns the phy intf config
   and phy intf status corresponding to it if they are valid. If they don't exist or
   are stale, ( None, None ) is returned.'''
   if phyIntfData.statusPath:
      assert phyIntfData.configPath
      phyIntfStatus = None
      phyIntfConfig = None
      try:
         phyIntfStatus = sysdbRoot.entity[ phyIntfData.statusPath ]
         phyIntfConfig = sysdbRoot.entity[ phyIntfData.configPath ]
      except KeyError:
         pass
      else:
         if phyIntfStatus.generation.valid and \
            phyIntfStatus.generation == phyIntfConfig.generation:
            return phyIntfConfig, phyIntfStatus
      # If we couldn't retrive one of the entities, check if we at least have a
      # phyStatus that indicates it doesn't have a phyConfig.
      if phyIntfStatus and phyIntfStatus.generation.valid and \
            getattr( phyIntfStatus, 'ignoreConfig', False ):
         return None, phyIntfStatus

   return None, None

def mountPhyTopology( em ):
   mg = em.mountGroup()
   mg.mount( 'hardware/phyChip/config', 'Tac::Dir', 'ri' )
   mg.mount( 'hardware/phy/status', 'Tac::Dir', 'ri' )
   mg.mount( 'hardware/phy/config', 'Tac::Dir', 'ri' )
   mg.mount( 'hardware/phyChip/status', 'Tac::Dir', 'ri' )
   phyTopology = mg.mount( 'hardware/phy/topology/allPhys',
                           'Hardware::PhyTopology::AllPhyIntfStatuses', 'r' )
   mg.close( blocking=True )
   assert mg.cMg_.mountStatus == 'mountSuccess'
   return phyTopology
