#!/usr/bin/env python3
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string
# pylint: disable=singleton-comparison

import Tac
from CliModel import Model, Int, Float, Str
from CliModel import Bool, Submodel, Enum
from CliModel import Dict, GeneratorDict, List
from ArnetModel import IpGenericAddress, IpGenericPrefix
from IntfModels import Interface
import datetime
import PimModelLib

def iPrint( indent, obj ): 
   """Prints an indented string"""

   string = ' ' * indent + "%s" % obj
   print( string )

def timeElapsedString( utcTime, reverse=False ):
   """Returns a human readable string that represents the time elapsed since the
   (UTC) time passed in"""

   currentTime = Tac.utcNow()
   if reverse:
      delta = utcTime - currentTime
   else:
      delta = currentTime - utcTime
   td = datetime.timedelta( seconds=int( delta ) )

   if td > datetime.timedelta( days=1 ) :
      # If time elapsed is > 1 day display the time in XdXXh format
      return str( td.days ) + "d" + str( td.seconds // 3600 ) + "h"
   else:
      # Display time elapsed in HH:MM:SS format
      return str( td )

class ElectedBsr( Model ):
   """Elected Bootstrap Router"""

   eBsrZone = IpGenericPrefix( help="Administrative Scope Zone" )
   eBsrAddress = IpGenericAddress( help="Elected BSR Address" )
   eBsrPriority = Int( help="BSR priority" )
   eBsrHashMaskLen = Int( help="BSR Hash Mask Length" )
   eBsrCreationTime = Float( help="UTC time at which the BSR was elected" )
   eBsrUpTime = Float( help="Time for which this BSR has been elected" )
   eBsrExpiryTime = Float( help="UTC time at which the BSR will expire" )
   eBsrExpires = Float( help="Time after which this BSR might expire" )
   local = Bool( help="Indicates of this router is the BSR", optional=True )
   scope = Str( help="Hints if the BSR is global or admin scoped", optional=True )

   def render( self ):
      if self.scope == 'admin' and self.eBsrZone.stringValue == '224.0.0.0/4':
         scopeStr = 'Admin - NonScoped'
      else:
         scopeStr = self.scope.capitalize()

      indent = 2
      iPrint( indent, f"Zone : {self.eBsrZone} ( {scopeStr} )" )
      indent += 2
      if self.local:
         iPrint( indent, 'This system is the Bootstrap Router (BSR)' )
      iPrint( indent, 'BSR address: %s' % self.eBsrAddress )
      iPrint( indent, 'Uptime:      %s, BSR Priority: %d, Hash mask length: %d' % \
                 ( timeElapsedString( self.eBsrCreationTime ), self.eBsrPriority,
                   self.eBsrHashMaskLen ) )
      iPrint( indent, 'Next bootstrap message in %s' % \
                 timeElapsedString( self.eBsrExpiryTime, reverse=True ) )

   def initFromTacc( self, eBsr, isLocal, scope ):
      if not eBsr:
         return
      self.eBsrZone = eBsr.zone
      self.eBsrAddress = eBsr.ipAddr
      self.eBsrPriority = eBsr.priority
      self.eBsrHashMaskLen = eBsr.hashMaskLen
      self.eBsrCreationTime = eBsr.setupTime + Tac.utcNow() - Tac.now()
      self.eBsrUpTime = eBsr.upTime
      self.eBsrExpiryTime = eBsr.expiryTime + Tac.utcNow() - Tac.now()
      self.eBsrExpires = eBsr.expires
      self.local = isLocal
      self.scope = scope

class BootstrapRouter( Model ):

   electedBsrs = GeneratorDict( help="Map of Bootstrap Routers per Administrative"
                                " scoped Zone", keyType=IpGenericPrefix,
                                valueType=ElectedBsr )

   def generateElectedBsrs( self, pimBsrStatus, zonePrefix ):
      ebsr = pimBsrStatus.bootstrapRouter
      cbsr = pimBsrStatus.cbsr
      scope = 'global'
      globalScope = [ '224.0.0.0/4', 'ff00::/8' ]

      if ( len( ebsr ) > 1 or
           ( len( ebsr ) == 1 and
             # pylint: disable-next=stop-iteration-return
             next( iter( ebsr ) ).stringValue not in globalScope ) ):
         scope = 'admin'

      if zonePrefix:
         zone = zonePrefix
         if zone in ebsr:
            model = ElectedBsr()
            isLocal = False if zone not in cbsr else \
                ebsr[ zone ].ipAddr == cbsr[ zone ].address
            model.initFromTacc( ebsr[ zone ], isLocal, scope )
            yield zone, model
      else:
         for zone in ebsr:
            model = ElectedBsr()
            isLocal = ebsr.get( zone ) != None and \
                cbsr.get( zone ) != None and \
                ebsr[ zone ].ipAddr == cbsr[ zone ].address
            model.initFromTacc( ebsr.get( zone ), isLocal, scope )
            if not ebsr.get( zone ):
               continue
            yield zone, model

   def render( self ):
      for _, ebsr in self.electedBsrs:
         ebsr.render()

   def generate( self, pimBsrStatus, zonePrefix=None ):
      self.electedBsrs = self.generateElectedBsrs( pimBsrStatus,
                                                   zonePrefix )

class PimCrpPrefix( Model ):
   """C-RP prefix"""
   prefix  = IpGenericPrefix( help="Group Prefix" )

   def render( self ):
      indent = 4
      iPrint( indent, 'Group: %s' % self.prefix )

   def initFromTacc( self, prefix ):
      self.prefix = prefix

class PimCrpPrefixOptions( Model ):
   """C-RP prefix options"""
   prefix = IpGenericPrefix( help="Group Prefix" )
   priority = Int( help="Group RP Priority" )

   def render( self ):
      indent = 4
      iPrint( indent, f'Group: {self.prefix}  Priority: {self.priority}' )
   
   def initFromTacc( self, prefix, prefixOptions ):
      self.prefix = prefix
      self.priority = prefixOptions.priority

class PimRpCandidate( Model ):
   rpAddress = IpGenericAddress( help="IP address of RP" )
   holdTime = Int( help="C-RP Holdtime" )
   prefixSet = GeneratorDict( help="List of group prefixes supported by C-RP",
                              keyType=IpGenericPrefix, valueType=PimCrpPrefix )
   prefixOptions = GeneratorDict( help="List of priority associated with prefixes",
                              keyType=IpGenericPrefix, 
                              valueType=PimCrpPrefixOptions )

   def generatePrefix( self, prefixSet ):
      for prefix in prefixSet:
         model = PimCrpPrefix()
         model.initFromTacc( prefix )
         yield prefix, model

   def generatePrefixOptions( self, prefixOptions ):
      for prefix in prefixOptions:
         model = PimCrpPrefixOptions()
         model.initFromTacc( prefix, prefixOptions[ prefix ] )
         yield prefix, model

   # pylint: disable-msg=W0221
   def render( self, detail=False ):
      indent = 0
      iPrint( indent, 'Candidate RP Address: %s' % self.rpAddress )
      indent += 2
      iPrint( indent, 'CRP Holdtime: %d' % self.holdTime )

   def initFromTacc( self, crp ):
      self.rpAddress = crp.address
      self.holdTime = crp.interval * 3
      self.prefixSet = self.generatePrefix( crp.prefixSet )
      self.prefixOptions = self.generatePrefixOptions( crp.prefixOptions )

class PimRpCandidates( Model ):
   crps = GeneratorDict( help="Map of Candidate RP local to "
                         "this router", keyType=IpGenericAddress,
                         valueType=PimRpCandidate )
   def generateCrps( self, crpTable ):
      for address in crpTable:
         model = PimRpCandidate()
         model.initFromTacc( crpTable[ address ] )
         yield address, model

   def generate( self, crpTable ):
      self.crps = self.generateCrps( crpTable )

   def render( self ):
      for _, crp in self.crps:
         crp.render()
         for _, prefix in crp.prefixSet:
            prefix.render()

class PimBsrErrorCounters( Model ):
   rxFilter = Int( help="Total packets received and dropped due to filters" )
   rxNbrRpfCheck = Int( help="Total BSMs dropped due to RPF check failure" )
   rxNoForwardBsm = Int( help="Total BSMs dropped due to no forward flag" )
   rxUnicastBsm = Int( help="Total BSMs dropped due to unicast destination" )
   rxScopeMismatch = Int( help="Total BSMs dropped due to scope mismatch" )
   rxInvalid = Int( help="Total invalid BSMs received" )

   def generate( self, counters ):
      self.rxFilter = counters.rxFilter
      self.rxNbrRpfCheck = counters.rxNbrRpfCheck
      self.rxNoForwardBsm = counters.rxNoForwardBsm
      self.rxUnicastBsm = counters.rxUnicastBsm
      self.rxScopeMismatch = counters.rxScopeMismatch
      self.rxInvalid = counters.rxInvalid

   def render( self ):
      indent = 0
      iPrint( indent, "Dropped packets classification")
      indent += 2
      iPrint( indent, "Filter configuraiton         : %d" % self.rxFilter )
      iPrint( indent, "RPF failure                  : %d" % self.rxNbrRpfCheck )
      iPrint( indent, "No-Forward BSM               : %d" % self.rxNoForwardBsm )
      iPrint( indent, "Unicast BSM                  : %d" % self.rxUnicastBsm )
      iPrint( indent, "Scope mismatch               : %d" % self.rxScopeMismatch )
      iPrint( indent, "Invalid                      : %d" % self.rxInvalid )

class PimBsrCounters( Model ):
   rx = Int( help="Total packets received", optional=True )
   rxError = Int( help="Total packets received and dropped due to errors",
                  optional=True )
   rxSuccess = Int( help="Total packets received and processed successfully",
                    optional=True )
   tx = Int( help="Total packets transmitted", optional=True )
   txSuccess = Int( help="Total packets transmitted successfully", optional=True )
   txError = Int( help="Total packets transmitted and dropped due to errors",
                  optional=True )
   rxBsm = Int( help="Total BSM's received", optional=True )
   rxBsmError = Int( help="Total BSMs received and dropped due to errors",
                     optional=True )
   rxBsmSuccess = Int( help="Total BSMs successfully processed", optional=True )
   rxCrpAdv = Int( help="Total CRP Advertisement messages received", optional=True )
   rxCrpAdvSuccess = Int( help="Total CRP Advertisements successfully processed",
                          optional=True )
   rxCrpAdvError = Int( help="Total Crp Advertisements received and dropped due to"
                        "errors or configuration", optional=True )
   errorCounters = Submodel( help="Error counters",
                             valueType=PimBsrErrorCounters, optional=True )

   def generate( self, counters ):
      self.rx = counters.rx
      self.rxError = counters.rxError
      self.rxSuccess = counters.rxSuccess
      self.tx = counters.tx
      self.txSuccess = counters.txSuccess
      self.txError = counters.txError
      self.rxBsm = counters.rxBsm
      self.rxBsmSuccess = counters.rxBsmSuccess
      self.rxBsmError = counters.rxBsmError
      self.rxCrpAdv = counters.rxCrpAdv
      self.rxCrpAdvSuccess = counters.rxCrpAdvSuccess
      self.rxCrpAdvError = counters.rxCrpAdvError
      self.errorCounters = PimBsrErrorCounters()
      self.errorCounters.generate( counters.errCounters )

   def render( self ):
      if self.rx is None:
         return
      indent = 0
      iPrint( indent, "Total Rx                     : %d" % self.rx )
      indent += 2
      iPrint( indent, "Success                    : %d" % self.rxSuccess )
      iPrint( indent, "Dropped                    : %d" % self.rxError )
      indent = 0
      iPrint( indent, "Total Rx BSM                 : %d" % self.rxBsm )
      indent += 2
      iPrint( indent, "Success                    : %d" % self.rxBsmSuccess )
      iPrint( indent, "Dropped                    : %d" % self.rxBsmError )
      indent = 0
      iPrint( indent, "Total Rx C-RP Advertisements : %d" % self.rxCrpAdv )
      indent += 2
      iPrint( indent, "Success                    : %d" % self.rxCrpAdvSuccess )
      iPrint( indent, "Dropped                    : %d" % self.rxCrpAdvError )
      indent = 0
      iPrint( indent, "Total Tx                     : %d" % self.tx )
      indent += 2
      iPrint( indent, "Success                    : %d" % self.txSuccess )
      iPrint( indent, "Dropped                    : %d" % self.txError )
      self.errorCounters.render()


class PimBsrBorder( Model ):
   prefix = IpGenericPrefix( help="Group Prefix" )
   intfs = List( help="Interface list", valueType=Interface )

   def render( self ):
      indent = 0
      iPrint( indent, "Group: %s" % self.prefix )
      indent += 2
      iPrint( indent, "Interfaces:" )
      indent += 2
      for intf in self.intfs:
         iPrint( indent, intf )

   def initFromTacc( self, border ):
      self.prefix = border.prefix
      for intf in border.intfId:
         self.intfs.append( intf )

class ZoneGroup( Model ):
   """ Administrative Scoped Zone Information """
   zone = IpGenericPrefix( help="Administrative Scoped Zone" )
   intfId = Interface( help="Interface Identifier" )

   def render( self ):
      indent = 0
      iPrint( indent, f"{self.zone} <--> {self.intfId}" )

   def initFromTacc( self, zone, intfId ):
      self.zone = zone
      self.intfId = intfId

class ZoneAcl( Model ):
   """ Administrative Scoped Zone Information """
   zone = Str( help="Administrative Scoped Zone" )
   intfId = Interface( help="Interface Identifier" )

   def render( self ):
      indent = 0
      iPrint( indent, f"{self.zone} <--> {self.intfId}" )

   def initFromTacc( self, zone, intfId ):
      self.zone = zone
      self.intfId = intfId

class PimBsrCandidate( Model ):
   zone = IpGenericPrefix( help="Zone" )
   address = IpGenericAddress( help="Candidate BSR IP" )
   intfId = Interface( help="Candidate BSR interface identifier" )
   priority = Int( help="Candidate BSR priority" )
   hashMaskLen = Int( help="Candidate BSR hash mask length" )
   bsPeriod = Int( help="Bootstrap message interval" )
   elected = Bool( help="Indicates if the candidate BSR is elected BSR" )
   state = Enum( help="Current state of candidate BSR as per RFC",
                 values=( "noInfoBsr", "cBsr", "pBsr", "eBsr" ) )

   def initFromTacc( self, cbsr ):
      self.zone = cbsr.zone
      self.address = cbsr.address
      self.intfId = cbsr.intfId
      self.priority = cbsr.priority
      self.hashMaskLen = cbsr.hashMaskLen
      self.bsPeriod = cbsr.bsPeriod
      self.elected = cbsr.elected
      self.state = cbsr.state

   def render( self ):
      indent = 0
      iPrint( indent, "Zone: %s" % self.zone )
      indent += 2
      iPrint( indent, "IP Address  : %s" % self.address )
      iPrint( indent, "Interface   : %s" % self.intfId )
      iPrint( indent, "Priority    : %d" % self.priority )
      iPrint( indent, "HashMaskLen : %d" % self.hashMaskLen )
      iPrint( indent, "Interval    : %d" % self.bsPeriod )
      iPrint( indent, "Elected     : %s" % "True" if self.elected else "False" )
      iPrint( indent, "State       : %s" % self.state )

class PimBsrRpCandidate( Model ):
   address = IpGenericAddress( help="Candidate RP IP addres" )
   intfId = Interface( help="Candidate RP interface identifier" )
   interval = Int( help="Candidate RP advertisement interval" )
   prefixSet = List( help="Group prefixes served by the candidate RP",
                     valueType=IpGenericPrefix )
   prefixOptions = Dict( help="Group prefixes and associated priorities", 
                       keyType=IpGenericPrefix, valueType=int )

   def initFromTacc( self, crp ):
      self.address = crp.address
      self.intfId = crp.intfId
      self.interval = crp.interval
      for prefix in crp.prefixSet:
         self.prefixSet.append( prefix )
      for prefix in crp.prefixOptions:
         self.prefixOptions[ prefix ] = crp.prefixOptions[ prefix ].priority

   def render( self ):
      indent = 0
      iPrint( indent, "C-RP Address  : %s" % self.address )
      indent += 2
      iPrint( indent, "Interface    : %s" % self.intfId )
      iPrint( indent, "Adv Interval : %d" % self.interval )
      indent += 2
      iPrint( indent, "Prefix Set   :   Priority   :" )
      for prefix in self.prefixOptions:
         string = ' ' * indent + "{}    {}".format( prefix,
                                                    self.prefixOptions[ prefix ] )
         print( string )

class PimBsrRpCandidateStatus( Model ):
   zone = IpGenericPrefix( help="Zone" )
   rpSetType = Enum( help="Type of RP Set",
                     values=( "noneRp", "staticRp", "bsrRp" ) )
   crpSet = GeneratorDict( help="C-RP set map", keyType=IpGenericPrefix,
                           valueType=PimModelLib.PimRpCandidateSet )

   def generateCrpSet( self, crpSet ):
      for prefix in crpSet:
         model = PimModelLib.PimRpCandidateSet()
         model.initFromTacc( prefix, bsrRpSet=crpSet[ prefix ] )
         yield prefix, model

   def initFromTacc( self, crpStatus ):
      self.zone = crpStatus.zone
      self.rpSetType = crpStatus.status.rpSetType
      self.crpSet = self.generateCrpSet( crpStatus.status.crpSet )

   def render( self ):
      indent = 0
      iPrint( indent, "Zone: %s" % self.zone )
      indent += 2
      iPrint( indent, "RP Set Type : %s" % self.rpSetType )
      for _, crpSet in self.crpSet:
         crpSet.render()
         for _, crp in crpSet.crp:
            crp.render()

class PimBsrDebug( Model ):
   af = Enum( help="Address Family supported in this instance",
              values=( "ipunknown", "ipv4", "ipv6" ) )
   bsTimeout = Int( help="Boorstrap Holdtime" )
   szTimeout = Int( help="Scope Zone timeout interval" )
   bsBorder = GeneratorDict( help="Pim BSR Border Map",
                             keyType=IpGenericPrefix, valueType=PimBsrBorder )
   zoneGroup = GeneratorDict( help="IntfId to Zone map",
                              keyType=IpGenericPrefix, valueType=ZoneGroup )
   zoneAcl = GeneratorDict( help="IntfId to Acl map",
                            keyType=str, valueType=ZoneAcl )
   bootstrapRouter = Submodel( help="Elected Bootstrap Router",
                               valueType=BootstrapRouter )
   cbsr = GeneratorDict( help="Candidate BSR Map",
                         keyType=IpGenericPrefix, valueType=PimBsrCandidate )
   crp = GeneratorDict( help="Candidate RP Map",
                        keyType=IpGenericAddress, valueType=PimBsrRpCandidate )
   crpPermit = List( help="Candidate RP permit list",
                     valueType=IpGenericPrefix )
   crpStatus = GeneratorDict( help="C-RP sets for G-RP mapping",
                              keyType=IpGenericPrefix,
                              valueType=PimBsrRpCandidateStatus )
   counters = Submodel( help="Pim BSR packet counters", valueType=PimBsrCounters )

   def generateBorder( self, borderTable ):
      for prefix in borderTable:
         model = PimBsrBorder()
         model.initFromTacc( borderTable[ prefix ] )
         yield prefix, model

   def generateZoneGrp( self, zoneGrpTable ):
      for zone in zoneGrpTable:
         model = ZoneGroup()
         model.initFromTacc( zone, zoneGrpTable[ zone ] )
         yield zone, model

   def generateZoneAcl( self, zoneAclTable ):
      for zone in zoneAclTable:
         model = ZoneAcl()
         model.initFromTacc( zone, zoneAclTable[ zone ] )
         yield zone, model

   def generateCBSR( self, cbsr ):
      for zone in cbsr:
         model = PimBsrCandidate()
         model.initFromTacc( cbsr[ zone ] )
         yield zone, model

   def generateCRP( self, crp ):
      for address in crp:
         model = PimBsrRpCandidate()
         model.initFromTacc( crp[ address ] )
         yield address, model

   def generateCRPStatus( self, crpStatus ):
      for zone in crpStatus:
         model = PimBsrRpCandidateStatus()
         model.initFromTacc( crpStatus[ zone ] )
         yield zone, model

   def generate( self, pimBsrStatus ):
      self.af = pimBsrStatus.af
      self.bsTimeout = pimBsrStatus.bsTimeout
      self.szTimeout = pimBsrStatus.szTimeout
      self.bsBorder = self.generateBorder( pimBsrStatus.bsBorder )
      self.zoneGroup = self.generateZoneGrp( pimBsrStatus.zoneGroup )
      self.zoneAcl = self.generateZoneAcl( pimBsrStatus.zoneAcl )
      self.bootstrapRouter = BootstrapRouter()
      self.bootstrapRouter.generate( pimBsrStatus )
      self.cbsr = self.generateCBSR( pimBsrStatus.cbsr )
      self.crp = self.generateCRP( pimBsrStatus.crp )
      for prefix in pimBsrStatus.crpPermit:
         self.crpPermit.append( prefix )
      self.crpStatus = self.generateCRPStatus( pimBsrStatus.crpStatus )
      self.counters = PimBsrCounters()
      if pimBsrStatus.counters:
         self.counters.generate( pimBsrStatus.counters )

   def render( self ):
      indent = 0
      iPrint( indent, "Address family       : %s" % self.af )
      iPrint( indent, "Bootstrap timeout    : %d" % self.bsTimeout )
      iPrint( indent, "Scope zone timeout : %d" % self.szTimeout )
      iPrint( indent, "-------------BSR Border Information----------------" )
      for _, border in self.bsBorder:
         border.render()
      iPrint( indent, "-------------Zone Mapping--------------------------" )
      for _, zoneGrp in self.zoneGroup:
         zoneGrp.render()
      for _, zoneAcl in self.zoneAcl:
         zoneAcl.render()
      iPrint( indent, "-------------Elected BSR Information---------------" )
      self.bootstrapRouter.render()
      iPrint( indent, "-------------Candidate BSR Information-------------" )
      for _, cbsr in self.cbsr:
         cbsr.render()
      iPrint( indent, "-------------Candidate RP Information--------------" )
      for _, crp in self.crp:
         crp.render()
      iPrint( indent, "-------------Candidate RP Permit Filter--------------" )
      for prefix in self.crpPermit:
         iPrint( indent, "%s" % prefix )
      iPrint( indent, "-------------Candidate RP Set----------------------" )
      for _, crpStatus in self.crpStatus:
         crpStatus.render()
      iPrint( indent, "-------------Pim BSR Counters----------------------" )
      self.counters.render()

