# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from IntfModels import Interface
from CliModel import Model, Enum, List
from ArnetModel import IpGenericAddress
import Tac

class RegisterSource( Model ):
   ''' Holds register source configuration-related information.. '''
   setting = Enum( help="Register source setting configured",
                  values=( 'no config', 'default', 'interface', 'iif' ) )
   intf = Interface( help="Interface name" )

   def fromTacc( self, pimRegConfig ):
      typeDict = { 'no config' : 'no config',
                   'defaultSrc' : 'default',
                   'interface' : 'interface',
                   'iif' : 'iif' }
      if pimRegConfig:
         self.setting = typeDict[ pimRegConfig.registerSrc ]
         self.intf = Tac.newInstance( "Arnet::IntfId", pimRegConfig.registerSrcIntf )
      else:
         self.setting = typeDict[ 'no config' ]
         self.intf = Tac.newInstance( "Arnet::IntfId" )

   def render( self ):
      ''' Generate output. '''
      printDict = { 'default' : 'Use source address of outgoing interface (default)',
                    'interface' : 'Use source address from specified interface',
                    'iif' : 'Use source address from incoming interface'
                  }

      if self.setting == 'no config':
         return

      # pylint: disable-next=consider-using-f-string
      print( "Register Source Setting: %s" % printDict [ self.setting ] )
      if self.setting == "interface":
         # pylint: disable-next=consider-using-f-string
         print( "Register Source Interface: %s" % self.intf )


class AnycastRpLocalStatus( Model ):
   anycastRpAddrs = List( valueType=IpGenericAddress,  
                         help="a List of Anycase RP addresses" )
   peerAddrs = List( valueType=IpGenericAddress,  
                     help="a List of Anycase peers addresses" )

   def render( self ):
      for k in self.anycastRpAddrs:
         # pylint: disable-next=consider-using-f-string
         print( "Anycast Rp address is %s" % k )
      for k in self.peerAddrs:
         # pylint: disable-next=consider-using-f-string
         print( "Anycast Peers address is %s" % k )

   def fromTacc( self, pimRegAnycastRpLocalStatus ):
      if not pimRegAnycastRpLocalStatus.coll:
         return 

      for rpAddr, anycastRpLocal in pimRegAnycastRpLocalStatus.coll.items():
         self.anycastRpAddrs.append( rpAddr )
         self.peerAddrs.append( anycastRpLocal.anycastAddr )
         for peerAddr in anycastRpLocal.peer:
            self.peerAddrs.append( peerAddr )
