#!/usr/bin/env python3
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
from CliModel import Bool, Int, Float, Str, Submodel, List
from CliModel import Dict, GeneratorDict, Enum, DeferredModel
from ArnetModel import IpGenericAddress, IpGenericPrefix
from IntfModels import Interface

class OifPrime( DeferredModel ):
   """ Interfaces not in the OIF set of an mroute """

   # Possible reasons why this interface isn't in the OIF set
   interface = Interface( help="Name of this interface" )
   incomingInterface = Bool( help="The incoming interface for the route" )
   notDr = Bool( help="Not a designated router on this interface" )
   assertLoser = Bool( help="Assert loser on this interface" )
   rptPruned = Bool( help="Pruned from the RP tree" )
   locallyExcluded = Bool( help="Locally excluded" )

class WcMroute( DeferredModel ):
   """(*,G) specific state"""

   # State summarization macros
   joinDesiredWc = Bool( help="Join Desired (*,G)" )
   joinDesiredWcRp = Bool( help="Join Desired (*,*,G)" )
   rptJoinDesiredWc = Bool( help="RPT Join Desired (*,G)" )
   nullImmediateOlistWc = Bool( help="No interfaces in immediate olist (*,G)" )
   nullImmediateOlistWcRp = Bool( help="No interfaces in immediate olist (*,*,G)" )

class SgMroute( DeferredModel ):
   """(S,G) speficic state"""

   # State summarization macros
   couldRegisterSg = Bool( help="CouldRegister(S,G): If true, we may send register"
                           " messages for this (S,G)" )
   registerStopSg = Bool( help="If true, we should send a register stop for this "
                          "(S,G) in response to a register message" )
   joinDesiredSg = Bool( help="Join Desired (S,G), if true we should send joins "
                         "for this source if a valid upstream neighbor exists" )
   pruneDesiredSgRpt = Bool( help="Prune Desired (S,G,Rpt), if true, we should "
                             " prune ourselves from the RP tree for this source. "
                             " This is to avoid receiving traffic both from the RP "
                             " tree and source tree." )
   nullImmediateOlistSg = Bool( help="No interfaces in immediate olist (S,G)" )
   nullInheritedOlistSg = Bool( help="No interfaces in immediate olist (S,G)" )

class MrouteDetails( DeferredModel ):

   oifPrimes = Dict( help="Map of interfaces not in OIF set",
                              valueType=OifPrime )

   fastFailovers = List( help="Deprecated", valueType=Interface )
   

   upstreamJoinState = Enum( help="Upstream joined state",
                             values=( "upNotJoined",
                                      "upJoined" ) )

   upstreamRptState = Enum( help="Upstream RPT joined state",
                            values=( "upRptNotPruned",
                                     "upRptNotJoined",
                                     "upRptPruned" ) )

   sgState = Submodel( help="State specific to (S,G) routes",
                       valueType=SgMroute, optional=True )
   wcState = Submodel( help="State specific to (*,G) routes",
                       valueType=WcMroute, optional=True )

class Rpf( DeferredModel ):

   rpfNeighbor = IpGenericAddress( help="RPF neighbor", optional=True )
   rpfPrefix = IpGenericPrefix( help="RPF route prefix" )
   rpfMetric = Int( help="RPF route metric" )
   rpfPreference = Int( help="RPF route preference" )
   rpfRib = Enum( help="RPF routing table",
                  values=( "none",
                           "U",
                           "M" ) )
   rpfAttached = Bool( help="If true, the source/RP is directly connected to this "
                       "router" )
   rpfEvpnTenantDomain = Bool( help="If true, the source/RP is inside Evpn "
                               "Tenant Domain ", optional=True )
   rpfMvpn = Bool( help="If true, the source/RP is inside the MVPN Domain ",
                   optional=True )

class Mroute( DeferredModel ):

   sourceAddress = IpGenericAddress( help="Source address" )
   rp = IpGenericAddress( help="RP(G): The Rendezvous Point for this group",
                          optional=True )
   rpf = Submodel( help="Reverse path forwarding information", valueType=Rpf,
                   optional=True )
   rpfInterface = Interface( help="Reverse path forwarding interface, packets for "
                             "this mroute must arrive on this interface to be "
                             "accepted/forwarded. If this value is set to "
                             "'Register', the RPF interface is the register tunnel",
                             optional=True )

   routeFlags = Str( help="Route flags" )
   creationTime = Float( help="UTC time at which the route was created" )
   oifList = List( help="Outgoing interface list", valueType=Interface )
   nonDrOifList = List( help="List of outgoing interfaces with egress drop rule "
                        "in effect on non-DR", 
                        valueType=Interface, 
                        optional=True )
   registerInOifList = Bool( help="If true, the register tunnel is part of the "
                             " outgoing interface list", optional=True )
   details = Submodel( help="Detailed information for this multicast route",
                       valueType=MrouteDetails, optional=True )

class Group( DeferredModel ):

   groupSources = Dict( help="Map of multicast routes for this group",
                                 keyType=IpGenericAddress, valueType=Mroute )

class Groups( DeferredModel ):

   groups = Dict( help="Map of sparse-mode Pim multicast groups",
                           keyType=IpGenericAddress, valueType=Group )

class SourceGroupCollection( DeferredModel ):
   sourceOrGroupAddrs = GeneratorDict( help="List of IP address", 
         keyType=IpGenericAddress, valueType=str )

class SourcesGroups( DeferredModel ):
   
   isSource = Bool( help="If true, we print S-> no of G" )
   sourceGroup = GeneratorDict( help="S-> no of G or G-> no of S mapping ", 
         keyType=int, valueType=SourceGroupCollection )

class AssertMetric( DeferredModel ):

   rptBit = Bool( help="Assert metric is for the RP tree" )
   preference = Int( help="Route preference to the source/RP" )
   metric = Int( help="Route metric to the source/RP" )
   address = IpGenericAddress( help="Address of router that generated the assert" )

class SgMrouteInterface( DeferredModel ):
   """(S,G,I) state"""

   # Assert state

   couldAssertSg = Bool( help="Could assert (S,G,I)" )
   assertTrackingDesiredSg = Bool( help="Assert tracking desired (S,G,I)" )
   lostAssertSg = Bool( help="Member of lost assert (S,G)" )
   lostAssertSgRpt = Bool( help="Member of lost assert (S,G,Rpt)" )
   myAssertMetricSg = Submodel( help="My assert metric (S,G,I)",
                                valueType=AssertMetric, optional=True )
   sptAssertMetric = Submodel( help="SPT assert metric (S,I)",
                               valueType=AssertMetric, optional=True )

   # State summarization macros

   localReceiverIncludeSg = Bool( help="Local receiver include (S,G,I)" )
   localReceiverExcludeSg = Bool( help="Local receiver exclude (S,G,I)" )
   joinsSg = Bool( help="Member of joins (S,G)" )
   prunesSgRpt = Bool( help="Member of prunes (S,G,Rpt)" )
   includeSg = Bool( help="Member of pim include (S,G)" )
   excludeSg = Bool( help="Member of pim exclude (S,G)" )
   immediateOlistSg = Bool( help="Member of immediate olist (S,G)" )
   inheritedOlistSg = Bool( help="Member of inherited olist (S,G)" )
   inheritedOlistSgRpt = Bool( help="Member of inherited olist (S,G,Rpt)" )

class WcMrouteInterface( DeferredModel ):
   """(*,G,I) state"""

   # Assert state

   couldAssertWc = Bool( help="Could assert (*,G,I)" )
   assertTrackingDesiredWc = Bool( help="Assert tracking desired (*,G,I)" )
   lostAssertWc = Bool( help="Member of lost assert (*,G)" )
   myAssertMetricWc = Submodel( help="My assert metric (*,G,I)",
                                valueType=AssertMetric, optional=True )
   rptAssertMetric = Submodel( help="RPT assert metric (G,I)",
                               valueType=AssertMetric, optional=True )

   # State summarization macros

   # Specific to (*,G) routes (and therefore optional)
   localReceiverIncludeWc = Bool( help="Local receiver include (*,G,I)" )
   joinsWc = Bool( help="Member of joins (*,G)" )
   joinsWcRp = Bool( help="Member of joins (*,*,G)" )
   includeWc = Bool( help="Member of pim include (*,G)" )
   immediateOlistWc = Bool( help="Member of immediate olist (*,G)" )
   immediateOlistWcRp = Bool( help="Member of immediate olist (*,*,G)" )

class MrouteInterfaceDetails( DeferredModel ):

   # Detailed assert information
   lastAssertAction = Int( help="Last assert action", optional=True )

   assertWinnerMetric = Submodel( help="Assert winner metric",
                                  valueType=AssertMetric, optional=True )
   sgState = Submodel( help="Interface state specific to (S,G) routes",
                       valueType=SgMrouteInterface, optional=True )
   wcState = Submodel( help="Interface state specific to (*,G) routes",
                       valueType=WcMrouteInterface, optional=True )

class MrouteInterface( DeferredModel ):

   address = IpGenericAddress( help="Interface IP address", optional=True )
   iAmDr = Bool( help="This Pim router is the designated router on this interface" )
   forward = Bool( help="Member of outgoing interface set" )

   markedForDeletion = Bool( help="If true, this interface has no (S,G,I)/(*,G,I)"
                             " state and is scheduled to be removed from the "
                             " route's interface list" )

   downstreamJoinState = Enum( help="Downstream join state",
                               values=( "noInfo",
                                        "prunePending",
                                        "joined" ) )
   lastJoinReceived = Float( help="UTC time at which the last downstream join "
                             "was received", optional=True )

   rptState = Enum( help="Downstream RPT state",
                    values=( "rptNoInfo",
                             "rptPruned",
                             "rptPrunePending",
                             "rptPruneTemp",
                             "rptPrunePendingTemp" ) )
   lastRptPruneReceived = Float( help="UTC time at which the last downstream RPT "
                                 "prune was received", optional=True )

   assertState = Enum( help="Assert state",
                       values=( "assertNoInfo",
                                "iAmAssertWinner",
                                "iAmAssertLoser" ) )

   details = Submodel( help="Detailed mroute interface information",
                       valueType=MrouteInterfaceDetails, optional=True )

class MrouteInterfaces( DeferredModel ):

   source = IpGenericAddress( help="Source address of multicast route" )
   group = IpGenericAddress( help="Group address of multicast route" )

   interfaces = Dict( help="Map of interfaces for multicast route",
                               keyType=Interface, valueType=MrouteInterface )

class MrouteCount( DeferredModel ):

   groups = Int( help="Number of groups" )
   sources = Int( help="Number of sources" )

   wcRoutes = Int( help="Number of (*,G) routes" )
   sgRoutes = Int( help="Number of (S,G) routes" )

   unresolved = Int( help="Number of routes for which RPF information has not been"
                           "determined" )
   toBeDeleted = Int( help="Number of routes currently scheduled for deletion" )
   switching = Int( help="Number of (S,G)s currently in the midst of switching from"
                          " the RP tree to the SP tree" )

   couldRegister = Int( help="Number of (S,G)s for which CouldRegister(S,G) is "
                        "True" )
   registerStop = Int( help="Number of (S,G)s for which the RP should generate"
                       " register stops on receipt of a register/null register" )
   registerOif = Int( help="Number of (S,G)s with the register tunnel in the OIF "
                      "set" )

   mbrDiscovered = Int( help="Number of (S,G)s discovered via MBR" )
   drDiscovered = Int( help="Number of routes discovered on the RP from a DR via a"
                          "a register message" )
   anycastRpDiscovered = Int( help="Number of routes discovered on the RP from from"
                              " an anycast RP via a register message" )
   msdpDiscovered = Int( help="Number of routes discovered on the RP via an MSDP "
                         "source advertisement" )

class NonDrDropRuleInterface( DeferredModel ):
   state = Enum( help="Non-DR drop rule state for interface",
         values=( "init",
                  "active",
                  "inactive",
                  "error",
                  "programming",
                  "unprogramming",
                  "unprogrammed" ) )
   inactiveReason = Enum( help="Reason why state is inactive or error",
         values=( "Drop rule creation failed",
                  "Drop rule programming failed on interface",
                  "Multicast routes forwarding",
                  "Interface is DR",
                  "Drop rule unprogramming failed on interface",
                  "Conflicting drop rule programmed on interface",
                  "Waiting for platform to program drop rule",
                  "Drop rule programming timed out",
                  "Drop rule unprogramming timed out",
                   ), optional=True )
   numRoutesForwarding = \
      Int( help="Number of multicast routes forwarding on this interface" )
   numRoutesReadyToForward = \
      Int( help="Number of multicast routes ready to forward on this interface" )

class NonDrDropRuleInterfaces( DeferredModel ):
   interfaces = Dict( help="Map of interfaces for non DR install oifs feature",
         keyType=Interface, valueType=NonDrDropRuleInterface )
