#!/usr/bin/env python3

# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import re
import sys
import time

import EosVersion
import Logging
import PlatformArchLogs

from PyClient import PyClient

SYSTEM_REQ_32 = "i686 required for {} but the image is x86_64"
SYSTEM_REQ_64 = "x86_64 required for {} but the image is i686"
PLATFORM_FILE_ERR = "Unable to read platform architecture data file"
VERSION_FILE_ERR = "Unable to read software version file"
DETERMINE_SKU_ERR = "Unable to determine platform SKU"
ACCESS_SKU_ERR = "Unable to access platform SKU"

PREFDL = "/etc/prefdl"
SWI_VERSION = "/etc/swi-version"
ARCH_EOS32 = "EOS32"
ARCH_EOS64 = "EOS64"
ARCH_DATA = "/usr/share/EosData/PlatformSupportedArch"
SKU_REGEX = r"^SKU:\s*(.+)\s*$"

def getSkuFromPrefdl():
   if not os.path.exists( PREFDL ):
      return None
   try:
      with open( PREFDL ) as f:
         prefdlData = f.read()
   except: # pylint: disable=bare-except
      return None

   skuMatch = re.search( SKU_REGEX, prefdlData, re.MULTILINE | re.IGNORECASE )
   if not skuMatch:
      return None

   return skuMatch.group( 1 )

# Some platforms like RaspberryIsland can take
# a long time to initialize Fru/Sysdb, so allow for
# 4 minutes of attempts.
MAX_ATTEMPTS = 25
SLEEP_TIMER = 10

def getCurrentPlatformSku():
   try:
      # entMib.root may not be initialized yet, so use
      # an attempt counter with a delay until it's ready
      # or we give up on attempts.
      pc = PyClient( "ar", "Sysdb" )
      attempts = 0
      sku = None
      while attempts < MAX_ATTEMPTS:
         try:
            root = pc.agentRoot()
            entMib = root[ 'hardware' ][ 'entmib' ]
            sku = entMib.root.modelName
            if not sku:
               time.sleep( SLEEP_TIMER )
               attempts += 1
               continue
            return sku
         except: # pylint: disable=bare-except
            time.sleep( SLEEP_TIMER )
            attempts += 1
      # Dual supes and some fixed systems may fail to fill in Sysdb
      # in hotswap / reload ptests, so retrieve the SKU from the prefdl
      if not sku:
         sku = getSkuFromPrefdl()
         if sku:
            return sku
   except: # pylint: disable=bare-except
      Logging.log( PlatformArchLogs.SYS_BOOT_INFO_ERROR, ACCESS_SKU_ERR )
   return None

def getArchAndOptimFromVersionFile():
   # Read in current version info and log which architecture is booted
   try:
      with open( SWI_VERSION ) as f:
         info = f.read()
   except:  # pylint: disable=bare-except
      Logging.log( PlatformArchLogs.SYS_BOOT_INFO_ERROR, VERSION_FILE_ERR )
      if __name__ == "__main__":
         sys.exit( 1 )
      else:
         return None, None

   versionInfo = EosVersion.VersionInfo( None, versionFile=info )

   return versionInfo.architecture(), versionInfo.swiOptimization()

def getRequiredArchs( sku ):
   reqArchs = None
   try:
      with open( ARCH_DATA ) as f:
         for l in f.readlines():
            if l.strip().startswith( '#' ):
               continue
            currSku, archs = l.strip().split( "=" )
            archs = archs.split( "," )
            if sku == currSku:
               reqArchs = archs
               break
   except:  # pylint: disable=bare-except
      Logging.log( PlatformArchLogs.SYS_BOOT_INFO_ERROR, PLATFORM_FILE_ERR )
      if __name__ == "__main__":
         sys.exit( 1 )
      else:
         return None

   return reqArchs

def checkArchitectureCompatibility( imageArch, reqArchs, sku, log=True ):
   if not reqArchs:
      return True

   if imageArch in [ "i686", "i386" ]:
      imageArch = ARCH_EOS32
   else:
      imageArch = ARCH_EOS64

   if imageArch not in reqArchs:
      if ARCH_EOS32 in reqArchs:
         if log:
            Logging.log( PlatformArchLogs.SYS_SWI_UNSUPPORTED,
                         SYSTEM_REQ_32.format( sku ) )
         return False
      else:
         if log:
            Logging.log( PlatformArchLogs.SYS_SWI_UNSUPPORTED,
                         SYSTEM_REQ_64.format( sku ) )
         return False

   return True

def main():
   imageArch, optim = getArchAndOptimFromVersionFile()

   # Skip arch check for non-EOS images (e.g. cEOS-lab)
   # which will have optimization set to None
   if optim == "None":
      return

   sku = getCurrentPlatformSku()
   if not sku:
      Logging.log( PlatformArchLogs.SYS_BOOT_INFO_ERROR, DETERMINE_SKU_ERR )
      sys.exit( 1 )

   reqArchs = getRequiredArchs( sku )

   _ = checkArchitectureCompatibility( imageArch, reqArchs, sku )


if __name__ == "__main__":
   try:
      main()
   except Exception as e:  # pylint: disable=broad-except
      Logging.log( PlatformArchLogs.SYS_BOOT_INFO_ERROR, e )
      sys.exit( 1 )
