#!/usr/bin/env arista-python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import json
import os
import sys

def depBasedEnvName( pluginType ):
   envName = {
      "PyPlugin": "PY_PLUGIN_LOADING_FROM_DEPS",
      "SysdbMountProfiles": "SYSDB_PROFILES_PLUGIN_LOADING_FROM_DEPS",
      "preinit": "PREINIT_PLUGIN_LOADING_FROM_DEPS",
      "DataPlugin": "DATA_PLUGIN_LOADING_FROM_DEPS",
      }.get( pluginType )
   assert envName, "Unsupported Plugin Type: %s" % ( pluginType )
   return envName

def checkOutdatedUsage():
   # We renamed A4_CURRENT_PACKAGE to A4PKG. To make this transition more obvious,
   # assert if both are set to different values.
   oldVar = os.environ.get( "A4_CURRENT_PACKAGE", "" )
   newVar = os.environ.get( "A4PKG", "" )
   if oldVar:
      if not newVar:
         errorMsg = \
            f"A4_CURRENT_PACKAGE has been deprecated: use A4PKG={oldVar} instead!"
      elif oldVar != newVar:
         errorMsg = \
            f"Different values A4_CURRENT_PACKAGE={oldVar} and A4PKG={newVar} " \
            "are set!"
      else:
         return

      assert False, errorMsg

class PluginTree:

   __slots__ = [ "plugins", "package", "permits" ]

   def __init__( self, depdir, pluginType ):
      '''
      1. Read /bld/deps/<pkg>/pluginType.json if it exists, ignoring if missing or
         malformed.
      2. If A4PKG is set, start walking the Requires and BuildRequires
         to determine which plugins we can load.  Add them to permits.
      '''
      checkOutdatedUsage()

      # We need to differentiate the follwoing two cases.
      #   1. Deps based plugin loading is disabled.
      #   2. Deps Based plugin loading is enabled and /bld/deps/<pkg>/pluginType.json
      #      has null dictionary ( {} )
      # In case 1, all plugins should be loaded.
      # In case 2, none of the plugins should be loaded.
      # Initialize self.plugins to None. If deps based plugin loading is enabled,
      # set self.plugins to contents of /bld/deps/<pkg>/pluginType.json - either {}
      # or non empty dict.
      self.plugins = None
      depBasedPluginEnabled = False
      self.package = os.environ.get( "A4PKG" )
      if self.package:
         depBasedPluginEnabled = True
      # Disable plugin based loading for a specific plugin type if the environment
      # variable is explicitly set to disable the plugin based loading.
      envName = depBasedEnvName( pluginType )
      if envName in os.environ and os.environ.get( envName ) == '0':
         depBasedPluginEnabled = False

      if self.package and depBasedPluginEnabled:
         # Expected format of json files is given below per plugin type:
         # PyPlugin.json format:
         # {
         #    "FooPlugin": [ "Foo.py" ],
         #    "BarPlugin": [ "FooBar.py" ]
         # },
         # Preinit.json format:
         # {
         #    "preinit": [ "Ebra,", "Strata", "Sand" ]
         # }
         # SysdbProfiles.json format:
         # {
         #    "SysdbMountProfiles": [ "Ebra,", "Strata", "Sand" ]
         # }
         # CppPlugin.json format:
         # {
         #    "StrataScPlugin": [ "ScPlugin1", "ScPlugin2" ],
         #    "FruPlugin": [ "FruPlugin1", "FruPlugin2" ],
         # }
         # DataPlugin.json format:
         # {
         #    "ConfigSession": [ "OpenConfigX.py" ],
         #    "CliExtension": [ "ExtensionX.yaml" ]
         # }

         fname = os.path.join( depdir, self.package, pluginType + ".json" )
         try:
            with open( fname ) as fp:
               self.plugins = json.load( fp )
         except ( OSError, ValueError ) as e:
            if not os.environ.get( 'A4_MOCK' ):
               sys.stderr.write(
                  "Ignoring plugin info from %s: %s\n" % ( fname, e ) )

   def canLoad( self, pluginTypeOrDir, pluginName ):
      '''
      If A4PKG is not set or the plugin file wasn't read, return True.
      Otherwise, return whether the plugin is permitted.
      '''

      pluginType = os.path.basename( pluginTypeOrDir.rstrip( "/" ) )
      if ( not self.package or self.plugins is None or
           not ( pluginTypeOrDir.startswith( ( "/usr/lib", "/usr/lib64",
                                               "/usr/share" ) ) ) ):
         return True

      return pluginType in self.plugins and pluginName in self.plugins[ pluginType ]
