# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import PowerFirmware
from struct import unpack

WRITE_PROTECT = 0x10
SEC_FW_REV = 0xef
IMAGE_ID = 0xdf

t0 = PowerFirmware.t0

class DeltaPowerSupply( PowerFirmware.PowerSupply ):
   deviceIdBase = 0x58
   upgradeTime = 4

   def __init__( self, slotId, smbusDesc, scdPciAddr ):
      PowerFirmware.PowerSupply.__init__( self, slotId, smbusDesc, scdPciAddr )
      self.currentSecFwRev = None

   def getPowerSupplyInfo( self ):
      PowerFirmware.PowerSupply.getPowerSupplyInfo( self )
      secFwRev = self.helper.readByteList( SEC_FW_REV, 8, 8 )
      self.currentSecFwRev = bytearray( secFwRev )
      # pylint: disable-next=consider-using-f-string
      t0( "Firmware Rev: %s" % self.currentSecFwRev )
      imageIdData = self.helper.read( IMAGE_ID, 4, self.addrSize )
      imageId = unpack( "<I", bytearray( imageIdData ) )[ 0 ]
      t0( "Image ID: 0x%08X" % imageId ) # pylint: disable=consider-using-f-string

   def shouldUpgrade( self ):
      raise NotImplementedError

   def upgradeSupply( self, firmwareFileName ):
      def doRead( register, count ):
         val = self.helper.read( register, count, count )
         if count == 1:
            return val[0]
         return val

      def doWrite( register, data, pec=None ):
         if isinstance( data, int ):
            assert 0 <= data <= 0xff
            data = chr( data )
         return self.helper.write( register, data, len( data ), pec=pec )

      # Disable write protect
      doWrite( WRITE_PROTECT, 0x00 )

      # Put the bootloader into a good state
      doWrite( 0xd2, 0x02, pec=True )
      Tac.runActivities( 0.1 )

      # Write bootloader key
      doWrite( 0xd1, "UBNC", pec=True )

      # Go to bootloader mode
      doWrite( 0xd2, 0x03, pec=True )
      Tac.runActivities( 1.5 )
      assert ( doRead( 0xd2, 1 ) & 0x40 ) == 0x40

      # Reset bootloader
      doWrite( 0xd2, 0x00, pec=True )
      Tac.runActivities( 0.1 )

      # Erase firmware
      doWrite( 0xd2, 0x04, pec=True )
      Tac.runActivities( 5 )

      # Write new firmware
      with open( firmwareFileName, "rb" ) as f:
         while True:
            data = f.read( 16 )
            if data:
               assert len( data ) == 16
               doWrite( 0xee, data, pec=True )
               Tac.runActivities( 0.02 )
            else:
               break

      # Verify checksum valid
      assert doRead( 0xd2, 1 ) == 0x41

      # Go to program mode
      doWrite( 0xd2, 0x01 )
      Tac.runActivities( 1 )

class DPS500AC_RED( DeltaPowerSupply ):
   mfrModel = 'DPS-500AB-40 A'
   firmwareFileName = '/usr/share/firmware/DPS-500AB-40_A_00.3E.0B.bin'

   def shouldUpgrade( self ):
      return self.currentSecFwRev in { b'00.3E.0A', b'00.00.0A', None }

class DPS500AC_BLUE( DeltaPowerSupply ):
   mfrModel = 'DPS-500AB-43 A'
   firmwareFileName = '/usr/share/firmware/DPS-500AB-43_A_00.3E.0B.bin'

   def shouldUpgrade( self ):
      return self.currentSecFwRev in { b'00.3E.0A', b'00.00.0A', None }

def Plugin( powerSupplyPlugin ):
   powerSupplyPlugin.registerPowerSupply( DPS500AC_RED )
   powerSupplyPlugin.registerPowerSupply( DPS500AC_BLUE )
