# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""Helpers for profiling Python code."""

import sys

class Profiler:
   """A class that makes it more convenient to profile small sections of code.  The
   class can be used in two ways:

   1. As a function decorator:

      @Profile.Profiler( stream=sys.stdout )
      def myfunc( ... ):
         ...

      The Profiler object is exposed as the .profiler attribute of the decorated
      function.

   2. As a context manager:

      with Profile.Profiler( filename='/tmp/mycode.prof' ) as p:
         ...

      The Profiler object can later be accessed as 'p'.

   Whenever the decorated function or block of code is executed, it will be profiled
   (using the cProfile module).

   If a stream is specified, the profiling data will be printed in human-readable
   form to that stream, sorted using the specified sort key (see
   http://docs.python.org/library/profile.html#pstats.Stats.sort_stats for a list of
   the valid sort keys).

   If a filename is specified, the raw profiling data will be stored in a file of
   that name.  The file will be overwritten if it already exists.

   If cumulative is True, the data will be accumulated over all executions of the
   function/code.  If cumulative is False (the default), the data will be cleared
   just before each time the function/code is executed.

   Note that to use cumulative mode with a context manager, the Profiler object must
   be stored in a global variable; otherwise a new Profiler object will be created
   each time the 'with' statement is executed.

   If onExit is True, the data will only be written to the stream and/or file at
   program termination (provided that Python exits cleanly) or if the print_stats()
   or dump_stats() methods are called explicitly.  If onExit is False (the default),
   the data will be written to the stream and/or file each time the function/code is
   executed; if the function/code is executed multiple times, only the data for the
   final run will be kept.

   Additional keyword arguments are passed through to the constructor of the
   underlying cProfile.Profile object.

   Use the pstats module to interpret the raw profile data; see
   <http://docs.python.org/library/profile.html#pstats.Stats> for documentation.
   """

   def __init__( self, stream=None, sort="stdname", filename=None,
                 cumulative=False, onExit=False, **kwargs ):
      import cProfile # pylint: disable=import-outside-toplevel
      self.profiler_ = cProfile.Profile( **kwargs )
      self.stream_ = stream
      self.sort_ = sort
      self.filename_ = filename
      self.cumulative_ = cumulative
      self.onExit_ = onExit
      self.depth_ = 0
      if self.onExit_:
         import atexit # pylint: disable=import-outside-toplevel
         atexit.register( self._print_and_dump_stats )

   def __call__( self, f ):
      import functools # pylint: disable=import-outside-toplevel
      @functools.wraps( f )
      def wrapper( *args, **kwargs ):
         with self:
            return f( *args, **kwargs )
      wrapper.profiler = self
      return wrapper

   def __enter__( self ):
      if not self.depth_:
         if not self.cumulative_:
            self.profiler_.clear()
         self.profiler_.enable()
      self.depth_ += 1
      return self

   def __exit__( self, ty, val, tb ):
      self.depth_ -= 1
      if not self.depth_:
         self.profiler_.disable()
         if not self.onExit_:
            self._print_and_dump_stats()

   def _print_and_dump_stats( self ):
      if self.stream_:
         self.print_stats()
      if self.filename_:
         self.dump_stats()

   def print_stats( self, stream=None, sort=None ):
      """Prints the profiling data in human-readable form to the specified stream,
      sorted using the specified sort key (see
      http://docs.python.org/library/profile.html#pstats.Stats.sort_stats for a list
      of the valid sort keys).  If no stream or sort key are specified, they default
      to the ones passed to the constructor."""
      if stream is None:
         stream = self.stream_
         if stream is None:
            raise ValueError( "Must specify a stream" )
      if sort is None:
         sort = self.sort_
      try:
         import pstats # pylint: disable=import-outside-toplevel
         p = pstats.Stats( self.profiler_, stream=stream )
         p.strip_dirs().sort_stats( sort ).print_stats()
      except OSError as e:
         # pylint: disable-next=consider-using-f-string
         print( "Warning: failed to print profile data to %r: %s" % (
            e.filename, e.strerror ), file=sys.stderr )

   def dump_stats( self, filename=None ):
      """Writes the raw profiling data in binary form to the specified filename.  If
      no filename is specified, it defaults to the one passed to the constructor."""
      if filename is None:
         filename = self.filename_
         if filename is None:
            raise ValueError( "Must specify a filename" )
      try:
         self.profiler_.dump_stats( filename )
      except OSError as e:
         # pylint: disable-next=consider-using-f-string
         print( "Warning: failed to dump profile data to %r: %s" % (
            e.filename, e.strerror ), file=sys.stderr )
