#!/usr/bin/env python3
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=singleton-comparison
# pylint: disable=consider-using-in

import Tac
from QosTypes import tacDirection, tacCMapNm, tacClassMapCpStaticType, \
      tacActionRateType, tacRateUnit, tacActionType, tacClassMapCpType, \
      tacEcnDelayThreshold, tacGuaranteedBw, tacGuaranteedBwVal, \
      tacPMapNm, tacPercent, tacShapeRate, tacBurstUnit, tacPolicerType, \
      tacTxQueueSchedulerProfileName
import Tracing
import MultiRangeRule
import AclCliLib
import QosLib
from QosLib import coppMapType, pdpMapType, qosMapType, rateUnitFromEnum, \
      isDefaultClass, coppStaticClassesByPrio, classNameFromCpStaticType, \
      classMapCpStaticType, pmapQosActionTypes, burstUnitFromEnum, \
      coppDynamicClassPresent, policyQosClassPresent, coppPerPortClassPresent, \
      coppStaticClassFromHwStatus, bsToBytes, isLagPort
import Cell
from CliPlugin.QosCliModel import PolicyMapModel, BandwidthModel, \
      ClassMapAllModel, PoliceModel, QosProfileModel, TxQueueProfileModel, \
      GuaranteedBwModel, ShapeRateModel, EcnParametersModel, PfcProfileModel, \
      EcnDelayConfigurationModel, WredParametersModel, PolicyQosModel, \
      PfcWatchdogPortTimerModel, DropPrecedenceThresholdsModel, \
      LatencyThresholdModel, MatchL2Params, BurstSizeModel, ProfileModel, \
      PolicerInstanceModel, InterfacePolicerModel, HwProgrammingEnumModel, \
      SchedulerCompensationModel, InterfacePolicingCounterModel, \
      IntfPoliceCounters, PoliceCounters, \
      PolicerPktSizeAdjProfileModel, PolicerPktSizeAdjModel, \
      EcnDelayTxQParametersModel, DecnParametersModel
from Toggles.QosToggleLib import toggleQosRateCounterEnabled

__defaultTraceHandle__ = Tracing.Handle( 'QosCli' )
t0 = Tracing.trace0

def populateL2ParamsModel( match, matchRule, vlanValue=None, innerVlanValue=None,
                           cosValue=None, deiValue=None, innerCosValue=None ):
   match.matchRule = matchRule
   vlanString = ''
   innerVlanString = ''
   cosString = ''
   deiString = ''
   innerCosString = ''
   valueString = ''
   modelL2Params = MatchL2Params()

   if vlanValue:
      modelL2Params.vlanMaskValid = vlanValue.maskValid
      if not vlanValue.maskValid:
         ids = set()
         for key in vlanValue.vlanColl:
            ids.update( list( range( key.min, key.max + 1 ) ) )
            vlanString = MultiRangeRule.multiRangeToCanonicalString( list( ids ) )
      else:
         modelL2Params.vlanMask = vlanValue.vlanMask
         vlanString = str( vlanValue.vlan )
      valueString = vlanString
      modelL2Params.vlanValue = vlanString

   if innerVlanValue:
      modelL2Params.innerVlanMaskValid = innerVlanValue.maskValid
      if not innerVlanValue.maskValid:
         ids = set()
         for key in innerVlanValue.innerVlanColl:
            ids.update( list( range( key.min, key.max + 1 ) ) )
            innerVlanString = MultiRangeRule.multiRangeToCanonicalString(
               list( ids ) )
      else:
         modelL2Params.innerVlanMask = innerVlanValue.innerVlanMask
         innerVlanString = str( innerVlanValue.innerVlan )
      valueString = innerVlanString if valueString == '' else valueString
      modelL2Params.innerVlanValue = innerVlanString

   if innerCosValue:
      ids = set()
      for key in innerCosValue.innerCosColl:
         ids.update( list( range( key.min, key.max + 1 ) ) )
      innerCosString = MultiRangeRule.multiRangeToCanonicalString( list( ids ) )
      modelL2Params.innerCosValue = innerCosString

   if cosValue:
      ids = set()
      for key in cosValue.cosColl:
         ids.update( list( range( key.min, key.max + 1 ) ) )
      cosString = MultiRangeRule.multiRangeToCanonicalString( list( ids ) )
      modelL2Params.cosValue = cosString

   if deiValue:
      deiString = str( deiValue.deiVal )
      modelL2Params.deiValue = deiString

   match.l2ParamsValue = modelL2Params
   valueString = cosString if valueString == '' else valueString
   return ( valueString, match )

def populateDscpEcnModel( match, matchRule, dscpValue ):
   match.matchRule = matchRule
   ecnName = AclCliLib.ecnNameFromValue( dscpValue.ecn )
   match.matchEcnValue = ecnName
   if dscpValue.dscp == Tac.Value( "Qos::DscpVal" ).invalid:
      if not dscpValue.dscpColl:
         # 'match ecn <ecnName>'
         value = ecnName
      else:
         # 'match dscp <comma-separated range>'
         match.matchDscpNameValid = dscpValue.dscpNameValid
         ids = set()
         for key in dscpValue.dscpColl:
            ids.update( list( range( key.min, key.max + 1 ) ) )
         value = \
                 MultiRangeRule.multiRangeToCanonicalString( list( ids ) )
   else:
      match.matchDscpNameValid = dscpValue.dscpNameValid
      if dscpValue.dscpNameValid:
         # 'match dscp <dscpName>'
         value = AclCliLib.dscpNameFromValue( dscpValue.dscp )
         match.dscpValue = dscpValue.dscp
      else:
         # 'match dscp <dscpId>'
         value = str( dscpValue.dscp )
   return( value, match )

def populateShapeRateModel( shapeRate, statusShapeRate=None, lagSize=None,
                            reference=None ):
   if shapeRate is None or shapeRate == tacShapeRate():
      return None
   modelShapeRate = ShapeRateModel()
   modelShapeRate.rate = shapeRate.rate
   modelShapeRate.unit = QosLib.shapeRateUnitFromEnum( shapeRate.unit )
   modelShapeRate.shared = shapeRate.shared
   if shapeRate.percent != tacPercent.invalid:
      modelShapeRate.percentOf = reference
      if statusShapeRate:
         modelShapeRate.rate = statusShapeRate.rate
      if lagSize and modelShapeRate.shared:
         modelShapeRate.percent = shapeRate.percent // lagSize
      else:
         modelShapeRate.percent = shapeRate.percent
   return modelShapeRate

def populateBurstSizeModel( burstSize ):
   modelBurstSize = BurstSizeModel()
   # We always want to store in bytes.
   modelBurstSize.value = bsToBytes( burstSize.value, burstSize.unit )
   modelBurstSize.unit = tacBurstUnit.burstUnitBytes
   return modelBurstSize

def populateSchedulerCompensationModel( schedulerCompensation ):
   modelSchedulerCompensation = SchedulerCompensationModel()
   modelSchedulerCompensation.value = schedulerCompensation
   modelSchedulerCompensation.unit = tacBurstUnit.burstUnitBytes
   return modelSchedulerCompensation

def populatePolicerPktSizeAdjModel( profileName, value ):
   modelPolicerPktSizeAdj = PolicerPktSizeAdjModel()
   modelPolicerPktSizeAdj.profile = profileName
   modelPolicerPktSizeAdj.value = value
   return modelPolicerPktSizeAdj

def populateGuaranteedBwModel( guaranteedBw, statusGuaranteedBw=None,
                               reference=None ):
   if guaranteedBw is None or guaranteedBw == tacGuaranteedBw():
      return None
   modelGuaranteedBw = GuaranteedBwModel()
   if guaranteedBw.unit == QosLib.tacGuaranteedBwUnit.guaranteedBwPercent:
      if statusGuaranteedBw and statusGuaranteedBw.bw != tacGuaranteedBwVal.invalid:
         # We want to use the absolute value instead of the raw percent if
         # possible, so we can show it to the user as the configured value
         modelGuaranteedBw.bandwidth = statusGuaranteedBw.bw
         modelGuaranteedBw.unit = QosLib.guaranteedBwUnitFromEnum(
            statusGuaranteedBw.unit )
      else:
         # If we don't have the absolute value yet, just use the percent since
         # we need to display something.
         modelGuaranteedBw.bandwidth = guaranteedBw.bw
         modelGuaranteedBw.unit = QosLib.guaranteedBwUnitFromEnum(
            guaranteedBw.unit )
      modelGuaranteedBw.percent = guaranteedBw.bw
      modelGuaranteedBw.percentOf = reference
   else:
      modelGuaranteedBw.bandwidth = guaranteedBw.bw
      modelGuaranteedBw.unit = QosLib.guaranteedBwUnitFromEnum(
            guaranteedBw.unit )
   return modelGuaranteedBw

def deleteInterfacePolicingConfig( qosInputConfig, intf ):
   intfPair = ( Tac.Value( "Qos::IntfPairKey", "", intf ) if isLagPort( intf ) else
                Tac.Value( "Qos::IntfPairKey", intf, "" ) )
   if qosInputConfig.policingConfig:
      t0( 'Deleting ', intfPair, ' from intf-Policer collection' )
      del qosInputConfig.policingConfig.intfPolicer[ intfPair ]
      del qosInputConfig.policingConfig.intfPolicerOutput[ intfPair ]
      del qosInputConfig.policingConfig.intfPolicerCpu[ intfPair ]

class PMapModelContainer:
   def __init__( self, qosConfig, qosAclConfig, status, hwStatus, sliceHwStatus,
                 aclHwStatus, aclSliceHwStatus, mapType, direction, hwEpochStatus,
                 defaultPdpPmapCfg, policyMapAllModel=None ):
      self.qosConfig_ = qosConfig
      self.qosAclConfig_ = qosAclConfig
      self.status_ = status
      self.hwStatus_ = hwStatus
      self.sliceHwStatus_ = sliceHwStatus
      self.aclHwStatus_ = aclHwStatus
      self.aclSliceHwStatus_ = aclSliceHwStatus
      self.mapType_ = mapType
      self.direction_ = direction
      self.hwEpochStatus_ = hwEpochStatus
      if self.mapType_ == coppMapType:
         self.direction_ = tacDirection.input
      self.spAppliedDirections = set()
      self.defaultPdpPmapCfg_ = defaultPdpPmapCfg
      self.policyMapAllModel = policyMapAllModel
      self.counterDetail = None
      self.classMaps = dir()
      if self.policyMapAllModel:
         self.policyMapAllModel.maxCmapIs( aclHwStatus.policyQosNumCMapSupported )
         self.policyMapAllModel.mapTypeIs( self.mapType_ )

   def populateAll( self, summary=None ):
      pmapList = sorted( self.qosAclConfig_.pmapType[ self.mapType_ ].pmap )
      if self.mapType_ == pdpMapType and self.defaultPdpPmapCfg_:
         pmapList.append( self.defaultPdpPmapCfg_.name )
      self.policyMapAllModel.summaryIs( summary )
      for name in pmapList:
         if self.counterDetail is None:
            self.populatePolicyMap( name, summary )
         else:
            self.populatePolicyMapCounters( name )

   def populatePolicyMap( self, pMap, summary=None ):
      policyMap = self._populatePolicyMap( pMap, interfaceOption=False,
                                           summary=summary )
      if policyMap:
         policyMap.mapType = self.mapType_
         self.policyMapAllModel.append( pMap, policyMap )

   def populatePolicyMapClassMap( self, pMap, cMap ):
      policyMap = self._populatePolicyMap( pMap, clMap=cMap )
      if policyMap:
         policyMap.mapType = self.mapType_
         self.policyMapAllModel.append( pMap, policyMap )

   def populatePolicyMapInterface( self, pMap ):
      policyMap = self._populatePolicyMap( pMap, interfaceOption=True,
                                           counterOption=self.counterDetail )
      if policyMap:
         policyMap.mapType = self.mapType_
         self.policyMapAllModel.append( pMap, policyMap )

   def populatePolicyMapCounters( self, pMap ):
      policyMap = self._populatePolicyMap( pMap, interfaceOption=True,
                                           counterOption=True )
      if policyMap:
         policyMap.mapType = self.mapType_
         self.policyMapAllModel.append( pMap, policyMap )

   def pmapHwStatusPresent( self, pMap, clMap=None ):
      if not self.direction_:
         return None
      key = Tac.newInstance( "Qos::PolicyMapHwStatusKey",
                             self.direction_, pMap.name )
      hwStatus = None
      for sliceHwStatus_ in self.sliceHwStatus_.values():
         if self.mapType_ in sliceHwStatus_.pmapType:
            hwStatus = sliceHwStatus_
            break

      for aclSliceHwStatus_ in self.aclSliceHwStatus_.values():
         if self.mapType_ in aclSliceHwStatus_.pmapType:
            if( coppDynamicClassPresent( self.aclHwStatus_, aclSliceHwStatus_, key,
                                         self.mapType_, clMap=clMap ) or
               policyQosClassPresent( aclSliceHwStatus_, key,
                                      self.mapType_, clMap=clMap ) or
               coppPerPortClassPresent( self.aclHwStatus_, aclSliceHwStatus_, key,
                                        self.mapType_, pMap=pMap, clMap=clMap ) ):
               hwStatus = aclSliceHwStatus_
               break

      if hwStatus is None:
         return None

      if self.mapType_ not in hwStatus.pmapType:
         return None

      if key in hwStatus.pmapType[ self.mapType_ ].pmap:
         pmapHwStatus = hwStatus.pmapType[ self.mapType_ ].pmap[ key ]
         return pmapHwStatus
      return None

   def isPolicyMapApplied( self, direction, pMap ):
      # Returns whether policy map is actually applied
      # along with the hwProgrammingStatus
      spKey = Tac.newInstance( "Qos::ServicePolicyKey", pMap.type,
                               direction, pMap.name )
      programmingStatus = ""
      applied = False
      if spKey in self.status_.servicePolicyStatus:
         applied = True
         programmingStatus = QosLib.programmedPMapHwStatus( self.hwStatus_,
                                                            self.aclHwStatus_,
                                                            self.sliceHwStatus_,
                                                            self.aclSliceHwStatus_,
                                                            self.mapType_,
                                                            pMap.name, direction )

      return applied, programmingStatus

   def getSpAppliedAndStatus( self, pMap ):
      if self.direction_ != None:
         directions = [ self.direction_ ]
         self.spAppliedDirections = { self.direction_ }
      else:
         directions = [ tacDirection.input, tacDirection.output ]
         self.spAppliedDirections = set()
      spApplied = {}
      status = {}
      for direction in directions:
         applied, prgmHwStatus = self.isPolicyMapApplied( direction, pMap )
         spApplied[ direction ] = applied
         status[ direction ] = prgmHwStatus
         if applied:
            self.spAppliedDirections.add( direction )
      return spApplied, status

   def populatePolicyMapQosInterface( self, pMap, intfServicePolicyQosModel=None ):
      if isinstance( pMap, str ):
         pMap = self.qosAclConfig_.pmapType[ self.mapType_ ].pmap.get( pMap )
         if pMap is None:
            return
      pmapHwStatus = self.pmapHwStatusPresent( pMap )
      if pmapHwStatus is None:
         return
      spApplied, status = self.getSpAppliedAndStatus( pMap )

      if True in spApplied.values():
         # If service policy is applied in either direction, populate the policy
         # with direction and programming status.
         for direction, isApplied in spApplied.items():
            if isApplied:
               policyQosModel = PolicyQosModel()
               policyQosModel.name = pMap.name
               policyQosModel.directionString = direction
               policyQosModel.status = status[ direction ]
               intfServicePolicyQosModel.intfPolicyMaps[ direction ] = policyQosModel

   def _populatePolicyMap( self, pMap, interfaceOption=False,
                           counterOption=False, clMap=None, summary=None ):
      policyMap = PolicyMapModel()
      if isinstance( pMap, str ):
         if pMap in self.qosAclConfig_.pmapType[ self.mapType_ ].pmap:
            pMap = self.qosAclConfig_.pmapType[ self.mapType_ ].pmap[ pMap ]
         elif self.mapType_ == pdpMapType and \
              self.defaultPdpPmapCfg_ and \
              pMap == self.defaultPdpPmapCfg_.name:
            pMap = self.defaultPdpPmapCfg_
         else:
            return None

      if pMap.shared:
         policyMap.shared = True
      pmapHwStatus = self.pmapHwStatusPresent( pMap, clMap=clMap )
      if( ( interfaceOption and self.direction_ ) and pmapHwStatus is None and
         self.mapType_ != pdpMapType ):
         return None
      spApplied, status = self.getSpAppliedAndStatus( pMap )
      policyMap.spApplied = spApplied
      if True in spApplied.values():
         # If service policy is applied in either direction, populate the policy
         # with direction and programming status.
         for direction, isApplied in spApplied.items():
            if isApplied:
               hardwareStatus = PolicyMapModel.HwStatus()
               if self.mapType_ == coppMapType:
                  hardwareStatus.unitsProgrammed = \
                     sum( 1 for phySlice in self.sliceHwStatus_.values()
                          if phySlice.physicalSlice )
               hardwareStatus.status = status[ direction ]
               policyMap.hwStatus[ direction ] = hardwareStatus

      if self.mapType_ == coppMapType and pMap.name == tacPMapNm.coppName and \
            self.hwStatus_.intfCoppSupported:
         policyMap.name = pMap.name + \
               " ( default applied per input port if not overridden )"
      else:
         policyMap.name = pMap.name

      # pylint: disable-msg=too-many-nested-blocks
      if summary:
         policyMap.summary = True
         for direction in spApplied:
            key = Tac.newInstance( "Qos::ServicePolicyKey", self.mapType_,
                                   direction, pMap.name )
            policyMap.configuredIntfs[ direction ] = \
                                                     PolicyMapModel.InterfaceList()
            if key in self.qosConfig_.servicePolicyConfig:
               configuredIntfs = self.qosConfig_.servicePolicyConfig[ key ].intfIds
               interfaceList = PolicyMapModel.InterfaceList()
               interfaceList.interfaces = []
               if configuredIntfs:
                  interfaceList.interfaces = [
                     intfName for intfName in configuredIntfs if
                     configuredIntfs[ intfName ] ]
               policyMap.configuredIntfs[ direction ] = interfaceList
            policyMap.activeIntfs[ direction ] = \
                                                 PolicyMapModel.InterfaceList()
            if key in self.status_.servicePolicyStatus:
               activeIntfs = self.status_.servicePolicyStatus[ key ].intfIds
               interfaceList = PolicyMapModel.InterfaceList()
               interfaceList.interfaces = []
               # activeIntfs is a list of IntfPairKeys
               for intf in activeIntfs:
                  if intf.lagIntf:
                     if intf.lagIntf not in interfaceList.interfaces:
                        interfaceList.interfaces.append( intf.lagIntf )
                  else:
                     interfaceList.interfaces.append( intf.intfId )
               policyMap.activeIntfs[ direction ] = interfaceList
            # list of interfaces on which the service policy is not active
            interfaceList = PolicyMapModel.InterfaceList()
            interfaceList.interfaces = list(
               set( policyMap.configuredIntfs[ direction ].interfaces ) -
               set( policyMap.activeIntfs[ direction ].interfaces ) )
            policyMap.inactiveIntfs[ direction ] = interfaceList
         # We continue generating the list of class maps even for
         # 'summary' case as we need the count of class maps anyways

      classMaps = {}
      clMapObj = None
      if clMap:
         if isinstance( clMap, str ):
            clMapName = clMap
            if clMapName in self.qosAclConfig_.cmapType[ self.mapType_ ].cmap:
               clMapObj = self.qosAclConfig_.cmapType[ self.mapType_ ].cmap[
                  clMapName ]
            elif isDefaultClass( self.mapType_, clMapName ):
               clMapObj = pMap.classDefault
         else:
            # Store class map name and object seperately as we may have a case where
            # class map is not registered.
            clMapName = clMap.name
            clMapObj = clMap
         if self.mapType_ == coppMapType and clMapObj == tacCMapNm.coppDrop:
            return None
         if isDefaultClass( self.mapType_, clMapName ) or \
                clMapName in pMap.classAction:
            classMap = PolicyMapModel.ClassMap()
            if clMapObj is not None:
               self._populateClassMap( clMapObj, pMap, interfaceOption,
                                       counterOption, classMap, 1, self.mapType_ )
               self.populateAction( pMap, clMapObj, interfaceOption, counterOption,
                                    classMap )
            else:
               self._populateClassMap( clMapName, pMap, interfaceOption,
                                       counterOption, classMap, 1, self.mapType_ )
               self.populateAction( pMap, clMapName, interfaceOption,
                                    counterOption, classMap )
            if classMap.matchCondition:
               policyMap.classMaps[ clMapName ] = classMap
               return policyMap

      self.populateClassMap( pMap, interfaceOption, counterOption,
                             classMaps=classMaps )
      for cMapName, cMap in classMaps.items():
         if cMapName != None:
            policyMap.classMaps[ cMapName ] = cMap
      return policyMap

   def populateClassMap( self, pMap, interfaceOption, counterOption, clMap=None,
                         classMaps=None ):
      if isinstance( pMap, str ):
         pMap = self.qosAclConfig_.pmapType[ self.mapType_ ].pmap.get( pMap )
         if pMap is None:
            return

      if clMap is None:
         # populate dynamic cmaps
         self.populateClassMapDynamic( pMap, interfaceOption, counterOption,
                                       classMaps )
         if self.mapType_ == coppMapType:
            # populate static cmaps
            self.populateClassMapStatic( pMap, interfaceOption, counterOption,
                  classMaps )
      else:
         if isinstance( clMap, str ):
            if clMap in self.qosAclConfig_.cmapType[ self.mapType_ ].cmap:
               clMap = self.qosAclConfig_.cmapType[ self.mapType_ ].cmap[ clMap ]
            elif isDefaultClass( self.mapType_, clMap ):
               clMap = pMap.classDefault
            else:
               return
         if self.mapType_ == coppMapType and clMap == tacCMapNm.coppDrop:
            return
         if isDefaultClass( self.mapType_, clMap ) or \
                clMap.name in pMap.classAction:
            classMap = PolicyMapModel.ClassMap()
            self._populateClassMap( clMap, pMap, interfaceOption, counterOption,
                                    classMap, 1, self.mapType_ )
            self.populateAction( pMap, clMap, interfaceOption, counterOption,
                                 classMap )

   def populateClassMapDynamic( self, pMap, interfaceOption, counterOption,
                                classMaps=None ):
      for _, clPrio in sorted( pMap.classPrio.items() ):
         classMap = PolicyMapModel.ClassMap()
         clName = clPrio.cmapName
         # ClassPrios' is a sequence of consecutive integers starting from 1
         # Not using the index as classPrio because it might lead to
         # nonconsecutive seq
         classPrio = len( classMaps ) + 1
         self._populateClassMap( clName, pMap, interfaceOption, counterOption,
                                 classMap, classPrio, self.mapType_ )
         self.populateAction( pMap, clName, interfaceOption, counterOption,
                              classMap )
         if classMap.matchCondition:
            classMaps[ clName ] = classMap
      # For qosMapType, display the default class
      if self.mapType_ != coppMapType:
         classMap = PolicyMapModel.ClassMap()
         clName = tacCMapNm.classDefault
         self._populateClassMap( clName, pMap, interfaceOption, counterOption,
                                 classMap, len( classMaps ) + 1, self.mapType_ )
         self.populateAction( pMap, clName, interfaceOption, counterOption,
                              classMap )
         if classMap.matchCondition:
            classMaps[ clName ] = classMap

   def populateClassMapStatic( self, pMap, interfaceOption, counterOption,
                               classMaps ):
      if not self.mapType_ == coppMapType:
         return
      numOfDynamicClasses = len( classMaps )
      idx = 1
      prioAndTypes = coppStaticClassesByPrio( self.sliceHwStatus_ )
      if prioAndTypes is None:
         # Ideally should never fall here
         return
      if pMap.name == tacPMapNm.coppName:
         for classPrioAndType in prioAndTypes:
            cmapCpStatic = classPrioAndType[ 1 ]
            if cmapCpStatic == tacClassMapCpStaticType.cmapCpStaticDrop:
               # BUG470644: Remove this exception for cmapCpStatiDrop once Drop VOQ
               # Mirroring is enabled in eos-trunk.
               continue
            cmapName = classNameFromCpStaticType( cmapCpStatic )
            classMap = PolicyMapModel.ClassMap()
            classPrio = numOfDynamicClasses + idx
            self._populateClassMap( cmapName, pMap, interfaceOption, counterOption,
                                    classMap, classPrio, self.mapType_ )
            self.populateAction( pMap, cmapName, interfaceOption, counterOption,
                                 classMap )
            if classMap.matchCondition:
               classMaps[ cmapName ] = classMap
               idx += 1
      else:
         for classPrio in pMap.coppStaticClassPrio:
            cmapCpStatic = classPrio
            cmapName = pMap.coppStaticClassPrio[ classPrio ].cmapName
            classMap = PolicyMapModel.ClassMap()
            classPrio = numOfDynamicClasses + idx
            self._populateClassMap( cmapName, pMap, interfaceOption, counterOption,
                                    classMap, classPrio, self.mapType_ )
            self.populateAction( pMap, cmapName, interfaceOption, counterOption,
                                 classMap )
            # Do not show empty converted class maps on Sand platforms
            if pMap.name == tacPMapNm.convertedPmapName and \
               not self.hwStatus_.coppActionPolicerSupported and \
               classMap.shape is None and classMap.bandwidth is None:
               continue
            if classMap.matchCondition:
               classMaps[ cmapName ] = classMap
               idx += 1

         # default class
         if pMap.classActionDefault.policer != None or \
            pMap.classActionDefault.policyAction:
            classMap = PolicyMapModel.ClassMap()
            cmapName = pMap.classDefault.name
            self._populateClassMap( cmapName, pMap, interfaceOption, counterOption,
                                    classMap, mapType=self.mapType_ )
            self.populateAction( pMap, cmapName, interfaceOption, counterOption,
                                 classMap )
            if classMap.matchCondition:
               classMaps[ cmapName ] = classMap

   def _populateClassMap( self, clMap, pMap, interfaceOption, counterOption,
                          classMap=None, classPrio=1, mapType='qos' ):
      if isinstance( clMap, str ):
         if clMap in self.qosAclConfig_.cmapType[ self.mapType_ ].cmap:
            clMap = self.qosAclConfig_.cmapType[ self.mapType_ ].cmap[ clMap ]
         elif isDefaultClass( self.mapType_, clMap ):
            clMap = pMap.classDefault
         else:
            # This class-map is not present in qosAclConfig, so for printing this
            # class-map, we need to populate it's attributes with dummy values.
            # Also match rule is not required for this class-map.
            classMap.matchCondition = 'matchConditionAny'
            classMap.name = clMap
            classMap.classPrio = classPrio
            classMap.mapType = mapType
            return

      clMapHwStatus = None
      pmapHwStatus = self.pmapHwStatusPresent( pMap, clMap=clMap )
      if pmapHwStatus != None:
         if clMap.name in pmapHwStatus.cmap:
            clMapHwStatus = pmapHwStatus.cmap[ clMap.name ]
      if( ( interfaceOption and self.direction_ ) and clMapHwStatus is None and
          self.mapType_ != pdpMapType ):
         return
      assert clMap.matchCondition == 'matchConditionAny'
      classMap.name = clMap.name
      classMap.matchCondition = clMap.matchCondition
      classMap.classPrio = classPrio
      classMap.mapType = mapType
      if not isDefaultClass( self.mapType_, clMap.name ):
         for matchRule in clMap.match:
            match = PolicyMapModel.ClassMap.Match()
            if matchRule in [ 'matchIpAccessGroup', 'matchIpv6AccessGroup',
                              'matchMacAccessGroup' ]:
               match.matchRule = matchRule
               classMap.match[ clMap.match[ matchRule ].strValue ] = match
            elif matchRule in [ 'matchL2Params' ]:
               vlanValue = clMap.match[ matchRule ].vlanValue
               innerVlanValue = clMap.match[ matchRule ].innerVlanValue
               cosValue = clMap.match[ matchRule ].cosValue
               deiValue = clMap.match[ matchRule ].deiValue
               innerCosValue = clMap.match[ matchRule ].innerCosValue
               ( l2ParamsString, match ) = populateL2ParamsModel(
                 match, matchRule, vlanValue, innerVlanValue, cosValue, deiValue,
                 innerCosValue )
               classMap.match[ l2ParamsString ] = match
            elif matchRule in [ 'matchDscpEcn' ]:
               dscpValue = clMap.match[ matchRule ].dscpEcnValue
               ( value, match ) = populateDscpEcnModel( match, matchRule, dscpValue )
               classMap.match[ value ] = match
            elif matchRule in [ 'matchMplsTrafficClass' ]:
               match.matchRule = matchRule
               mplsTrafficClassVal = clMap.match[ matchRule ].mplsTrafficClassVal
               ids = set()
               for key in mplsTrafficClassVal.mplsTrafficClassColl:
                  ids.update( list( range( key.min, key.max + 1 ) ) )
               mplsTrafficClassString = \
                  MultiRangeRule.multiRangeToCanonicalString( list( ids ) )
               classMap.match[ mplsTrafficClassString ] = match

   def populateAction( self, pMap, clMap, interfaceOption, counterOption,
                       classMap=None ):
      if isinstance( pMap, str ):
         pMap = self.qosAclConfig_.pmapType[ self.mapType_ ].pmap.get( pMap )
         if pMap is None:
            return
      clMapObj = None
      if isinstance( clMap, str ):
         clMapName = clMap
         if clMapName in self.qosAclConfig_.cmapType[ self.mapType_ ].cmap:
            clMapObj = self.qosAclConfig_.cmapType[ self.mapType_ ].cmap[ clMapName ]
         elif isDefaultClass( self.mapType_, clMapName ):
            clMapObj = pMap.classDefault
      else:
         clMapName = clMap.name
         clMapObj = clMap

      def isIntCounter( arg ):
         return arg in [ "outCount", "dropCount" ]

      def counterDir( pmapStatus, cmapName, direction, arg="outCount" ):
         count = 0.0
         cmapKey = Tac.newInstance( "Qos::CounterKey", cmapName, direction )
         clMapStatus = pmapStatus.cmapCounters.get( cmapKey )
         count = getattr( clMapStatus, arg, 0.0 )
         if isIntCounter( arg ):
            count = int( count )
         return count

      def counter( cmapName, arg ):
         count = {}
         for direction in self.spAppliedDirections:
            count[ direction ] = 0 if isIntCounter( arg ) else 0.0
         # Make sure class map is defined before accessing it attributes.
         if not clMapObj:
            return count

         pmapType = self.status_.pmapType.get( self.mapType_ )
         if pmapType is None:
            return count

         pmapStatus = pmapType.pmap.get( pMap.name )
         if pmapStatus is None:
            return count

         return { direction: counterDir( pmapStatus, cmapName, direction, arg )
                  for direction in self.spAppliedDirections }

      clMapHwStatus = None
      pmapHwStatus = self.pmapHwStatusPresent( pMap, clMap=clMapObj )
      if pmapHwStatus != None:
         if clMapName in pmapHwStatus.cmap:
            clMapHwStatus = pmapHwStatus.cmap[ clMapName ]
      # Interface option make sense only if the class-map is defined.
      if clMapObj and ( ( interfaceOption and self.direction_ ) and
         clMapHwStatus is None and self.mapType_ != pdpMapType ):
         return

      if isDefaultClass( self.mapType_, clMapName ):
         classAction = pMap.classActionDefault
      elif clMapName in pMap.classAction:
         classAction = pMap.classAction[ clMapName ]
      else:
         return

      def _populateActionCopp( _clMapObj ):
         maxPps = tacActionRateType.noValue
         minPps = tacActionRateType.noValue
         maxRu = minRu = None
         for policyAction in classAction.policyAction.values():
            if policyAction.actionType == tacActionType.actionSetShape:
               if policyAction.rate:
                  if clMapHwStatus:
                     maxPps = clMapHwStatus.shapeVal
                     maxRu = rateUnitFromEnum( policyAction.rate.rateUnit )
               if maxRu is None:
                  if self.hwStatus_.coppPpsSupported:
                     maxRu = rateUnitFromEnum( tacRateUnit.rateUnitPps )
                  elif self.hwStatus_.coppKbpsSupported:
                     maxRu = rateUnitFromEnum( tacRateUnit.rateUnitKbps )
            elif policyAction.actionType == tacActionType.actionSetBandwidth:
               if policyAction.rate:
                  if clMapHwStatus:
                     minPps = clMapHwStatus.bandwidthVal
                     minRu = rateUnitFromEnum( policyAction.rate.rateUnit )
               if minRu is None:
                  if self.hwStatus_.coppPpsSupported:
                     minRu = rateUnitFromEnum( tacRateUnit.rateUnitPps )
                  elif self.hwStatus_.coppKbpsSupported:
                     minRu = rateUnitFromEnum( tacRateUnit.rateUnitKbps )
         shape = None
         bandwidth = None
         outPackets = None
         dropPackets = None
         if maxPps != tacActionRateType.noValue:
            shape = ShapeRateModel( rate=maxPps, unit=maxRu )
         if minPps != tacActionRateType.noValue:
            bandwidth = BandwidthModel( rate=minPps, unit=minRu )
         if interfaceOption:
            if clMapHwStatus:
               # Check whether class map is defined before accessing it's
               # attributes as we may have undefined class as well.
               if _clMapObj and self.hwStatus_.coppOutCounterSupported:
                  outPackets = counter( clMapName, "outCount" )
               if _clMapObj and self.hwStatus_.coppDropCounterSupported:
                  dropPackets = counter( clMapName, "dropCount" )
         return ( shape, bandwidth, outPackets, dropPackets )

      def _populateActionQosPolicer( cmapName, _clMapObj ):
         actPolice = None
         if pMap.name == tacPMapNm.coppName and \
               self.mapType_ == coppMapType:
            if clMapHwStatus:
               policerRateVal = pmapHwStatus.cmap[ cmapName ].policerRateVal
               policerBurstVal = pmapHwStatus.cmap[ cmapName ].policerBurstVal
               policerRateUnit = pmapHwStatus.cmap[ cmapName ].policerRateUnit
               policerBurstUnit = pmapHwStatus.cmap[ cmapName ].policerBurstUnit
               actPolice = PoliceModel( cir=policerRateVal,
                                        cirUnit=rateUnitFromEnum( policerRateUnit ),
                                        bc=policerBurstVal,
                                        bcUnit=burstUnitFromEnum( policerBurstUnit ),
                                        pir=0, be=0 )
         else:
            policer = classAction.policer
            actPolice = PoliceModel(
               cir=policer.cir, cirUnit=rateUnitFromEnum( policer.cirUnit ),
               bc=policer.bc, bcUnit=burstUnitFromEnum( policer.bcUnit ),
               pir=policer.pir, pirUnit=rateUnitFromEnum( policer.pirUnit ),
               be=policer.be, beUnit=burstUnitFromEnum( policer.beUnit ) )

            actPolice.policerName = policer.name if policer.named else ''
            actPolice.yellowActions = PoliceModel.PolicerActions()
            yellowActions = policer.yellowActions
            if tacActionType.actionSetDropPrecedence in yellowActions:
               actPolice.yellowActions.dropPrecedence = True
            action = yellowActions.get( tacActionType.actionSetDscp )
            if action is not None:
               actPolice.yellowActions.dscp = action.value
               actPolice.yellowActions.dscpConfiguredAsName = \
                                             classAction.dscpConfiguredAsNameYellow

         def _populatePolicerCounterPerDirection( direction ):
            def _getPolicerCounters():
               if direction in actPolice.counters:
                  return actPolice.counters[ direction ]
               return PoliceCounters()

            def _getClMapStatus( cmapName, direction ):
               if not _clMapObj:
                  return None
               pmapType = self.status_.pmapType.get( self.mapType_ )
               if pmapType is None:
                  return None
               pmapStatus = pmapType.pmap.get( pMap.name )
               if pmapStatus is None:
                  return None
               cmapKey = Tac.newInstance( "Qos::CounterKey", cmapName, direction )
               return pmapStatus.cmapCounters[ cmapKey ]

            clMapStatus = _getClMapStatus( cmapName, direction )
            # Do not populate policer related counters in the model for PDP policy
            # classes.
            # We only publish hit counters for PDP policy classes when they have
            # 'count' action set.
            if _clMapObj and counterOption and clMapStatus and not \
               self.mapType_ == pdpMapType:
               # pktConform gets count of green packets
               pktConform = byteConform = pktExceed = byteExceed = 0
               pktYellow = byteYellow = 0
               rateConform = rateYellow = rateExceed = 0.0
               actPolice.redCounter = self.aclHwStatus_.redCounterSupported
               # accumulated, per-interface specific police counters
               for k, v in clMapStatus.policerStatus.items():
                  # conformed pkts = pktInCount (all hits) - pktDropCount (drops)
                  pktConform += ( v.pktInCount - v.pktDropCount )
                  byteConform += ( v.byteInCount - v.byteDropCount )
                  pktYellow += v.yellowPktInCount
                  byteYellow += v.yellowByteInCount
                  pktExceed += v.pktDropCount
                  byteExceed += v.byteDropCount
                  rateConform += v.outRate
                  rateYellow += v.eir
                  rateExceed += v.dropRate
                  actPolice.greenCounter = self.aclHwStatus_.greenCounterSupported
                  if self.counterDetail:
                     # per interface counters
                     policeCounters = _getPolicerCounters()
                     policeCounters.interfaces[ k ] = IntfPoliceCounters(
                          conformedPackets=( v.pktInCount - v.pktDropCount ),
                          conformedBytes=( v.byteInCount - v.byteDropCount ),
                          yellowPackets=( v.yellowPktInCount ),
                          yellowBytes=( v.yellowByteInCount ),
                          exceededPackets=v.pktDropCount,
                          exceededBytes=v.byteDropCount,
                          droppedBitsRate=v.dropRate,
                          conformedBitsRate=v.outRate,
                          exceededBitsRate=v.eir,
                          updateInterval=self.qosConfig_.loadInterval.val )
                     actPolice.counters[ direction ] = policeCounters
               if self.aclHwStatus_.sharedPolicerSupported and \
                  self.mapType_ == qosMapType and not clMapStatus.policerStatus:
                  actPolice.redCounter = self.aclHwStatus_.redCounterSupported
                  actPolice.greenCounter = self.aclHwStatus_.greenCounterSupported
                  actPolice.perSviRedCounterWithSharedPolicer = \
                     self.aclHwStatus_.perSviRedCounterWithSharedPolicerSupported

                  exceededPackets = exceededBytes = 0
                  if self.aclHwStatus_.redCounterSupported:
                     exceededPackets = clMapStatus.dropCount
                     exceededBytes = clMapStatus.dropBytes
                  policeCounters = _getPolicerCounters()
                  policeCounters.interfaces[ 'all' ] = IntfPoliceCounters(
                     conformedPackets=clMapStatus.outCount,
                     conformedBytes=clMapStatus.outBytes,
                     exceededPackets=exceededPackets,
                     exceededBytes=exceededBytes,
                     yellowPackets=clMapStatus.yellowCount,
                     yellowBytes=clMapStatus.yellowBytes )
                  actPolice.counters[ direction ] = policeCounters
               elif not self.counterDetail:
                  policeCounters = _getPolicerCounters()
                  # accumulated police counters for all interfaces
                  policeCounters.interfaces[ 'all' ] = IntfPoliceCounters(
                     conformedBytes=byteConform, exceededPackets=pktExceed,
                     conformedPackets=pktConform, exceededBytes=byteExceed,
                     yellowPackets=pktYellow,
                     yellowBytes=byteYellow, droppedBitsRate=rateExceed,
                     conformedBitsRate=rateConform,
                     exceededBitsRate=rateYellow )
                  actPolice.counters[ direction ] = policeCounters
         # get policer counter per direction
         for direction in self.spAppliedDirections:
            _populatePolicerCounterPerDirection( direction )
         return actPolice

      def _createActionQos( actStrSetCos, actStrSetDscp, actStrSetDrop,
                            actStrSetTc, actStrSetDropPrecedence,
                            dropActionRestricted, dscpConfiguredAsName ):
         qosParams = PolicyMapModel.ClassMap.QosParams()
         qosParams.cos = actStrSetCos
         qosParams.dscp = actStrSetDscp
         qosParams.drop = actStrSetDrop
         qosParams.trafficClass = actStrSetTc
         qosParams.dropPrecedence = actStrSetDropPrecedence
         qosParams.dropActionRestricted = dropActionRestricted
         qosParams.dscpConfiguredAsName = dscpConfiguredAsName
         return qosParams

      def _populateActionQos( _clMapObj ):
         def addAction( actionProgrammed ):
            if interfaceOption:
               if self.direction_:
                  if actionProgrammed:
                     return True
               else:
                  return True
            else:
               return True
            return False

         def _getClMapStatus( cmapName, direction ):
            if not _clMapObj:
               return None
            pmapType = self.status_.pmapType.get( self.mapType_ )
            if pmapType is None:
               return None
            pmapStatus = pmapType.pmap.get( pMap.name )
            if pmapStatus is None:
               return None
            cmapKey = Tac.newInstance( "Qos::CounterKey", cmapName, direction )
            return pmapStatus.cmapCounters[ cmapKey ]

         actPolice = None
         if classAction.policer:
            actPolice = _populateActionQosPolicer( clMapName, _clMapObj )
         actions = classAction.policyAction
         # Maintain the order during display
         actStrSetCos = None
         actStrSetDscp = None
         actStrSetDrop = None
         actStrSetTc = None
         actStrSetDropPrecedence = None
         matchedPackets = None
         dropRate = None
         outRate = None
         eir = None
         dropActionRestricted = False
         dscpConfiguredAsName = None
         if self.hwEpochStatus_.isAclTcamRestricted() and \
               Cell.cellType() != 'supervisor':
            dropActionRestricted = True
         for actionType in pmapQosActionTypes:
            if actionType not in actions:
               continue
            if actionType == tacActionType.actionSetDscp:
               if addAction( clMapHwStatus and clMapHwStatus.dscpPrgmd ):
                  action = actions[ actionType ]
                  actStrSetDscp = action.value
                  dscpConfiguredAsName = classAction.dscpConfiguredAsName
            if actionType == tacActionType.actionSetDrop:
               if addAction( clMapHwStatus and clMapHwStatus.dropPrgmd ):
                  action = actions[ actionType ]
                  actStrSetDrop = True
            elif actionType == tacActionType.actionSetCos:
               if addAction( clMapHwStatus and clMapHwStatus.cosPrgmd ):
                  action = actions[ actionType ]
                  actStrSetCos = action.value
            elif actionType == tacActionType.actionSetTc:
               if addAction( clMapHwStatus and clMapHwStatus.tcPrgmd ):
                  action = actions[ actionType ]
                  actStrSetTc = action.value
            elif actionType == tacActionType.actionSetDropPrecedence:
               if addAction( clMapHwStatus and clMapHwStatus.dropPrecedencePrgmd ):
                  action = actions[ actionType ]
                  actStrSetDropPrecedence = action.value
         if counterOption:
            matchedPackets = {}
            for direction in self.spAppliedDirections:
               # Sand Platforms and in case of unshared mode
               clMapStatus = _getClMapStatus( clMapName, direction )
               if self.aclHwStatus_.sharedPolicerSupported and clMapStatus \
                     and clMapStatus.policerStatus:
                  matchedPackets[ direction ] = {}
                  for intf, policerStatus in clMapStatus.policerStatus.items():
                     matchedPackets[ direction ][ intf ] = policerStatus.pktInCount
               elif clMapStatus and clMapStatus.counterIntfStatus:
                  matchedPackets[ direction ] = {}
                  matchedPackets[direction][ "all" ] = 0
                  for intf, counterIntfStatus \
                        in clMapStatus.counterIntfStatus.items():
                     matchedPackets[direction][ "all" ] += \
                           counterIntfStatus.outCount
                     matchedPackets[direction][ intf ] = counterIntfStatus.outCount
               else:
                  matchedPacketsAll = counter( clMapName, "outCount" )
                  for matchDir in matchedPacketsAll:
                     matchedPackets[ matchDir ] = {}
                     matchedPackets[ matchDir ][ "all" ] = \
                           matchedPacketsAll[ matchDir ]
            if self.aclHwStatus_.sharedPolicerSupported and \
                  toggleQosRateCounterEnabled():
               cmapStatsNeeded = True
               for direction in self.spAppliedDirections:
                  clMapStatus = _getClMapStatus( clMapName, direction )
                  if clMapStatus and clMapStatus.policerStatus:
                     cmapStatsNeeded = False
               if cmapStatsNeeded:
                  dropRate = counter( clMapName, "dropRate" )
                  outRate = counter( clMapName, "outRate" )
                  eir = counter( clMapName, "eir" )

         qosParams = _createActionQos( actStrSetCos, actStrSetDscp,
                                       actStrSetDrop, actStrSetTc,
                                       actStrSetDropPrecedence,
                                       dropActionRestricted, dscpConfiguredAsName )
         return actPolice, qosParams, matchedPackets, dropRate, outRate, eir

      def _populateMatchedPackets( classMap, matchedPackets ):
         if not matchedPackets:
            classMap.matchedPackets = None
            return
         for direction in matchedPackets:
            pktCounter = \
                  PolicyMapModel.ClassMap.ClassMapMatchPacketCounterCollection()
            for intf in matchedPackets[ direction ]:
               pktCounter.interfaces[ intf ] = \
                     PolicyMapModel.ClassMap.ClassMapMatchPacketCounterCollection.\
                     ClassMapMatchPacketCounter()
               pktCounter.interfaces[ intf ].packets = \
                     matchedPackets[ direction ][ intf ]
            classMap.matchedPackets[ direction ] = pktCounter

      def _getClassMapStatistics( classMap, direction ):
         if direction in classMap.classMapStatistics:
            return classMap.classMapStatistics[ direction ]
         return PolicyMapModel.ClassMap.ClassMapStatistics()

      def _populateClassMapStatistics( classMap, dropRate, outRate, eir ):
         if dropRate is None or outRate is None or eir is None:
            classMap.classMapStatistics = None
            return
         for direction in dropRate:
            classMapStats = _getClassMapStatistics( classMap, direction )
            classMapStats.droppedBitsRate = dropRate[ direction ]
            classMapStats.conformedBitsRate = outRate[ direction ]
            classMapStats.exceededBitsRate = eir[ direction ]
            classMapStats.updateInterval = self.qosConfig_.loadInterval.val
            classMap.classMapStatistics[ direction ] = classMapStats

      def _populateIntfPacketCounters( classMap, outPackets, dropPackets ):
         if outPackets is None or dropPackets is None:
            classMap.intfPacketCounters = None
            return
         for direction in outPackets:
            intfPktCounter = PolicyMapModel.ClassMap.IntfPacketCounters()
            intfPktCounter.outPackets = outPackets[ direction ]
            intfPktCounter.dropPackets = dropPackets[ direction ]
            classMap.intfPacketCounters[ direction ] = intfPktCounter

      if self.mapType_ == coppMapType:
         classMap.coppActionPoliceSupported = \
                                 self.hwStatus_.coppActionPolicerSupported
         if self.hwStatus_.coppActionPolicerSupported:
            if ( classAction.policer or pMap.name == tacPMapNm.coppName ):
               classMap.police = _populateActionQosPolicer( clMapName, clMapObj )
            else:
               classMap.police = None
         else:
            classMap.shape, classMap.bandwidth, outPackets, dropPackets = \
               _populateActionCopp( clMapObj )
            _populateIntfPacketCounters( classMap, outPackets, dropPackets )
      else:
         classMap.police, classMap.qosParams, matchedPackets, dropRate, \
             outRate, eir = _populateActionQos( clMapObj )
         # Populate matchedPackets attr in the model if it is not a PDP policy
         if self.mapType_ != pdpMapType:
            _populateMatchedPackets( classMap, matchedPackets )
         if self.aclHwStatus_.sharedPolicerSupported and \
               toggleQosRateCounterEnabled() and self.mapType_ == qosMapType:
            _populateClassMapStatistics( classMap, dropRate, outRate, eir )
         # Populate count attr in the model if 'count' action is set.
         # Populate matchedPackets attr in the model if the PDP policy class has
         # 'count' action set.
         if classAction.count:
            classMap.count = True
            _populateMatchedPackets( classMap, matchedPackets )

class CMapModelContainer:
   def __init__( self, aclConfig, hwStatus, aclHwStatus, sliceHwStatus,
                 cmapType,
                 classMapAllModel ):
      self.qosAclConfig_ = aclConfig
      self.hwStatus_ = hwStatus
      self.aclHwStatus_ = aclHwStatus
      self.sliceHwStatus_ = sliceHwStatus
      self.mapType_ = cmapType
      self.classMapAllModel_ = classMapAllModel

   def populateAll( self ):
      classPrio = 1
      for name in sorted( self.qosAclConfig_.cmapType[ self.mapType_ ].cmap ):
         if self.populateClassMap( name, classPrio ):
            classPrio += 1

   def populateClassMap( self, clMap, classPrio ):
      if isinstance( clMap, str ):
         if clMap in self.qosAclConfig_.cmapType[ self.mapType_ ].cmap:
            clMap = self.qosAclConfig_.cmapType[ self.mapType_ ].cmap[ clMap ]
         else:
            return False

      classMap = ClassMapAllModel.ClassMapModel()
      if clMap.type == coppMapType:
         if clMap.cpType == tacClassMapCpType.cmapCpStatic:
            cmapCpStaticType = classMapCpStaticType( self.qosAclConfig_, clMap.name )
            coppStaticClass = coppStaticClassFromHwStatus( self.sliceHwStatus_ )
            if cmapCpStaticType not in coppStaticClass:
               return False
            if cmapCpStaticType == tacClassMapCpStaticType.cmapCpStaticDrop:
               return False

      assert clMap.matchCondition == 'matchConditionAny'
      classMap.matchCondition = clMap.matchCondition
      classMap.classPrio = classPrio
      if not isDefaultClass( self.mapType_, clMap.name ):
         for matchRule in clMap.match:
            match = ClassMapAllModel.ClassMapModel.Match()
            if matchRule in [ 'matchIpAccessGroup', 'matchIpv6AccessGroup',
                              'matchMacAccessGroup' ]:
               match.matchRule = matchRule
               classMap.match[ clMap.match[ matchRule ].strValue ] = match
            elif matchRule in [ "matchL2Params" ]:
               vlanValue = clMap.match[ matchRule ].vlanValue
               innerVlanValue = clMap.match[ matchRule ].innerVlanValue
               cosValue = clMap.match[ matchRule ].cosValue
               deiValue = clMap.match[ matchRule ].deiValue
               innerCosValue = clMap.match[ matchRule ].innerCosValue
               ( l2ParamsString, match ) = populateL2ParamsModel(
                 match, matchRule, vlanValue, innerVlanValue, cosValue, deiValue,
                 innerCosValue )
               classMap.match[ l2ParamsString ] = match
            elif matchRule in [ 'matchDscpEcn' ]:
               dscpValue = clMap.match[ matchRule ].dscpEcnValue
               ( value, match ) = populateDscpEcnModel( match, matchRule, dscpValue )
               classMap.match[ value ] = match
            elif matchRule in [ 'matchMplsTrafficClass' ]:
               match.matchRule = matchRule
               mplsTrafficClassVal = clMap.match[ matchRule ].mplsTrafficClassVal
               ids = set()
               for key in mplsTrafficClassVal.mplsTrafficClassColl:
                  ids.update( list( range( key.min, key.max + 1 ) ) )
               mplsTrafficClassString = \
                  MultiRangeRule.multiRangeToCanonicalString( list( ids ) )
               classMap.match[ mplsTrafficClassString ] = match

      self.classMapAllModel_.append( clMap.name, classMap )
      return True

class QosProfileModelContainer:
   def __init__( self, config, hwStatus, qosProfileAllModel=None ):
      self.config_ = config
      self.hwStatus_ = hwStatus
      self.qosProfileAllModel = qosProfileAllModel

   def populateAll( self ):
      for qosProfileName in sorted( self.config_ ):
         self.populateQosProfile( qosProfileName )

   def populateQosProfile( self, qosProfileName ):
      qosProfile = self._populateQosProfile( qosProfileName )
      if qosProfile:
         self.qosProfileAllModel.append( qosProfileName, qosProfile )

   def _populateQosProfile( self, qosProfileName ):
      qosProfile = QosProfileModel()
      profileName = qosProfileName
      if isinstance( qosProfileName, str ):
         if qosProfileName in self.config_:
            qosProfileName = self.config_[ qosProfileName ]
         else:
            return None

      qosProfileInConfig = self.config_[ profileName ].qosProfile

      qosProfile.trustMode = str( qosProfileInConfig.trustMode )

      if self.hwStatus_.defaultCosSupported:
         qosProfile.defaultCosSupported = True
         qosProfile.defaultCos = int( qosProfileInConfig.defaultCos )

      if self.hwStatus_.defaultDscpSupported:
         qosProfile.defaultDscpSupported = True
         qosProfile.defaultDscp = int( qosProfileInConfig.defaultDscp )

      if self.hwStatus_.perPortSchedulerCompensationSupported:
         qosProfile.schedulerCompensationSupported = True
         qosProfile.configuredSchedulerCompensation = \
            populateSchedulerCompensationModel(
               qosProfileInConfig.schedulerCompensation )

      qosProfile.configuredPortShapeRate = \
         populateShapeRateModel( qosProfileInConfig.shapeRate,
                                 reference='unknown' )
      qosProfile.configuredPortGuaranteedBw = \
         populateGuaranteedBwModel( qosProfileInConfig.guaranteedBw,
                                    reference='unknown' )

      if self.hwStatus_.dscpPreserveIpMplsEncapSupported:
         qosProfile.dscpPreserveIpMplsEncapMode = \
            qosProfileInConfig.dscpPreserveIpMplsEncapMode

      # Pfc
      pfcPortConfig = qosProfileInConfig.pfcPortConfig
      if pfcPortConfig:
         priorities = pfcPortConfig.priorities
         qosProfile.pfcWatchdogEnabled = pfcPortConfig.watchdogEnabled
         qosProfile.pfcWatchdogPortAction = pfcPortConfig.watchdogPortAction

         if pfcPortConfig.enabled:
            qosProfile.pfcEnabled = True
         for priority in range( 8 ):
            bit = 1 << priority
            if bit & priorities:
               pfcModel = PfcProfileModel()
               if bit & qosProfileInConfig.fabricPfcDlb:
                  pfcModel.dynamicLoadBalancing = True
               qosProfile.pfcPriorities[ priority ] = pfcModel
         if pfcPortConfig.portTimerConfig.usePerPortTimerValues:
            qosProfile.pfcWatchdogPortTimerValues = PfcWatchdogPortTimerModel()
            portTimerConfig = pfcPortConfig.portTimerConfig
            portTimerModel = qosProfile.pfcWatchdogPortTimerValues
            portTimerModel.timeout = portTimerConfig.portWatchdogTimeout
            portTimerModel.pollingInterval = \
                                       portTimerConfig.portWatchdogPollingInterval
            portTimerModel.recoveryTime = \
                           portTimerConfig.portWatchdogRecoveryCfg.recoveryTime
            portTimerModel.forcedRecovery = \
                           portTimerConfig.portWatchdogRecoveryCfg.forcedRecovery

      # Qos Service Policies
      servicePolicyConfig = qosProfileInConfig.servicePolicyConfig
      if servicePolicyConfig:
         key = servicePolicyConfig.key
         qosProfile.configuredServicePolicy = PolicyQosModel()
         qosProfile.configuredServicePolicy.name = key.pmapName
         qosProfile.configuredServicePolicy.directionString = key.direction
         qosProfile.configuredServicePolicy.status = \
             QosLib.tacPMapHwPrgmStatus.hwPrgmStatusSuccess

      qosProfileTxQueueConfig = self.config_[ profileName ].\
            qosProfile.txQueueConfig
      qosProfileTxQueueCounterConfig = self.config_[ profileName ].\
         qosProfile.ecnTxQueueCounterConfig
      qosProfile.numTxQueueSupported = self.hwStatus_.numTxQueueSupported

      for qoshwtxqid in range( qosProfile.numTxQueueSupported ):
         if self.hwStatus_.hwTxQueue.get( qoshwtxqid ) is None:
            continue
         txQueue = self.hwStatus_.hwTxQueue[ qoshwtxqid ].txQueue
         txQueueType = txQueue.type
         txQueueId = txQueue.id

         txQueueConfig = qosProfileTxQueueConfig.get( txQueue )
         ecnTxQueueCounterConfig = qosProfileTxQueueCounterConfig.get( txQueueId )

         txQueueProfileModel = TxQueueProfileModel()
         txQueueModel = TxQueueProfileModel.TxQueue()
         txQueueModel.ecnWeightSupported = self.hwStatus_.ecnWeightSupported
         txQueueModel.wredWeightSupported = self.hwStatus_.wredWeightSupported
         txQueueModel.nonEctWeightSupported = self.hwStatus_.nonEctWeightSupported
         txQueueModel.ecnMarkProbSupported = self.hwStatus_.ecnMarkProbSupported
         # Priority
         txQueuePriority = Tac.Type( 'Qos::TxQueuePriority' )

         if txQueueConfig is None:
            if ecnTxQueueCounterConfig:
               txQueueModel.ecnCounter = ecnTxQueueCounterConfig.counterEnable
               qosProfile.append( txQueueId, txQueueType, txQueueModel,
                                  txQueueProfileModel )
            continue

         # Scheduler profile
         if txQueueConfig.schedulerProfile != \
               tacTxQueueSchedulerProfileName.defaultProfile:
            txQueueModel.schedulerProfile = txQueueConfig.schedulerProfile

         # For subinterfaces, qos profile supports explicit mention of strict
         # priority, hence checking invalid priority is not enough to confirm there
         # is no priority set. Check for strict priority as well.
         if txQueueConfig.priority != txQueuePriority.priorityInvalid and \
            txQueueConfig.priority != txQueuePriority.priorityStrict:
            txQueueModel.noPriority = True

         # Wrr
         bandwidthPercent = Tac.Type( 'Qos::Percent' )
         if txQueueConfig.bandwidth != bandwidthPercent.invalid:
            txQueueModel.wrrBandwidthPercent = txQueueConfig.bandwidth

         # Shape
         txQueueModel.shapeRate = populateShapeRateModel( txQueueConfig.shapeRate,
                                                          reference='unknown' )

         # Guaranteed Bw
         invalidGuaranteedBw = Tac.Value( 'Qos::GuaranteedBw' )
         if txQueueConfig.guaranteedBw and \
               txQueueConfig.guaranteedBw != invalidGuaranteedBw:
            txQueueModel.guaranteedBw = GuaranteedBwModel()
            txQueueModel.guaranteedBw.bandwidth = txQueueConfig.guaranteedBw.bw
            txQueueModel.guaranteedBw.unit = QosLib.guaranteedBwUnitFromEnum(
                  txQueueConfig.guaranteedBw.unit )

         # Ecn
         if txQueueConfig.ecnConfig:
            txQueueModel.ecn = EcnParametersModel()
            txQueueModel.ecn.minThreshold = txQueueConfig.ecnConfig.minThd
            txQueueModel.ecn.maxThreshold = txQueueConfig.ecnConfig.maxThd
            txQueueModel.ecn.unit = txQueueConfig.ecnConfig.unit
            txQueueModel.ecn.maxDroprate = txQueueConfig.ecnConfig.maxDroprate
            txQueueModel.ecn.weight = txQueueConfig.ecnConfig.weight

         if txQueueConfig.bufferBasedDecnConfig:
            decnModel = DecnParametersModel()
            decnModel.offset = txQueueConfig.bufferBasedDecnConfig.offset
            decnModel.floor = txQueueConfig.bufferBasedDecnConfig.floor
            decnModel.unit = txQueueConfig.bufferBasedDecnConfig.unit
            txQueueModel.decn = decnModel

         if txQueueConfig.ecnDelayConfig:
            txQueueModel.ecnDelayConfig = EcnDelayTxQParametersModel()
            txQueueModel.ecnDelayConfig.maxThreshold = \
               txQueueConfig.ecnDelayConfig.maxThd
            txQueueModel.ecnDelayConfig.unit = txQueueConfig.ecnDelayConfig.unit

         if ecnTxQueueCounterConfig:
            txQueueModel.ecnCounter = ecnTxQueueCounterConfig.counterEnable

         if self.hwStatus_.ecnDelaySupported and txQueueConfig.delayEcnEnabled:
            txQueueModel.ecnDelay = EcnDelayConfigurationModel()
            if txQueueConfig.ecnDelayThreshold != tacEcnDelayThreshold:
               txQueueModel.ecnDelay.source = "local"
               txQueueModel.ecnDelay.configuredThreshold = \
                     txQueueConfig.ecnDelayThreshold.threshold / 1000.0
            else:
               txQueueModel.ecnDelay.source = "global"

         # Wred
         if txQueueConfig.wredConfig:
            txQueueModel.wred = WredParametersModel()
            txQueueModel.wred.minThreshold = txQueueConfig.wredConfig.minThd
            txQueueModel.wred.maxThreshold = txQueueConfig.wredConfig.maxThd
            txQueueModel.wred.unit = txQueueConfig.wredConfig.unit
            txQueueModel.wred.maxDroprate = txQueueConfig.wredConfig.maxDroprate
            txQueueModel.wred.weight = txQueueConfig.wredConfig.weight

         # Dp Wred
         if txQueueConfig.dpConfig:
            for dp in txQueueConfig.dpConfig.dpWredConfig:
               dpWredConfig = txQueueConfig.dpConfig.dpWredConfig[ dp ]
               dpWred = WredParametersModel()
               dpWred.minThreshold = dpWredConfig.minThd
               dpWred.maxThreshold = dpWredConfig.maxThd
               dpWred.unit = dpWredConfig.unit
               dpWred.maxDroprate = dpWredConfig.maxDroprate
               dpWred.weight = dpWredConfig.weight
               dpWred.dp = dp
               txQueueModel.dpWred[ dp ] = dpWred

         # Queue Weight
         QueueWeight = Tac.Value( 'Qos::QueueWeight' )
         if txQueueConfig.weightConfig != QueueWeight.defaultWeight:
            txQueueModel.weightConfig = txQueueConfig.weightConfig

         # DP Thresholds
         for dp in txQueueConfig.dropThresholds:
            dpThreshold = DropPrecedenceThresholdsModel()
            dpThreshold.percent = txQueueConfig.dropThresholds[ dp ]
            txQueueModel.dpThresholds[ dp ] = dpThreshold

         qosProfile.append( txQueueId, txQueueType, txQueueModel,
                            txQueueProfileModel )
         # NonEct
         if txQueueConfig.nonEctConfig:
            txQueueModel.nonEct = EcnParametersModel()
            txQueueModel.nonEct.minThreshold = txQueueConfig.nonEctConfig.minThd
            txQueueModel.nonEct.maxThreshold = txQueueConfig.nonEctConfig.maxThd
            txQueueModel.nonEct.unit = txQueueConfig.nonEctConfig.unit
            txQueueModel.nonEct.weight = txQueueConfig.nonEctConfig.weight

         txQueueModel.latencyThreshold = LatencyThresholdModel()
         txQueueModel.latencyThreshold.threshold = \
               txQueueConfig.latencyThreshold.configThreshold()
         txQueueModel.latencyThreshold.unit = \
               txQueueConfig.latencyThreshold.configUnit

         qosProfile.append( txQueueId, txQueueType, txQueueModel,
                            txQueueProfileModel )
      return qosProfile

class PolicingModelContainer:
   def __init__( self, qosInputConfig, qosConfig, qosStatus,
                 intfPolicingHwStatus, qosSliceHwStatus, lagInputConfig,
                 modePolicingModel=None ):
      self.qosConfig = qosConfig
      self.qosStatus = qosStatus
      self.intfPolicingHwStatus = intfPolicingHwStatus
      self.modePolicingModel = modePolicingModel
      self.qosSliceHwStatus = qosSliceHwStatus
      self.lagInputConfig = lagInputConfig

   def getConfiguredLagMembers( self, lagIntf ):
      if Tac.Type( "Arnet::SubIntfId" ).isSubIntfId( lagIntf ):
         lagIntf = Tac.Type( "Arnet::SubIntfId" ).parentIntfId( lagIntf )
      lagMem = []
      for phyIntf, phyIntfVal in self.lagInputConfig.phyIntf.items():
         if phyIntfVal.lag and phyIntfVal.lag.intfId == lagIntf:
            lagMem.append( phyIntf )
      return lagMem

   # pylint: disable-next=inconsistent-return-statements
   def populateAllPolicerProfiles( self, profileOutput, policerOutput,
                                   pktSizeAdjProfileOutput ):
      if not self.qosStatus.policingStatus:
         return self.modePolicingModel
      if profileOutput and policerOutput and pktSizeAdjProfileOutput:
         self.modePolicingModel.populateAllOutput( True )
      if profileOutput:
         profileList = sorted( self.qosStatus.policingStatus.profile )
         for profile in profileList:
            self.populateProfile( profile )
      if policerOutput:
         policerList = sorted( self.qosStatus.policingStatus.policerInstance )
         for policerName in policerList:
            self.populatePolicerInstance( policerName )
      if pktSizeAdjProfileOutput:
         profileList = sorted(
            self.qosStatus.policingStatus.policerPktSizeAdjProfile )
         for profile in profileList:
            self.populatePolicerPktSizeAdjProfile( profile )

   def populatePolicerProfileIntfAssociations( self, intfPair, policerProfile ):
      for policingHwStatus in self.intfPolicingHwStatus.values():
         if intfPair in policingHwStatus.intfPolicerHwStatus:
            intfPairStatus = policingHwStatus.intfPolicerHwStatus[ intfPair ].status
            if intfPair.lagIntf:
               if intfPair.lagIntf in policerProfile.configuredIngressIntfs:
                  confIntfStatus = policerProfile.configuredIngressIntfs[
                     intfPair.lagIntf ]
                  if not confIntfStatus:
                     return
                  elif confIntfStatus == intfPairStatus:
                     return
               policerProfile.configuredIngressIntfs[
                  intfPair.lagIntf ] = intfPairStatus
               return
            else:
               policerProfile.configuredIngressIntfs[
                  intfPair.intfId ] = intfPairStatus
               return
      if intfPair.lagIntf:
         policerProfile.configuredIngressIntfs[ intfPair.lagIntf ] = False
      else:
         policerProfile.configuredIngressIntfs[ intfPair.intfId ] = False

   def populateProfile( self, profileName ):
      profile = ProfileModel()
      profile.profileName = profileName
      if profileName in self.qosStatus.policingStatus.profile:
         profileConfig = self.qosStatus.policingStatus.profile[
            profileName ].profileConfig
         profile.police = PoliceModel( cir=profileConfig.cir,
                                 cirUnit=rateUnitFromEnum( profileConfig.cirUnit ),
                                 bc=profileConfig.bc,
                                 bcUnit=burstUnitFromEnum( profileConfig.bcUnit ),
                                 pir=profileConfig.pir,
                                 pirUnit=rateUnitFromEnum( profileConfig.pirUnit ),
                                 be=profileConfig.be,
                                 beUnit=burstUnitFromEnum( profileConfig.beUnit ) )
         if profileName in self.qosStatus.qosPolicingMap.profileToPolicerInstanceMap:
            profile.policerAssociations = \
               list( self.qosStatus.qosPolicingMap.profileToPolicerInstanceMap[
                  profileName ].policerInstanceMap )
         intfPairList = []
         if profileName in self.qosStatus.qosPolicingMap.profileToIntfMap:
            intfPairList = list( self.qosStatus.qosPolicingMap.profileToIntfMap[
               profileName ].intfMap )
         profile.configuredIngressIntfs = {}
         for intfPair in intfPairList:
            self.populatePolicerProfileIntfAssociations( intfPair, profile )
      self.modePolicingModel.populateProfile( profileName, profile )

   def checkIntfPolicerPktSizeAdjHwProfile( self, intf, profile ):
      for sliceHwStatus in self.qosSliceHwStatus.values():
         if sliceHwStatus.intfToPolicerPktSizeAdjProfile.get( intf ) ==\
            profile.profileName:
            return True
      return False

   def populatePolicePktSizeAdjProfileIntfAssociations( self, intf, profile ):
      if isLagPort( intf ):
         lagMem = self.getConfiguredLagMembers( intf )
         activeIntfs = sum( self.checkIntfPolicerPktSizeAdjHwProfile( member,
                            profile ) for member in lagMem )
         if activeIntfs == len( lagMem ):
            state = 'active'
         elif activeIntfs == 0:
            state = 'inactive'
         else:
            state = 'partial'
      elif self.checkIntfPolicerPktSizeAdjHwProfile( intf, profile ):
         state = 'active'
      else:
         state = 'inactive'
      profile.configuredIngressIntfs[ intf ] =\
            HwProgrammingEnumModel( hwState=state )

   def populatePolicerPktSizeAdjProfile( self, profileName ):
      profile = PolicerPktSizeAdjProfileModel()
      profile.profileName = profileName
      if profileName in self.qosStatus.policingStatus.policerPktSizeAdjProfile:
         pktSizeAdj = self.qosStatus.policingStatus.\
            policerPktSizeAdjProfile[ profileName ].policerPktSizeAdj
         profile.pktSizeAdj = pktSizeAdj
         intfMap = []
         if profileName in self.qosStatus.qosPolicingMap.\
            policerPktSizeAdjProfileToIntfMap:
            intfMap = self.qosStatus.qosPolicingMap.\
               policerPktSizeAdjProfileToIntfMap[ profileName ].intfMap
         profile.configuredIngressIntfs = {}
         for intf in intfMap:
            self.populatePolicePktSizeAdjProfileIntfAssociations( intf, profile )
         self.modePolicingModel.populatePktSizeAdjProfile( profileName, profile )

   def populatePolicerInstance( self, policerName ):
      policer = PolicerInstanceModel()
      policer.policerName = policerName
      if policerName in self.qosStatus.policingStatus.policerInstance:
         policer.profileName = self.qosStatus.policingStatus.policerInstance[
            policerName ].profileName
         intfPairList = []
         if policerName in self.qosStatus.qosPolicingMap.policerInstanceToIntfMap:
            intfPairList = list(
               self.qosStatus.qosPolicingMap.policerInstanceToIntfMap[
                  policerName ].intfMap )
         policer.configuredIngressIntfs = {}
         for intfPair in intfPairList:
            self.populatePolicerProfileIntfAssociations( intfPair, policer )
      self.modePolicingModel.populatePolicerInstance( policerName, policer )

class IntfPolicerModelContainer:
   def __init__( self, qosAclHwStatus, qosConfig, qosStatus,
                 intfPolicingHwStatus, intfPolicerAllModel ):
      self.qosAclHwStatus = qosAclHwStatus
      self.qosConfig = qosConfig
      self.qosStatus = qosStatus
      self.intfPolicingHwStatus = intfPolicingHwStatus
      self.intfPolicerAllModel = intfPolicerAllModel

   def setIntfPolicerAllModelAttr( self, detail, policerDir ):
      self.intfPolicerAllModel.detailIs( detail )
      self.intfPolicerAllModel.policerDirIs( policerDir )
      self.intfPolicerAllModel.ingPolicerSupportedIs(
         self.qosAclHwStatus.interfacePolicingGroupPolicerSupported or
         self.qosAclHwStatus.interfacePolicingDedicatedPolicerSupported )
      self.intfPolicerAllModel.egrPolicerSupportedIs(
         self.qosAclHwStatus.interfaceEgressPolicingDedicatedPolicerSupported )
      self.intfPolicerAllModel.cpuPolicerSupportedIs(
         self.qosAclHwStatus.interfaceControlPlanePolicingDedicatedPolicerSupported )

   # pylint: disable-next=inconsistent-return-statements
   def populateAllIntfPolicers( self, detail, policerDir ):
      if not self.qosStatus.policingStatus:
         return self.intfPolicerAllModel
      self.setIntfPolicerAllModelAttr( detail, policerDir )

      if policerDir == 'input' or policerDir == 'all':
         intfList = list( self.qosStatus.policingStatus.intfPolicer )
         for intfPair in intfList:
            intf = intfPair.lagIntf if intfPair.lagIntf else intfPair.intfId
            self.populateIntfPolicers( intf, 'input' )

      if policerDir == 'output' or policerDir == 'all':
         intfEgrList = list( self.qosStatus.policingStatus.intfPolicerOutput )
         for intfPair in intfEgrList:
            intf = intfPair.lagIntf if intfPair.lagIntf else intfPair.intfId
            self.populateIntfPolicers( intf, 'output' )

      if policerDir == 'cpu' or policerDir == 'all':
         intfCpuList = list( self.qosStatus.policingStatus.intfPolicerCpu )
         for intfPair in intfCpuList:
            intf = intfPair.lagIntf if intfPair.lagIntf else intfPair.intfId
            self.populateIntfPolicers( intf, 'cpu' )

   # pylint: disable-next=inconsistent-return-statements
   def populateIntfRangePolicers( self, intfs, detail, policerDir ):
      if not self.qosStatus.policingStatus:
         return self.intfPolicerAllModel
      self.setIntfPolicerAllModelAttr( detail, policerDir )

      for intf in intfs:
         if policerDir == 'input' or policerDir == 'all':
            self.populateIntfPolicers( intf.name, 'input' )
         if policerDir == 'output' or policerDir == 'all':
            self.populateIntfPolicers( intf.name, 'output' )
         if policerDir == 'cpu' or policerDir == 'all':
            self.populateIntfPolicers( intf.name, 'cpu' )

   def populatePortChannelStatus( self, intfPairStatus, intfPolicerInfo ):
      if intfPolicerInfo.isConfigured is None:
         hwState_ = 'active' if intfPairStatus else 'inactive'
         intfPolicerInfo.isConfigured = HwProgrammingEnumModel( hwState=hwState_ )
      elif intfPolicerInfo.isConfigured.hwState == 'inactive':
         if intfPairStatus:
            intfPolicerInfo.isConfigured.hwState = 'partial'
         else:
            return
      elif intfPolicerInfo.isConfigured.hwState == 'active':
         if intfPairStatus:
            return
         else:
            intfPolicerInfo.isConfigured.hwState = 'partial'
      else:
         return

   def populateMemberAndStatus( self, intfPairKey, policingHwStatus,
                                intfPolicerInfo, policerDir ):
      if policerDir == 'cpu':
         intfPairStatus = \
            policingHwStatus.intfPolicerCpuHwStatus[ intfPairKey ].status
      elif policerDir == 'output':
         intfPairStatus = \
            policingHwStatus.intfPolicerOutputHwStatus[ intfPairKey ].status
      else:
         intfPairStatus = \
            policingHwStatus.intfPolicerHwStatus[ intfPairKey ].status

      if intfPairStatus:
         intfPolicerInfo.portChannelMembers[
            intfPairKey.intfId ] = HwProgrammingEnumModel( hwState='active' )
      else:
         intfPolicerInfo.portChannelMembers[
            intfPairKey.intfId ] = HwProgrammingEnumModel( hwState='inactive' )
      self.populatePortChannelStatus( intfPairStatus, intfPolicerInfo )

   def populatePortChannelMembers( self, intf, intfPolicerInfo, policerDir ):
      if policerDir == 'cpu':
         intfPolicerColl = self.qosStatus.policingStatus.intfPolicerCpu
      elif policerDir == 'output':
         intfPolicerColl = self.qosStatus.policingStatus.intfPolicerOutput
      else:
         intfPolicerColl = self.qosStatus.policingStatus.intfPolicer

      for intfPairKey in intfPolicerColl:
         if intf == intfPairKey.lagIntf:
            for policingHwStatus in self.intfPolicingHwStatus.values():
               if policerDir == 'input' and \
                  intfPairKey in policingHwStatus.intfPolicerHwStatus:
                  self.populateMemberAndStatus( intfPairKey, policingHwStatus,
                                                intfPolicerInfo, policerDir )
                  break
               if policerDir == 'output' and \
                  intfPairKey in policingHwStatus.intfPolicerOutputHwStatus:
                  self.populateMemberAndStatus( intfPairKey, policingHwStatus,
                                                intfPolicerInfo, policerDir )
                  break
               if policerDir == 'cpu' and \
                  intfPairKey in policingHwStatus.intfPolicerCpuHwStatus:
                  self.populateMemberAndStatus( intfPairKey, policingHwStatus,
                                                intfPolicerInfo, policerDir )
                  break
            else:
               intfPolicerInfo.portChannelMembers[
                  intfPairKey.intfId ] = HwProgrammingEnumModel(
                     hwState='inactive' )
               self.populatePortChannelStatus( intfPairStatus=False,
                                               intfPolicerInfo=intfPolicerInfo )

   def populateInterfaceStatus( self, intfPair, intfPolicerInfo, policerDir ):
      for policingHwStatus in self.intfPolicingHwStatus.values():
         if policerDir == 'cpu':
            intfPolicerHwStatus = policingHwStatus.intfPolicerCpuHwStatus
         elif policerDir == 'output':
            intfPolicerHwStatus = policingHwStatus.intfPolicerOutputHwStatus
         else:
            intfPolicerHwStatus = policingHwStatus.intfPolicerHwStatus

         if intfPair in intfPolicerHwStatus:
            intfPairStatus = intfPolicerHwStatus[ intfPair ].status
            if intfPairStatus:
               intfPolicerInfo.isConfigured = HwProgrammingEnumModel(
                  hwState='active' )
            else:
               intfPolicerInfo.isConfigured = HwProgrammingEnumModel(
                  hwState='inactive' )
            break
      else:
         intfPolicerInfo.isConfigured = HwProgrammingEnumModel( hwState='inactive' )

   def populateIntfPolicers( self, intf, policerDir ):
      intfPolicerInfo = InterfacePolicerModel()

      if policerDir == 'cpu':
         intfPolicerColl = self.qosStatus.policingStatus.intfPolicerCpu
      elif policerDir == 'output':
         intfPolicerColl = self.qosStatus.policingStatus.intfPolicerOutput
      else:
         intfPolicerColl = self.qosStatus.policingStatus.intfPolicer

      for interface in intfPolicerColl:
         intfStatusInfo = intfPolicerColl[ interface ]
         intfStatusPairKey = intfStatusInfo.intfPairKey
         if intf == intfStatusPairKey.intfId or intf == intfStatusPairKey.lagIntf:
            if tacPolicerType.dedicatedPolicer in intfStatusInfo.policerName:
               intfPolicerInfo.profileName = intfStatusInfo.policerName[
                  tacPolicerType.dedicatedPolicer ]
            if tacPolicerType.policerInstance in intfStatusInfo.policerName:
               intfPolicerInfo.policerName = intfStatusInfo.policerName[
                  tacPolicerType.policerInstance ]
            if not isLagPort( intf ):
               self.populateInterfaceStatus(
                  intfStatusPairKey, intfPolicerInfo, policerDir )
            else:
               intfPolicerInfo.portChannelMembers = {}
               self.populatePortChannelMembers( intf, intfPolicerInfo, policerDir )
            self.intfPolicerAllModel.populateIntfPolicerInfo(
               intf, intfPolicerInfo, policerDir )
            break


class PolicingCountersModelContainer:
   def __init__( self, qosConfig, qosStatus, aclHwStatus, intfPolicingHwStatus,
                 policingCounterAllModel ):
      self.qosConfig = qosConfig
      self.qosStatus = qosStatus
      self.aclHwStatus = aclHwStatus
      self.intfPolicingHwStatus = intfPolicingHwStatus
      self.policingCounterAllModel = policingCounterAllModel

   # pylint: disable-next=inconsistent-return-statements
   def populateIntfPolicingCountersForIntfs( self, intfs, policerType ):
      """
      Populate the counter information for a set of interface. If the interface is
      a port-channel, then get the intfPolicerInfo status for all the members.
      If there are no interfaces specified, then populate for all interfaces.
      """
      if not self.qosStatus.policingStatus:
         return self.policingCounterAllModel

      def populateIntfPolicingCountersByDirection( intfPolicerColl, policerType ):
         intfsToDisplay = [ intf.name for intf in intfs ]
         portChannelMembers = {}

         for intfPairKey, intfPolicerInfo in intfPolicerColl.items():
            if intfPairKey.lagIntf in intfsToDisplay:
               intf = intfPairKey.lagIntf
               portChannelMembers.setdefault( intf, [] )
               portChannelMembers[ intf ].append( intfPolicerInfo )
               # Do not pop intf from intfsToDisplay so that we can find all members.

            else:
               if intfPairKey.intfId in intfsToDisplay and not intfPairKey.lagIntf:
                  intf = intfPairKey.intfId
                  # For FPP-subintf, populate the model directly.
                  intfsToDisplay.remove( intf )
                  intfCounterInfoModel = self.populateIntfPolicingCounterModel(
                     intfPolicerInfo )
                  self.policingCounterAllModel.populateIntfCounterInfo(
                     intf, intfCounterInfoModel, policerType )

            if not intfsToDisplay:
               # No Port-Channel intfs to show.
               return

         # Iterate over remaining Port-Channel and populate the counter info.
         for intf, intfPolicerStatus in portChannelMembers.items():
            intfCounterInfoModel = self.populateIntfPolicingCounterModel(
               intfPolicerStatus )
            self.policingCounterAllModel.populateIntfCounterInfo(
               intf, intfCounterInfoModel, policerType )

      # We only populate counters for policing directions where counter reporting for
      # that specific policer direction is supported

      if ( ( policerType == 'input' or policerType == 'all' ) and
           self.aclHwStatus.interfacePolicingCountersSupported ):
         intfPolicerColl = self.qosStatus.policingStatus.intfPolicer
         populateIntfPolicingCountersByDirection( intfPolicerColl, 'input' )

      if ( ( policerType == 'output' or policerType == 'all' ) and
           self.aclHwStatus.interfaceEgressPolicingCountersSupported ):
         intfPolicerColl = self.qosStatus.policingStatus.intfPolicerOutput
         populateIntfPolicingCountersByDirection( intfPolicerColl, 'output' )

      if ( ( policerType == 'cpu' or policerType == 'all' ) and
           self.aclHwStatus.interfaceControlPlanePolicingCountersSupported ):
         intfPolicerColl = self.qosStatus.policingStatus.intfPolicerCpu
         populateIntfPolicingCountersByDirection( intfPolicerColl, 'cpu' )

   def populateIntfPolicingCounterModel( self, intfPolicerInfoStatuses ):
      """
      Given a list of intfPolicerStatus objects, populate the
      InterfacePolicingCounterModel. If the interface is a lagIntf,
      then either sum up the counter values if per-member counter
      is supported, or just take the counter value from one of the members.

      intfPolicerInfoStatuses is a single intfPolicerInfo object in case of
      FPP subintf, and a list of intfPolicerInfo objects in case of lag subintf.
      """
      if not isinstance( intfPolicerInfoStatuses, list ):
         intfPolicerInfoStatuses = [ intfPolicerInfoStatuses ]

      intfPolicerInfoStatus = intfPolicerInfoStatuses[ 0 ]

      intfCounterInfo = InterfacePolicingCounterModel()
      intfCounterInfo.profileName = intfPolicerInfoStatus.policerName.get(
         tacPolicerType.dedicatedPolicer )
      intfCounterInfo.policerName = intfPolicerInfoStatus.policerName.get(
         tacPolicerType.policerInstance )

      # Get the policerMode from the policer's policer modes.
      # If either of the 2 policers is in TrTCM mode, then we show the output in
      # TrTCM mode.
      intfCounterInfo.mode = "committed"
      if "trtcm" in [ self.getPolicerProfileMode( intfCounterInfo.profileName ),
                      self.getPolicerMode( intfCounterInfo.policerName ) ]:
         intfCounterInfo.mode = "trtcm"

      # If the interface is a lag, then either sum up the counter info, or
      # just populate the counter info for one of its member.
      intfPairKey = intfPolicerInfoStatus.intfPairKey
      if isLagPort( intfPairKey.lagIntf ):
         if self.aclHwStatus.interfacePolicingCountersPerLagMemberCounterInfo:
            # Then sum up the counterInfo and populate the counter info.
            counterModel = IntfPoliceCounters()
            for intfStatus in intfPolicerInfoStatuses:
               counterStatus = intfStatus.counter
               counterModelForIntf = self.populateIntfPoliceCountersModel(
                  counterStatus )
               self.sumCounterModel( counterModel, counterModelForIntf )

            intfCounterInfo.counters = counterModel
         else:
            # Get counter data from the active member and populate counter info.
            activeIntfPairKey, status = self.getActiveIntfPairKey( intfPairKey )
            if not status:
               intfCounterInfo.counters = self.populateIntfPoliceCountersModel(
                  counterStatus=None )
            else:
               for intfStatus in intfPolicerInfoStatuses:
                  intfKey = intfStatus.intfPairKey
                  if( intfKey.intfId == activeIntfPairKey.intfId and
                      intfKey.lagIntf == activeIntfPairKey.lagIntf ):
                     counterStatus = intfStatus.counter
                     counterModel = self.populateIntfPoliceCountersModel(
                        counterStatus )
                     intfCounterInfo.counters = counterModel
                     break
      else:
         counterStatus = intfPolicerInfoStatus.counter
         counterModel = self.populateIntfPoliceCountersModel( counterStatus )
         intfCounterInfo.counters = counterModel

      return intfCounterInfo

   def sumCounterModel( self, original, tmp ):
      original.conformedPackets = original.conformedPackets + tmp.conformedPackets
      original.conformedBytes = original.conformedBytes + tmp.conformedBytes
      original.yellowPackets = original.yellowPackets + tmp.yellowPackets
      original.yellowBytes = original.yellowBytes + tmp.yellowBytes
      original.exceededPackets = original.exceededPackets + tmp.exceededPackets
      original.exceededBytes = original.exceededBytes + tmp.exceededBytes

   def getPolicerMode( self, policerName ):
      if policerName is None:
         return ""
      profile = self.qosStatus.policingStatus.policerInstance[
         policerName ].profileName
      return self.getPolicerProfileMode( profile )

   def getPolicerProfileMode( self, profileName ):
      if profileName is None:
         return ""

      profileConfig = self.qosStatus.policingStatus.profile[
         profileName ].profileConfig
      if profileConfig.pir:
         return "trtcm"

      return "committed"

   def getActiveIntfPairKey( self, intfPairKey ):
      '''
      This method returns intfPairKey for an active member whose lagIntf is equal to
      intfPairKey's lagIntf. This method is used to display counter values for lag
      subIntfs, with values derived from the active member's counter values.
      '''
      assert intfPairKey.lagIntf
      for policingHwStatus in self.intfPolicingHwStatus.values():
         for intfPair in policingHwStatus.intfPolicerHwStatus:
            if intfPair.lagIntf == intfPairKey.lagIntf:
               if policingHwStatus.intfPolicerHwStatus[ intfPair ].status:
                  return intfPair, True
      return intfPairKey, False

   def populateIntfPoliceCountersModel( self, counterStatus ):
      c = counterStatus
      if not c:
         return IntfPoliceCounters(
            conformedPackets=0, conformedBytes=0, yellowPackets=0, yellowBytes=0,
            exceededPackets=0, exceededBytes=0, droppedBitsRate=0.0,
            conformedBitsRate=0.0, exceededBitsRate=0.0 )

      return IntfPoliceCounters(
         conformedPackets=c.pktInCount, conformedBytes=c.byteInCount,
         yellowPackets=c.yellowPktInCount, yellowBytes=c.yellowByteInCount,
         exceededPackets=c.pktDropCount, exceededBytes=c.byteDropCount,
         droppedBitsRate=c.dropRate, conformedBitsRate=c.outRate,
         exceededBitsRate=c.eir )
