#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AirStreamLib
import Cell
import GnmiSetCliSession
import LazyMount
import re
import Tac
import Tracing

from QosTypes import tacClassMapType, tacMatchOption, tacDscp

t0 = Tracing.Handle( "OpenConfigQos" ).trace0

tacAclType = Tac.Type( 'Acl::AclType' )
tacAclApiType = Tac.Type( 'AclApi::ApiType' )
tacAclApiState = Tac.Type( 'AclApi::ApiState' )
tacAclAction = Tac.Type( 'Acl::Action' )
tacClassifierType = Tac.Type( "QosOc::Classifier::Type" )

def registerClassifiersHandler( entMan ):
   # pre-commit handlers
   # - Classifier config
   class ToNativeClassifierConfigSyncher( GnmiSetCliSession.PreCommitHandler ):
      externalPathList = [ 'qos/openconfig/config/classifiers' ]
      nativePathList = [ 'qos/acl/input/cli', 'acl/config/cli' ]

      @classmethod
      def run( cls, sessionName ):
         t0( 'running ToNativeClassifierConfigSyncher' )
         classifierConfig = AirStreamLib.getSessionEntity(
            entMan, sessionName, 'qos/openconfig/config/classifiers' )
         qosAclCliConfig = AirStreamLib.getSessionEntity(
            entMan, sessionName, 'qos/acl/input/cli' )
         aclCliConfig = AirStreamLib.getSessionEntity(
            entMan, sessionName, 'acl/config/cli' )
         statusDpDir = LazyMount.mount( entMan,
                                        f"cell/{Cell.cellId()}/acl/status/dp",
                                        "Tac::Dir", "ri" )

         classifierMetaData = Tac.newInstance( "QosOc::ClassifierMetaData" )

         def followsClassifierNaming( objName ):
            result = re.search( r"^__yang_\[([^\]]+)\]_\[([^\]]+)\]$", objName )
            if result is None:
               result = re.search( r"^__yang_\[([^\]]+)\]_\[\]$", objName )
            return result

         # Collections to maintain reverse mappings
         aclNameToAclType = {}
         cmapNameToMatchOption = {}

         # Populate the reverse maps for ACL config
         for aclType in [ tacAclType.ip, tacAclType.ipv6 ]:
            for aclName in aclCliConfig.config[ aclType ].acl:
               if followsClassifierNaming( aclName ):
                  aclNameToAclType[ aclName ] = aclType

         # Populate the reverse maps for Class-map config
         cmapConfig = qosAclCliConfig.cmapType[ tacClassMapType.mapQos ].cmap
         for cmapName in cmapConfig:
            if followsClassifierNaming( cmapName ):
               cmapNameToMatchOption[ cmapName ] = cmapConfig[
                  cmapName ].match.keys()[ 0 ]

         def getAclApi( aclName, aclType, apiType ):
            aclApiSm = Tac.Type( 'AclApi::ApiSm' )(
               aclName, aclType, False, apiType, aclCliConfig, None, None, None,
               None, statusDpDir.force(), None )
            aclApiSm.inConfigSession = True
            return aclApiSm

         def getMatchOptionFromClassifierType( classifierType ):
            matchOption = None
            if classifierType == tacClassifierType.IPV4:
               matchOption = tacMatchOption.matchIpAccessGroup
            elif classifierType == tacClassifierType.IPV6:
               matchOption = tacMatchOption.matchIpv6AccessGroup
            else:
               raise AirStreamLib.ToNativeSyncherError(
                  sessionName, 'ToNativeClassifierConfigSyncher',
                  'Unsupported classifier type' )
            return matchOption

         def getAclTypeFromClassifierType( classifierType ):
            aclType = None
            if classifierType == tacClassifierType.IPV4:
               aclType = tacAclType.ip
            elif classifierType == tacClassifierType.IPV6:
               aclType = tacAclType.ipv6
            else:
               raise AirStreamLib.ToNativeSyncherError(
                  sessionName, 'ToNativeClassifierConfigSyncher',
                  'Unsupported classifier type' )
            return aclType

         def ipRuleConfig( dscp=None ):
            ipFilter = Tac.Value( 'Acl::IpFilter' )
            ruleConfig = Tac.Value( 'Acl::IpRuleConfig' )
            ruleConfig.action = tacAclAction.permit
            addrAny = Tac.Value( 'Arnet::IpAddrWithFullMask',
                                 Tac.Value( 'Arnet::IpAddr', 0 ), 0 )
            ipFilter.source = addrAny
            ipFilter.destination = addrAny
            if dscp is not None and dscp != tacDscp.invalid:
               ipFilter.dscp = dscp
               ipFilter.dscpMask = ipFilter.dscpMaskAll
               ipFilter.matchDscp = True
            ruleConfig.filter = ipFilter
            return ruleConfig

         def ip6RuleConfig( dscp=None ):
            ip6Filter = Tac.Value( 'Acl::Ip6Filter' )
            ruleConfig = Tac.Value( 'Acl::Ip6RuleConfig' )
            ruleConfig.action = tacAclAction.permit
            ip6AddrAny = Tac.Value( 'Arnet::Ip6Addr', 0, 0, 0, 0 )
            addrAny = Tac.Value( 'Arnet::Ip6AddrWithFullMask', ip6AddrAny,
                                 ip6AddrAny )
            ip6Filter.sourceFullMask = addrAny
            ip6Filter.destinationFullMask = addrAny
            if dscp is not None and dscp != tacDscp.invalid:
               ip6Filter.tc = dscp << 2
               ip6Filter.tcMask = ip6Filter.dscpMaskAll << 2
            ruleConfig.filter = ip6Filter
            return ruleConfig

         def getAclRuleConfigList( termCondition, aclType ):
            ruleConfigList = []
            ruleConfigFn = None
            if termCondition is None or ( termCondition.config.dscp is None
                                          and not termCondition.config.dscpSet ):
               # The second check is for handling the `ip/ipv6 any any` case.
               # Commenting it out now, opened BUG783704 for the same.
               return ruleConfigList
            if aclType == tacAclType.ip:
               ruleConfigFn = ipRuleConfig
            elif aclType == tacAclType.ipv6:
               ruleConfigFn = ip6RuleConfig
            if termCondition.config.dscp is not None:
               ruleConfig = ruleConfigFn( termCondition.config.dscp )
               ruleConfigList.append( ruleConfig )
            for dscp in termCondition.config.dscpSet.values():
               ruleConfig = ruleConfigFn( dscp )
               ruleConfigList.append( ruleConfig )
            return ruleConfigList

         def deleteAclConfig( aclName, aclApiSm ):
            apiState = aclApiSm.remove()
            if apiState in ( tacAclApiState.succeeded,
                             tacAclApiState.errAclDoesNotExist ):
               t0( f"ACL: {aclName} has been deleted successfully" )
            else:
               raise AirStreamLib.ToNativeSyncherError(
                  sessionName, 'ToNativeClassifierConfigSyncher',
                  f'Error in ACL {aclName} deletion' )

         def createAclConfig( aclName, aclApiSm, ruleConfigList, aclType ):
            for ruleConfig in ruleConfigList:
               if aclType == tacAclType.ip:
                  aclApiSm.addIpRule( 0, ruleConfig )
               elif aclType == tacAclType.ipv6:
                  aclApiSm.addIp6Rule( 0, ruleConfig )
            apiState = aclApiSm.commit()
            if apiState == tacAclApiState.succeeded:
               t0( f"ACL: {aclName} has been created" )
            else:
               raise AirStreamLib.ToNativeSyncherError(
                  sessionName, 'ToNativeClassifierConfigSyncher',
                  f'Error in ACL {aclName} creation' )

         def configureAcl( aclName, classifierType, termConditions ):
            aclType = getAclTypeFromClassifierType( classifierType )
            deleteAclApiSm = getAclApi( aclName, aclType, tacAclApiType.remove )
            # To avoid inconsistencies in rule ordering, if ACL of same name exists,
            # we first delete the ACL then create the ACL again.
            deleteAclConfig( aclName, deleteAclApiSm )
            if termConditions is None:
               # The conditions container under the term is not present. Do not
               # create any ACL config
               return
            commitAclApiSm = getAclApi( aclName, aclType, tacAclApiType.commit )
            if aclType == tacAclType.ip:
               conditions = termConditions.ipv4
            else:
               conditions = termConditions.ipv6
            ruleConfigList = getAclRuleConfigList( conditions, aclType )
            createAclConfig( aclName, commitAclApiSm, ruleConfigList, aclType )

         def configureClassMap( className, classifierType=None ):
            cmapType = tacClassMapType.mapQos
            matchOption = tacMatchOption.matchIpAccessGroup
            if classifierType is not None:
               matchOption = getMatchOptionFromClassifierType( classifierType )
            cmapTypeConfig = qosAclCliConfig.cmapType.newMember( cmapType )
            if cmapTypeConfig:
               cmapConfig = cmapTypeConfig.cmap.newMember( className, cmapType )
               if cmapConfig:
                  classMapMatch = cmapConfig.match.newMember( matchOption )
                  if classMapMatch.strValue != className:
                     classMapMatch.strValue = className
                     cmapConfig.version += 1

         def populateClassifierMetaData( classifierName, termId, termActions ):
            t0( f'populateClassifierMetaData: classifierName = {classifierName} '
                f'termId = {termId}, termActions = {termActions}' )
            if not termActions:
               return

            targetGroup = ''
            if termActions.config:
               targetGroup = termActions.config.targetGroup
            classifierInfo = \
                  classifierMetaData.classifierInfo.newMember( classifierName )
            termActionInfo = Tac.Value(
                  'QosOc::TermActionInfo', termId, classifierName, targetGroup )
            classifierInfo.termActionInfo.addMember( termActionInfo )

         # Collections to maintain forward mapping of config
         classifierTermConfigToType = {}
         # Forward mapping of the config
         for classifierName, classifier in classifierConfig.classifier.items():
            typeConfig = classifier.config
            className = f'__yang_[{classifierName}]_[]'
            if not typeConfig.type:
               t0( f"No type configued for the classifier {classifier}" )
               # Creating a class-map with only the classifier name for one-to-one
               # mapping between classifier config and EOS config
               configureClassMap( className )
               classifierTermConfigToType[ className ] = None
               continue

            # Creating a class-map with only the classifier name for one-to-one
            # mapping between classifier config and EOS config.
            configureClassMap( className, typeConfig.type )
            classifierTermConfigToType[ className ] = typeConfig.type

            termConfig = classifier.terms
            if not termConfig:
               continue
            if termConfig.term:
               for termId, term in termConfig.term.items():
                  objName = f'__yang_[{classifierName}]_[{termId}]'
                  classifierTermConfigToType[ objName ] = typeConfig.type
                  termConditions = term.conditions
                  configureAcl( objName, typeConfig.type, termConditions )
                  configureClassMap( objName, typeConfig.type )
                  populateClassifierMetaData( classifierName, termId, term.actions )

         # Remove stale config for ACLs
         for aclName, aclType in list( aclNameToAclType.items() ):
            if aclName not in classifierTermConfigToType:
               del aclCliConfig.config[ aclType ].acl[ aclName ]
               del aclNameToAclType[ aclName ]

         # Remove stale config for class-maps
         for cmapName, matchOption in list( cmapNameToMatchOption.items() ):
            if cmapName not in classifierTermConfigToType:
               del qosAclCliConfig.cmapType[ tacClassMapType.mapQos ].cmap[
                  cmapName ]
               del cmapNameToMatchOption[ cmapName ]
            else:
               if len( qosAclCliConfig.cmapType[ tacClassMapType.mapQos ].cmap[
                     cmapName ].match ) > 1:
                  # This means that the type of an exisiting class-map has changed.
                  # We delete the older match option of the class-map.
                  del qosAclCliConfig.cmapType[ tacClassMapType.mapQos ].cmap[
                     cmapName ].match[ matchOption ]

   GnmiSetCliSession.registerPreCommitHandler( ToNativeClassifierConfigSyncher )
