# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from Radius import radsecproxyInternalPort
import Tac
import Arnet

RADSEC_MAX_LOG_LEVEL = 5
RADSEC_PORT = 2083
COA_TLS_BLOCK_NAME = "coa-tls-config-block"
COA_RESP_REWRITE_BLOCK = "coa-resp-rewrite-block"
# These values are copied from freeradius-client/freeradius-client.h
ARISTA_VENDOR_ID = 30065
ARISTA_ORIGIN_IDENTIFIER = 12
TlsConstants = Tac.Type( "Mgmt::Security::Ssl::Constants" )

# pylint: disable-msg=W1401,W0105
def generateTlsSection( certKeyPath, trustedCertsPath, cipherSuite,
                        cipherSuiteV1_3, tlsVersion, sslProfile,
                        opensslHashPath="", crlsPath="" ):
   dstConf = ""
   if not certKeyPath:
      return ( False, "Certificate or Key doesn't exist" )
   if not trustedCertsPath:
      return ( False, "Trusted Certificate doesn't exist" )
   dstConf += "tls %s {\n" % sslProfile

   # Adding the path to search for CA or intermediate certificates.
   # This path contains files named <opensslHashOfCert>.<int>
   # opensslHashOfCert -> Hash of certificates
   # <int> -> Numerical value for the hashes like 0,1,2.. ( hash collision case)
   # This is to add in Tls segment of radsecproxy.conf
   # So as to enable Radsec over TCP
   # If both the arg is provided i.e CACertificatePath and CACertificateFile
   # then CACertificateFile takes the lead. Hence we have added if else case.
   if opensslHashPath:
      dstConf += "\tCACertificatePath\t%s\n" % opensslHashPath
   else:
      dstConf += "\tCACertificateFile\t%s\n" % trustedCertsPath
   dstConf += "\tCertificateFile\t%s\n" % certKeyPath
   dstConf += "\tCertificateKeyFile\t%s\n" % certKeyPath
   dstConf += "\tCipherList\t%s\n" % cipherSuite
   dstConf += "\tCipherSuites\t%s\n" % cipherSuiteV1_3

   tlsVersionMap = [ ( TlsConstants.tlsv1_3,
                       "TLS1_3" ),
                     ( TlsConstants.tlsv1_2,
                       "TLS1_2" ),
                     ( TlsConstants.tlsv1_1,
                       "TLS1_1" ),
                     ( TlsConstants.tlsv1,
                       "TLS1" ) ]
   min_version = None
   max_version = None
   for mask, version in tlsVersionMap:
      # Check from 1_3 to 1_0
      if tlsVersion & mask:
         if not max_version:
            max_version = version
         min_version = version
   versionStr = "%s:%s" % ( min_version, max_version )
   dstConf += "\tTlsVersion\t%s\n" % versionStr

   if crlsPath:
      dstConf += "\tCRLCheck on\n"
   dstConf += "}\n"
   return ( True, dstConf )

"""This helper function can be utilised by the tests and the SuperServerPlugin to
generate the radsecproxy configuration file. Arguments info:
1. radiusStatus: Radsec::Status entity which lists the configured radius servers
in a particular VRF. It also contains the SSL profile information for a
particular hostspec.
2. listenPort: The listening port radsecproxy will listen on for UDP requests
3. logPath: Logging path for radsecproxy. Determined by the caller.
4. clients: A list of client IP addresses and corresponding addresses.
"""

def generateRadsecProxyConf( radsecStatus, logPath, clients,
                             listenPort=radsecproxyInternalPort,
                             coaEnabled=False ):
   if not isinstance( clients, list ):
      clients = [ clients ]

   dstConf = ""
   # 1. Add the listening section
   dstConf += "ListenUDP\t127.0.0.1:%d\n" % listenPort
   dstConf += "ListenUDP\t[::1]:%d\n" % listenPort

   if radsecStatus.dynAuthTlsEnabled and coaEnabled:
      # Listen on default Radsec port for CoA/Disconnect requests
      dstConf += "ListenTLS\t*:%d\n" % RADSEC_PORT

   # 2. Add the logging section
   dstConf += "LogLevel\t%d\n" % RADSEC_MAX_LOG_LEVEL
   dstConf += "LogDestination\tfile://%s\n" % logPath

   # Status has per host tls profile. So default profile is not needed.
   # This can be further optimized by having only one tls profile section
   # per ssl profile. As of now there is one tls section per host even if
   # multiple hosts uses the same profile.
   numTlsHosts = len( radsecStatus.host )
   if not numTlsHosts:
      return ( False, "TLS hosts don't exist" )

   profileList = []
   for spec in radsecStatus.host:
      h = radsecStatus.host[ spec ]
      # 3. Add the tls section if its not already present.
      if h.sslProfile not in profileList:
         tlsProfile = generateTlsSection( h.certKeyPath, h.trustedCertsPath,
                                          h.cipherSuite, h.cipherSuiteV1_3,
                                          h.tlsVersion, h.sslProfile,
                                          h.opensslHashPath, h.crlsPath )
         if not tlsProfile[ 0 ]:
            return tlsProfile
         else:
            tlsProfileStr = tlsProfile[ 1 ]
            dstConf += tlsProfileStr
            profileList.append( h.sslProfile )
      # 4. Add the rewrite section
      serverIpv6 = False
      dstConf += "rewrite %s {\n" % h.hostname
      try:
         hostIp = Arnet.IpGenAddr( h.hostname )
         if hostIp.af == "ipv4":
            t = h.hostname.split( "." )
            dstConf += "\tmodifyAttribute\t1:/^(.*)@%s\\.%s\\.%s\\.%s$/\\1/\n" % (
               t[ 0 ], t[ 1 ], t[ 2 ], t[ 3 ] )
         else:
            serverIpv6 = True
            dstConf += "\tmodifyAttribute\t1:/^(.*)@%s$/\\1/\n" % h.hostname
      except ValueError:
         dstConf += "\tmodifyAttribute\t1:/^(.*)@(.*)/\\1/\n"
      dstConf += "}\n"

      # 5. Add the server section
      if serverIpv6:
         # From the man page: literal IPv6 addresses must be enclosed in brackets.
         dstConf += "server [%s] {\n" % h.hostname
      else:
         dstConf += "server %s {\n" % h.hostname
      dstConf += "\ttype\ttls\n"
      dstConf += "\tport\t%d\n" % h.port
      dstConf += "\tsecret\tradsec\n"
      dstConf += "\tStatusServer\toff\n"
      dstConf += "\ttcpKeepalive\ton\n"
      dstConf += "\ttls\t%s\n" % h.sslProfile
      dstConf += "\trewriteOut\t%s\n" % h.hostname
      dstConf += "\tCertificateNameCheck\toff\n"
      if h.srcAddr:
         if h.srcAddr.af == "ipv4":
            dstConf += "\tSource\t%s\n" % str( h.srcAddr )
         else:
            dstConf += "\tSource\t[%s]\n" % str( h.srcAddr )
      # We listen for dynauth requests on the existing TLS connection to all radsec
      # servers by default
      if coaEnabled:
         dstConf += "\tDynAuthClient\ton\n"
      dstConf += "}\n"

      # 6. Add the realm section
      try:
         hostIp = Arnet.IpGenAddr( h.hostname )
         if hostIp.af == "ipv4":
            t = h.hostname.split( "." )
            dstConf += "realm /@%s\\.%s\\.%s\\.%s$ {\n" % ( t[ 0 ], t[ 1 ], t[ 2 ],
                                                            t[ 3 ] )
         else:
            dstConf += "realm /@%s$ {\n" % h.hostname
      except ValueError:
         parts = h.hostname.split( "." )
         regex = parts[ 0 ]
         if len( parts ) > 1:
            for p in parts[ 1 : ]:
               regex += r"\.%s" % p
         dstConf += "realm /@%s$ {\n" % regex
      if serverIpv6:
         dstConf += "\tserver\t[%s]\n" % h.hostname
         dstConf += "\tAccountingServer\t[%s]\n" % h.hostname
      else:
         dstConf += "\tserver\t%s\n" % h.hostname
         dstConf += "\tAccountingServer\t%s\n" % h.hostname
      dstConf += "}\n"

   # If dynAuth over TLS is enabled, add configured radsec servers as dynamic auth
   # clients and add a tls block to be used for incoming connections.
   if radsecStatus.dynAuthTlsEnabled and coaEnabled:
      # Only a single common dynauth ssl profile is supported for all hosts, so just
      # add a single tls config block and reuse it for all dynauth clients
      dynAuthTlsProfile = generateTlsSection( radsecStatus.dynAuthCertKeyPath,
                                              radsecStatus.dynAuthTrustedCertsPath,
                                              radsecStatus.dynAuthCipherSuite,
                                              radsecStatus.dynAuthCipherSuiteV1_3,
                                              radsecStatus.dynAuthTlsVersion,
                                              COA_TLS_BLOCK_NAME,
                                              radsecStatus.dynAuthOpensslHashPath )
      if not dynAuthTlsProfile[ 0 ]:
         return dynAuthTlsProfile

      dstConf += dynAuthTlsProfile[ 1 ]

      for spec in radsecStatus.host:
         h = radsecStatus.host[ spec ]
         try:
            hostIp = Arnet.IpGenAddr( h.hostname )
            if hostIp.af == "ipv4":
               dstConf += "client %s {\n" % h.hostname
            else:
               dstConf += "client [%s] {\n" % h.hostname
         except ValueError:
            # hostname is a domain name, not an ip address
            dstConf += "client %s {\n" % h.hostname
         dstConf += "\ttype\ttls\n"
         dstConf += "\tsecret\tradsec\n"
         dstConf += "\ttcpKeepalive\toff\n"
         dstConf += "\ttls\t%s\n" % COA_TLS_BLOCK_NAME
         dstConf += "\tCertificateNameCheck\toff\n"
         dstConf += "}\n"

   # 7. Add the clients
   if clients:
      for ( clientIp, clientSecret ) in clients:
         if Arnet.IpGenAddr( clientIp ).af == "ipv4":
            dstConf += "client %s {\n" % clientIp
         else:
            # From the man page: literal IPv6 addresses must be enclosed in brackets.
            dstConf += "client [%s] {\n" % clientIp
         dstConf += "\ttype\tudp\n"
         dstConf += "\tsecret\t%s\n" % clientSecret
         dstConf += "\tCertificateNameCheck\toff\n"
         dstConf += "}\n"

   # 11. Add configured clients as dynamic auth servers as well. Also, add realm
   # section for the dynauth servers with a single realm = arista.com
   # We do this even if dynauth over TLS is not enabled because its assumed each
   # radsec server can send a dynamic authorization request over an established
   # TLS connection by default.
   if clients and coaEnabled:
      # Add the rewrite section to remove/ignore Message-Authenticator attribute sent
      # by Dot1x in CoA responses, which will make radsecproxy skip this attribute in
      # CoA response it proxies ahead.
      # Message-Authenticator has RADIUS attribute code 80.
      dstConf += "rewrite %s {\n" % COA_RESP_REWRITE_BLOCK
      dstConf += "\tRemoveAttribute\t80\n"
      dstConf += "}\n"

      for ( dynAuthIp, dynAuthSecret ) in clients:
         if Arnet.IpGenAddr( dynAuthIp ).af == 'ipv4':
            dstConf += "server %s {\n" % dynAuthIp
         else:
            dstConf += "server [%s] {\n" % dynAuthIp
         # dynAuthPort is the port on which Dot1x is listening for UDP CoA requests
         dstConf += "\tport\t%d\n" % radsecStatus.dynAuthPort
         dstConf += "\ttype\tudp\n"
         dstConf += "\tsecret\t%s\n" % dynAuthSecret
         dstConf += "\tCertificateNameCheck\toff\n"
         dstConf += "\tOriginIpAttribute\t%d:%d\n" % ( ARISTA_VENDOR_ID,
                                                       ARISTA_ORIGIN_IDENTIFIER )
         dstConf += "\trewriteIn\t%s\n" % COA_RESP_REWRITE_BLOCK
         dstConf += "}\n"

      # Add realm section for dynAuth servers
      dstConf += "realm * {\n"
      for ( dynAuthIp, _ ) in clients:
         if Arnet.IpGenAddr( dynAuthIp ).af == 'ipv4':
            dstConf += "\tdynAuthServer\t%s\n" % dynAuthIp
         else:
            dstConf += "\tdynAuthServer\t[%s]\n" % dynAuthIp
      dstConf += "}\n"

   return dstConf
