# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from contextlib import contextmanager
import traceback

import Agent
import BothTrace
import Cell
from RcfCompiler import (
   RcfCompiler,
   RcfCompileRequest,
)
from RcfDiagCommon import RcfDiagGeneric
from RcfInstantiatingCollHelpers import (
   InstantiatingCollectionGV,
   InstantiatingCollectionCleaner
)
from CodeUnitMapping import (
   CodeUnitMapping,
   generateCodeUnitsForDomain,
)
from RcfAgentHeartbeat import RcfAgentHeartbeat
from RcfCommandTagHelpers import getExcludedCodeUnits
from RcfOpenConfigFunctionTextGen import generateOpenConfigFunctionTexts
from RcfOpenConfigFunctionValidator import (
   RcfOpenConfigFunctionValidationRequest,
   validateOpenConfigFunctions,
)
import RcfTypeFuture as Rcf
from Task import (
   Task,
   TaskRunResult,
)
import Logging
import Tac
import Tracing

from Toggles import (
   ConfigTagToggleLib,
   RcfLibToggleLib,
)

RcfAgentName = 'Rcf'

__defaultTraceHandle__ = Tracing.Handle( 'RcfAgent' )
bt0 = BothTrace.tracef0
t0 = Tracing.t0

MAX_U32 = 0xFFFFFFFF

RCF_UNEXPECTED_COMPILATION_ERROR = Logging.LogHandle(
              "RCF_UNEXPECTED_COMPILATION_ERROR",
              severity=Logging.logError,
              fmt="An unexpected processing error occurred while compiling "
                  "the RCF configuration.",
              explanation="An unexpected processing error occurred while compiling "
                          "the RCF configuration."
                          "When this occurs, no RCF functions will be applied "
                          "on the affected router.",
              recommendedAction="Revert to a previously working version of RCF "
                                "config, and report this error to Arista's "
                                "customer support team."
)

RCF_COMPILATION_ERROR = Logging.LogHandle(
              "RCF_COMPILATION_ERROR",
              severity=Logging.logError,
              fmt="The RCF code in the running-config failed to compile.",
              explanation="The RCF code in the running-config failed to compile. "
                          "When this occurs, RCF functions may not be applied "
                          "on the affected router. "
                          "For more details, run 'show router rcf errors'.",
              recommendedAction="Revert to a previously working version of RCF code"
)

redundancyStatus = None
redundancyStatusReactor = None

tagState = Tac.Type( "ConfigTag::ConfigTagState" )

@contextmanager
def enableTracing( trace ):
   originalTraceSetting = Tracing.traceSetting()
   Tracing.traceSettingIs( originalTraceSetting + ',' + trace )
   yield
   Tracing.traceSettingIs( originalTraceSetting )

class RcfCompilerWrapper:
   """ Wrapper for the data required by the RCF compiler. Provides an interface
   for reactors to schedule a compilation.

   Must be setup by RcfCompilerManager before compilation will run.
   """

   def __init__( self, rcfConfig, rcfAllPoaConfig, ipv6PfxList, rcfAllFunctionStatus,
                 status, debugSymbols, configTagIdState, configTagStatus,
                 strictMode=False ):
      bt0( "RcfCompilerWrapper: init" )
      self.rcfConfig = rcfConfig
      self.rcfAllPoaConfig = rcfAllPoaConfig
      self.ipv6PfxList = ipv6PfxList
      self.rcfAllFunctionStatus = rcfAllFunctionStatus
      self.rcfStatus = status
      self.debugSymbols = debugSymbols
      self.configTagIdState = configTagIdState
      self.configTagStatus = configTagStatus
      self.strictMode = strictMode
      self.instantiatingCollectionCleaner = None
      self.compilationTask = None
      self.completedFirstCompilation = False
      self.rcfCompiler = None

   def _changeAgent2AgentMountCounter( self ):
      # At this point, we know /<sysname>/Agent2Agent/RcfAllFunctionStatus has been
      # created in RcfAgent, so we bump up the counter in rcfStatus to kick ArBgp.
      if self.rcfStatus.agent2AgentMountChangedCounter == MAX_U32:
         bt0( "RcfAgent: agent2AgentMountChangedCounter wrapped" )
         self.rcfStatus.agent2AgentMountChangedCounter = 1
      else:
         self.rcfStatus.agent2AgentMountChangedCounter += 1
      bt0( "RCF agent2AgentMountChangedCounter %i" %
            self.rcfStatus.agent2AgentMountChangedCounter )

   def compileRcfImpl( self, codeUnitMapping, rcfCodeVersion, diag ):
      compileResult = None
      try:
         compileRequest = RcfCompileRequest( codeUnitMapping,
                                             rcfCodeVersion,
                                             strictMode=self.strictMode )
         compileResult = self.rcfCompiler.compile( compileRequest )
         diag.update( compileResult.diag )
         if compileResult.diag.allErrors:
            # CLI published non-compiling RCF text into RcfConfig.
            bt0( "RcfAgent: AET generation failed!" )
            Logging.log( RCF_COMPILATION_ERROR )
      except Exception as e: # swallow everything # pylint: disable=broad-except
         bt0( 'RcfAgent: Rcf compiler error!' )
         self.traceError( e, codeUnitMapping.codeUnitRcfTexts )
      return compileResult

   def compileRcf( self, shouldYield ):
      # Note we do not check if we should yield during compilation
      if self.rcfConfig.rcfCodeUnitsUpdatesInProgress():
         bt0( "RcfAgent: rcf text is updating" )
         # The version updating will scheduled us again.
         return TaskRunResult( shouldReschedule=False, workDone=0 )

      bt0( "RcfAgent: compiling" )
      codeUnitsForCompiler = {}
      compilerStageResults = []
      diag = RcfDiagGeneric()

      # Handle user-defined code units
      userDefinedCodeUnitTexts = dict( self.rcfConfig.rcfCodeUnitText )
      rcfCodeVersion = self.rcfConfig.rcfCodeVersion

      # Filter command-tag disabled code units
      if ConfigTagToggleLib.toggleRcfConfigTagSupportEnabled():
         self.filterCodeUnits( userDefinedCodeUnitTexts )
      codeUnitsForCompiler.update( generateCodeUnitsForDomain(
         Rcf.Metadata.FunctionDomain.USER_DEFINED, userDefinedCodeUnitTexts ) )

      # Handle POA wrapper functions
      if RcfLibToggleLib.toggleRcfFunctionArgsAtPOAEnabled():
         poaWrapperCodeUnitTexts = self.getPoaWrapperCodeUnitTexts()
         codeUnitsForCompiler.update( generateCodeUnitsForDomain(
            Rcf.Metadata.FunctionDomain.POA_WRAPPER, poaWrapperCodeUnitTexts ) )

      # Generate OpenConfig functions
      if RcfLibToggleLib.toggleRcfPolicyDefinitionsEnabled():
         success = self.handleOpenConfigFunctions( codeUnitsForCompiler,
                                                   rcfCodeVersion, diag )
         compilerStageResults.append( success )

      # Compile
      codeUnitMapping = CodeUnitMapping( codeUnitsForCompiler )
      compileResult = self.compileRcfImpl( codeUnitMapping, rcfCodeVersion, diag )
      compilerStageResults.append( compileResult and compileResult.success )

      allStagesSucceeded = all( compilerStageResults )
      allWarnings = [ warning.render( codeUnitMapping )
                      for warning in diag.allWarnings ]
      allErrors = [ error.render( codeUnitMapping ) for error in diag.allErrors ]

      self.rcfStatus.lastCompilationWarning = '\n'.join( allWarnings )
      self.rcfStatus.lastCompilationError = '\n'.join( allErrors )
      if allStagesSucceeded:
         compileResult.publish( self.rcfAllFunctionStatus,
                                self.rcfStatus,
                                self.debugSymbols,
                                self.instantiatingCollectionCleaner )

      # If compilation completed (regardless of if the compilation result was a
      # success or not), update the status's code version to indicate that
      # RcfStatus is now in sync with RcfConfig.
      self.rcfStatus.rcfCodeVersion = rcfCodeVersion

      if not self.completedFirstCompilation:
         bt0( 'First compilation completed' )
         # Once the first compilation has completed, with success or failure,
         # update client agents to remount the agent2agent mount and reevaluate.
         # Doing this before the first compilation completion could result in the
         # client agent flapping routes on Rcf agent restart as AETs disapear
         # briefly between remounting and compilation completing.
         self._changeAgent2AgentMountCounter()
         self.completedFirstCompilation = True

      # Compilation is complete, the task does not need to run again
      return TaskRunResult( shouldReschedule=False, workDone=1 )

   def filterCodeUnits( self, rcfCodeUnitTexts ):
      """Filter command-tag disabled code units. The input copy of rcfCodeUnitText is
      modified in-place.

      Args:
         rcfCodeUnitTexts (dict): current code units
      """
      codeUnitNames = set( rcfCodeUnitTexts )
      excludedCodeUnits = getExcludedCodeUnits(
         codeUnitNames, self.rcfConfig.codeUnitToConfigTag, self.configTagIdState,
         self.configTagStatus )
      for excludedCodeUnitName in excludedCodeUnits:
         del rcfCodeUnitTexts[ excludedCodeUnitName ]

   def getPoaWrapperCodeUnitTexts( self ):
      """Generate code unit texts for all POA configs that require a POA wrapper
      """
      poaWrapperCodeUnitTexts = {}
      for rcfPoaConfig in self.rcfAllPoaConfig.rcfPoaConfig.values():
         if rcfPoaConfig.usesWrapperFunction():
            poaWrapperCodeUnitTexts[ rcfPoaConfig.codeUnitNameForWrapper() ] = (
               rcfPoaConfig.functionTextForWrapper() )
      return poaWrapperCodeUnitTexts

   def handleOpenConfigFunctions( self, codeUnitsForCompiler, rcfCodeVersion, diag ):
      """Perform OpenConfig function validation and mix the results into
      codeUnitsForCompiler in-place if applicable.

      Args:
         codeUnitsForCompiler (dict): current code units
         rcfCodeVersion (int): code version of current config
      Returns:
         RcfOpenConfigFunctionValidationResult for error handling
      """
      ocValidationResult = None
      try:
         openConfigValidationRequest = RcfOpenConfigFunctionValidationRequest(
            dict( self.rcfConfig.openConfigFunction ), rcfCodeVersion )
         ocValidationResult = validateOpenConfigFunctions(
               openConfigValidationRequest )
         diag.update( ocValidationResult.diag )
         if not ocValidationResult.success:
            # Cli/ConfigSessionPlugin pushed a non-compiling openconfig function
            # to Rcf::Config
            bt0( "RcfAgent: OpenConfig function validation failed!" )
            Logging.log( RCF_COMPILATION_ERROR )
      except Exception as e: # swallow everything # pylint: disable=broad-except
         bt0( "RcfAgent: OpenConfig function validation error!" )
         self.traceError( e, codeUnitsForCompiler )

      if ocValidationResult and ocValidationResult.success:
         openConfigFunctionTexts = generateOpenConfigFunctionTexts(
               ocValidationResult.openConfigFunctions, self.ipv6PfxList )
         codeUnitsForCompiler.update( generateCodeUnitsForDomain(
               Rcf.Metadata.FunctionDomain.OPEN_CONFIG, openConfigFunctionTexts ) )
      return ocValidationResult and ocValidationResult.success

   def traceError( self, exception, codeUnitsForCompiler ):
      Logging.log( RCF_UNEXPECTED_COMPILATION_ERROR )
      # now try to get some information about the error.
      with enableTracing( 'RcfAgent/0' ):
         bt0( str( exception ) )
         for line in traceback.format_exc().split( '\n' ):
            t0( line )
         for unitKey, unitText in codeUnitsForCompiler.items():
            t0( f'RCF code unit {unitKey.codeUnitName}' )
            t0( unitText )
         for functionName, openConfigFunction in \
               self.rcfConfig.openConfigFunction.items():
            t0( f'RCF OpenConfig function {functionName}' )
            t0( openConfigFunction )

   def scheduleCompilation( self ):
      if self.compilationTask is not None:
         self.compilationTask.schedule()
      else:
         bt0( "RcfAgent: compiler not set up." )

class RcfCompilerManager:
   """ Provides an API for setting up and cleaning up the compiler data in
   RcfCompilerWrapper.
   """

   def __init__( self, rcfCompilerWrapper, rcfStatus, rcfAllFunctionStatus,
                 debugSymbols, rcfExternalConfig=None ):
      bt0( "RcfCompilerManager: init" )
      self.compiler = rcfCompilerWrapper
      self.rcfStatus = rcfStatus
      self.rcfAllFunctionStatus = rcfAllFunctionStatus
      self.debugSymbols = debugSymbols
      self.rcfExternalConfig = rcfExternalConfig

   def setupCompiler( self ):
      InstantiatingCollectionGV.setup( self.rcfAllFunctionStatus.aetNodes,
                                       self.debugSymbols.allDebugSymbolNodes )
      self.compiler.instantiatingCollectionCleaner = InstantiatingCollectionCleaner()
      self.compiler.rcfCompiler = \
         RcfCompiler( self.compiler.rcfAllFunctionStatus,
                      self.compiler.instantiatingCollectionCleaner,
                      rcfExternalConfig=self.rcfExternalConfig )
      self.compiler.compilationTask = Task(
            'RcfAgent-compilationTask', self.compiler.compileRcf )
      bt0( "RcfAgent: triggering compilation" )
      self.compiler.scheduleCompilation()
      self.rcfStatus.enable()

   def cleanupCompiler( self ):
      bt0( "RcfAgent: going down, cleanup" )
      InstantiatingCollectionGV.teardown()
      self.compiler.instantiatingCollectionCleaner.teardown()
      self.compiler.rcfCompiler = None
      self.compiler.compilationTask.suspend()
      self.compiler.compilationTask = None
      # Cleanup any published state and instantiating collections.
      self.rcfAllFunctionStatus.reset()
      self.debugSymbols.reset()
      self.rcfStatus.reset()

      self.rcfStatus.disable()

class RcfAgentConfigReactor( Tac.Notifiee ):
   """Reacts to changes in rcf/config/rcfCodeVersion by telling RcfCompilerWrapper
   to schedule a compilation.
   """
   notifierTypeName = "Rcf::Config"

   def __init__( self, notifier, rcfCompilerWrapper ):
      bt0( "RcfAgentConfigReactor: init" )
      Tac.Notifiee.__init__( self, notifier )
      self.compiler = rcfCompilerWrapper

   # CLI is directing us to process new RCF code unit config
   @Tac.handler( "rcfCodeVersion" )
   def handleRcfCodeVersion( self ):
      bt0( "RcfAgentConfigReactor: rcfCodeVersion", self.notifier_.rcfCodeVersion )
      self.compiler.scheduleCompilation()

class RcfAgentConfigEnabledReactor( Tac.Notifiee ):
   """Reacts to changes in rcf/config/enabled by telling RcfCompilerManager to
   either setup or cleanup the compiler.
   """

   notifierTypeName = "Rcf::Config"

   def __init__( self, notifier, rcfCompilerManager ):
      bt0( "RcfAgentConfigEnabledReactor: init" )
      Tac.Notifiee.__init__( self, notifier )
      self.compilerManager = rcfCompilerManager
      if self.notifier_.enabled:
         self.handleEnabled()

   # CLI is directing us to either bring up or tear down the RCF agent
   @Tac.handler( "enabled" )
   def handleEnabled( self ):
      bt0( "RcfAgentConfigEnabledReactor: rcfConfig.enabled changed" )
      if self.notifier_.enabled:
         self.compilerManager.setupCompiler()
      else:
         self.compilerManager.cleanupCompiler()

class RcfAgentAllPoaConfigReactor( Tac.Notifiee ):
   """Reacts to changes in routing/rcf/allPoaConfig by telling
   RcfCompilerWrapper to schedule a compilation.
   """
   notifierTypeName = "Rcf::Poa::AllRcfPoaConfig"

   def __init__( self, notifier, rcfCompilerWrapper ):
      bt0( "RcfAgentAllPoaConfigReactor: init" )
      Tac.Notifiee.__init__( self, notifier )
      self.compiler = rcfCompilerWrapper

   # CLI is directing us to process new RCF POA config
   @Tac.handler( "rcfPoaConfig" )
   def handleRcfPoaConfig( self, poaKey ):
      bt0( "RcfAgentAllPoaConfigReactor: poaKey: ", poaKey.toStrep() )
      self.compiler.scheduleCompilation()

class RedundancyStatusReactor( Tac.Notifiee ):
   """Reacts to redundancy/status.mode changing. This is needed to
   bump up the agent2AgentmountChangedCounter when the supervisor
   becomes the active supervisor.
   The bump up is done by blindly invoking the callback function.
   """
   notifierTypeName = "Redundancy::RedundancyStatus"

   def __init__( self, redStatusEntity, callback ):
      Tac.Notifiee.__init__( self, redStatusEntity )
      self.callback = callback

   @Tac.handler( "mode" )
   def handleRedundancyMode( self ):
      self.callback()

class ConfigTagEntryReactor( Tac.Notifiee ):
   """Reacts to changes in command-tag operState in configTag/configTagStatus by
   incrementing the compilationRequestCounter in rcf/compilationRequest. This will
   cause the RcfAgentCompilationRequestReactor to recompile the Rcf code.
   """
   notifierTypeName = "ConfigTag::ConfigTagEntry"

   def __init__( self, configTagEntry, rcfConfig, rcfCompilerWrapper ):
      Tac.Notifiee.__init__( self, configTagEntry )
      self.rcfConfig = rcfConfig
      self.configTagEntry = configTagEntry
      self.compiler = rcfCompilerWrapper
      t0( "ConfigTagEntryReactor: instatiated for tag", configTagEntry.tag )
      self.checkForRecompilation()

   def checkForRecompilation( self ):
      operState = self.configTagEntry.operState
      if self.configTagEntry.tagId in self.rcfConfig.configTagToCodeUnit:
         t0( "ConfigTagEntryReactor:", self.configTagEntry.tag, operState )
         self.compiler.scheduleCompilation()

   @Tac.handler( "operState" )
   def handleConfigTagState( self ):
      self.checkForRecompilation()

   @Tac.handler( "tagId" )
   def handleConfigTagId( self ):
      # tagId will be set after the ConfigTagEntry is created. This will catch
      # the case where the reactor is created before tagId is set.
      self.checkForRecompilation()

   def close( self ):
      t0( "ConfigTagEntryReactor deleted:", self.configTagEntry.tag )
      # Check if removing this command-tag will cause any code units to be enabled
      self.checkForRecompilation()
      Tac.Notifiee.close( self )


class RcfAgent( Agent.Agent ):
   """ Rcf Agent reacts to ConfigAgent writing valid Rcf text and compiles it
   down to its AET form, that ArBgp can run.

   RcfAgent writes to the AllFunctionStatus's AET collection, and ArBgp directly
   reads updates from it through an Agent2Agent mount.

   This agent is managed by Launcher, and will only run when RcfConfig's
   RcfCodeUnitText (that ConfigAgent writes) is not empty.
   """
   agentDirName = RcfAgentName
   def __init__( self, entityManager ):
      bt0( "Initialize RcfAgent" )
      self.sysname = entityManager.sysname()
      bt0( f"Starting agent {RcfAgentName} in system {self.sysname}" )
      self.entityManager = entityManager

      # config/input
      self.rcfConfig = None
      self.rcfAllPoaConfig = None

      # external references
      self.aclListConfig = None

      # config tag
      self.configTagIdState = None
      self.configTagStatus = None

      # status/output
      self.rcfAllFunctionStatus = None
      self.rcfStatus = None
      self.debugSymbols = None

      # compiler management
      self.rcfCompilerManager = None
      self.rcfCompilerWrapper = None

      Agent.Agent.__init__( self, entityManager, agentName=RcfAgentName )

   def doInit( self, entityManager ):
      bt0( "RcfAgent: doInit" )
      global redundancyStatus

      mg = entityManager.mountGroup()

      # config/input
      redundancyStatus = mg.mount( Cell.path( 'redundancy/status' ),
                                 'Redundancy::RedundancyStatus',
                                 mode='r' )
      self.rcfConfig = mg.mount( 'routing/rcf/config',
                                 'Rcf::Config',
                                 mode='rS' )

      if RcfLibToggleLib.toggleRcfFunctionArgsAtPOAEnabled():
         self.rcfAllPoaConfig = mg.mount( 'routing/rcf/allPoaConfig',
                                          'Rcf::Poa::AllRcfPoaConfig',
                                          mode='r' )

      # external references
      self.aclListConfig = mg.mount( 'routing/acl/config',
                                     'Acl::AclListConfig',
                                     mode='rS' )

      # config tag
      if ConfigTagToggleLib.toggleRcfConfigTagSupportEnabled():
         self.configTagIdState = mg.mount( 'configTag/configTagIdState',
                                           'ConfigTag::ConfigTagIdState',
                                           mode='r' )
         self.configTagStatus = mg.mount( 'configTag/status',
                                          'ConfigTag::ConfigTagStatus',
                                          mode='r' )

      # status/output
      agent2AgentRcfDir = Tac.root[ self.sysname ].mkdir( 'Agent2AgentRcf' )
      self.rcfAllFunctionStatus = agent2AgentRcfDir.newEntity(
                                    "Rcf::AllFunctionStatus",
                                    "RcfAllFunctionStatus"
                                 )
      self.rcfStatus = mg.mount( Cell.path( 'routing/rcf/status' ),
                                 'Rcf::Status',
                                 mode='w' )
      self.debugSymbols = mg.mount( Cell.path( 'routing/rcf/debugSymbols' ),
                                    'Rcf::Debug::Symbols',
                                    mode='wS' )

      mg.close( self.doMountComplete )

   def isActiveSupervisor( self ):
      global redundancyStatusReactor
      # redundancyStatus got mounted. If a reactor for this is not created yet,
      # create one now.
      if not redundancyStatusReactor:
         redundancyStatusReactor = RedundancyStatusReactor( redundancyStatus,
                                                            self.doMountComplete )
      # In a dual-sup situation, only proceed if we are the active supervisor
      if redundancyStatus.mode != "active":
         bt0( "RcfAgent: isActiveSupervisor: no" )
         return False
      bt0( "RcfAgent: isActiveSupervisor: yes" )
      return True

   def doMountComplete( self ):
      bt0( "RcfAgent: doMountComplete" )
      if not self.isActiveSupervisor():
         return

      RcfAgentHeartbeat.enableManualPunching()
      self.rcfStatus.reset()
      self.debugSymbols.reset()

      bt0( "RcfAgent: initialize rcfCompilerWrapper" )
      ipv6PfxList = self.aclListConfig.ipv6PrefixList
      self.rcfCompilerWrapper = RcfCompilerWrapper( self.rcfConfig,
                                                    self.rcfAllPoaConfig,
                                                    ipv6PfxList,
                                                    self.rcfAllFunctionStatus,
                                                    self.rcfStatus,
                                                    self.debugSymbols,
                                                    self.configTagIdState,
                                                    self.configTagStatus )
      self.rcfCompilerManager = RcfCompilerManager( self.rcfCompilerWrapper,
                                                    self.rcfStatus,
                                                    self.rcfAllFunctionStatus,
                                                    self.debugSymbols )

      bt0( "RcfAgent: initialize rcfConfigReactor" )
      self.entityManager.rcfConfigReactor = RcfAgentConfigReactor(
            self.rcfConfig, self.rcfCompilerWrapper )

      bt0( "RcfAgent: initialize rcfConfigEnabledReactor" )
      self.entityManager.rcfConfigEnabledReactor = RcfAgentConfigEnabledReactor(
            self.rcfConfig, self.rcfCompilerManager )

      if RcfLibToggleLib.toggleRcfFunctionArgsAtPOAEnabled():
         bt0( "RcfAgent: initialize rcfAllPoaConfigReactor" )
         self.entityManager.rcfAllPoaConfigReactor = RcfAgentAllPoaConfigReactor(
               self.rcfAllPoaConfig, self.rcfCompilerWrapper )

      if ConfigTagToggleLib.toggleRcfConfigTagSupportEnabled():
         bt0( "RcfAgent: initialize configTagEntryCollReactor" )
         self.entityManager.configTagEntryCollReactor = Tac.collectionChangeReactor(
            self.configTagStatus.configTagEntry, ConfigTagEntryReactor,
            reactorArgs=( self.rcfConfig, self.rcfCompilerWrapper ) )

def main():
   container = Agent.AgentContainer( [ RcfAgent ] )
   container.runAgents()

