#!/usr/bin/env python3
# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac

class RcfAgentHeartbeat:
   '''
   Today RCF compilation is done as one piece of work, from lexing to publish.
   For large configurations compilation can exceed the heartbeat period for the
   RCF agent, resulting in agent restart.

   To avoid restarting during compilation, the RCF agent will manually punch its
   heartbeat at regular intervals during compilation.
   This is typically done on per function operations in each compilation phase.
   Manually punching the heartbeat is requred for RCF as parsing is a significant
   part of compilation time and this is done with a third party library. It's not
   pratical to break down the users config to be per function for parsing and so
   customers with all of their RCF code in a single code unit could exceed
   the heartbeat period from parsing alone.
   In Rcf.g4 a hook is suplied to punch the heartbeat manually when a function
   node is parsed, meaning we punch per function during parsing.
   Please see aid/9874 for more details.
   '''

   # watchdogReview is the object responsible for maintaining the agent's heartbeat.
   # This is only set in this class by the Rcf agent at startup, ensuring we only
   # punch the heartbeat manually in the Rcf agent.
   watchdogReview = None

   @classmethod
   def punch( cls ):
      if cls.watchdogReview:
         cls.watchdogReview.doPunchExternalWatchdog()

   @classmethod
   def enableManualPunching( cls ):
      watchdogReactor = Tac.singleton( 'ProcMgr::WatchdogReactorAgent' )
      cls.watchdogReview = watchdogReactor.watchdogReview
