#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

""" AST definition.

This module holds the definition of the RCF AST, and visitor to walk it.

!Rules (don't change unless discussing with authors first)

   - Follow the open/closed principle: The AST data model should be closed
     for modification, and open for extension.

   - Don't change AST attributes, don't add AST nodes, don't implement
     static, class, or member function: use a dedicated visitor instead.

   @author: matthieu (rcf-dev)
"""

class Label:
   def __init__( self, label ):
      self.token = label
      self.name = label.text

class Node:
   """ Abstract Syntax Tree node base type.

   This node holds all the information common to every node in the AST.

   Attributes:
      context (Antlr Context): the antlr context from parse-tree.
      evalType (RcfType): type evaluated during AST construction.
      scope (Scope): scope where the node sits in the symbol table.
      promoteToType (RcfType): Rcf type destination, if the semantic analysis
                               deemed to promote.
      label (Label): Rcf label (applicable to blocks and conditions)

   """
   def __init__( self, context ):
      self.context = context
      self.evalType = None
      self.scope = None
      self.promoteToType = None

   def resolveType( self ):
      if self.promoteToType is None:
         return self.evalType
      else:
         return self.promoteToType

class Root( Node ):
   """ Rcf ROOT AST node.

   The RCF Root holds a list of functions (class Function)

   Attributes:
      functions (list[ Function ]): list of function definitons.
      builtInFunctions (list[ Function ]): list of built in function definitons.
   """
   def __init__( self, builtinFunctions=None ):
      super().__init__( None )
      self._functions = []
      self._builtInFunctions = builtinFunctions or []
      self._bottomUpOrder = tuple()

   def addFunctions( self, functions ):
      self._functions += functions

   def validFunctions( self ):
      yield from ( func for func in self.validAndInvalidFunctions() if func.valid )

   def validAndInvalidFunctions( self, *, builtInFirst=True ):
      if self._bottomUpOrder:
         assert builtInFirst, ( "Cannot explicitly set builtInFirst to false"
                                " after _bottomUpOrder is defined" )
         yield from self._bottomUpOrder
      elif builtInFirst:
         yield from self._builtInFunctions
         yield from self._functions
      else:
         yield from self._functions
         yield from self._builtInFunctions

   def setBottomUpOrder( self, bottomUpOrder ):
      self._bottomUpOrder = bottomUpOrder

class Function( Node ):
   """ Rcf Function node

   A function has a name, and an execution block.

   Attributes:
      name (str): function name.
      codeUnitKey (CodeUnitKey): name and domain of the code unit that this
         function is defined in
      block (Block): function block definiton.
      symbol (RcfSymbol): function symbol after definition.
      funcParams (list[ FunctionParam ] ): list of function parameters
   """
   # pylint: disable-next=dangerous-default-value
   def __init__( self, context, name, codeUnitKey, block, funcParams=[] ):
      super().__init__( context )
      self.name = name
      self.codeUnitKey = codeUnitKey
      self.block = block
      self.funcParams = funcParams
      self.symbol = None
      self.functionCalls = []
      self.externalRefs = []
      self.valid = True

class FunctionParam( Node ):
   """ Rcf Function Parameter node

   A function parameter has a name and a type.

   For user defined functions with parameters, @typeStr will be the user provided
   string (e.g. "int_type"). For builtin functions with parameters, @typeStr will be
   the Rcf typename from the metadata (e.g. "Int").

   Attributes:
      name (str): parameter name or None (builtin function parameters)
      typeStr (str): parameter type as provided by the user (display name) or
         RcfMetadata.yaml (builtin function parameters)
      symbol (RcfSymbol): parameter symbol after definition
   """
   def __init__( self, context, name, typeStr ):
      super().__init__( context )
      self.name = name
      self.typeStr = typeStr
      self.symbol = None

class Block( Node ):
   """ Rcf block

   A block represents a list of instruction. We can find block
   in functions, and if statements.

   Attributes:
      stmts (list[ Nodes ]): list of statements.
      label: Label
   """
   def __init__( self, context, stmts, label ):
      super().__init__( context )
      self.stmts = stmts
      self.label = label

class StmtExpr( Node ):
   """ Rcf Statement

   An Rcf statement with an expression.

   Attributes:
      expr: The expression for the statement
      openStmtPoint: Delimiter indicating where the statement begins
      closeExprPoint: Delimiter indicating where the expression ends
   """
   def __init__( self, context, expr, openStmtPoint, closeExprPoint ):
      super().__init__( context )
      self.expr = expr
      self.openStmtPoint = openStmtPoint
      self.closeExprPoint = closeExprPoint

class IfStmt( StmtExpr ):
   """ Rcf If statement

   Represents a branch in the AST.

   Attributes:
      condition (Node): the condition AST expression.
      thenBlock (Block): branch when the condition is true.
      elseBlock (Block, optional): branch when the condition is false.
      ifLbl (Label, optional): @this_label if (... )
      elseLbl (Label, optional): if (... ) else @this_label { }
   """
   def __init__( self, context, condition, thenBlock, elseBlock,
                 openStmtPoint, closeExprPoint ):
      super().__init__( context, condition, openStmtPoint,
                        closeExprPoint )
      self.condition = condition
      self.thenBlock = thenBlock
      self.elseBlock = elseBlock
      self.ifLbl = Label( context.ifLbl ) if context.ifLbl else None
      self.elseLbl = Label( context.elseLbl ) if context.elseLbl else None

class Expr( Node ):
   """ Rcf expression

   An Rcf expression can be enclosing in parenthesis, e.g. '(' expr  ')'.
   This base type allows the details of those parenthesis to be stored.

   Attributes:
      openingParens: Parse Context, '('
      closingParens: Parse Context, ')'
   """
   def __init__( self, context ):
      super().__init__( context )
      self.explicitParens = None
      self.openingParens = []
      self.closingParens = []

class Call( Expr ):
   """ Rcf call statement

   A call references another function by its name, and it's symbol in the AST.
   The function symbol itself has a reference to the AST node of this function
   defintion in the AST. (functionSymbol is an RCF Symbol).

   The symbol is resolved in later phases, after AST creation.

   Attributes:
      funcName (str): function name.
      functionSelf (Attribute): The attribute this function is called on as a method
      funcArgs  ([Node]): list of value nodes corresponding to function arguments.
      funcSymbol (Symbol): function symbol in the symbol table.
   """
   def __init__( self, context, funcName, funcArgs ):
      super().__init__( context )
      self.funcName = funcName
      self.functionSelf = None
      self.funcArgs = funcArgs
      self.symbol = None

class ExternalRefOp( Expr ):
   """ Compare external represents a comparison with an external entity through
   a 'match', 'match_covered' or 'match-exact' condition.

   Attributes:
      attribute (Attribute): the Attribute (AST node) on which the match operates.
      isExact (bool): whether it is an exact match or not.
      isMatchCovered (bool): whether we are using 'match_covered' instead of match.
         Only one of isExact or isMatchCovered is allowed to be true.
      rhs (ExternalRef|Variable): the external reference AST node (may be a variable)
      op (string): string representation of the operation
   """
   def __init__( self, context, attribute, isExact, isMatchCovered, rhs ):
      super().__init__( context )
      self.attribute = attribute
      self.isExact = isExact
      self.isMatchCovered = isMatchCovered
      self.rhs = rhs
      assert not isExact or not isMatchCovered
      if isExact:
         self.op = "match_exact"
      elif isMatchCovered:
         self.op = "match_covered"
      else:
         self.op = "match"
      self.additionalAttributes = []

class ExternalRef( Node ):
   """ External refrence

   Represents an external reference to another Cli construct.

   e.g prefix_list_v4 NETFLIX

   Attriubtes:
      name (str): the name of the external construct (e.g NETFLIX).
      etype (str): the type of the external construct (e.g: prefix_list_v4).
      isIpV4 (bool/None): whether the current external ref refers to an IPv4
                          or None if it doesn't apply.
                             e.g:
                               prefix_list_v4 -> true
                               prefix_list_v6 -> false
                               community_list -> None
   """
   def __init__( self, context, name, etype ):
      super().__init__( context )
      self.name = name
      self.etype = etype
      self.ipVersioned = (
         any( ipv in etype for ipv in ( 'v4', 'v6' ) )
      )
      self.isIpV4 = 'v4' in etype if self.ipVersioned else None

class Assign( Node ):
   """ Rcf assignment node.

   e.g: med += 4;

   Attributes:
      attribute (Attribute): AST node Attribute.
      value (Constant): AST node Constant.
      op (string): operation (e.g '+=' or '-=', or '=')
   """
   def __init__( self, context, attribute, value, op ):
      super().__init__( context )
      self.attribute = attribute
      self.value = value
      self.op = op

class Return( StmtExpr ):
   """ Rcf return node.

   e.g: return POLICYA() and med >= 10;

   Attributes:
      expr (Node): AST subtree expression. (in the base class)
   """

class Exit( StmtExpr ):
   """ Rcf exit node.

   e.g: exit POLICYA() and med >= 10;

   Attributes:
      expr (Node): AST subtree expression.
   """

class Directive( Node ):
   """ Rcf directive node.

   e.g:  !FAIL_COMPILE;
   """

   def __init__( self, context, value ):
      super().__init__( context )
      self.value = value

class CommunityValue( Expr ):
   """ Rcf CommunityValue node
   Represents a CommunityValue (e.g: link bandwidth, route target etc. )
   in Rcf. Each section in itself is a node.
   Attributes:
      sections : sections of the community
   BUG888714: This class is meant to eventually hold all types of communities
   (standard/extended/large). For now, it just supports extended communities
   """
   class Type:
      """
      Introducing this Type class for extendebility.
      We could also hard code this in the type attribute
      """
      immediateExtCommunityValue = 'immediateExtCommunityValue'

   def __init__( self, context, sections, ctype ):
      super().__init__( context )
      self.sections = sections
      self.type = ctype

      # The symbol that is on the lhs for the operation
      self.parentLhsSymbol = None
      # The operator this immediate is on the RHS of
      self.parentOperator = None

   def __lt__( self, other ):
      return self.getSectionValues() < other.getSectionValues()

   @property
   def extCommType( self ):
      assert self.type == self.Type.immediateExtCommunityValue
      return self.sections[ 0 ].value

   @staticmethod
   def getSectionValue( section ):
      """returns the value if the section is of Constant type.
      Else, returns the name if it is of Attribute type"""
      if isinstance( section, Constant ):
         return section.value
      elif isinstance( section, Attribute ):
         return section.name
      else:
         assert False, "unsupported AST type for Community sections"

   def getSectionValues( self ):
      """returns the value if the section is of Constant type.
      Else, returns the name if it is of Attribute type, for each section
      """
      return tuple( self.getSectionValue( section ) for section in self.sections )

class Constant( Expr ):
   """ Rcf constant (a.k.a 'Immediate' value )

   e.g: 42, or 'unknown'

   Attributes:
      value (?): holds the value of the constant.
      ctype (Constant.Type): the constant type.
   """

   class Type:
      integer = 'integer'
      bandwidth = 'bandwidth'
      boolean = 'boolean'
      trilean = 'trilean'
      prefix = 'prefix'
      asDot = 'asDot'
      asNumber = 'asNumber'
      esi = 'esi'
      interface = 'interface'
      ipAddress = 'ipAddress'
      macAddress = 'macAddress'
      immediateCommunityValue = 'immediateCommunityValue'
      immediateLargeCommunityValue = 'immediateLargeCommunityValue'
      isisLevelValue = 'isisLevelValue'
      string = 'string'
      none = 'none'
      # More will be automatically added here (eg. enums like Origin) when the
      # metadata is processed during initialization.

   def __init__( self, context, value, ctype ):
      super().__init__( context )
      self.value = value
      self.type = ctype

class Range( Node ):
   """ Rcf range

   e.g: x to y, inclusive of x and y

   Attributes:
      lowerBound (Constant): the lower value in the range
      upperBound (Constant): the upper value in the range
   """

   def __init__( self, context, lowerBound, upperBound ):
      super().__init__( context )
      self.lowerBound = lowerBound
      self.upperBound = upperBound

class Collection( Node ):
   """ Rcf collection

   e.g: set or list, { ... }

   Attributes:
      values (List of Constants, Ranges): holds the contents of the collection.
      ctype (Collection.Type): the collection type.
   """

   class Type:
      empty = 'empty'
      ImmediateList = 'ImmediateList'
      ImmediateSet = 'ImmediateSet'
      AsPathImmediate = 'AsPathImmediate'
      # More will be automatically added here (eg. ImmediateIntSet) when the
      # metadata is processed during initialization.

   def __init__( self, context, values, ctype ):
      super().__init__( context )
      self.values = values
      self.type = ctype

      # The symbol that is on the lhs for the operation
      self.parentLhsSymbol = None
      # The operator this immediate is on the RHS of
      self.parentOperator = None
      # Contents of the collection that are ranges. These are handled differently
      # at the AET phase and so we split them out.
      self.ranges = []

class Attribute( Node ):
   """ Rcf Attribute node

   Represents an Attribute (e.g: med, prefix, or community.len) in Rcf.

   Attributes:
      name (str): the attribute name (e.g 'prefix').
      symbol (RcfSymbol): Associated RcfSymbol (set when resolved).
   """
   def __init__( self, context, name ):
      super().__init__( context )
      self.name = name
      self.symbol = None

class Variable( Node ):
   """ Rcf Variable node

   Attributes:
      name (str): the variable name (e.g. '$var1').
      symbol (RcfSymbol): Associated RcfSymbol (set when resolved).
   """
   def __init__( self, context, name ):
      super().__init__( context )
      self.name = name
      self.symbol = None

class BinOp( Expr ):
   """ Binary Operation in Rcf

   e.g: prefix is 10.10.10.0/24
        ^~~~   ^~ ^~~~~~~~~~~~~
        lhs    op rhs

   Attributes:
      operator (string): e.g 'is', '<'.
      rhs (Node): right hand side node.
      lhs (Node): left hand side node.
   """
   def __init__( self, context, operator, lhs, rhs ):
      super().__init__( context )
      self.operator = operator
      self.rhs = rhs
      self.lhs = lhs

class LogicalOp( Expr ):
   """ Logical Operation in Rcf
   A sequence of the same logical opeator

   e.g: true and false
        ^~~~ ^~~ ^~~~~
        expr op  expr

   e.g: false or false or true
        ^~~~~ ^~ ^~~~~ ^~ ^~~~
        expr  op expr  op expr

   Attributes:
      operator (string): e.g 'and', 'or', or '??'
      expressionList (list of expression contexts):
         Two or more expression in the logical operation. MUST be one more expression
         that operator.
      operatorList (list of operator contexts):
         One or more operator of the same type in the logical expression.
   """
   def __init__( self, context, operator, expressionList, operatorList ):
      super().__init__( context )
      self.operator = operator
      self.expressionList = expressionList
      self.operatorList = operatorList
      assert len( self.expressionList ) == len( self.operatorList ) + 1
      operatorType = type( self.operatorList[ 0 ] )
      assert all( isinstance( x, operatorType ) for x in self.operatorList )

class Not( Expr ):
   """ Rcf boolean inverter

   Simply holds an expression (Ast Node), and inverses it's
   boolean value:
   e.g:  not ( false and true )
             ^~~~~~~~~~~~~~~~~~ (expr)

   Attributes:
      expr (Node): Rcf expression AST tree.
   """
   def __init__( self, context, expr ):
      super().__init__( context )
      self.expr = expr
