# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from collections import namedtuple

import RcfAst
import RcfAstListener
import RcfAstWalker
import RcfTypeFuture as Rcf

AccessibleAttributes = namedtuple( 'AccessibleAttributes',
                                   [ 'readAttrs', 'writeAttrs' ] )

# See BUG824666
readWriteOp = [
   "+=",
   "-=",
   "*=",
   "add",
   "remove",
   "retain",
   "prepend",
]

class AccessibleAttributesGenerator( RcfAstListener.Listener ):
   """AccessableAttributesGenerator

   Create an AccessableAttributes Type for each function containing the access type
   for each attributes in the function

   Attributes:
       readAttrs (Dict): Attribute read in the current function
       writeAttrs (Dict): Attribute modified in the current function
   """
   def __init__( self ):
      super().__init__()
      self.readAttrs = None
      self.writeAttrs = None

   def _processAttributeName( self, attribute ):
      if attribute.symbol.rootAttribute is not None:
         attribute = attribute.symbol.rootAttribute
      return attribute.name.removeprefix( "input." ).replace( ".", "_" )

   def _setReadFlag( self, attribute ):
      if isinstance( attribute, RcfAst.Attribute ):
         name = self._processAttributeName( attribute )
         assert name is not None
         self.readAttrs[ name ] = True
      elif isinstance( attribute, RcfAst.Collection ):
         self._processCollection( attribute )

   def _setWriteFlag( self, attribute ):
      assert isinstance( attribute, RcfAst.Attribute )
      name = self._processAttributeName( attribute )
      assert name is not None
      self.writeAttrs[ name ] = True

   def _processCollection( self, collection ):
      for value in collection.values:
         self._setReadFlag( value )

   def listenFunction( self, function, action, **kwargs ):
      if action is RcfAstWalker.Action.ENTRY:
         self.readAttrs = {}
         self.writeAttrs = {}
      if action is RcfAstWalker.Action.EXIT:
         readAttrs = Rcf.Eval.AttributeSet( **self.readAttrs )
         writeAttrs = Rcf.Eval.ModifiableAttributeSet( **self.writeAttrs )
         accessAttrs = Rcf.Eval.AccessibleAttributes( readAttrs, writeAttrs )
         function.accessibleAttributes = accessAttrs

   def listenExternalRefOp( self, externalRefOp, action, **kwargs ):
      self._setReadFlag( externalRefOp.attribute )

   def listenAssign( self, assign, action, **kwargs ):
      self._setWriteFlag( assign.attribute )
      if assign.op in readWriteOp:
         self._setReadFlag( assign.attribute )
      self._setReadFlag( assign.value )

   def listenBinOp( self, binOp, action, **kwargs ):
      self._setReadFlag( binOp.lhs )
      self._setReadFlag( binOp.rhs )

   def listenCall( self, call, action, **kwargs ):
      if action is RcfAstWalker.Action.EXIT:
         for arg in call.funcArgs:
            self._setReadFlag( arg )

   def listenCommunityValue( self, commValue, action, **kwargs ):
      for section in commValue.sections:
         self._setReadFlag( section )
